// ------------ dfwindow.cpp

#include "dflatpp.h"
#include "frame.h"
#include "desktop.h"

// -------- common constructor initialization code
void DFWindow::InitWindow(int lf, int tp,
                int ht, int wd, DFWindow *par)
{
    windowtype = DFlatWindow;
    if (lf == -1)
        lf = (desktop.screen().Width()-wd)/2;
    if (tp == -1)
        tp = (desktop.screen().Height()-ht)/2;
    if (ht == -1)
        ht = desktop.screen().Height();
    if (wd == -1)
        wd = desktop.screen().Width();
    attrib = restored_attrib = 0;
    title = NULL;
    ctlmenu = NULL;
    videosave = NULL;
    visible = False;
    clipoverride = False;
    first = NULL;
    last = NULL;
    next = NULL;
    prev = NULL;
    prevcapture = NULL;
    parent = par;
    BorderAdj = TopBorderAdj = BottomBorderAdj = 0;
    Rect rcc(lf, tp, lf+wd-1, tp+ht-1);
    restored_rc = rect = rcc;
    SetColors();
    clearch = ' ';
    DblBorder = True;
    Enqueue();
    if (parent == NULL)
        SetAttribute(SAVESELF);
    windowstate = ISRESTORED;
}

void DFWindow::InitWindow(char *ttl, int lf, int tp,
            int ht, int wd, DFWindow *par)
{
    InitWindow(lf, tp, ht, wd, par);
    attrib |= TITLEBAR;
    title = new String(ttl);
}

void DFWindow::OpenWindow()
{
    if (windowstate == CLOSED)
        InitWindow(*title, Left(), Top(),
            Height(), Width(), parent);
}

void DFWindow::CloseWindow()
{
    windowstate = ISCLOSING;
    Hide();
    // ------- close window's children
    DFWindow *Wnd = first;
    while (Wnd != NULL)    {
        Wnd->CloseWindow();
        Wnd = Wnd->next;
    }
    // ------ delete this window's memory
    if (title != NULL)
        delete title;
    if (videosave != NULL)
        delete videosave;
    DeleteCtlMenu();
    if (this == desktop.InFocus())    {
        if (desktop.FocusCapture() == this)
            ReleaseFocus();
        else if (parent == NULL ||
                parent->windowstate == ISCLOSING)
            desktop.SetFocus(parent ? parent : NULL);
        else    {
            NextSiblingFocus();
            if (this == desktop.InFocus())
                if (!parent->SetFocus())
                    desktop.SetFocus(NULL);
        }
    }
    Dequeue();
    windowstate = CLOSED;
}

// -------- set the fg/bg colors for the window 
void DFWindow::SetColors()
{
    colors.fg = 
    colors.sfg = 
    colors.ffg = 
    colors.hfg = WHITE;
    colors.bg = 
    colors.sbg = 
    colors.fbg = 
    colors.hbg = BLACK;
}

// ---------- display the window
void DFWindow::Show()
{
    if (attrib & SAVESELF)    {
        Rect rc = ShadowedRect();
        if (videosave == NULL)    {
            int sz = rc.Height() * rc.Width() * 2;
            videosave = new char[sz];
        }
        if (!visible)
            desktop.screen().GetBuffer(rc, videosave);
    }
    visible = True;
    clipoverride = True;
    Paint();
    Border();
    Shadow();
    clipoverride = False;
    if (windowstate != ISMINIMIZED)    {
        // --- show the children of this window
        DFWindow *Wnd = first;
        while (Wnd != NULL)    {
            if (Wnd->windowstate != ISCLOSING)
                Wnd->Show();
            Wnd = Wnd->next;
        }
    }
}

Rect DFWindow::ShadowedRect()
{
    Rect rc = rect;
    if (attrib & SHADOW)    {
        rc.Right()++;
        rc.Bottom()++;
    }
    return rc;
}

void DFWindow::Hide()
{
    if (visible)    {
        Rect rc = rect;
        Bool HasShadow = (Bool)((attrib & SHADOW) != 0);
        if (HasShadow)    {
            rc.Bottom()++;
            rc.Right()++;
        }
        visible = False;
        // ----- hide the children
        DFWindow *Wnd = first;
        while (Wnd != NULL)    {
            Wnd->Hide();
            Wnd = Wnd->next;
        }
        if (videosave != NULL)    {
            desktop.screen().PutBuffer(rc, videosave);
            delete videosave;
            videosave = NULL;
        }
        else if (parent != NULL)    {
            if (parent->isVisible())    {
                parent->Paint(ShadowedRect());
                PaintOverLappers();
            }
        }
    }
}

void DFWindow::Keyboard(int key)
{
    switch (key)    {
        case CTRL_F4:
            CloseWindow();
            break;
        case ALT_HYPHEN:
            OpenCtlMenu();
            break;
        default:
            // --- send all unprocessed keystrokes 
            //     to the parent window
            if (parent != NULL)
                parent->Keyboard(key);
            break;
    }
}

void DFWindow::ShiftChanged(int sk)
{
    if (parent != NULL)
        parent->ShiftChanged(sk);
}

void DFWindow::DoubleClick(int mx, int my)
{
    if (HitControlBox(mx, my))
        CloseWindow();
}

void DFWindow::LeftButton(int mx, int my)
{
    if (my == Top())    {
        // ----- hit the top border
        int x = mx-Left();
        int wd = Width();
        if (x == wd-2)    {
            // ---- hit the restore or maximize box
            if (windowstate != ISRESTORED)
                Restore();
            else if (attrib & MAXBOX)
                Maximize();
        }
        else if (x == wd-3)    {
            // ----- hit the minimize box
            if (windowstate != ISMINIMIZED &&
                    (attrib & MINBOX))
                Minimize();
        }
        else if (HitControlBox(mx, my) &&
                (attrib & CONTROLBOX))
            // ------- hit the control box
            OpenCtlMenu();
        else if ((attrib & MOVEABLE) &&
                windowstate != ISMAXIMIZED)
            // ---- none of the above, move the window
            new Frame(this, mx);
    }
    else if ((attrib & SIZEABLE) && windowstate == ISRESTORED)
        if (mx == Right() && my == Bottom())
            // --- hit the lower right corner, size the window
            new Frame(this);
    prevmouseline = my;
    prevmousecol = mx;
}

void DFWindow::ButtonReleased(int, int)
{
    prevmouseline = -1;
    prevmousecol = -1;
}

Rect DFWindow::ClientRect()
{
    Rect rc(ClientLeft(), ClientTop(),
        ClientRight(), ClientBottom());
    return rc;
}

// ------------ move a window
void DFWindow::Move(int x, int y)
{
    int xdif = x - Left();
    int ydif = y - Top();
    if (xdif == 0 && ydif == 0)
        return;
    Bool wasVisible = visible;
    if (wasVisible)
        Hide();
    int ht = Height();
    int wd = Width();
    rect.Left() = x;
    rect.Top() = y;
    rect.Right() = Left()+wd-1;
    rect.Bottom() = Top()+ht-1;
    if (windowstate == ISRESTORED)
        restored_rc = rect;

    DFWindow *Wnd = first;
    while (Wnd != NULL)    {
        Wnd->Move(Wnd->Left()+xdif, Wnd->Top()+ydif);
        Wnd = Wnd->next;
    }
    if (wasVisible)
        Show();
}

// ------------ size a window
void DFWindow::Size(int x, int y)
{
    int xdif = x - Right();
    int ydif = y - Bottom();
    if (xdif == 0 && ydif == 0)
        return;
    Bool wasVisible = visible;
    if (wasVisible)
        Hide();
    rect.Right() = x;
    rect.Bottom() = y;
    if (windowstate == ISRESTORED)
        restored_rc = rect;

    DFWindow *Wnd = first;
    while (Wnd != NULL)    {
        Wnd->ParentSized(xdif, ydif);
        Wnd = Wnd->next;
    }
    if (wasVisible)
        Show();
}

void DFWindow::Minimize()
{
    if (windowstate == ISRESTORED)
        restored_rc = rect;
    Rect rc = PositionIcon();
    Hide();
    windowstate = ISMINIMIZED;
    Move(rc.Left(), rc.Top());
    Size(rc.Right(), rc.Bottom());
    Show();
}

void DFWindow::Maximize()
{
    restored_rc = rect;
    Rect rc(0, 0, desktop.screen().Width()-1,
                    desktop.screen().Height()-1);
    if (parent != NULL)
        rc = parent->ClientRect();
    Hide();
    windowstate = ISMAXIMIZED;
    Move(rc.Left(), rc.Top());
    Size(rc.Right(), rc.Bottom());
    Show();
}

void DFWindow::Restore()
{
    Hide();
    Move(restored_rc.Left(), restored_rc.Top());
    Size(restored_rc.Right(), restored_rc.Bottom());
    windowstate = ISRESTORED;
    if (this == desktop.InFocus())
        Show();
    else 
        SetFocus();
}

// ---- compute lower right icon space in a rectangle
static Rect LowerRight(Rect &prc)
{
    Rect rc(prc.Right()-IconWidth,prc.Bottom()-IconHeight,0,0);
    rc.Right() = rc.Left() + IconWidth-1;
    rc.Bottom() =  rc.Top() + IconHeight-1;
    return rc;
}

// ----- compute a position for a minimized window icon
Rect DFWindow::PositionIcon()
{
    Rect rc(desktop.screen().Width()-IconWidth,
            desktop.screen().Height()-IconHeight,
            desktop.screen().Width()-1,
            desktop.screen().Height()-1);
    if (parent != NULL)    {
        Rect prc = parent->rect;
        rc = LowerRight(prc);
        // --- search for icon available location
        DFWindow *Wnd = parent->first;
        while (Wnd != NULL)    {
            if (Wnd->windowstate == ISMINIMIZED)    {
                Rect rc1= Wnd->rect;
                if (rc1.Left() == rc.Left() &&
                        rc1.Top() == rc.Top())    {
                    rc.Left() -= IconWidth;
                    rc.Right() -= IconWidth;
                    if (rc.Left() < prc.Left()+1)   {
                        rc.Left() =
                            prc.Right()-IconWidth;
                        rc.Right() =
                            rc.Left()+IconWidth-1;
                        rc.Top() -= IconHeight;
                        rc.Bottom() -= IconHeight;
                        if (rc.Top() < prc.Top()+1)
                            return LowerRight(prc);
                    }
                    break;
                }
            }
            Wnd = Wnd->next;
        }
    }
    return rc;
}

