// ------------ dfwindow.h

#ifndef DFWINDOW_H
#define DFWINDOW_H

#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <stdlib.h>

#include "dflatdef.h"
#include "rectangl.h"
#include "strings.h"

// -------- window attribute flags
const int MOVEABLE   = 0x0001;
const int SIZEABLE   = 0x0002;
const int BORDER     = 0x0004;
const int TITLEBAR   = 0x0008;
const int CONTROLBOX = 0x0010;
const int MINBOX     = 0x0020;
const int MAXBOX     = 0x0040;
const int SHADOW     = 0x0080;
const int SAVESELF   = 0x0100;
const int NOCLIP     = 0x0200;
const int MENUBAR    = 0x0400;
const int STATUSBAR  = 0x0800;
const int VSCROLLBAR = 0x1000;
const int HSCROLLBAR = 0x2000;
const int FRAMEWND   = 0x4000;

// --------------- Color Macros
enum Colors {
    BLACK,
    BLUE,
    GREEN,
    CYAN,
    RED,
    MAGENTA,
    BROWN,
    LIGHTGRAY,
    DARKGRAY,
    LIGHTBLUE,
    LIGHTGREEN,
    LIGHTCYAN,
    LIGHTRED,
    LIGHTMAGENTA,
    YELLOW,
    WHITE
};

// ------ window shadow attributes
const unsigned char ShadowFG = DARKGRAY;
const unsigned char ShadowBG = BLACK;

// ----- minimized icon dimensions
const int IconWidth = 10;
const int IconHeight = 3;

// --------------- border characters
const unsigned char FOCUS_NW   = '\xc9';
const unsigned char FOCUS_NE   = '\xbb';
const unsigned char FOCUS_SE   = '\xbc';
const unsigned char FOCUS_SW   = '\xc8';
const unsigned char FOCUS_SIDE = '\xba';
const unsigned char FOCUS_LINE = '\xcd';
const unsigned char NW         = '\xda';
const unsigned char NE         = '\xbf';
const unsigned char SE         = '\xd9';
const unsigned char SW         = '\xc0';
const unsigned char SIDE       = '\xb3';
const unsigned char LINE       = '\xc4';

// ----------------- title bar characters
const unsigned char CONTROLBOXCHAR = '\xf0';
const unsigned char MAXPOINTER     = 24;
const unsigned char MINPOINTER     = 25;
const unsigned char RESTOREPOINTER = 18;

// ----------- window states
enum WndState     {
    OPENING,
    ISRESTORED,
    ISMINIMIZED,
    ISMAXIMIZED,
    ISCLOSING,
    CLOSED
};

// ---------- window colors
struct Color {
    Colors fg, bg;      // standard colors
    Colors sfg, sbg;    // selected text colors
    Colors ffg, fbg;    // window frame colors
    Colors hfg, hbg;    // highlighted text colors
};

class Application;
class StatusBar;
class PopDown;

class DFWindow    {
protected:
    WndType windowtype;   // window type
    int prevmouseline;    // holders for
    int prevmousecol;     // mouse coordinates
private:
    String *title;              // window title
    // -------------- window attributes
    int restored_attrib;        // attributes when restored
    Bool clipoverride;          // True to override clipping
    Rect restored_rc;           // restored state rect
    // ------- for painting overlapping windows
    void PaintOverLappers();
    // ----- control menu
    PopDown *ctlmenu;
    void OpenCtlMenu();
    void DeleteCtlMenu();
	// -------- for clearing the window
    char clearch;
    // --------- common window contructor code
    void InitWindow(char *ttl,
        int lf, int tp, int ht, int wd, DFWindow *par);
    void InitWindow(int lf, int tp, int ht, int wd,
        DFWindow *par);
    virtual void SetColors();
    void Enqueue();
    void Dequeue();
    Bool ClipParent(int &x, int y, String *ln);
    void WriteChar(int ch, int x, int y,
        Rect &rc, int fg, int bg);
    void WriteString(String &ln, int x, int y,
        Rect &rc, int fg, int bg);
    Rect PositionIcon();
    friend class StatusBar;
protected:
    // --------------- video memory save data
    char *videosave;      // video save buffer
    Bool visible;         // True = window has been shown
    int attrib;           // Window attribute flags
    Bool DblBorder;       // True = dbl border on focus
    Color colors;         // window colors
    WndState windowstate; // Restored, Maximized,
                          // Minimized, Closing
    Rect rect;            // window coordinates
                          // (0/0 to 79/24)
    // ------ previous capture focus handle
    DFWindow *prevcapture;
    // -------------- window geneology
    DFWindow *parent;           // parent window
    // -------- children
    DFWindow *first;            // first child window
    DFWindow *last;             // last child window
    // -------- siblings
    DFWindow *next;             // next sibling window
    DFWindow *prev;             // previous sibling window
    // -------- system-wide
    void NextSiblingFocus();
    void PrevSiblingFocus();
    void WriteClientString(String &ln,int x,int y,int fg,int bg);
    void WriteWindowString(String &ln,int x,int y,int fg,int bg);
    void WriteWindowChar(int ch,int x,int y,int fg,int bg);
    void WriteClientChar(int ch,int x,int y,int fg,int bg);
    // ------------- client window coordinate adjustments
    virtual void AdjustBorders();
    int BorderAdj;              // adjust for border
    int TopBorderAdj;           // adjust for top border
    int BottomBorderAdj;        // adjust for bottom border
    // -----
    Bool HitControlBox(int x, int y)
        { return (Bool)(x-Left() == 2 && y-Top() == 0 &&
            (attrib & CONTROLBOX)); }
    friend void DispatchEvents(Application *ApWnd);
    friend DFWindow *MouseWindow();
    friend DFWindow *inWindow(int x, int y, int &fg, int &bg);
public:
    // -------- constructors
    DFWindow(char *ttl, int lf, int tp, int ht, int wd,
                DFWindow *par)
        { InitWindow(ttl, lf, tp, ht, wd, par); }
    DFWindow(char *ttl, int ht, int wd, DFWindow *par)
        { InitWindow(ttl, -1, -1, ht, wd, par); }
    DFWindow(int lf, int tp, int ht, int wd, DFWindow *par)
        { InitWindow(lf, tp, ht, wd, par); }
    DFWindow(int ht, int wd, DFWindow *par)
        { InitWindow(-1, -1, ht, wd, par); }
    DFWindow(char *ttl, DFWindow *par = NULL)
        { InitWindow(ttl, 0, 0, -1, -1, par); }
    // -------- destructor
    virtual ~DFWindow()
        { if (windowstate != CLOSED) CloseWindow(); }
    // ------- window dimensions and position
    Rect WindowRect()    { return rect; }
    Rect ShadowedRect();
    int Right()          { return rect.Right(); }
    int Left()           { return rect.Left(); }
    int Top()            { return rect.Top(); }
    int Bottom()         { return rect.Bottom(); }
    int Height()         { return Bottom() - Top() + 1; }
    int Width()          { return Right() - Left() + 1; }
    // ------ client space dimensions and position
    Rect ClientRect();
    int ClientRight()    { return Right()-BorderAdj; }
    int ClientLeft()     { return Left()+BorderAdj; }
    int ClientTop()      { return Top()+TopBorderAdj; }
    int ClientBottom()   { return Bottom()-BottomBorderAdj; }
    int ClientHeight()   { return Height()-TopBorderAdj-
                              BottomBorderAdj; }
    int ClientWidth()    { return Width()-BorderAdj*2; }

    DFWindow *Parent()   { return parent; }
    DFWindow *First()    { return first; }
	DFWindow *Last()     { return last; }
	DFWindow *Next()     { return next; }
	DFWindow *Prev()     { return prev; }

    Bool isVisible()     { return visible; }
    int Attribute()      { return attrib; }
    void SetAttribute(int atr)
                         { attrib |= atr; AdjustBorders(); }
    void ClearAttribute(int atr)
                         { attrib &= ~atr; AdjustBorders(); }
    WndType WindowType() { return windowtype; }
    // ----- Control Menu messages
    void CtlMenuMove();
    void CtlMenuSize();
    // -------- API messages
    virtual void OpenWindow();
    virtual void CloseWindow();
    virtual void Show();
    virtual void Hide();
    virtual Bool SetFocus();
    virtual void ResetFocus();
    virtual void EnterFocus(DFWindow *child) {}
    virtual void LeaveFocus(DFWindow *child) {}
    void CaptureFocus();
    void ReleaseFocus();
    virtual void Paint();
    virtual void Paint(Rect rc);
    virtual void Border();
    virtual void Shadow();
    virtual void Title();
    virtual void ClearWindow();
    virtual void ShiftChanged(int sk);
    virtual void Keyboard(int key);
	virtual void KeyReleased() {}
    virtual void DoubleClick(int mx, int my);
    virtual void LeftButton(int mx, int my);
    virtual void ButtonReleased(int mx, int my);
    virtual void MouseMoved(int mx, int my) {}
    virtual void Move(int x, int y);
    virtual void Size(int x, int y);
    virtual void ParentSized(int, int) {}
    virtual void ClockTick() {}
    void Minimize();
    void Maximize();
    void Restore();
    WndState State() { return windowstate; }
    Rect &VisibleRect();
    Colors ClientFG()    { return colors.fg; }
    Colors ClientBG()    { return colors.bg; }
    Colors SelectedFG()  { return colors.sfg; }
    Colors SelectedBG()  { return colors.sbg; }
    Colors FrameFG()     { return colors.ffg; }
    Colors FrameBG()     { return colors.fbg; }
    Colors HighlightFG() { return colors.ffg; }
    Colors HighlightBG() { return colors.fbg; }
	void SetClearChar(char ch) { clearch = ch; }
};

inline DFWindow *inWindow(int x, int y)
{
    int fg, bg;
    return inWindow(x, y, fg, bg);
}

inline void DFWindow::WriteClientString(String &ln,
                    int x,int y,int fg,int bg)
{
    WriteString(ln,x+ClientLeft(),y+ClientTop(),
        ClientRect(),fg,bg);
}

inline void DFWindow::WriteWindowString(String &ln,
                    int x,int y,int fg,int bg)
{
    WriteString(ln,x+Left(),y+Top(),
        ShadowedRect(),fg,bg);
}

inline void DFWindow::WriteWindowChar(int ch,int x,int y,int fg,int bg)
{
    WriteChar(ch, x+Left(), y+Top(),
        ShadowedRect(), fg, bg);
}

inline void DFWindow::WriteClientChar(int ch,int x,int y,int fg,int bg)
{
    WriteChar(ch, x+ClientLeft(),y+ClientTop(),
        ClientRect(),fg,bg);
}

#endif


