// ------------- editbox.cpp

#include <ctype.h>
#include "desktop.h"
#include "editbox.h"

// ----------- common constructor code
void EditBox::OpenWindow()
{
	windowtype = EditboxWindow;
	if (windowstate == CLOSED)
		TextBox::OpenWindow();
	column = 0;
	changed = False;
	text = new String(bufflen);
	BuildTextPointers();
}

void EditBox::CloseWindow()
{
	// destructor code
	// .....
    TextBox::CloseWindow();
}

Bool EditBox::SetFocus()
{
	Bool rtn = TextBox::SetFocus();
	if (rtn)	{
		ResetCursor();
		desktop.cursor().Show();
	}
	return rtn;
}

void EditBox::ResetFocus()
{
	desktop.cursor().Hide();
	TextBox::ResetFocus();
}

// -------- process keystrokes
void EditBox::Keyboard(int key)
{
	int shift = desktop.keyboard().GetShift();
	if ((shift & ALTKEY) == 0)	{
    	switch (key)    {
        	case HOME:
            	Home();
				return;
        	case END:
            	End();
				return;
        	case CTRL_FWD:
            	NextWord();
				return;
        	case CTRL_BS:
            	PrevWord();
				return;
        	case FWD:
            	Forward();
				return;
        	case BS:
            	Backward();
				return;
        	case RUBOUT:
				if (column == 0)
					break;
            	Backward();
				// --- fall through
        	case DEL:
            	DeleteCharacter();
            	return;
        	default:
            	if (!isprint(key))
                	break;
            	// --- all printable keys get processed by editbox
				CharacterEntry(key);
            	return;
    	}
	}
	TextBox::Keyboard(key);
}

// -------- paint the editbox
void EditBox::Paint()
{
	TextBox::Paint();
	ResetCursor();
}

// -------- move the editbox
void EditBox::Move(int x, int y)
{
	TextBox::Move(x, y);
	ResetCursor();
}

// --------- clear the text from the editbox
void EditBox::ClearText()
{
	TextBox::ClearText();
	column = 0;
	changed = False;
	ResetCursor();
}

void EditBox::Home()
{
	column = wleft = 0;
	Paint();
	ResetCursor();
}

void EditBox::End()
{
	column = textwidth;
	if (column - wleft >= ClientWidth())	{
		wleft = column - ClientWidth() + 1;
		Paint();
	}
	ResetCursor();
}

void EditBox::NextWord()
{
	while ((*text)[column] != ' ' && (*text)[column])
		Forward();
	while ((*text)[column] == ' ')
		Forward();
}

void EditBox::PrevWord()
{
	Backward();
	while ((*text)[column] == ' ' && column != 0)
		Backward();
	while ((*text)[column] != ' ' && column != 0)
		Backward();
	if ((*text)[column] == ' ')
		Forward();
}

void EditBox::Forward()
{
	if ((*text)[column])	{
		column++;
		if (column-wleft == ClientWidth())
			ScrollLeft();
		ResetCursor();
	}
}

void EditBox::Backward()
{
	if (column)	{
		if (column == wleft)
			ScrollRight();
		--column;
		ResetCursor();
	}
}

void EditBox::DeleteCharacter()
{
	if ((*text)[column])	{
		String *ns = new String;
		if (column)
			*ns = text->left(column);
		int rt = text->Strlen()-column-1;
		if (rt > 0)
			*ns += text->right(rt);
		delete text;
		text = ns;
		BuildTextPointers();
		Paint();
		changed = True;
	}
}

void EditBox::CharacterEntry(int key)
{

	// --- test typing at end of buffer
	if (column == bufflen)	{
		desktop.speaker().Beep();
		return;
	}

	if (desktop.keyboard().InsertMode() && (*text)[column])	{
		// ---- shift the text to make room for new character
		String *ns = new String;
		if (column)
			*ns = text->left(column);
		*ns += " ";
		int rt = text->Strlen()-column;
		if (rt > 0)
			*ns += text->right(rt);
		delete text;
		text = ns;
	}
	(*text)[column] = (char) key;
	BuildTextPointers();
	Forward();
	Paint();
	changed = True;
}

void EditBox::SetCursor(int x, int y)
{
	desktop.cursor().SetPosition(
		x+ClientLeft()-wleft, y+ClientTop()-wtop);
}

void EditBox::LeftButton(int mx, int my)
{
	if (ClientRect().Inside(mx, my))	{
		column = min(text->Strlen(), mx-ClientLeft()+wleft);
		ResetCursor();
	}
	else
		TextBox::LeftButton(mx, my);
}



