/********************************************************************
 *
 *    CDROM.C
 *
 *    Copyright 1991 by Jim Harper.
 *    A CD-ROM redirector for High Sierra and ISO 9660 disks.
 *    usage:  CDROM drivelet: [scsiid]
 *    Example:  CDROM F: 3
 *    To unload, type CDROM -U
 *    Requires that the ST01 SCSI transport layer be present.
 *
 ********************************************************************
 *    Written in Microsoft QuickC 2.5.  Should work with C 6.0.
 *    Tabstop = 4
 *    To compile:  qcl cdrom.c
 ********************************************************************/
#pragma check_stack(off)
#pragma pack(1)

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <dos.h>
#include <bios.h>
#include <memory.h>
#include <errno.h>
#include "scsi.h"
#include "cdrom.h"
#include "redir.h"

#define STACKSIZE	128
#define CACHESIZE	10
#define MAXCLOSEALL	30
#define SetC(X)  	(X) |=  0x01
#define ClrC(X)  	(X) &= ~0x01

/* Globals */
extern unsigned _psp,       /* Runtime gives us these */
				end;

/* I/O Buffer ptr */
char *IOBuf;

/* Table of saved open sft's for DoCloseAll() */
struct SFT _far *CloseTab[MAXCLOSEALL];

unsigned StkSeg,
		 DataSeg,
		 DriveNo,
		 ScsiId,
		 DriveFlags,
		 TsrStkSeg,
		 TsrStkPtr,
		 AppStkSeg,
		 AppStkPtr,
		 CDType,
		 FIDoff,
		 Nameoff,
		 Dateoff,
		 Flagsoff,
		 Blkoff,
		 Sizeoff,
		 BlkSize,
		 ChainFlag,
		 MyStack[STACKSIZE];

unsigned _AX,
		 _BX,
		 _CX,
		 _DX,
		 _DS,
		 _ES,
		 _DI,
		 _FLAGS;

struct IntRegs {
	unsigned ES;
	unsigned DS;
	unsigned DI;
	unsigned SI;
	unsigned BP;
	unsigned SP;
	unsigned BX;
	unsigned DX;
	unsigned CX;
	unsigned AX;
	unsigned IP;
	unsigned CS;
	unsigned FLAGS;
};

int Active = 0;

/* Protos */
void MsgOut(char *),
	 DoScsi(void),
	 ToIBM(char *,int,char *),
	 DeInstall(void),
	 DoChDir(void),
	 DoClose(void),
	 DoRead(void),
	 DoGetSpace(void),
	 DoGetAttr(void),
	 DoOpen(void),
	 DoFindFirst(void),
	 DoFindNext(void),
	 DoSeek(void),
	 DoCloseAll(void),
	 DoPathName(void),
	 ToFront(struct DirEnt *),
	 ToHex(unsigned);

unsigned ScsiRead(unsigned long),
		 ScsiDirect(char _huge *,unsigned long,unsigned),
		 ToDosDate(char *),
		 ToDosTime(char *);

int Match(char *,char *),
	Lookup(struct DirEnt **),
	DirLook(struct DirEnt *,struct DirEnt **,char *);

void (_interrupt _far *Old2F)(void);
void _interrupt _far New2F(struct IntRegs);

struct isoVolDesc	*isoVolDescp;
struct isoDirRec	*isoDp;
struct hsVolDesc	*hsVolDescp;
struct hsDirRec		*hsDp;
struct Cmd			Cmd;
struct DirEnt		RootEnt,
			  		DirCache[CACHESIZE];

/* Important pointers */
struct SDB _far *SDBp;				/* Ptr to Dos Search Data Blk	*/
struct FDB _far *FDBp;				/* Ptr to Dos Found Data Blk	*/
struct LOL _far *LOLp;				/* Ptr to list of lists			*/
struct CDS _far *CDSp;				/* Ptr to cur dir tab entry		*/

/* These pointers are set according to DOS 3.xx or 4.xx				*/
char _far  *SWAPp,					/* Ptr to Dos swap area			*/
	 _far  *FN1p,					/* Ptr to Dos resolved name		*/
	 _far  * _far *DTApp,			/* Ptr to Ptr to Current DTA	*/
	 _far  * _far *SFTpp,			/* Ptr to Ptr to Current SFT	*/
	 _far  *DosDp,					/* Ptr to dir ent for file      */
	 _far  *Sattrp,					/* Ptr to search attr			*/
	 _far  *OpenModep;				/* Ptr to open mode				*/

unsigned _far  *PSPp;				/* Ptr to current PSP			*/

char *HiSierra = "HISIERRA   ",
	 *Iso9660  = "ISO9660    ";


main(int argc, char *argv[])
{
	union REGS		regs;
	struct SREGS	sregs;
	unsigned _far	*EnvBlkp;
	int				i,
					Junk,
					CdsLen,
					ProgSize;

	DriveNo = 999;
	ScsiId = 0;
	if (argc > 1) {
		if (!strcmp(argv[1],"-u") || ! strcmp(argv[1],"-U")) {
			regs.h.ah = 0x11;
			regs.h.al = DEINSTALL;
			int86x(INT2F,&regs,&regs,&sregs);
			exit(0);
		}
		if (argv[1][0] >= 'a' && argv[1][0] <= 'z')
			argv[1][0] -= 32;
		if (argv[1][0] >= 'A' && argv[1][0] <= 'Z' &&
				argv[1][1] == ':' &&
					argv[1][2] == '\0') {
			DriveNo = argv[1][0] - 'A';
		}
		if (argc > 2) {
			if ((argv[2][0] >= '0' && argv[2][0] <= '7') && argv[2][1] == '\0')
				ScsiId = argv[2][0] - '0';
			else
				ScsiId = 999;
		}
	}

	if (DriveNo > 26 || ScsiId > 7) {
		MsgOut("usage: cdrom [A-Z]: [ScsiId]\r\n");
		exit(1);
	}

	/* Get our stack and data segments */
	segread(&sregs);
	StkSeg = sregs.ss;
	DataSeg = sregs.ds;

	/* Check if Scsi tsr is present */
	regs.h.ah = FUNCID;
	regs.h.al = INSTALLCHK;
	int86x(INT2F,&regs,&regs,&sregs);

	if (regs.h.al != 0xff) {
		MsgOut("Scsi tsr not found!\r\n");
		exit(1);
	}

	/*
	 * Check if there's a High Sierra or ISO9660 disk
	 * in the drive.  Decide which it is.
	 */
	if (ScsiRead(0x10L)) {
		MsgOut("IO error.\r\n");
		exit(1);
	}
	hsVolDescp = (struct hsVolDesc *) IOBuf;
	isoVolDescp = (struct isoVolDesc *) IOBuf;
	CDType = UNKNOWN;
	Blkoff = 2;
	Sizeoff = 10;
	Dateoff = 18;
	FIDoff= 32;
	Nameoff= 33;
	strcpy(RootEnt.FName,"ROOT-CDROM ");
	if (strncmp(hsVolDescp->ID,"CDROM",5) == 0) {
		CDType = HIGHSIERRA;
		Flagsoff = 24;
		hsDp = (struct hsDirRec *)hsVolDescp->DirRec;
		RootEnt.Fattr = _A_SUBDIR;
		RootEnt.FTime = ToDosTime(hsDp->Date);
		RootEnt.FDate = ToDosDate(hsDp->Date);
		RootEnt.BlkNo = hsDp->ExtLocLSB;
		RootEnt.FSize = hsDp->DataLenLSB;
		RootEnt.ParentBlk = hsDp->ExtLocLSB;
		BlkSize = hsVolDescp->BlkSizeLSB;
		MsgOut("High Sierra disk...\r\n");
	}
	if (strncmp(isoVolDescp->ID,"CD001",5) == 0) {
		CDType = ISO9660;
		Flagsoff = 25;
		isoDp = (struct isoDirRec *)isoVolDescp->DirRec;
		RootEnt.Fattr = _A_SUBDIR;
		RootEnt.FTime = ToDosTime(isoDp->Date);
		RootEnt.FDate = ToDosDate(isoDp->Date);
		RootEnt.BlkNo = isoDp->ExtLocLSB;
		RootEnt.FSize = isoDp->DataLenLSB;
		RootEnt.ParentBlk = isoDp->ExtLocLSB;
		BlkSize = isoVolDescp->BlkSizeLSB;
		MsgOut("ISO 9660 disk...\r\n");
	}
	if (CDType == UNKNOWN) {
		MsgOut("Unknown disk type..\r\n");
		exit(1);
	}

	/* Get Address of List of Lists */
	regs.h.ah = 0x52;
	int86x(0x21,&regs,&regs,&sregs);

	FP_SEG(LOLp) = sregs.es;
	FP_OFF(LOLp) = regs.x.bx;

	/* Get address of Dos Swap area */
	regs.x.ax = 0x5d06;
	int86x(0x21,&regs,&regs,&sregs);
	FP_SEG(SWAPp) = sregs.ds;
	FP_OFF(SWAPp) = regs.x.si;

	if (DriveNo > LOLp->LastDrive) {
		MsgOut("Drive # to high.\r\n");
		exit(1);
	}

	MsgOut("DOS version ");
	ToHex(_osmajor);
	MsgOut(".");
	ToHex(_osminor);

	/* Now set the offsets within Dos according to 3.3x, 4.xx or 5.xx */
	if (_osmajor == 3 && _osminor >= 30) {
		CdsLen = 0x51;
		PSPp = (unsigned _far *)(SWAPp + 0x10U);
		FN1p = SWAPp + 0x0092U;
		Sattrp = SWAPp + 0x023aU;
		DosDp = SWAPp + 0x01a7U;
		SDBp = (struct SDB _far *) (SWAPp + 0x0192U);
		DTApp = (char _far * _far *)(SWAPp + 0x000cU);
		SFTpp = (char _far * _far *)(SWAPp + 0x0268U);
	} else if (_osmajor == 4 || _osmajor == 5) {
		CdsLen = 0x58;
		PSPp = (unsigned _far *)(SWAPp + 0x10U);
		FN1p = SWAPp + 0x009eU;
		Sattrp = SWAPp + 0x024dU;
		DosDp = SWAPp + 0x01b3U;
		SDBp = (struct SDB _far *) (SWAPp + 0x019eU);
		DTApp = (char _far * _far *)(SWAPp + 0x000cU);
		SFTpp = (char _far * _far *)(SWAPp + 0x027eU);
	} else {
		MsgOut("Not DOS 3.3x, 4.xx or 5.xx\r\n");
		exit(1);
	}

	/* Cast ptr to table entry pointer */
	CDSp = (struct CDS _far *) (LOLp->CDS + DriveNo * CdsLen);

	/* Turn on network & physical bits */
	DriveFlags = CDSp->Flags;
	CDSp->Flags = 0xC000;
	CDSp->RootOff = 2;

	/* Set to root */
	CDSp->CurDir[0] = 'A' + DriveNo;
	CDSp->CurDir[1] = ':';
	CDSp->CurDir[2] = '\\';
	CDSp->CurDir[3] = '\0';

	/* Initialize our stack */
	for (i = 0; i < STACKSIZE; i++)
		MyStack[i] = 0x4141;

	/* Initialize stack and bottom of program. */
	TsrStkSeg = DataSeg;
	TsrStkPtr = (unsigned)&MyStack[STACKSIZE];
	/* Make sure stack is on a word boundry */
	if (TsrStkPtr & 0x1U)
		TsrStkPtr--;

	/* Program size in paragraphs w/o a heap */
	ProgSize = (StkSeg + (((unsigned)&end) >> 4)) - _psp + 1;

	/* Initialize cache */
	for (i = 1; i < CACHESIZE - 1; i++) {
		DirCache[i].Forw = &DirCache[i+1];
		DirCache[i].Back = &DirCache[i-1];
	}
	DirCache[0].Forw = &DirCache[1];
	DirCache[0].Back = &RootEnt;
	DirCache[CACHESIZE-1].Forw = &RootEnt;
	DirCache[CACHESIZE-1].Back = &DirCache[CACHESIZE-2];

	/* Root dirent provides anchor into the cache */
	RootEnt.Forw = &DirCache[0];
	RootEnt.Back = &DirCache[CACHESIZE-1];

	/* Close files */
	_dos_close(0);		/* stdin  */
	_dos_close(1);		/* stdout */
	_dos_close(2);		/* stderr */

	/* Grab multiplex interrupt */
	Old2F = _dos_getvect(INT2F);
	_dos_setvect(INT2F,New2F);

	/* Free the environment */
	FP_SEG(EnvBlkp) = _psp;
	FP_OFF(EnvBlkp) = 0x2c;
	_dos_freemem(*EnvBlkp);

	/* Shrink our program size */
	_dos_setblock(ProgSize,_psp,&Junk);

	/* TSR ourself */
	_dos_keep(0,ProgSize);
}

/*
 *  Our interrupt 2F handler.
 */
void _interrupt _far New2F(struct IntRegs IntRegs)
{
	/* See if we handle this function */
	if ((IntRegs.AX >> 8U) != 0x11 || Active)
		_chain_intr(Old2F);

	/* Install check?? */
	if ((IntRegs.AX & 0xff) == INSTALLCHK) {
		IntRegs.AX = 0x00ff;
		return;
	}

	/* Set flag saying we're active */
	Active++;

	/* Don't chain out by default */
	ChainFlag = 0;

	/* Save needed regs from stack */
	_AX = IntRegs.AX;
	_BX = IntRegs.BX;
	_CX = IntRegs.CX;
	_DX = IntRegs.DX;
	_DS = IntRegs.DS;
	_ES = IntRegs.ES;
	_DI = IntRegs.DI;
	_FLAGS = IntRegs.FLAGS;

	/* Switch to own stack */
	_asm
	{
		cli							; Interrupts off
		mov  WORD PTR AppStkPtr,sp	; Save app stack
		mov  WORD PTR AppStkSeg,ss
		mov  sp,WORD PTR TsrStkPtr	; Load new stack
		mov  ss,WORD PTR TsrStkSeg
		sti							; Interrupts on
	}

	/* Process command */
	switch(_AX & 0xff) {
		case DEINSTALL:
			DeInstall(); break;
		case CHDIR:
			DoChDir(); break;
		case CLOSE:
			DoClose(); break;
		case READ:
			DoRead(); break;
		case GETSPACE:
			DoGetSpace(); break;
		case GETATTR:
			DoGetAttr(); break;
		case OPEN:
			DoOpen(); break;
		case FINDFIRST:
			DoFindFirst(); break;
		case FINDNEXT:
			DoFindNext(); break;
		case SEEK:
			DoSeek(); break;
		case CLOSEALL:
			DoCloseAll(); break;
		case PATHNAME:
			DoPathName(); break;
		case 0x25:
		default:
			ChainFlag = 1; break;
	}

	/* Switch back to app stack */
	_asm
	{
		cli							; Interrupts off
		mov  sp,WORD PTR AppStkPtr	; Load app stack
		mov  ss,WORD PTR AppStkSeg
		sti							; Interrupts on
	}

	/* If anyone set the chain flag, chain out */
	if (ChainFlag) {
		Active = 0;
		_chain_intr(Old2F);
	}

	/* Restore (possibly modifed) registers */
	IntRegs.AX = _AX;
	IntRegs.BX = _BX;
	IntRegs.CX = _CX;
	IntRegs.DX = _DX;
	IntRegs.FLAGS = _FLAGS;

	/* Clear Active Flag */
	Active = 0;
}

void DeInstall(void)
{
	unsigned AppPsp;

	/* See if anyone grabbed our vector */
	if (_dos_getvect(INT2F) != New2F) {
		MsgOut("cdrom driver can't deinstall.\r\n");
		_AX = 0xffff;
		SetC(_FLAGS);
		return;
	}

	/* Restore interrupts */
	_dos_setvect(INT2F,Old2F);

	/* Un-network the drive */
	CDSp->Flags = DriveFlags;

	/* Get foreground's psp */
	AppPsp = *PSPp;
	/* Set current psp to ours */
	*PSPp = _psp;
	/* Free our prog */
	_dos_freemem(_psp);
	/* Restore foregound psp */
	*PSPp = AppPsp;
	MsgOut("cdrom redirector uninstalled.\r\n");
	_AX = 0x0000;
	ClrC(_FLAGS);
}

#define LPT1 0
#define LPT2 2

void MsgOut(char *Msg)
{
#if 0
	while (*Msg)
		_bios_printer(_PRINTER_WRITE,LPT1,*Msg++);
#else
	union REGS		regs;
	struct SREGS	sregs;
	char			Buf[2];

	while (*Msg) {
		Buf[0] = *Msg++;
		Buf[1] = '$';
		regs.h.ah = 0x09;
		sregs.ds = DataSeg;
		regs.x.dx = (unsigned)Buf;
		intdos(&regs,&regs);
	}
#endif
}

void DoOpen(void)
{
	struct SFT _far			*SFTp;
	struct DirEnt			*Dp;
	char _far				*Fnp;
	int						i,
							Err;

	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}

	/* Get system file table pointer */
	SFTp = (struct SFT _far *) *SFTpp;

	/* Check closeall table */
	for (i = 0; i < MAXCLOSEALL && SFTp != CloseTab[i]; i++)
		;

	/* If not in table, put it there (if it fits) */
	if (i >= MAXCLOSEALL) {
		for (i = 0; i < MAXCLOSEALL && CloseTab[i]; i++)
			;
		if (!CloseTab[i])
			CloseTab[i] = SFTp;
	}

	/* Look up filename */
	if ((Err = Lookup(&Dp)) != 0) {
		_AX = Err;
		SetC(_FLAGS);
		return;
	}

	/* Gotta be a file, not a dir */
	if (Dp->Fattr & _A_SUBDIR) {
		_AX = FILENOTFOUND;
		SetC(_FLAGS);
		return;
	}

	/* Fill in sft */
	for (i = 0; i < 11; i++)
		SFTp->Name[i] = Dp->FName[i];
	if (SFTp->Mode & 0x8000)
		SFTp->Mode |= 0x70;					/* Network FCB, read */
	else
		SFTp->Mode |= 0x40;					/* Read */
	SFTp->DirAttrib = Dp->Fattr;
	SFTp->Flags = 0x8000 | 0x40 | DriveNo;	/* Net, written, DriveNo */
	SFTp->FilPos = 0L;
	SFTp->FilSiz = Dp->FSize;
	SFTp->DCB = Dp->BlkNo;
	SFTp->OwnerPSP = *PSPp;

	ClrC(_FLAGS);
}

void DoChDir(void)
{
	int				Err;
	struct DirEnt	*Dp;
	char _far		*Fnp;

	/* For us? */
	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}

	/* Are you there? */
	if ((Err = Lookup(&Dp)) != 0) {
		_AX = Err;
		SetC(_FLAGS);
		return;
	}

	/* Gotta be a dir, not a file */
	if ((Dp->Fattr & _A_SUBDIR) == 0x00) {
		_AX = PATHNOTFOUND;
		SetC(_FLAGS);
		return;
	}

	_AX = 0;
	ClrC(_FLAGS);
	return;
}

void DoFindFirst(void)
{
	struct DirEnt			*Dp;
	int						i,
							Ch,
							Err;
	char					*Chp,
							_far *Fnp,
							WildCard[11],
							TemPlate[11];

	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}
	
	SDBp = (struct SDB _far *) *DTApp;
	FDBp = (struct FDB _far *) (*DTApp + sizeof(struct SDB));

	SDBp->DriveLet = (DriveNo | 0x80);

	/* Handle vol id */
	if ((*Sattrp == _A_VOLID) ||
			(FN1p[3] == '0' && (*Sattrp & _A_VOLID))) {
		for (i = 0; i < 11; i++)
			SDBp->TemPlate[i] = '?';
		SDBp->Sattr = *Sattrp;
		SDBp->Entry = 0;
		if (CDType == HIGHSIERRA) {
			for (i = 0, Chp = HiSierra; *Chp; Chp++)
				FDBp->FName[i++] = *Chp;
		} else {
			for (i = 0, Chp = Iso9660; *Chp; Chp++)
				FDBp->FName[i++] = *Chp;
		}
		FDBp->Fattr = _A_VOLID;
		FDBp->FDate = 0;
		FDBp->FTime = 0;
		FDBp->FSize = 0;

		_AX = 0;
		ClrC(_FLAGS);
		return;
	}

	/*
	 * Handle getfirst by setting up for findnext.
	 */
	/* Find end of path string */
	for (i = 0; FN1p[i]; i++)
		;

	/* find last path separator */
	while (FN1p[i] != PATHSEPARATOR)
		i--;

	/* Cut off wildcard at end */
	if (i == 2)
		i++;

	Ch = FN1p[i];
	FN1p[i] = '\0';

	/* Now find directory */
	if ((Err = Lookup(&Dp)) != 0) {
		_AX = Err;
		SetC(_FLAGS);
		return;
	}

	/* Restore full pathname */
	FN1p[i] = Ch;

	/* Gotta be a dir, not a file */
	if ((Dp->Fattr & _A_SUBDIR) == 0x00) {
		_AX = PATHNOTFOUND;
		SetC(_FLAGS);
		return;
	}

	/* Point to Wildcard */
	if (Ch == PATHSEPARATOR)
		i++;
	Fnp = &FN1p[i];

	/* Copy */
	for (i = 0; *Fnp; i++)
		WildCard[i] = *Fnp++;
	WildCard[i] = '\0';

	/* Convert to IBM style, in dos record */
	ToIBM(TemPlate,12,WildCard);

	/* Copy to DOS's SDB record */
	for (i = 0; i < 11; i++)
		SDBp->TemPlate[i] = TemPlate[i];

	SDBp->Sattr = *Sattrp & 0x1e;
	SDBp->Entry = 0;

	/* Save startblk of parent directory */
	SDBp->ParentBlk = Dp->BlkNo;

	/* Save size of parent */
	SDBp->ParentSize = Dp->FSize;

	DoFindNext();
}

void DoFindNext(void)
{
	int						Entry,
							Drive,
							Flags,
							ExtRecLen,
							i;
	char					*Chp,
							IBMName[11],
							TemPlate[11];
	unsigned long 			BlkNo,
							EndBlk;

	SDBp = (struct SDB _far *) *DTApp;
	FDBp = (struct FDB _far *) (*DTApp + sizeof(struct SDB));

	/* Is this for us? */
	if ((Drive = SDBp->DriveLet & 0x1f) != DriveNo) {
		ChainFlag = 1;
		return;
	}

	/* Get copy of wildcard */
	for (i = 0; i < 11; i++)
		TemPlate[i] = SDBp->TemPlate[i];

	/* Where's the end of the dir? */
	EndBlk = SDBp->ParentBlk + (SDBp->ParentSize / BlkSize) - 1;

	/* Loop through parent dir, counting entries */
	Entry = 0;
	for (BlkNo = SDBp->ParentBlk; BlkNo <= EndBlk; BlkNo++) {
		ScsiRead(BlkNo);
		for (Chp = IOBuf; *Chp; Chp += *Chp, Entry++) {
			/* Ignore entries < our start entry # */
			if (Entry < SDBp->Entry)
				continue;
			/* Convert to IBM style name */
			ToIBM(IBMName,*(Chp+FIDoff),Chp+Nameoff);

			/* Convert flags */
			Flags = (*(Chp+Flagsoff) & DIR) ? _A_SUBDIR : _A_RDONLY;

			/* Check for normal files _only_ */
			if (SDBp->Sattr == 0x00)
				if (Flags == _A_RDONLY && Match(IBMName,TemPlate))
					goto FoundIt;

			/* Check for dirs & normal files */
			if (SDBp->Sattr != 0 && Match(IBMName,TemPlate))
					goto FoundIt;
		}
		/* Sneaky patch to speed up searchs */
		SDBp->ParentSize -= BlkSize;
		SDBp->ParentBlk = BlkNo + 1;
		SDBp->Entry = 0;
		Entry = 0;
	}

	/* At end of dir w/o a match */
	_AX = NOMOREFILES;
	SetC(_FLAGS);
	return;

FoundIt:
	/* Save start point for next time */
	SDBp->Entry = Entry + 1;

	/* Give to DOS */
	for (i = 0; i < 11; i++)
		FDBp->FName[i] = IBMName[i];

	/* Convert to PC attribute */
	FDBp->Fattr = (*(Chp+Flagsoff) & DIR) ? _A_SUBDIR : _A_RDONLY;

	/* Convert date, time & size to PC structures (use macros instead?) */
	FDBp->FDate = ToDosDate(Chp + Dateoff);
	FDBp->FTime = ToDosTime(Chp + Dateoff);
	FDBp->FSize = *((unsigned long *)(Chp + Sizeoff));

	_AX = 0;
	ClrC(_FLAGS);
}

void DoGetSpace(void)
{
	char _far		*Chp;

	/* For us? */
	FP_SEG(Chp) = _ES;
	FP_OFF(Chp) = _DI;

	if (*Chp != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}

	_CX = BlkSize;	/* # bytes/sector */
	_DX = 0U;		/* # clusters available */
	_BX = 0U;		/* # clusters on drive */
	_AX = 1U;		/* # sectors/cluster */
	ClrC(_FLAGS);
}

void DoRead(void)
{
	struct SFT _far		*SFTp;
	char _huge			*DTAp;
	char				*Orgp;
	unsigned			Offset,
						NumRead,
						ReadLen,
						N;
	unsigned long		BlkNo;

	/* Get system file table pointer */
	SFTp = (struct SFT _far *) *SFTpp;

	/* See if this is for us */
	if ((SFTp->Flags & 0x3f) != DriveNo) {
		ChainFlag = 1;
		return;
	}

	/* Get DTA ptr */
	DTAp = (char _far *) *DTApp;

	/* Cant't read past EOF */
	if (SFTp->FilPos >= SFTp->FilSiz) {
		_CX = 0x00;
		ClrC(_FLAGS);
		return;
	}

	/* Chop read back if too long */
	if ((SFTp->FilPos + _CX) > SFTp->FilSiz)
		_CX = SFTp->FilSiz - SFTp->FilPos;

	/* Keep track of how much we've read */
	NumRead = 0;

	while (NumRead < _CX) {
		/* Calc blk w/start of data */
		BlkNo = SFTp->DCB + (SFTp->FilPos / BlkSize);
		ReadLen = _CX - NumRead;

#if 0
		MsgOut("ReadLen = $");
		ToHex(ReadLen);
		MsgOut("\r\n$");
#endif

		/* Special case - read direct (on blk boundry, over 1 blk) */
		if ((SFTp->FilPos % BlkSize) == 0 && (ReadLen / BlkSize) > 0) {
			ReadLen = (ReadLen / BlkSize) * BlkSize;
			ScsiDirect(DTAp,BlkNo,ReadLen/BlkSize);
			SFTp->FilPos += ReadLen;
			NumRead += ReadLen;
			DTAp += ReadLen;
		} else {	/* Partial block */
			Offset = SFTp->FilPos % BlkSize;
			N = min(BlkSize - Offset,_CX - NumRead);
			ScsiRead(BlkNo);
			Orgp = IOBuf + Offset;
			movedata((INT16)DataSeg,
					 (INT16)(Orgp),
					 FP_SEG(DTAp),
					 FP_OFF(DTAp),
					 (UINT16)N);
			SFTp->FilPos += N;
			NumRead += N;
			DTAp += N;
		}
	}

	ClrC(_FLAGS);
	return;
}

void DoClose(void)
{
	struct SFT _far		*SFTp;
	int					i;

	/* For us? */
	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}

	/* Get system file table pointer */
	SFTp = (struct SFT _far *) *SFTpp;

	if (SFTp->RefCnt)
		SFTp->RefCnt--;

	/* Update private file table */
	if (SFTp->RefCnt == 0) {
		for (i = 0; i < MAXCLOSEALL && SFTp != CloseTab[i]; i++)
			;
		if (i < MAXCLOSEALL)
			CloseTab[i] = (struct SFT _far *)NULL;
	}

	ClrC(_FLAGS);
	return;
}

void DoGetAttr(void)
{
	struct DirEnt			*Dp;
	char _far				*Fnp;
	int						Err,
							i;

	/* For us? */
	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}

	/* Look up filename (full path) */
	if ((Err = Lookup(&Dp)) != 0) {
		_AX = Err;
		SetC(_FLAGS);
		return;
	}

	/* Get attributes */
	_AX = Dp->Fattr;

	ClrC(_FLAGS);
}

void DoSeek(void) /* What it _really_ wants to do is to seek() from the end.. */
{
	struct SFT _far		*SFTp;
	unsigned long		Offset;
	int					i;

	/* Get system file table pointer */
	SFTp = (struct SFT _far *) *SFTpp;

	/* See if this is for us */
	if ((SFTp->Flags & 0x3f) != DriveNo) {
		ChainFlag = 1;
		return;
	}

	/* Where (from end) to seek to */
	FP_SEG(Offset) = _CX;
	FP_OFF(Offset) = _DX;

	SFTp->FilPos = SFTp->FilSiz - Offset;

	/* return current position in DX:AX */
	_DX = (unsigned) (SFTp->FilPos >> 16U);
	_AX = (unsigned) SFTp->FilPos;
	ClrC(_FLAGS);
}

void DoCloseAll(void)
{
	int		i;

	/* Close all open files, mark all as closed */
	for (i = 0; i < MAXCLOSEALL; i++)
		if (CloseTab[i]) {
			CloseTab[i]->RefCnt = 0;
			CloseTab[i] = (struct SFT _far *)NULL;
		}
	/* Chain to any other redirs, to allow them to do so, too */
	ChainFlag = 1;
}

void DoPathName(void)
{
	/* See if this is for us */
	if (*FN1p != ('A' + DriveNo)) {
		ChainFlag = 1;
		return;
	}
	/* Error out if it is (makes DOS do it instead) */
	SetC(_FLAGS);
	return;
}

static unsigned long BufBlk1 = 0xffffffffL;
static char Buf1[2048];
			
/*
 * Read a sector
 */
unsigned ScsiRead(UINT32 BlkNo)
{
	union REGS		regs;
	struct SREGS	sregs;
	int				TryCount;

	if (BlkNo == BufBlk1) {
		IOBuf = Buf1;
		return(0U);
	}

	IOBuf = Buf1;
	for (TryCount = 0; TryCount < RETRYCOUNT; TryCount) {
		regs.h.ah = FUNCID;
		regs.h.al = DOCMD;
		sregs.ds = DataSeg;
		regs.x.dx = (unsigned)&Cmd;
		Cmd.ID = ScsiId;						/* Scsi ID */
		Cmd.CDB[0] = 0x28;						/* Read */
		Cmd.CDB[1] = 0x00;						/* Lun */
		Cmd.CDB[2] = (BlkNo >> 24U) & 0xffL;
		Cmd.CDB[3] = (BlkNo >> 16U) & 0xffL;
		Cmd.CDB[4] = (BlkNo >> 8U) & 0xffL;
		Cmd.CDB[5] = BlkNo & 0xffL;
		Cmd.CDB[7] = 0x00;
		Cmd.CDB[8] = 0x01;						/* 1 block */
		Cmd.CDB[9] = 0x00;						/* Control Byte */
		Cmd.DSeg = DataSeg;
		Cmd.DOff = (unsigned)IOBuf;
		Cmd.Count = 2048;
		int86x(INT2F,&regs,&regs,&sregs);

		if (Cmd.Stat == 0x00) {
			BufBlk1 = BlkNo;
			break;
		}
	}

	return((unsigned)(Cmd.Stat & 0xff));
}

unsigned ScsiDirect(char _huge *Cp,unsigned long BlkNo,unsigned NumBlks)
{
	union REGS		regs;
	struct SREGS	sregs;

	regs.h.ah = FUNCID;
	regs.h.al = DOCMD;
	sregs.ds = DataSeg;
	regs.x.dx = (unsigned)&Cmd;
	Cmd.ID = ScsiId;						/* Scsi ID */
	Cmd.CDB[0] = 0x28;						/* Read */
	Cmd.CDB[1] = 0x00;						/* Lun */
	Cmd.CDB[2] = (BlkNo >> 24U) & 0xffL;
	Cmd.CDB[3] = (BlkNo >> 16U) & 0xffL;
	Cmd.CDB[4] = (BlkNo >> 8U) & 0xffL;
	Cmd.CDB[5] = BlkNo & 0xffL;
	Cmd.CDB[7] = NumBlks >> 8U;
	Cmd.CDB[8] = NumBlks;
	Cmd.CDB[9] = 0x00;						/* Control Byte */
	Cmd.DSeg = ((unsigned long) Cp) >> 16U;
	Cmd.DOff = (unsigned) Cp;
	Cmd.Count = 2048 * NumBlks;
	int86x(INT2F,&regs,&regs,&sregs);

	return((unsigned)(Cmd.Stat & 0xff));
}

void ToIBM(char *IBMName,int FIDLen,char *Chp) /* 9660 name to DOS name */
{
	int		i,
			j;

	for (i = 0; i < 11; i++)
		IBMName[i] = ' ';

	/* return '.' for self */
	if (FIDLen == 1 && *Chp == 0x00) {
		IBMName[0] = '.';
		return;
	}

	/* return '..' for parent */
	if (FIDLen == 1 && *Chp == 0x01) {
		IBMName[0] = '.';
		IBMName[1] = '.';
		return;
	}

	for (i = j = 0;
			j < 11 &&
				i < FIDLen &&
					Chp[i] != '.' &&
						Chp[i] != ';' &&
							Chp[i] != '\0';
				i++)
		IBMName[j++] = Chp[i];
	
	if (i < FIDLen && Chp[i++] == '.')
		for (j = 8;
			i < FIDLen &&
				j < 11 &&
					Chp[i] != ';' &&
						Chp[i] != '\0';
				i++)
			IBMName[j++] = Chp[i];
}

int Match(char *Name,char *TemPlate) /* Check name against wildcard */
{
	int i;

	for (i = 0; i < 11; i++) {
		if (TemPlate[i] == '?')
			continue;
		if (TemPlate[i] != Name[i])
			return(0);
	}

	return(1);
}

unsigned ToDosDate(char *Date)
{
	unsigned TheDate;

	TheDate = ((Date[0] - 80) << 9U) | (Date[1] << 5U) | Date[2];
	return(TheDate);
}

unsigned ToDosTime(char *Date)
{
	unsigned TheTime;

	TheTime = (Date[3] << 11U) | (Date[4] << 5U) | (Date[5] >> 2U);
	return(TheTime);
}

int Lookup(struct DirEnt **Dpp) /* Find file in dir if it exists */
{
	struct DirEnt	*pDp,
					*cDp;
	int				i,
					Err;
	char _far		*Pathp,
					*Chp;
	char			Name[13];

	/* Check for null - return Root */
	if (FN1p[3] == '\0') {
		*Dpp = &RootEnt;
		return(0);
	}

	/* Skip drive letter, ':' and '\'. ex: 'F:\' */
	Pathp = FN1p + 3;

	/* Start at root */
	pDp = cDp = &RootEnt;

	while (*Pathp) {
		Chp = Name;
		while(*Pathp && *Pathp != PATHSEPARATOR)
			*Chp++ = *Pathp++;
		*Chp++ = '\0';

		/* Look up Name */
		if ((Err = DirLook(pDp,&cDp,Name)) != 0)
			return(Err);

		/* Move down a level */
		pDp = cDp;

		if (*Pathp == PATHSEPARATOR)
			Pathp++;
	}
	*Dpp = cDp;
	return(0);
}

/*
 * See if Name is present in pDp, if so return a DirEnt struct using cDpp
 * pDp - Parent DirEnt structure pointer.
 * cDpp - Child DirEnt structure pointer pointer.
 */
int DirLook(struct DirEnt *pDp,struct DirEnt **cDpp,char *Name)
{
	struct DirEnt	*Dp;
	char 			*Chp,
					WantName[11],
					FName[11];
	int 			ExtRecLen;
	unsigned long	BlkNo,
					EndBlk;

	/* Is a dir? */
	if ((pDp->Fattr & _A_SUBDIR) == 0)
		return(PATHNOTFOUND);

	/* Convert Name to IBM style */
	ToIBM(WantName,12,Name);

	/* Check cache */
	for (Dp = RootEnt.Forw; Dp != &RootEnt; Dp = Dp->Forw) {
		if (Dp->ParentBlk == pDp->BlkNo && !strncmp(WantName,Dp->FName,11)) {
			ToFront(Dp);
			*cDpp = Dp;
			return(0);
		}
	}

	/* Where's the end of the dir ? */
	EndBlk = pDp->BlkNo + (pDp->FSize / BlkSize) - 1;

	/* Read dir, scan for match */
	for (BlkNo = pDp->BlkNo; BlkNo <= EndBlk; BlkNo++) {
		ScsiRead(BlkNo);
		for (Chp = IOBuf; *Chp; Chp += *Chp) {

			/* Convert to IBM style name */
			ToIBM(FName,*(Chp+FIDoff),Chp+Nameoff);

			if (Match(FName,WantName))
				goto FoundIt;
		}
	}

	/* So sad.. nothing found */
	return(FILENOTFOUND);

FoundIt:
	/* Take from tail of cache queue */
	Dp = RootEnt.Back;
	strncpy(Dp->FName,FName,11);
	ExtRecLen = *(Chp+1) & 0xff;
	Dp->Fattr = (*(Chp+Flagsoff) & DIR) ? _A_SUBDIR : _A_RDONLY;
	Dp->FTime = ToDosTime(Chp + Dateoff);
	Dp->FDate = ToDosDate(Chp + Dateoff);
	Dp->BlkNo = *((unsigned long *)(Chp + Blkoff)) + ExtRecLen;
	Dp->FSize = *((unsigned long *)(Chp + Sizeoff));
	Dp->ParentBlk = pDp->BlkNo;

	/* Move Dp to front of cache queue */
	ToFront(Dp);

	*cDpp = Dp;
	return(0);
}

void ToFront(struct DirEnt *Dp) /* Move cache entry to front */
{
	Dp->Forw->Back = Dp->Back;		/* Unlink */
	Dp->Back->Forw = Dp->Forw;

	Dp->Forw = RootEnt.Forw;		/* Link in after RootEnt */
	RootEnt.Forw = Dp;
	Dp->Back = Dp->Forw->Back;
	Dp->Forw->Back = Dp;
}

void ToHex(unsigned Num) /* Output number in hex (brute force) */
{
	char XBuf[8];
	int i;

	XBuf[0] = (Num >> 12U) & 0x0f;
	XBuf[1] = (Num >> 8U) & 0x0f;
	XBuf[2] = (Num >> 4U) & 0x0f;
	XBuf[3] = Num & 0x0f;
	for (i = 0; i < 4; i++)
		XBuf[i] = (XBuf[i] < 9) ? '0' + XBuf[i] : 'a' + XBuf[i] - 10;
	XBuf[4] = '\0';
	strcat(XBuf,"$");
	MsgOut(XBuf);
}
