/********************************************************************
 *
 *    ST01.C
 *
 *    Copyright 1991 by Jim Harper.
 *    Simple SCSI tranport TSR for CDROM.C
 *    usage:  ST01 [-b baseseg]
 *			  baseseg values are:
 *                de00, ce00, c800 and ca00
 *    To unload, type ST01 -u
 *
 ********************************************************************
 *    Written in Microsoft QuickC 2.5.  Should work with C 6.0.
 *    Tabstop = 4
 *    To compile:  qcl st01.c
 ********************************************************************/
#pragma check_stack(off)
#pragma pack(1)

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <memory.h>
#include "scsi.h"

#define STACKSIZE	64
#define CLOCKINT	0x1c
#define SetC(X)  	(X) |=  0x01
#define ClrC(X)  	(X) &= ~0x01

/* Globals */
extern unsigned _psp,       /* Runtime gives us these */
				end;

union REGS regs;
struct SREGS sregs;

unsigned ss,
		 ds,
		 ProgSize;

unsigned _AX,
		 _BX,
		 _CX,
		 _DX,
		 _DS,
		 _FLAGS;

unsigned volatile Timer1;

struct IntRegs {
	unsigned ES;
	unsigned DS;
	unsigned DI;
	unsigned SI;
	unsigned BP;
	unsigned SP;
	unsigned BX;
	unsigned DX;
	unsigned CX;
	unsigned AX;
	unsigned IP;
	unsigned CS;
	unsigned FLAGS;
};

int Active = 0;

/* Protos */
void MsgOut(char *),
	 DoScsi(void),
	 SetPsp(unsigned),
	 DeInstall(void),
	 Delay(void);

unsigned GetPsp(void);

void (_interrupt _far *OldClk)(void);
void _interrupt _far NewClk(void);
void (_interrupt _far *Old2F)(void);
void _interrupt _far New2F(struct IntRegs);

char _far *AppStk,
	 _far *Base = (char _far *) 0xca000000L;

char _far *Datap,
	 _far *Endp;

/* Pointers to the memory mapped I/O ports */
char volatile _far *DataPort;
char volatile _far *RegPort;
char _far *TsrStk;

#define REGISTERS	0x1a00L
#define DATA		0x1c00L
#define RAMSTART	0x1800L
#define RAMEND		0x19feL
#define HZ			18U			/* Clock ticks/sec */


/*
 *	Important masks.
 */

/* Status register */
#define BUSYBIT		0x01
#define MSG			0x02
#define IO			0x04
#define CD			0x08
#define REQ			0x10
#define PERROR		0x40
#define ARBITDONE	0x80

/* Control Register */
#define RESET		0x01
#define SEL			0x02
#define BUSY		0x04
#define ATTN		0x08
#define ARBITSTART	0x10
#define PENABLE		0x20
#define INTENABLE	0x40
#define BUSENABLE	0x80

/* Phase Definitions */
#define BUSFREE		0x00
#define DATAOUT		(BUSYBIT | REQ                )
#define COMMAND		(BUSYBIT | REQ |       CD     )
#define DATAIN		(BUSYBIT | REQ |            IO)
#define STATUS		(BUSYBIT | REQ |       CD | IO)
#define MSGOUT		(BUSYBIT | REQ | MSG | CD     )
#define MSGIN		(BUSYBIT | REQ | MSG | CD | IO)

/* Phase Mask */
#define PHASEMASK	(BUSYBIT | REQ | MSG | CD | IO)

main(int argc, char *argv[])
{
	unsigned _far *EnvBlkp;
	int i,
		Junk;

	if (argc == 2 && strcmp(argv[1],"-u") == 0) {
		regs.h.ah = FUNCID;
		regs.h.al = DEINSTALL;
		int86x(INT2F,&regs,&regs,&sregs);
		exit(0);
	}
	if (argc == 3 && strcmp(argv[1],"-b") == 0) {
		FP_SEG(Base) = (unsigned)strtol(argv[2],NULL,16);
		if (Base != (char _far *) 0xde000000L &&
				Base != (char _far *) 0xce000000L &&
					Base != (char _far *) 0xc8000000L &&
						Base != (char _far *) 0xca000000L) {
			MsgOut("Invalid base address.\r\n$");
			MsgOut("Allowed values are: de00, ce00, c800 and ca00.\r\n$");
			exit(1);
		}
	}
			
	/* Set up port pointers */
	DataPort = Base + DATA;
	RegPort = Base + REGISTERS;

	/* Use on-board RAM for the stack */
	TsrStk = Base + RAMSTART;

	/* Is the RAM there? */
	for (i = 0; i < 128; i++)
		TsrStk[i] = i;
	for (i = 0; i < 128; i++)
		if ((TsrStk[i] & 0xff) != i) {
			MsgOut("ST01 may not be present, or is at incorrect address.\r\n$");
			MsgOut("TSR not installed.\r\n$");
			exit(1);
		}
	TsrStk = Base + RAMEND;
	
	/* Get our stack and data segments */
	segread(&sregs);
	ss = sregs.ss;
	ds = sregs.ds;

	/* Already installed? */
	regs.h.ah = FUNCID;
	regs.h.al = INSTALLCHK;
	int86x(INT2F,&regs,&regs,&sregs);
	if (regs.h.al == 0xff) {
		MsgOut("Already installed.\r\n$");
		exit(1);
	}

	/* Program size in paragraphs w/o a heap */
	ProgSize = (ss + (((unsigned)&end) >> 4)) - _psp + 1;

	/* Grab multiplex interrupt */
	Old2F = _dos_getvect(INT2F);
	_dos_setvect(INT2F,New2F);

	/* Grab clock interrupt */
	OldClk = _dos_getvect(CLOCKINT);
	_dos_setvect(CLOCKINT,NewClk);

	/* Free the environment */
	FP_SEG(EnvBlkp) = _psp;
	FP_OFF(EnvBlkp) = 0x2c;
	_dos_freemem(*EnvBlkp);

	/* Shrink our program size */
	_dos_setblock(ProgSize,_psp,&Junk);

	/* TSR ourself */
	_dos_keep(0,ProgSize);
}

void _interrupt _far New2F(struct IntRegs IntRegs)
{
	/* See if we handle this function */
	if ((IntRegs.AX >> 8U) != FUNCID || Active)
		_chain_intr(Old2F);

	/* Install check?? */
	if ((IntRegs.AX & 0xff) == INSTALLCHK) {
		IntRegs.AX = 0x00ff;
		return;
	}

	/* Set flag saying we're active */
	Active++;

	/* Save needed regs from stack */
	_AX = IntRegs.AX;
	_BX = IntRegs.BX;
	_CX = IntRegs.CX;
	_DX = IntRegs.DX;
	_DS = IntRegs.DS;
	_FLAGS = IntRegs.FLAGS;

	/* Switch to own stack */
	_asm
	{
		cli							; Interrupts off
		mov  WORD PTR AppStk[0],sp	; Save app stack
		mov  WORD PTR AppStk[2],ss
		mov  sp,WORD PTR TsrStk[0]	; Load new stack
		mov  ss,WORD PTR TsrStk[2]
		sti							; Interrupts on
	}

	switch(_AX & 0xff) {
		case DEINSTALL:
			DeInstall();
			break;
		case DORESET:
			*RegPort = (RESET | BUSENABLE);
			for (Timer1 = HZ; Timer1;)
				;
			*RegPort = 0x00;
			ClrC(_FLAGS);
			break;
		case DOCMD:
			DoScsi();
			break;
	}

	/* Switch back to app stack */
	_asm
	{
		cli							; Interrupts off
		mov  sp,WORD PTR AppStk[0]	; Load app stack
		mov  ss,WORD PTR AppStk[2]
		sti							; Interrupts on
	}

	/* Restore (possibly modifed) registers */
	IntRegs.AX = _AX;
	IntRegs.BX = _BX;
	IntRegs.CX = _CX;
	IntRegs.DX = _DX;
	IntRegs.FLAGS = _FLAGS;

	/* Clear Active Flag */
	Active = 0;
}

void _interrupt _far NewClk(void)
{
	if (Timer1)
		Timer1--;
	_chain_intr(OldClk);
}

void DoScsi(void)
{
	struct Cmd _far *Cmdp;
	unsigned Phase,
			 NumBytes = 512,
			 Byte = 0,
			 i;

	FP_SEG(Cmdp) = _DS;
	FP_OFF(Cmdp) = _DX;

	FP_SEG(Datap) = Cmdp->DSeg;
	FP_OFF(Datap) = Cmdp->DOff;
	Endp = Datap;
	Cmdp->Count = 0L;

	/* Clear control reg */
	*RegPort = 0x00;

retry:
	/* Bus has gotta be free */
	if ((*RegPort & BUSYBIT) != 0x00) {
		*RegPort = (RESET | BUSENABLE);
		for (Timer1 = HZ; Timer1;)
			;
		*RegPort = 0x00;
		for (Timer1 = HZ * 15; Timer1;)
			;
		goto retry;
	}

	/* Clear control reg */
	*RegPort = 0x00;

	/* Assert HBA's address */
	*DataPort = 0x80;

	/* Set the arbit bit */
	*RegPort = (ARBITSTART | PENABLE);

	/* Wait for arbit complete */
	for (Timer1 = HZ * 3; (*RegPort & ARBITDONE) == 0x00;) {
		if (!Timer1) {
			Cmdp->Stat = 0x81;
			return;
		}
		Delay();
	}

	/* OR the target & our id bits into the data reg */
	*DataPort = 0x80 | (0x01 << (unsigned)Cmdp->ID);

	/* Assert SEL, bus enable, deassert arbit */
	*RegPort = (SEL | PENABLE | BUSENABLE);

	/* Wait for BUSY */
	for (Timer1 = HZ * 3; (*RegPort & BUSYBIT) == 0x00;) {
		if (!Timer1) {
			*RegPort = 0x00;
			Cmdp->Stat = 0x82;
			return;
		}
	}

	/* Drop Select */
	*RegPort = (PENABLE | BUSENABLE);

	Cmdp->Stat = 0x00;
	for (Timer1 = HZ * 60; Timer1;) {		/* Cmd must complete in 10s */
		switch(*RegPort & PHASEMASK) {
			case COMMAND:
				do {
					*DataPort = Cmdp->CDB[Byte++];
				} while ((*RegPort & PHASEMASK) == COMMAND);
				break;
			case DATAIN:
				do {
					*Datap++ = *DataPort;
				} while ((*RegPort & PHASEMASK) == DATAIN);
				break;
			case DATAOUT:
				do {
					*DataPort = *Datap++;
				} while ((*RegPort & PHASEMASK) == DATAOUT);
				break;
			case STATUS:
				Cmdp->Stat = *DataPort;
				break;
			case MSGIN:
				Cmdp->Sense = *DataPort;
				break;
			case BUSFREE:
				*RegPort = 0x00;
				Cmdp->Count += Datap - Endp;
				return;
			default:
				/* Short delay */
				Delay();
		}
	}
	Cmdp->Stat = 0x84;
	return;
}

void DeInstall(void)
{
	unsigned AppPsp;

	/* See if anyone grabbed our vectors */
	if (_dos_getvect(INT2F) != New2F || _dos_getvect(CLOCKINT) != NewClk) {
		MsgOut("st01 driver can't deinstall.\r\n$");
		_AX = 0xffff;
		SetC(_FLAGS);
		return;
	}

	/* Restore interrupts */
	_dos_setvect(INT2F,Old2F);
	_dos_setvect(CLOCKINT,OldClk);

	/* Get foreground's psp */
	AppPsp = GetPsp();
	/* Set current psp to ours */
	SetPsp(_psp);
	/* Free our prog */
	_dos_freemem(_psp);
	/* Restore foregound psp */
	SetPsp(AppPsp);
	MsgOut("st01 driver uninstalled.\r\n$");
	_AX = 0x0000;
	ClrC(_FLAGS);
}

void MsgOut(char *Msg)
{
	regs.h.ah = 0x09;
	sregs.ds = ds;
	regs.x.dx = (unsigned)Msg;
	intdos(&regs,&regs);
}

unsigned GetPsp(void)
{
	regs.h.ah = 0x62;
	intdos(&regs,&regs);
	return(regs.x.bx);
}

void SetPsp(unsigned Psp)
{
	regs.h.ah = 0x50;
	regs.x.bx = Psp;
	intdos(&regs,&regs);
}

void Delay(void) { }
