/*  DATECONV.C
 *  Universal date conversion functions
 *  Last mod.: 1992-10-10
 */

#include <STDLIB.H>     /*  for labs()  */
#include <CTYPE.H>
#undef toupper          /*  so as to use function, not macro  */

#include "DATECONV.H"

int month_length[14]
  = {
    0, 31, 28, 31, 30, 31, 30,
    31, 31, 30, 31, 30, 31, 0
    };
/*  months 0 and 13 have 0 days  */

/*  the calendar parameter is always 'G' or 'J'
 *  G = Gregorian, J = Julian
 */

/*-------------------------*/
int is_leap_year_c(long year,
                   char calendar)
{
calendar = (char)toupper((int)calendar);

if ( year%4 )           /*  if year not divisible by 4  */
    return ( FALSE );
else
    {
    if ( calendar == 'J' )
        return ( TRUE );
    else    /*  calendar == 'G' */
        return ( ( year%100 != 0L || year%400 == 0L ) ?
                 TRUE : FALSE );
    }
}

/*-----------------------*/
int is_leap_year(long year)
{
return ( is_leap_year_c(year,'G') );
}

/*------------------------*/
void set_feb_length(long yr,
                    char calendar)
{
month_length[2] = 28 + is_leap_year_c(yr,calendar) ;
}

/*-----------------------*/
void reset_feb_length(void)
{
month_length[2] = 28;
}

/*  function to convert date to Gregorian day number
 *  sets valid flag to FALSE if date invalid, otherwise
 *  returns the number of days before or after the day
 *  the Gregorian calendar came into effect (15-OCT-1582)
 */
/*----------------------*/
void date_to_gdn(Date *dt,       /*  dt is a pointer to a structure  */
                 char calendar)  /*  'G' or 'J'  */
{
int day = dt->day;
int month = dt->month;
long year = dt->year;
long gdn;

calendar = (char)toupper((int)calendar);

set_feb_length(year,calendar);

if ( month < 1  || month > 12
        || day < 1  || day > month_length[month] )
    dt->valid = FALSE;
else
    {
    /*  calculate the number of days before or after
     *  October 15, 1582 (Gregorian)
     */
    gdn = (year-1)*365 + lfloor(year-1,4L);
    if ( calendar == 'G' )
        gdn += lfloor(year-1,400L) - lfloor(year-1,100L);

    while (--month)
        gdn += month_length[month];
    gdn += day - 577736L - 2*(calendar=='J');

    dt->gdn = gdn;
    dt->valid = TRUE;
    }

reset_feb_length();
}

/*  function to convert gregorian day number to date  */
/*----------------------*/
void gdn_to_date(Date *dt,
                 char calendar)
{
int month, i, exception;
long year, gdn, y4, y100, y400;

calendar = (char)toupper((int)calendar);

gdn = dt->gdn;
gdn += 577735L + 2*(calendar=='J');

y400 = 146100L - 3*(calendar=='G');
y100 =  36525L -   (calendar=='G');
y4   =   1461L;

exception = FALSE;
year = 400*lfloor(gdn,y400);        /*  400-year periods  */
gdn -= y400*lfloor(gdn,y400);
if ( gdn > 0L )
    {
    year += 100*lfloor(gdn,y100);   /*  100-year periods  */
    gdn -= y100*lfloor(gdn,y100);
    exception = ( gdn == 0L && calendar == 'G' );
    if ( gdn > 0L )
        {
        year += 4*lfloor(gdn,y4);   /*  4-year periods  */
        gdn -= y4*lfloor(gdn,y4);
        if ( gdn > 0L )
            {
            i = 0;
            while ( gdn > 365 && ++i < 4 )
                {
                year++;
                gdn -= 365L;
                }
            }
        }
    }

if ( exception )
    gdn = 366L;
   /*  occurs once every hundred years with Gregorian calendar  */
else
    {
    year++;
    gdn++;
    }

set_feb_length(year,calendar);
month = 1;
while ( month < 13 && gdn > month_length[month] )
    gdn -= month_length[month++];

if ( month == 13 )
    {
    month = 1;
    year++;
    }

reset_feb_length();

dt->day = (int)gdn;
dt->month = month;
dt->year = year;
dt->valid = TRUE;
}

/*---------------*/
long lfloor(long a,
            long b)               /*  assumes b positive  */
{
return ( a >= 0L ? a/b : ( a%b == 0L ) - 1 - labs(a)/b );
/*  labs() returns the absolute value of its long int argument  */
}

/*  returns day of the week for a given
 *  Gregorian day number;
 *  0 = Sunday, 6 = Saturday
 */
/*---------------------*/
int day_of_week(long gdn)
{
return ((int)(((gdn%7)+12)%7));
}
