;----------------------------------------------------------
;
;   FLASHDEV.ASM  created 02/16/93  by Drew Gislason
;
;   FAT-Flash Disk Driver
;   Dr. Dobbs Journal, May 1993
;
;   This Flash Disk Driver is designed as a Write Once
;   Read Many (WORM) FAT-style disk driver.  Normal DOS
;   commands such as COPY, DIR, and TYPE will work as
;   expected.  DOS commands like RENAME, ERASE and
;   "ECHO text >>file" will fail.
;
;   The 1st part of the device driver is generic to all
;   disk devices.  It could be used as a template for
;   any FAT-style disk device driver.
;
;   The 2nd part of the device driver actually reads and
;   programs Flash memory.  It is written for Intel Flash
;   memory, but could be "ported" to understand almost
;   any Flash memory mechanism.
;
;----------------------------------------------------------


;----------------------------------------------------------
;
;	PART 1 - GENERIC DISK DRIVER CODE
;
;   This generic disk driver relies on a number of routines
;   to read and write Flash memory.  All of these routines
;   could be rewritten to accomodate _ANY_ kind of Flash
;   memory, or even ROM, RAM or an actual disk for that
;   matter.
;
;   The routines that MUST be supplied to make this generic
;   driver into a working Flash Disk Driver are defined (in
;   C syntax) as follows:
;
;   bool meminit(char far *cmdline, unsigned *endseg);
;   int  memchanged(void);
;   bool memread(long offset, unsigned len, 
;				char far * buffer);
;   bool memwrite(long offset, unsigned len,
;				char far * buffer);
;
;   Note that all the routines follow C conventions: that
;   is, the last item "pushed" on the stack is the 1st
;   argument, and each routine MUST preserve SI,DI,BP and
;   all segment registers besides ES.
;
;----------------------------------------------------------


;------------------------------------------------
;  Device Requests
;
;  When DOS calls the device driver, the "request"
;  (pointed to by ES:BX) may contain some or all of
;  the following fields:
;
REQ_CMDLEN	equ	00H	; length of command
REQ_UNIT	equ	01H	; unit # for disks
REQ_CMD		equ	02H	; the command itself
REQ_STATUS	equ	03H	; status word
REQ_MEDIA	equ	0DH	; media byte (disks)
REQ_TRANS	equ	0EH	; transfer address
REQ_COUNT	equ	12H	; # sectors for rd/wr
REQ_START	equ	14H	; start sector for rd/wr
REQ_DRIVE	equ	16H	; drive # (0=A, 1=B, etc)


;------------------------------------------------
;  Device Error Codes
;
;  If the device must return an error, the following
;  list defines the possible error conditions:
;
ERR_WRITEPROT		equ	   00H
ERR_BADUNIT		equ	   01H
ERR_NOTREADY		equ	   02H
ERR_BADCMD		equ	   03H
ERR_CRC			equ	   04H
ERR_BADSTRUCT		equ	   05H
ERR_SEEK		equ	   06H
ERR_BADMEDIA		equ	   07H
ERR_SECTORNOTFOUND	equ	   08H
ERR_NOPAPER		equ	   09H
ERR_WRITEFAULT		equ	   0AH
ERR_READFAULT		equ	   0BH
ERR_GENERALFAIL		equ	   0CH


;------------------------------------------------
;  C Function Arguments
;
;  In small CODE models (that is, where CS never
;  changes), the 1st argument begins at 4 bytes
;  above the Base Pointer.
;
;  +--------+
;  | Return |
;  +--------+
;  |   BP   |
;  +--------+
;
P	equ	4


;------------------------------------------------
;  Segment Ordering
;
;  SEGMENT ORDERING is defined here for convenience.
;
;  The linker gathers segments in the order it first
;  encounters them.  By defining the segment ordering
;  immediately, the segments will automatically appear
;  in the correct order as they do here.
;
;  The segment alignment and combine type are specified
;  here to simplify segment directives.
;
_TEXT	segment	byte public 'CODE'	; Code
	assume CS:_TEXT
_TEXT	ends

_DATA	segment word public 'DATA'	; Initialized data
_DATA	ends

_BSS	segment word public 'BSS'	; Uninitialized data
	DGROUP group _DATA, _BSS
	assume DS:DGROUP
_BSS	ends

_ENDDEV	segment para public 'ENDDEV'	; End of device driver
_ENDDEV	ends


;------------------------------------------------
;   Device Header
;
;   Every DOS device driver, whether for Character or
;   Disk devices, MUST begin with a device header.
;
_TEXT	segment

	public	flash_header
flash_header:
	dw	-1, -1		; Pointer to next header, set by DOS
	dw	0000H		; Disk device, not removable
	dw	strategy	; Strategy routine
	dw	service		; Service routine
maxdrv	db	1,0		; For future expansion

	; DOS makes "requests" to device driver to perform
	; the following actions...
cmdtbl:
	dw	flash_init	; 0 - initialize device
	dw	flash_media	; 1 - detect media change
	dw	flash_bpb	; 2 - build a bpb
	dw	dev_cmd_err	; 3 - ioctl input (not implemented)
	dw	flash_read	; 4 - read from device
	dw	dev_cmd_err	; 5 - non-destructive read (unused)
	dw	dev_cmd_err	; 6 - input status (not implemented)
	dw	dev_cmd_err	; 7 - input flush (not implemented)
	dw	flash_write	; 8 - write to device
	dw	flash_write	; 9 - write to device w/ verify
cmdend:

_TEXT	ends


;------------------------------------------------
;   Device Strategy and Service Routines
;
;   DOS makes two calls to a device driver for each
;   request, one to the "strategy" routine and one to
;   the service routine.
;
;   The first call, to the device driver's "strategy"
;   routine, was intended for multi-tasking that was
;   never implemented in DOS and so, can be ignored.
;
;   The second call, to the device driver's "service"
;   routine, does the real work.
;
_BSS	segment
	public	_dos_req_ptr
	_dos_req_ptr	DD	?

	stack_bottom	DW	512 dup (?)	; 1K stack
	local_stack	DW	?

	org_stack	DD	?
_BSS	ends

_TEXT	segment
	public strategy
strategy:
	retf

	public service
service:
	; save all registers we might change
	pushf
	push	AX
	push	BX
	push	CX
	push	DX
	push	SI
	push	DI
	push	DS
	push	ES

	; save current stack
	mov	DX, DGROUP
	mov	DS, DX
	mov	WORD PTR DGROUP:org_stack, SP
	mov	WORD PTR DGROUP:org_stack+2, SS

	; switch to our stack
	mov	AX, OFFSET DGROUP:local_stack
	cli
	mov	SS, DX
	mov	SP, AX
	sti

	; ES:BX points to "request" from DOS, save it
	mov	word ptr [_dos_req_ptr], BX
	mov	word ptr [_dos_req_ptr+2], ES

	; load some basics that ALL commands use into registers
	mov	CX, word ptr ES:[BX+REQ_COUNT]	; CX = count
	mov	DX, word ptr ES:[BX+REQ_START]	; DX = start sector
	mov	AL, byte ptr ES:[BX+REQ_CMD]	; command code
	cmp	AL, (OFFSET cmdend - OFFSET cmdtbl)/2
	jae	dev_cmd_err			; bad command
	cbw
	shl	AX, 1				; get index into cmd table
	mov	SI, offset cmdtbl
	add	SI, AX
	les	DI, dword ptr ES:[BX+REQ_TRANS]	; ES:DI = transfer buffer

;	we've set up the following:
;	--------------------------
;	DS	= DGROUP
;	CX	= count
;	DX	= start sector
;	ES:DI	= transfer area

	; do the command
	jmp	word ptr CS:[SI]


	; Unknown command error
	public	dev_cmd_err
dev_cmd_err:
	mov	AL, 3

	; Mark error status, error code is in AL
	public	dev_err_exit
dev_err_exit:
	mov	AH, 81H
	jmp	SHORT err1

	; Normal exits (no error) occur through here
	public	dev_exit
dev_exit:
	mov	AH, 01H		; done bit set

	; save status (error or not) in "request" header from DOS
err1:	les	BX, [_dos_req_ptr]
	mov	WORD PTR ES:[BX+REQ_STATUS], AX

	; restore original stack
	mov	AX, WORD PTR DGROUP:org_stack
	mov	DX, WORD PTR DGROUP:org_stack+2
	cli
	mov	SS, DX
	mov	SP, AX
	sti

	; restore all registers we may have changed
	pop	ES
	pop	DS
	pop	DI
	pop	SI
	pop	DX
	pop	CX
	pop	BX
	pop	AX
	popf
	retf

_TEXT	ends


;------------------------------------------------
;  Device Data Area
;
;  The device driver requires
;
_BSS	segment

	; if an error occurs, what is it?
	_memerr	db	?

	PUBLIC	start_bpb
	start_bpb	LABEL BYTE
	sec_size 	DW	?	; bytes per sector
	sec_per_clus	DB	?	; sectors/cluster
	res_sec		DW	?	; reserved sectors
	num_fat		DB	?	; number of fats (2 for a floppy)
	dir_ent		DW	?	; number of directory entries
	num_sec		DW	?	; total number of sectors
	media_desc	DB	?	; media descriptor
	sec_per_fat	DW	?	; sectors per fat
	sec_per_trak	DW	?	; sectors/track
	heads		DW	?	; number of heads
	spec_res	DW	?	; reserved for disk partitioning
	end_bpb		LABEL BYTE

_BSS	ends

_DATA	segment

	PUBLIC	bpbptr
	bpbptr	DW	offset DGROUP:start_bpb

_DATA	ends


;------------------------------------------------
;  Initialize Device (COMMAND=0)
;
;  The "request" made by DOS defines the following
;  fields:
;
;  On Entry
;	+-------------------------+
;	| 13-Byte header          |	* set by DOS
;	+-------------------------+
;	| BYTE   Number of units  |	(undefined)
;	+-------------------------+
;	| DWORD  End Address      |	(undefined)
;	+-------------------------+
;	| DWORD  Cmdline Pointer  |	* ptr to "DEVICE=" command line
;	+-------------------------+
;	| BYTE   Block Device #   |	* this disk is... (0=A, 1=B, etc)
;	+-------------------------+
;
;	device= \dev\name.sys /1
;	       ^
;	Cmdline points here, string ends in a CR (0DH) or LF (0AH)
;
;  On Exit
;	+-------------------------+
;  00H	| 13-Byte header          |	(set by DOS on entry)
;	+-------------------------+
;  0DH	| BYTE   Number of units  |	* set by this driver
;	+-------------------------+
;  0EH	| DWORD  End Address      |	* set by this driver
;	+-------------------------+
;  12H	| DWORD  Ptr to BPB array |	* set by this driver
;	+-------------------------+
;  16H	| BYTE   Block Device #   |	(set by DOS on entry)
;	+-------------------------+
;
_DATA	segment
	endseg	dw	SEG _ENDDEV
_DATA	ends

_TEXT	segment
	public flash_init
flash_init:

	; initialze the Flash memory hardware
	; meminit(char far * cmdline, unsigned *endseg)
	mov	AX, offset DGROUP:endseg
	push	AX
	les	BX, [_dos_req_ptr]
	push	WORD PTR [BX+REQ_COUNT+2]
	push	WORD PTR [BX+REQ_COUNT]
	call	_meminit
	add	SP, 6
	or	AX, AX
	jz	init_err

	; read the BPB from Flash memory into device driver BPB
	call	readbpb
	or	AX, AX
	jz	init_err

	; set up the i/o packet
	les	BX, _dos_req_ptr
	mov	BYTE PTR ES:[BX+REQ_MEDIA], 1		; 1 unit (drive)
	mov	WORD PTR ES:[BX+REQ_COUNT+2], DS	; point to BPB list
	mov	WORD PTR ES:[BX+REQ_COUNT], OFFSET DGROUP:bpbptr
	mov	AX, word ptr endseg
	mov	WORD PTR ES:[BX+REQ_TRANS], 0
	mov	WORD PTR ES:[BX+REQ_TRANS+2], AX
	jmp	dev_exit

	; error occured during init, fail
init_err:
	mov	AL, ERR_GENERALFAIL
	jmp	dev_err_exit


;------------------------------------------------
;  Detect Media Change (COMMAND 1)
;
;  Has the flash memory been changed somehow?  (A
;  PCMCIA card might be pulled out and swapped for
;  another).
;
;  int memchanged(void)
;  Returns -1 if Device HAS changed
;  Returns  0 if Device MAY have changed
;  Returns  1 if Device has NOT changed
;
	public	flash_media
flash_media:
	call	_memchanged
	les	BX, [_dos_req_ptr]
	mov	byte ptr [BX+REQ_TRANS], AL
	jmp	dev_exit


;------------------------------------------------
;  Build a BPB (COMMAND 2)
;
;  DOS makes this "request" when it believes the disk
;  has or may have been changed.  This will detect
;
	public	flash_bpb
flash_bpb:
	; read the BPB from Flash memory into device driver BPB
	call	readbpb 
	or	AX, AX
	jnz	gotbpb

	; we failed to read BPB
	mov	AL, _memerr
	jmp	dev_err_exit

	; return BPB ptr in request header
gotbpb:	les	BX, [_dos_req_ptr]
	mov	WORD PTR ES:[BX+REQ_COUNT], offset DGROUP:start_bpb
	mov	WORD PTR ES:[BX+REQ_COUNT+2], DS
	jmp	dev_exit

readbpb:
	; read the BPB from the disk into our structure
	; memread(long offset, unsigned len, char far * buffer)
	push	DS
	push	bpbptr
	mov	AX, (OFFSET end_bpb - OFFSET start_bpb)
	push	AX
	xor	DX, DX
	mov	AX, 11		; offset 11L (start of BPB on disk)
	push	DX
	push	AX
	call	_memread
	add	SP, 10
	ret


;------------------------------------------------
;  Read from Device (COMMAND 4)
;
;  Read from Flash memory.  DOS "requests" that
;  this device driver read 1 or more sectors from
;  the Flash memory.  The device driver, in turn,
;  calls "memread" to do the real work.
;
;  Upon Entry
;	DS	= ROM-DOS data segment
;	ES:DI	= buffer data
;	CX	= # of sectors to read
;	DX	= starting sector
;
	public flash_read
flash_read:
	mov	SI, OFFSET _memread

	; This generic read/write routine is passed a
	; pointer to the read or write routine in "SI"
rdwr:
	; convert # of sectors to # of bytes
	push	DX
	mov	BX, OFFSET DGROUP:start_bpb
	mov	AX, [BX]	; sector size (512)
	mul	CX
	mov	CX, AX		; CX = bytes to transfer
	pop	DX

	; calculate starting offset
	mov	AX, [BX]	; sector size (512)
	mul	DX		; DX:AX = offset (in bytes)

	; call memread(offset,len,buffer) to do the work
	push	ES
	push	DI
	push	CX
	push	DX
	push	AX
	call	SI		; call read or write routine
	add	SP, 10
	or	AX, AX
	jz	rdwr_failed
	mov	AL, 0
	jmp	dev_exit

rdwr_failed:
	mov	AL, byte ptr _memerr
	jmp	dev_err_exit


;------------------------------------------------
;  Write to Device (COMMAND 8 or 9)
;
;
;  Write to Flash memory.  DOS "requests" that
;  this device driver write 1 or more sectors to
;  the Flash memory.  The device driver, in turn,
;  calls "memwrite" to do the real work.
;
;  Upon Entry
;	DS	= ROM-DOS data segment
;	ES:DI	= buffer data
;	CX	= # of sectors to read
;	DX	= starting sector
;
	public	flash_write
flash_write:
	mov	SI, OFFSET _memwrite
	jmp	short rdwr
	
_TEXT	ENDS


;----------------------------------------------------------
;
;	PART 2 - FLASH MEMORY I/O
;
;   The actual code in this device driver that interacts
;   with the Flash memory is confined to this section.
;
;   This code will write once to the Flash memory, but
;   cannot erase the memory.  On startup, it assumes the
;   Flash memory is either completely erased or contains
;   a valid Flash Disk image.
;
;   When writing to or reading from the disk, all bits
;   are inverted. 
;
;   The program runs "as is" with the SeaLevel System's
;   PROM III memory card containing a 28F010 (128K x 8)
;   Intel Flash chip.
;
;----------------------------------------------------------

_BSS	SEGMENT

	; delays for flashing/writing Flash memory
	public	_delay_10mil, _delay_10mic
	_calc_ticks	dw	?
	_delay_10mil	dw	? ; 10 * 1/1000 second
	_delay_10mic	dw	? ; 10 * 1/1,000,000 second

_BSS	ENDS

_TEXT	SEGMENT

;---------------------------------------------------------
;  void interrupt (far * getvect(int num))();
;
_getvect PROC	NEAR
	push	BP
	mov	BP, SP
	mov	BX, word ptr [BP+4]
	shl	BX, 1
	shl	BX, 1
	xor	AX, AX
	mov	ES, AX		; ES:BX pts to int vector
	cli
	mov	AX, ES:[BX]
	mov	DX, ES:[BX+2]
	sti
	pop	BP
	ret
_getvect	ENDP

;---------------------------------------------------------
;  void setvect(int num, void interrupt (far * isr)());
;
_setvect PROC	NEAR
	push	BP
	mov	BP, SP
	mov	BX, word ptr [BP+4]
	shl	BX, 1
	shl	BX, 1
	xor	AX, AX
	mov	ES, AX		; ES:BX pts to int vector
	mov	AX, [BP+6]
	mov	DX, [BP+8]	; DX:AX ptr to new vector
	cli
	mov	ES:[BX], AX
	mov	ES:[BX+2], DX
	sti
	pop	BP
	ret
_setvect	ENDP

;---------------------------------------------------------
;  void interrupt tick_interrupt(void);
;
;  Used exclusively by calc_delay().  Simply increments a
;  tick count.
;	
_tick_interrupt	PROC	FAR
	push	AX
	push	DS
	mov	AX, DGROUP
	mov	DS, AX
	inc	WORD PTR _calc_ticks
	pop	DS
	pop	AX
	iret
_tick_interrupt	endp

;---------------------------------------------------------
;  void calc_delay(void);
;
;  Calculate a delay iteration value for 10 milliseconds
;  and 10 microseconds.  Used for timing Flash memory
;  writes.
;	
	public	_calc_delay
_calc_delay	proc	near
	push	BP
	mov	BP, SP
	sub	SP, 8
   ;	
   ;	{
   ;		unsigned long totalcount = 0;
   ;	
   	xor	AX, AX
	mov	WORD PTR [BP-2], AX
	mov	WORD PTR [BP-4], AX
   ;	
   ;	void interrupt tick_interrupt(void);
   ;	void (interrupt far *oldvect)(void);
   ;	
   ;	/* set up our new interrupt vector */
   ;	calc_ticks = 0;
   ;	oldvect = getvect(0x1c);
   ;	
	mov	_calc_ticks, AX
	mov	AX, 1CH
	push	AX
	call	_getvect
	pop	CX
	mov	WORD PTR [BP-6], DX
	mov	WORD PTR [BP-8], AX
   ;	
   ;	setvect(0x1c,tick_interrupt);
   ;	
	push	CS
	mov	AX, offset _tick_interrupt
	push	AX
	mov	AX, 1CH
	push	AX
	call	_setvect
	add	SP, 6
	jmp	short calc1
calc0:
   ;	
   ;	/* wait for 9 ticks (approx 1/2 second) */
   ;	while(calc_ticks < 9)
   ;	{
   ;	   delay(0);
   ;	   ++totalcount;
   ;	}
   ;	
	mov	AX, 1
	push	AX
	call	_delay
	pop	CX
	add	WORD PTR [BP-4], 1
	adc	WORD PTR [BP-2], 0
calc1:	cmp	_calc_ticks, 10
	jb	short calc0
   ;	
   ;	setvect(0x1c,oldvect);
   ;	
	push	WORD PTR [BP-6]
	push	WORD PTR [BP-8]
	mov	AX, 1CH
	push	AX
	call	_setvect
	add	SP, 6
   ;	
   ;	delay_10mil = totalcount / 5;
   ;	if((totalcount % 5) >= 3)
   ;	   ++delay_10mil;
   ;	
   	mov	AX, WORD PTR [BP-4]
   	mov	DX, WORD PTR [BP-2]
   	cmp	DX, 5
	jb	calc2	; totalcount maximum of 4FFFFH
	mov	AX, 0FFFFH 
	mov	DX, 4
calc2:	mov	CX, 5
	div	CX
	mov	_delay_10mil, AX
	cmp	DX, 3
	jb	short calc3
	inc	_delay_10mil
calc3:
   ;	
   ;	delay_10mic = delay_10mil / 1000;
   ;	if((delay_10mil % 1000) >= 500)
   ;	   ++delay_10mic;
   ;	
	xor	DX, DX
	mov	AX, _delay_10mil
	mov	CX, 1000
	div	CX
	mov	_delay_10mic, AX
	cmp	DX, 500
	jb	calc4
	inc	_delay_10mic
calc4:
   ;	
   ;	}
   ;	
	mov	SP, BP
	pop	BP
	ret	
_calc_delay	endp

;---------------------------------------------------------
;  void delay(unsigned count);
;
;  Delay for a small period of time.  The time's involved
;  are milliseconds and microseconds, hence a software loop
;  rather than the timer tick.
;
	public	_delay
_delay	proc	near
	push	BP
	mov	BP, SP
delay:	mov	AX, word ptr [BP+4]
	dec	word ptr [BP+4]
	or	AX, AX
	jnz	delay
	pop	BP
	ret
_delay	endp


;---------------------------------------------------------
;  void invert_bits(char far * buffer, unsigned len);
;
;  Invert a number of bytes - changes all 1 bits to 0s,
;  all 0 bits to 1s (example: changes 41H to BFH).
;
	public	_invert_bits
_invert_bits	PROC	NEAR
	push	BP
	mov	BP, SP
	push	SI
	mov	SI, WORD PTR [BP+8]
	les	BX,DWORD PTR [BP+4]
	jmp	short invert2
   ;	
   ;	{
   ;	   while(len--)
   ;	   {
   ;	   *buffer = ~(*buffer);
   ;	   ++buffer;
   ;	   }
   ;	}
   ;	
invert1:
	not	BYTE PTR ES:[BX]
	inc	BX
invert2:
	mov	AX, SI
	dec	SI
	or	AX, AX
	jne	short invert1
	pop	SI
	pop	BP
	ret	
_invert_bits	ENDP


;---------------------------------------------------------
;  int byte_cmp(char *buff1, char * buff2, int len)
;
	public	_byte_cmp
_byte_cmp	PROC	NEAR
	push	BP
	mov	BP,SP
	push	SI
	push	DI
	mov	SI,WORD PTR [BP+4]
	mov	DI,WORD PTR [BP+6]
	mov	DX,WORD PTR [BP+8]
	jmp	short cmp3
   ;	
   ;	{
   ;	   while(len--)
   ;	   {
   ;	      if(*buff1 != *buff2)
   ;	         return 0;
   ;	      ++buff1;
   ;	      ++buff2;
   ;	   }
   ;	   return 1;
   ;	}
   ;	
cmp1:
	mov	AL, BYTE PTR [SI]
	cmp	AL, BYTE PTR [DI]
	je	short cmp2
	xor	AX, AX
	jmp	short cmp4
cmp2:	inc	SI
	inc	DI
cmp3:	mov	AX, DX
	dec	DX
	or	AX, AX
	jne	short cmp1
	mov	AX, 1
cmp4:	pop	DI
	pop	SI
	pop	BP
	ret	
_byte_cmp	ENDP

_TEXT	ENDS

_DATA	SEGMENT
	cr	DB	13,0
_DATA	ENDS

_TEXT	SEGMENT

;---------------------------------------------------------
;  void bios_puts(char *s);
;
;  Display a NULL terminated string using BIOS INT 10H
;
	public	_bios_puts
_bios_puts	PROC	NEAR
	push	BP
	mov	BP, SP
	push	SI
	mov	SI,WORD PTR [BP+4]
	jmp	short bput3
   ;	
   ;	{
   ;	static char cr[2] = { '\r', 0 };
   ;	while(*s)
   ;	{
   ;	   if(*s == '\n')
   ;	      bios_puts(cr);
   ;
bput1:
	cmp	BYTE PTR [SI], 10
	jne	short bput2
	mov	AX, OFFSET DGROUP:cr
	push	AX
	call	_bios_puts
	pop	CX
bput2:
   ;	
   ;	   _AL = *s;
   ;	   _AH = 0xE;
   ;	   _BX = 7;
   ;	   geninterrupt(0x10);
   ;	
	mov	AL, BYTE PTR [SI]
	mov	AH, 0EH
	mov	BX, 7
	int	10H
   ;	
   ;	   ++s;
   ;	   }
   ;	}
   ;	
	inc	SI
bput3:
	cmp	BYTE PTR [SI],0
	jne	short bput1
	pop	SI
	pop	BP
	ret	
_bios_puts	ENDP


;---------------------------------------------------------
;  Program Flash Device
;
;  This section deals exclusively with programming the
;  Flash device itself.  All the code here is unique to
;  the particular Flash hardware and must be changed if
;  support for other hardware is desired.
;
;  The program runs "as is" with the SeaLevel System's
;  PROM III memory card containing a 28F010 (128K x 8)
;  Intel Flash chip.
;

PAGECMD   equ 0D000H	; Page command byte
PAGESTART equ 0D200H	; Start of paged Flash Memory
PAGESIZE  equ 8*1024	; Page size of Flash Memory in K bytes
SECTORS   equ 256	; 128K = 256 sectors

FLSH_MAX_CYCLES equ	100
FLSH_WRITE	 equ	40H
FLSH_VERIFY	 equ	0C0H
FLSH_RESET	 equ	0FFH
FLSH_READ	 equ	00H

_DATA	SEGMENT
flash_page_cmd	DW	0
		DW	PAGECMD
_flash_disk_seg	DW	PAGESTART
_flash_pagesize	DW	PAGESIZE
_flash_sectors	DW	SECTORS
_flash_BPB	DW	200H	; unsigned   sec_size;
		DB	1	; uchar      sec_per_clus;
		DW	1	; unsigned   res_sec;
		DB	1	; uchar      num_fats;
		DW	64	; unsigned   dir_ent;
		DW	256	; unsigned   num_sec;
		DB	0F0H	; uchar      media;
		DW	1	; unsigned   fat_size;
		DW	0F000H	; unsigned   sec_per_trak;
		DW	2	; unsigned   sides;

	public	_in_root
	_in_root DB	0
_DATA	ENDS

;---------------------------------------------------------
;  void setpage(int page);
;
;  The 128K x 8 Intel Flash memory chip is banked into
;  the 8K address space of D200:0 - D200:1FFF on the
;  Sealevel System's PROM III memory board.  This routine
;  sets the page.
;	
	public _setpage
_setpage	PROC	NEAR
	push	BP
	mov	BP, SP
   ;	
   ;	{
   ;		static unsigned char far * page_cmd = 
   ;			(unsigned char far *)(0xD0000000L);
   ;		*page_cmd = page;
   ;	}
   ;	
	les	BX, dword ptr flash_page_cmd
	mov	AL, BYTE PTR [BP+4]
	mov	BYTE PTR ES:[BX], AL
	pop	BP
	ret	
_setpage	ENDP

;---------------------------------------------------------
;  int program_one_byte(unsigned char far * valptr,
;			unsigned char value);
;
;  This function will program (write) a single byte to
;  the Flash memory.
;	
	public	_program_one_byte
_program_one_byte	PROC	NEAR
	push	BP
	mov	BP, SP
	sub	SP, 6
	push	SI
   ;	
   ;	{
   ;	   unsigned program_count = 0;
   ;	   unsigned cmdseg = ((unsigned long)valptr >> 16);
   ;	
	xor	si,si
	mov	AX, WORD PTR [BP+6]
	mov	WORD PTR [BP-2], AX
   ;
   ;	   unsigned char far * cmdptr = (unsigned 
   ;		char far *)((unsigned long)cmdseg << 16);
   ;
	mov	AX,WORD PTR [BP-2]
	mov	WORD PTR [BP-4],AX
	mov	WORD PTR [BP-6],0
	jmp	short prog1
prog0:
   ;	
   ;	while(program_count++ < FLSH_MAX_CYCLES)
   ;	{
   ;	   *cmdptr = FLSH_WRITE;
   ;	   disable();
   ;	   *valptr = value;
   ;	
	les	BX, DWORD PTR [BP-6]
	mov	BYTE PTR ES:[BX], FLSH_WRITE
	cli
	les	BX, DWORD PTR [BP+4]
	mov	AL, BYTE PTR [BP+8]
	mov	BYTE PTR ES:[BX], AL
   ;	
   ;	   delay(delay_10mic);
   ;	   *cmdptr = FLSH_VERIFY;
   ;	   enable();
   ;	
	push	WORD PTR _delay_10mic
	call	_delay
	pop	CX
	les	BX, DWORD PTR [BP-6]
	mov	BYTE PTR ES:[BX], FLSH_VERIFY
	sti
   ;	
   ;	   delay(delay_10mic);
   ;	   *cmdptr = FLSH_READ;
   ;	
	push	WORD PTR _delay_10mic
	call	_delay
	pop	CX
	les	BX, DWORD PTR [BP-6]
	mov	BYTE PTR ES:[BX], FLSH_READ
   ;	
   ;	   /* really make SURE the byte got burned in */
   ;	   if(*valptr == value)
   ;	   {
   ;	
	les	BX, DWORD PTR [BP+4]
	mov	AL, BYTE PTR ES:[BX]
	cmp	AL, BYTE PTR [BP+8]
	jne	short prog1
   ;
   ;	      delay(delay_10mil);
   ;	      *cmdptr = FLSH_READ;
   ;
	push	WORD PTR _delay_10mil
	call	_delay
 	pop	CX
	les	BX, DWORD PTR [BP-6]
	mov	BYTE PTR ES:[BX], FLSH_READ
   ;
   ;	      delay(delay_10mic);
   ;	      if(*valptr == value)
   ;	         break;
   ;	   }
   ;	
	push	WORD PTR _delay_10mic
	call	_delay
	pop	CX
	les	BX, DWORD PTR [BP+4]
	mov	AL, BYTE PTR ES:[BX]
	cmp	AL, BYTE PTR [BP+8]
	je	short prog2
prog1:
   ;
   ;	}
   ;
	mov	ax, si
	inc	si
	cmp	ax, FLSH_MAX_CYCLES
	jb	short prog0
prog2:
   ;	
   ;	return (*valptr == value);
   ;	
	les	BX, DWORD PTR [BP+4]
	mov	AL, BYTE PTR ES:[BX]
	cmp	AL, BYTE PTR [BP+8]
	jne	short prog3
	mov	AX, 1
	jmp	short prog4
prog3:
	xor	AX, AX
prog4:
   ;	
   ;	}
   ;	
	pop	SI
	mov	SP, BP
	pop	BP
	ret	
_program_one_byte	ENDP

;---------------------------------------------------------
;  int program_bytes(char far *buffer, char far *flashmem, 
;			unsigned len);
;
;  This function programs (writes) a number of bytes to
;  Flash memory.  
;	
	public	_program_bytes
_program_bytes	PROC	NEAR
	push	BP
	mov	BP, SP
	push	SI
	mov	SI, WORD PTR [BP+12]
	jmp	short progb2
progb0:
   ;	
   ;	while(len--)
   ;	{
   ;	   if(!program_one_byte(flashmem, *buffer) && !in_root)
   ;          return 0;
   ;	
	les	BX,DWORD PTR [BP+4]
	mov	AL,BYTE PTR ES:[BX]
	push	AX
	push	WORD PTR [BP+10]
	push	WORD PTR [BP+8]
	call	_program_one_byte
	add	SP,6
	or	AX, AX
	jne	short progb1
	cmp	BYTE PTR DGROUP:_in_root, 0
	jnz	progb1
	xor	AX, AX
	jmp	short progb4
progb1:
   ;	
   ;	   ++flashmem;
   ;	   ++buffer;
   ;	}
   ;	
	inc	WORD PTR [BP+8]
	inc	WORD PTR [BP+4]
progb2:	mov	AX, SI
	dec	SI
	or	AX, AX
	jne	short progb0
   ;
   ;	return 1;
   ;	
	mov	AX, 1
progb4:
	pop	SI
	pop	BP
	ret	
_program_bytes	ENDP


;---------------------------------------------------------
;  int read_bytes(char far *dst, char far *src,
;					unsigned len)
;
;  Just a simple memory copy.
;
	public	_read_bytes
_read_bytes	PROC	NEAR
	push	BP
	mov	BP, SP
	push	SI
	push	DI
	push	DS
   ;	
   ;	{
   ;	   while(len--)
   ;	      *dst++ = *src++;
   ;	   return 1;
   ;	}
   ;
	mov	CX, WORD PTR [BP+12]
	jcxz	readb1
	lds	SI, DWORD PTR [BP+8] 
	les	DI, DWORD PTR [BP+4] 
	cld
	rep	MOVSB
readb1:
	mov	AX, 1
	pop	DS
	pop	DI
	pop	SI
	pop	BP
	ret
_read_bytes	ENDP

_TEXT	ENDS

_BSS	SEGMENT
	buff	DB	512 dup (?)
_BSS	ENDS

_TEXT	SEGMENT

;---------------------------------------------------------
;  int disk_is_zero(void);
;
;  Checks if the entire Flash memory is erased.  Erased
;  means logical 0s (inverted FFs).  Returns TRUE if so,
;  FALSE if not.
;
	public	_disk_is_zero
_disk_is_zero	PROC	NEAR
	push	BP
	mov	BP, SP
	sub	SP, 4
	push	SI
	push	DI
   ;	
   ;	{
   ;	int i,j;
   ;	long position;
   ;
   ;	position = 0L;
   ;	for(i=0; i < flash_sectors; ++i)
   ;	{
   ;
	xor	DI, DI
	mov	word ptr [bp-2], DI
	mov	word ptr [bp-4], DI
	jmp	short diskz6
   ;	
   ;	   if(!memread(position, 512, (void far *)buff))
   ;	      return 0;
   ;	
diskz1:
	push	DS
	mov	AX, OFFSET DGROUP:buff
	push	AX
	mov	AX, 512
	push	AX
	push	WORD PTR [BP-2]
	push	WORD PTR [BP-4]
	call	_memread
	add	SP, 10
	or	AX, AX
	jz	short diskz7
   ;	
   ;	for(j=0; j<512; ++j)
   ;	   if(buff[j])
   ;	      return 0;
   ;	
	xor	si, si
	jmp	short diskz5
diskz3:
	cmp	BYTE PTR DGROUP:[buff+SI], 0
	je	short diskz4
	xor	AX, AX
	jmp	short diskz7
diskz4:	inc	SI
diskz5:	cmp	SI, 512
	jl	short diskz3
	inc	DI
diskz6:	cmp	DI, WORD PTR DGROUP:_flash_sectors
	jb	short diskz1
   ;	
   ;	}
   ;	return 1;
   ;	
	mov	AX,1
diskz7:
   ;	
   ;	}
   ;	
	pop	DI
	pop	SI
	mov	SP, BP
	pop	BP
	ret	
_disk_is_zero	ENDP

_TEXT	ENDS


;---------------------------------------------------------
;  Generic Memory Functions
;
;  bool meminit(char far *cmdline, unsigned *endseg);
;  int  memchanged(void);
;  bool memread(long offset, unsigned len, 
;				char far * buffer);
;  bool memwrite(long offset, unsigned len,
;				char far * buffer);
;
;  This section deals exclusively with programming the
;  Flash device itself.  All the code here is unique to
;  the particular Flash hardware and must be changed if
;  support for other hardware is desired.
;
;  The program runs "as is" with the SeaLevel System's
;  PROM III memory card containing a 28F010 (128K x 8)
;  Intel Flash chip.
;
_DATA	SEGMENT
		public	_signon
	_signon	DB "FAT-Flash Disk Driver",10
		DB "Dr. Dobbs Journal, May 1993",10,10,0

	; error and status messages
	cant_read  DB "Failed to read Flash memory!",10,0
	cant_write DB "Failed to write Flash memory!",10,0
	found	   DB "Found Valid Flash Memory Disk",10,0
	creating   DB "Creating Flash Memory Disk",10,0
	erased	   DB "Flash Memory must be erased before "
 		   DB "disk can be created!",10,0

	valid_name DB	"FLASHDSK"
	valid_id   DB	0EBH, 3CH, 90H
	fat_id     DB	0F0H, 0FFH, 0FFH
	vol_label  DB	"Flash Disk ",28H

_DATA	ENDS

_BSS	SEGMENT
	chk_name DB	8 dup (?)
	chk_id	 DB	3 dup (?)
_BSS	ENDS

_TEXT	SEGMENT

;---------------------------------------------------------
;   bool meminit(char far *cmdline, unsigned *endseg);
;
;  Initialize the flash memory.  The parameters "cmdline"
;  and "endseg" are not presently used, but are still
;  passed by the calling device driver init routine.
;	
	public	_meminit
_meminit	PROC	NEAR
	push	BP
	mov	BP, SP
	sub	SP, 4
   ;	
   ;	{
   ;	unsigned char far * cmdptr = (unsigned char far *)
   ;		((long)flash_disk_seg << 16);
   ;	
	mov	AX, WORD PTR _flash_disk_seg
	mov	WORD PTR [BP-2], AX
	mov	WORD PTR [BP-4], 0
   ;	
   ;	static char chk_name[8];
   ;	static char chk_id[3];
   ;	static char valid_name[8] =
   ;		{ 'F','L','A','S','H','D','S','K' };
   ;	static char valid_id[3]   = { 0xEB, 0x3C, 0x90 };
   ;	static char fat_id[3] = { 0xF0, 0xFF, 0xFF };
   ;	static char vol_label[12] = { 'F','l','a','s','h',
   ;			' ','D','i','s','k',' ',0x28 };
   ;	
   ;	/* sign on */
   ;	bios_puts("FAT-Flash Disk Driver\n"
   ;			"Dr. Dobbs Journal, May 1993\n\n");
   ;	
	mov	AX, OFFSET DGROUP:_signon
	push	AX
	call	_bios_puts
	pop	CX
   ;	
   ;	/* reset the flash disk software */
   ;	calc_delay();
   ;	setpage(0);
   ;	*cmdptr = FLSH_RESET;
   ;	*cmdptr = FLSH_RESET;
   ;	*cmdptr = FLSH_READ;
   ;	delay(delay_10mil);
   ;	
	call	_calc_delay
	xor	AX, AX
	push	AX
	call	_setpage
	pop	CX
	les	BX,DWORD PTR [BP-4]
	mov	BYTE PTR ES:[BX], FLSH_RESET
	mov	BYTE PTR ES:[BX], FLSH_RESET
	mov	BYTE PTR ES:[BX], FLSH_READ
	push	WORD PTR _delay_10mil
	call	_delay
	pop	CX
   ;	
   ;	/* check for valid Flash disk in Flash memory */
   ;	if(!memread(0L, 3, (void far *)id) ||
   ;			!memread(3L, 8, (void far *)name))
   ;
	push	DS
	mov	AX, OFFSET DGROUP:chk_id
	push	AX
	mov	AX,3
	push	AX
	xor	AX, AX
	push	AX
	push	AX
	call	_memread
	add	SP, 10
	or	AX, AX
	je	short init1
	push	DS
	mov	AX, OFFSET DGROUP:chk_name
	push	AX
	mov	AX, 8
	push	AX
	xor	AX, AX
	mov	DX, 3
	push	AX
	push	DX
	call	_memread
	add	SP, 10
	or	AX, AX
	jne	short init2
init1:
   ;	
   ;	{
   ;	   bios_puts("Cannot read Flash memory!\n");
   ;	   return 0;
   ;	}
   ;	
	mov	AX, OFFSET DGROUP:cant_read
	push	AX
	call	_bios_puts
	pop	CX
	xor	AX, AX
	jmp	init10
init2:
   ;
   ;	/* if not a valid disk, create one */
   ;	if(!byte_cmp(id,valid_id,3) ||
   ;			!byte_cmp(name,valid_name,8))
   ;	{
   ;	   bios_puts("Creating Flash Memory Disk\n");
   ;
	mov	AX, 3
	push	AX
	mov	AX, OFFSET DGROUP:valid_id
	push	AX
	mov	AX, OFFSET DGROUP:chk_id
	push	AX
	call	_byte_cmp
	add	SP,6
	or	AX, AX
	je	short init3
	mov	AX, 8
	push	AX
	mov	AX, OFFSET DGROUP:valid_name
	push	AX
	mov	AX, OFFSET DGROUP:chk_name
	push	AX
	call	_byte_cmp
	add	SP, 6
	or	AX, AX
	je	init3
	jmp	init8
init3:	mov	AX, OFFSET DGROUP:creating
	push	AX
	call	_bios_puts
	pop	CX
   ;	
   ;	/* whole disk MUST be erased */
   ;	if(!disk_is_zero())
   ;	{
   ;	   bios_puts("Flash Memory must be erased before "
   ;			"disk can be created!\n");
   ;	   return 0;
   ;	}
   ;
	call	_disk_is_zero
	or	AX, AX
	jne	short init4
	mov	AX, OFFSET DGROUP:erased
	push	AX
	call	_bios_puts
	pop	CX
	xor	AX,AX
	jmp	init10
init4:
   ;	
   ;	/* write out boot sector */
   ;	if(!memwrite(0L, 3, (void far *)valid_id) ||
   ;	   !memwrite(3L, 8, (void far *)valid_name) ||
   ;	   !memwrite(11L, 17, (void far *)flash_BPB))
   ;	
	push	DS
	mov	AX, OFFSET DGROUP:valid_id
	push	AX
	mov	AX, 3
	push	AX
	xor	AX, AX
	xor	DX, DX
	push	AX
	push	DX
	call	_memwrite
	add	SP, 10
	or	AX, AX
	je	short init5
	push	DS
	mov	AX, OFFSET DGROUP:valid_name
	push	AX
	mov	AX, 8
	push	AX
	xor	AX, AX
	mov	DX, 3
	push	AX
	push	DX
	call	_memwrite
	add	SP, 10
	or	AX, AX
	je	short init5
	push	DS
	mov	AX, OFFSET DGROUP:_flash_BPB
	push	AX
	mov	AX, 17
	push	AX
	xor	AX, AX
	mov	DX, 11
	push	AX
	push	DX
	call	_memwrite
	add	SP, 10
	or	AX, AX
	jne	short init6
init5:
   ;	
   ;	{
   ;	write_failed:
   ;	   bios_puts("Failed to write Flash memory!\n");
   ;	   return 0;
   ;	}
   ;	
	mov	AX, OFFSET DGROUP:cant_write
	push	AX
	call	_bios_puts
	pop	CX
	xor	AX, AX
	jmp	short init10
init6:
   ;	
   ;	/* write out FAT */
   ;	if(!memwrite(512L, 3, (void far *)fat_id))
   ;	    goto write_failed;
   ;	
	push	DS
	mov	AX, OFFSET DGROUP:fat_id
	push	AX
	mov	AX, 3
	push	AX
	xor	AX, AX
	mov	DX, 512
	push	AX
	push	DX
	call	_memwrite
	add	SP, 10
	or	AX, AX
	jne	init7
	jmp	short init5
init7:
   ;	
   ;	/* write out Volume Label */
   ;	if(!memwrite(1024L, 12, (void far *)vol_label))
   ;	    goto write_failed;
   ;	
	push	DS
	mov	AX, OFFSET DGROUP:vol_label
	push	AX
	mov	AX, 12
	push	AX
	xor	AX, AX
	mov	DX, 1024
	push	AX
	push	DX
	call	_memwrite
	add	SP, 10
	or	AX, AX
	jne	init9
	jmp	short init5
   ;	
   ;	else
   ;	   bios_puts("Found Valid Flash Memory Disk\n");
   ;	
init8:
	mov	AX, OFFSET DGROUP:found
	push	AX
	call	_bios_puts
	pop	CX
init9:
   ;	
   ;	return 1;
   ;	}
   ;	
	mov	AX, 1
init10:
	mov	SP, BP
	pop	BP
	ret	
_meminit	ENDP


;---------------------------------------------------------
;  int  memchanged(void);
;
	public	_memchanged
_memchanged	PROC	NEAR
	mov	AX, 1	; can NEVER be removed
	ret
_memchanged	ENDP

;---------------------------------------------------------
;  bool memrdwr(long offset, unsigned len,
;			char far * buffer, int (*func)())
;
;  This helper function does the work for both memread()
;  and memwrite().
;	
_memrdwr	PROC	NEAR
	push	BP
	mov	BP, SP
	sub	SP, 8
	push	SI
	push	DI
	mov	SI, WORD PTR [BP+8]	; len
   ;
   ;	{
   ;	int page;
   ;	unsigned off;
   ;	unsigned tmplen;
   ;	unsigned char far * disk = (unsigned char far *)
   ;		((long)flash_disk_seg << 16);
   ;	
	mov	AX, WORD PTR _flash_disk_seg
	mov	WORD PTR [BP-6], AX
	mov	WORD PTR [BP-8], 0
	jmp	short rdwr5
rdwr1:
   ;	
   ;	/* read/write the bytes */
   ;	while(len)
   ;	{
   ;	   /* calculate page */
   ;	   page = offset / flash_pagesize;
   ;	   off  = offset % flash_pagesize;
   ;	
	mov	AX, WORD PTR [BP+4]
	mov	DX, WORD PTR [BP+6]
	div	WORD PTR _flash_pagesize
	mov	WORD PTR [BP-2], AX
	mov	WORD PTR [BP-4], DX
   ;	
   ;	   if(off+len <= flash_pagesize)
   ;	      tmplen = len;
   ;	   else
   ;	      tmplen = flash_pagesize - off;
   ;	
	mov	AX, WORD PTR [BP-4]
	add	AX, SI
	cmp	AX, WORD PTR _flash_pagesize
	ja	short rdwr2
	mov	DI, SI
	jmp	short rdwr3
rdwr2:
	mov	AX, WORD PTR _flash_pagesize
	sub	AX, WORD PTR [BP-4]
	mov	DI, AX
rdwr3:
   ;	
   ;	   /* copy the data to/from the flash memory */
   ;	   setpage(page);
   ;	
	push	WORD PTR [BP-2]
	call	_setpage
	pop	CX
   ;	
   ;	   if(!(*func)(buffer,disk+off,tmplen))
   ;	      return 0;
   ;
	push	DI
	mov	AX, WORD PTR [BP-8]
	add	AX, WORD PTR [BP-4]
	push	WORD PTR [BP-6]
	push	AX
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	call	WORD PTR [BP+14]
	add	SP, 10
	or	AX, AX
	jne	short rdwr4
	xor	AX, AX
	jmp	short rdwr6
rdwr4:
   ;	
   ;	   len -= tmplen;
   ;	   offset += tmplen;
   ;	   buffer += tmplen;
   ;	}
   ;	
	sub	SI, DI
	add	WORD PTR [BP+4], DI
	adc	WORD PTR [BP+6], 0
	add	WORD PTR [BP+10], DI
rdwr5:
	or	SI, SI
	jne	short rdwr1
   ;	
   ;	return 1;
   ;	}
   ;
	mov	AX, 1
rdwr6:
	pop	DI
	pop	SI
	mov	SP, BP
	pop	BP
	ret	
_memrdwr	ENDP

;---------------------------------------------------------
;  bool memread(long offset, unsigned len,
;				char far * buffer);
;
;  Interface to Generic Disk Device Driver.  Read bytes
;  from Flash memory - uses lower level function memrdwr().
;	
	public	_memread
_memread	PROC	NEAR
	push	BP
	mov	BP,SP
	push	SI
	push	DI
	mov	SI, WORD PTR [BP+8]
   ;	
   ;	{
   ;	bool gotit;
   ;	gotit = memrdwr(offset,len,buffer,read_bytes);
   ;	
	mov	AX, OFFSET _read_bytes
	push	AX
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	push	SI
	push	WORD PTR [BP+6]
	push	WORD PTR [BP+4]
	call	_memrdwr
	add	SP, 12
	mov	DI, AX
   ;	
   ;	if(gotit)
   ;	   invert_bits(buffer,len);
   ;	
	or	DI, DI
	je	short memrd1
	push	SI
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	call	_invert_bits
	add	SP, 6
memrd1:
   ;	
   ;	return gotit;
   ;	}
   ;	
	mov	AX, DI
	pop	DI
	pop	SI
	pop	BP
	ret	
_memread	ENDP

;---------------------------------------------------------
;  bool memwrite(long offset, unsigned len,
;			char far * buffer);
;
;  Interface to Generic Disk Device Driver.  Write bytes
;  to Flash memory - uses lower level function memrdwr().
;	
	public	_memwrite
_memwrite	PROC	NEAR
	push	BP
	mov	BP, SP
	dec	SP
	dec	SP
	push	SI
	mov	SI, WORD PTR [BP+8]
   ;
   ;	{
   ;	bool gotit;
   ;	invert_bits(buffer,len);
   ;	
	push	SI
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	call	_invert_bits
	add	SP, 6
   ;
   ;	in_root = 0;
   ;	if(offset >= 0x400 && offset < 0xC00)
   ;	   in_root = 1;
   ;
	mov	BYTE PTR DGROUP:_in_root, 0
	cmp	WORD PTR [BP+6], 0
	jnz	memwr0
	cmp	WORD PTR [BP+4], 0400H
	jb	memwr0
	cmp	WORD PTR [BP+4], 0C00H
	jae	memwr0
	mov	BYTE PTR DGROUP:_in_root, 1
memwr0:	
   ;	
   ;	gotit = memrdwr(offset,len,buffer,program_bytes);
   ;	
	mov	AX, OFFSET _program_bytes
	push	AX
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	push	SI
	push	WORD PTR [BP+6]
	push	WORD PTR [BP+4]
	call	_memrdwr
	add	SP, 12
	mov	WORD PTR [BP-2], AX
   ;	
   ;	invert_bits(buffer,len);	/* change bits back to normal */
   ;	
	push	SI
	push	WORD PTR [BP+12]
	push	WORD PTR [BP+10]
	call	_invert_bits
	add	SP, 6
   ;	
   ;	return gotit;
   ;	}
   ;	
	mov	AX,WORD PTR [BP-2]
	pop	SI
	mov	SP, BP
	pop	BP
	ret	
_memwrite	ENDP

_TEXT	ENDS


;------------------------------------------------
;  Print a Message to the screen during assembly
;
outstr MACRO msg
    IFDEF ??version
	%out msg
    ELSE
	IF1
	    %out msg
	ENDIF
    ENDIF
ENDM
	outstr	<* Flash Disk Driver>
	outstr	<* Dr. Dobbs Journal, May 1993>

	END	flash_header
