/*
  @Project: The cross-platform plug-in toolkit (XPIN).
  @Project-Version: 1.2
  @Project-Date: December 1992.
  @Filename: XPINGLUE.C

  @Author: Ramin Firoozye - rp&A Inc.
  @Copyright: (c) 1992, rp&A Inc. - San Francisco, CA.
  @License.
  The portable plug-in toolkit source and library code is
  presented without any warranty or assumption of responsibility
  from rp&A Inc. and the author. You can not sell this toolkit in
  source or library form as a standalone product without express
  permission of rp&A Inc. You are, however, encouraged to use this
  code in your personal or commercial application without any
  royalties. You can modify the sources as long as the license
  and credits are included with the source.

  For bugs, updates, thanks, or suggestions you can contact the
  author at:
	CompuServe: 70751,252
	Internet: rpa@netcom.com

  @End-License.

  @Description.
  These are glue routines used by any XPIN caller facility to
  load and invoke the plug-ins.

  The publicly accessible routines are:

	XPINInit:	Initializes the XPIN environment and
			loads plug-ins from the specified directory
			(note, this could be reinvoked at runtime
			to rescan the directory).
	XPINCall:	To invoke a plug-in.
	XPINCount:	Returns the number of plug-ins detected.
	XPINLabel:	Returns the label attached to a given plug-in.
	XPINDesc:	Returns the description attached to a
			given plug-in.
	XPINDone:	Called when all is done, to clean things up.

  Problems/Enhancements/Notes:
  Some System 7'isms are used in xpinFindFiles that can be
  made backward compatible with a few changes:

  - Files, when located, are being identified with their FSSpec
    which is particular to Sys7. This was done to avoid messing
    with working directories when plug-ins are actually being
    loaded from the disk. To revert to previous system versions,
    replace FSSpec with vRefNum, parID and fileNames. You have
    to save and restore working directories every time an plug-in
    is loaded (which, come to think of it, is exactly what the DOS
    version does).
  - Path substitution for $SYSTEM uses the new FindFolder call
    to locate the "blessed folder".
  - Path substitution for $HOME uses the Process Manager calls
    to locate the path to the current "process" (i.e. the main
    application).

  Possible Enhancements:
  - Use a hashed table to allow plug-ins to be invoked by name
    as well as index.
  - The path substitution mechanism could use some bullet-proofing.
  - Support for alternate paths to look into.
  - Support for multiple finder file types.

  @End-Description.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "XCONFIG.H"

#ifdef OS_MAC
#include <Types.h>
#include <Files.h>
#include <Events.h>
#include <Errors.h>
#include <Aliases.h>
#include <Traps.h>
#include <Memory.h>
#include <ToolUtils.h>
#include <OSUtils.h>
#include <Folders.h>
#include <Processes.h>
#include <Resources.h>
#ifdef COMPILER_MPW
#include <Strings.h>
#endif
#define	XFUNC	pascal void
#endif
#ifdef OS_WIN
#include <dos.h>
#include <errno.h>
#define	XFUNC	void
#endif
#ifdef COMPILER_MSC
#include <direct.h>
#endif
#ifdef COMPILER_BORLAND
#include <dir.h>		// DOS only
#include <alloc.h>		// DOS only
#endif

#include "XPINTYPE.h"		// XPIN data types
#include "XPINSHR.h"		// Common Caller-XPIN def's
#include "XPINCALL.h"		// Caller only def's

#ifndef MAXDIR
#define MAXDIR	128
#endif

/*
 * Local function prototypes, keeps the compiler smiling.
 * Functions accessible externally are defined in XPINCALL.H
 */
XPINErr	xpinAllocate(XElement **);
XFUNC  xpinCallExternal(XPINFunction code, XBlock *xb);
void		xpinCleanPath(PathSpec *path);
void		xpinDeallocate(XElement *);
XPINErr 	xpinExecute(XPIN *xpin, long index, XBlock *xb, int action);
XPINErr	xpinFindFiles(char *dirName, XPIN *xpin);
XPINErr	xpinFindDirectory(PathSpec *thePath, PathSpec *retPath);
XPINErr	xpinGetAppDirectory(PathSpec *path);
XPINErr	xpinGetDefault(PathSpec *path);
XPINErr	xpinGetFilesMatchingType(XPIN *xpin);
XPINErr	xpinGetFolder(XPIN *xpin, char *givenPath, PathSpec *fullPath);
XPINErr	xpinInitialize(XPIN *xpin);
XPINErr	xpinLoad(XPIN *xpin, long id, XBlock *xb);
void		xpinRelease(XPIN *xpin, long id);
XPINErr	xpinSetDefault(PathSpec *path);
/*
 * These all have to do with in-memory list management. They're
 * sort of generic. We use them for managing XElement's in memory.
 */
void		xpinListAppend(XPIN *xpin, XElement *item);
long		xpinListCount(XPIN *xpin);
XElement	*xpinListDelete(XPIN *xpin, long index);
void		xpinListDispose(XPIN *xpin);
XElement	*xpinListNth(XPIN *xpin, long index);

/********************************************************************
 *
 * External API functions.
 *
 ********************************************************************/

/*
 * XPINInit
 * Initializes the XPIN environment. The specified directory is searched for
 * plug-in files. The filename has to be either a valid absolute path in the
 * local system's syntax or a relative path with the following prefixes:
 *
 *		$HOME - Shorthand for directory from which application is launched.
 *		$SYSTEM - Shorthand for system directory ("System Folder" for MacOS,
 *			the Windows installation directory under Win3).
 *
 * The following are all valid directories under Windows:
 *
 *	$HOME:plug-in - The plug-in folder inside the application directory.
 *	$SYSTEM:XPIN - Expands to "C:\WIN\XPIN"
 *	C:\XPINDIR - The absolute path on the given hard-disk.
 *
 * The following are all valid directories under MacOS:
 *
 *	$HOME:plug-in. - The plug-in. folder inside the application directory.
 *	$SYSTEM:XPIN - Expands to "System Folder:XPIN"
 *	Volume::XPINfolder - The absolute path on the given hard-disk.
 *
 * The file type is a four byte file type under MacOS. The default is
 * 'xpnF'. Under Win3, the file type is a valid DOS wildcard sequence to search
 * for. The default is "*.XPIN".
 * NOTE: Under MacOS, it is customary to pass the four characters of the
 * FileType inside a "long". Under DOS, the filetype is the address of a 
 * null-terminated string. To make things consistent everywhere, we have
 * the Mac filetype passed as a pointer to a string. We just take its
 * first four characters and convert it to a long.
 *
 */
XPINErr XPINInit(XPIN *xpin, char *directory, FType fileType)
{
XPINErr	status;
Bool		initStatus;
Ulong		 badCount;
PathSpec	oldDirectory, newDirectory;

/*
  * First: If the xpin area is already initialized, we go around and release all
  * locked down functions, zero out the area and rescan from the beginning, 
  * otherwise, we initialize the header and get on with it...
  */
	if (xpin->magic == XPINMAGIC) {
		xpinRelease(xpin, -1);
	 }
	 // Clear out all entries and reinitialize from scratch
	memset(xpin, (int) 0, sizeof(XPIN));
	xpin->magic = XPINMAGIC;
	xpin->max = XPINMAX;

	// Now load the "current" path, which we assume is the application's
	// default directory. This is used to translate the $HOME setting.
	// We also get the default Windows directory. This is used for the
	// $SYSTEM setting.
	// Mac NOTE: We don't use default paths. We take advantage of the cool
	// new FSSpec-based routines under System 7.
#ifdef OS_WIN
	xpinGetDefault(&xpin->appPath);
	GetWindowsDirectory((LPSTR) &xpin->sysPath, sizeof(xpin->sysPath));
	xpinCleanPath(&xpin->sysPath);
#endif

    // Make a copy of the file type...
	strcpy((char *) &(xpin->fileType),  (char *) fileType);

  /* Find all plug-in files. */
	status = xpinFindFiles(directory, xpin);

 /* If any found, Initialize! */
	if (status == XPINOK)
		if (xpin->count > 0)
			status = xpinInitialize(xpin);

	return(status);
}

/*
 * This routine allocates space on the heap for the individual elements.
 * xpinDeallocate frees it up. The element should be initialized to all zeros.
 */
XPINErr	xpinAllocate(XElement **element)
{
XPINErr		status = XPINNOMEM; // assume the worst
XElement	*one;

#ifdef OS_WIN
	one = (XElement *) calloc((size_t) 1, sizeof(XElement));
#endif
#ifdef OS_MAC
	one = (XElement *) NewPtrClear(sizeof(XElement));
#endif
	if (one != NULL) {
		status = XPINOK;
		*element = one;
	}
	return(status);
}

void	xpinDeallocate(XElement *element)
{
#ifdef OS_WIN
	free(element);
#endif
#ifdef OS_MAC
	DisposePtr((Ptr) element);
#endif
}

/*
  * XPINCount -
  * Returns the count of the number of plug-ins loaded so far.
  * To access all the plug-ins, run from 0 to XPINCount-1.
  */
long	XPINCount(XPIN *xpin)
{
	return(xpin->count);
}

/*
  * XPINLabel -
  * Returns a pointer to the label declared under DescribeXPIN inside the plug-in. The first time
  * XOP_INIT opcode copies it over to the caller's address space.
  * NOTE: We return a pointer to the internal storage space. The caller should make a copy
  * of the label if it is going to alter it somehow. The label is stored as a null-terminated string
  * of length XPINLABEL. If it is to be passed on to the toolbox, make a copy then use C2P on the copy.
  */
Ubyte *XPINLabel(XPIN *xpin, long index)
{
XElement	*item;
Ubyte	*text;

	item = xpinListNth(xpin, index);
	if (item != NIL) 
		text = item->label;
	else
		text = NIL;
	return(text);
}

/*
  * XPINDesc -
  * Same as XPINLabel except the XPINDESC sized description is returned.
  */
Ubyte *XPINDesc(XPIN *xpin, long index)
{
XElement	*item;
Ubyte	*text;

	item = xpinListNth(xpin, index);
	if (item != NIL) 
		text = item->desc;
	else
		text = NIL;
	return(text);
}

/*
  * XPINVersion -
  * Returns a copy of the  version number for the plug-in (defined by the plug-in using the
  * DescribeXPIN macro. The major and minor numbers can be accessed together as v.L or
  * individually as v.major and v.minor.
  */
void	XPINVersion(XPIN *xpin, long index, XVersion *v)
{
XElement	*item;
Ubyte	*text;

	item = xpinListNth(xpin, index);
	if (item != NIL) 
		*v = item->version;
}

/*
  * XPINCall/XPINExecute
  *
  * The main interface to plug-ins.
  *
  * What happens:
  *	-	An plug-in is accessed by its "id" which is the index into the xpin->items array.
  *		The plug-in should be initialized by now (i.e. loaded, initialized and called at least
  *		once).
  *	-	A plug-in is called with a parameter block (XBlock defined in XPINSHR.H).
  *		It contains an action opcode, a number of parameters and  a result
  *		field. The caller loads the parameter block and allocates all space that needs to be
  *		sent down to the plug-in. The macros XSETx and XGETx can also be used to
  *		load the parameter block. The macros XSETSTATx and XGETSTATx can be used
  *		to access the result value.
  *	-	The user usually never sees any XOP's. The plug-in recognizes the negtive ones as control
  *		functions and doesn't pass them on to the regular code. The positive ones actually do
  *		something. In this case, there's only one defined: XOP_DO that executes the actual
  *		plug-in code, but of course, it could be extended for custom uses. The two control
  *		opcodes defined are XOP_INIT and XOP_DONE. XOP_INIT returns the label, description, and
  *		version fields. The XOP_DONE gives the plug-in a chance to deallocate any space allocated
  *		by the control opcodes.
  *
  * XPINExecute is the internal routine that does all the work. XPINCall is just a wrapper
  * that does a user action (rather than a control action).
  *
  * NOTE: It is assumed that the caller allocates all persistent storage and deallocates 
  * the space.
  *
  */
XPINErr	XPINCall(XPIN *xpin, long index,  XBlock *xb)
{
    return(xpinExecute(xpin, index, xb, XOP_DO));
}

XPINErr	xpinExecute(XPIN *xpin, long index, XBlock *xb, int action)
{
XPINErr		status;
XPINFunction	code;
XElement		*item;

	if (index > xpin->count) return(XPINBADINDEX);	// Invalid ID

/*
  * Load and lock the plug-in, if there's a problem, release it and return.
  */
	status = xpinLoad(xpin, index, xb);
	if (status != XPINOK) {
		xpinRelease(xpin, index);
		return(status);
	}
/*
  * Lock the code and jump to it. On return, release it.
  */

	item = xpinListNth(xpin, index);
	xb->action = action;
	code = item->code;
	xpinCallExternal(code, xb);
	xpinRelease(xpin, index);
    	status = XGETSTATw(xb);		// IMPLICIT ASSUMPTION: XPINError is a short (w).
	return(status);
}

/*
  * XPINDone
  *
  * Called to close things off and release all resources.
  * NOTE: The caller is responsible for deallocating all local memory that may have been allocated
  * and to send any persistent plug-ins a shutdown message. We send each open and locked down
  * plug-in an XOP_DONE signal to let it drop off peacefully.
  *
  */
void		XPINDone(XPIN *xpin)
{
long		start, end, idx;
XBlock		xblock;
XPINErr		status;
XElement	*item;

	/*
	  * First we send down an XOP_DONE to every plug-in that is currently locked (don't bother
	  * if it's unlocked and out on the disk...
	  */
	for (idx = 0; idx < xpin->count; idx++)
		status = xpinExecute(xpin, idx, &xblock, XOP_DONE);

	xpinRelease(xpin, -1);			// release all locked up code resources
	xpinListDispose(xpin);			// Now the entire list structure....
	/*
	  * We just zero out the XPIN structure. The main caller is responsible for
	  * disposing of the structure itself.
	  */
	memset((void *) xpin, (int) 0, sizeof(XPIN));
}

/*
  * xpinLoad -
  *
  * This function makes sure an plug-in is loaded into memory and is locked.
  * If it isn't, it loads it back up from the resource file and locks it down.
  *
  * NOTE:  We don't want to hog space in memory and cause heap fragmentation, so whenever we want to 
  * access an plug-in, we open up the resource file associated with it, and load it in manually.
  * We use Get1IndResource to only load the first resource of the given type. This presumes that each
  * file contains only a single plug-in function.
  * Note that the resource file is kept open when the plug-in is loaded. During the call to plug-in,
  * it may need another resource (a dialog, icon or something else) out of its own resource file. It also
  * helps support multi-segment code resources and partial resources that may need to refer to the file during
  * the call. again. The high-level function XPINCall calls xpinRelease to unload the resource and close the file.
  *
  * Parameters:
  *		xpin -		The internal xpin structure
  *		index -	Index to the plug-in to load and lock. If it's -1, everything is loaded.
  * Returns:
  *		XPINErr	Standard Resource and Memory manager errors. noErr if all is OK.
  *
  * Windows Note: Code is actually "loaded" through calls to LoadLibrary and
  * GetProcAddress. It doesn't need to be locked down by us.
  * Mac Note:  Code is stored in a plug-in resource file as a standard resource.
  * We have to load and lock it down before executing it...
  */
XPINErr xpinLoad(XPIN *xpin, long index, XBlock *xb)
{
XPINErr		status;
short		resRef, resStat;
XPINFunction	xpinCode;
LibSpec		pluginID;
XElement		*item;
PathSpec		oldPath, realFile;
#ifdef OS_MAC
OSErr		err;
Boolean		wasAliased, wasFolder;
#endif

	if (index < 0 || index > xpin->count) return(XPINBADINDEX);	// error index out of bounds

    // Get where we currently are, then jump over to the XPIN directory.
#ifdef OS_WIN
	xpinGetDefault(&oldPath);
	xpinSetDefault(&xpin->xpinPath);
#endif

	item = xpinListNth(xpin, index);
#ifdef OS_WIN
	 pluginID = (LibSpec) LoadLibrary((LPSTR) item->pathSpec);
	 if (pluginID < 32) {		// Bad error on LoadLibrary
		xpinRelease(xpin, index);		// unlock everything
		xpinSetDefault(&oldPath);
		return(XPINBADPIN);
		}
#elif OS_MAC
/* Resolve the filename, in case it's an alias */
	err = ResolveAliasFile(&item->pathSpec,TRUE,&wasFolder, &wasAliased);
	if (err != noErr) 
		return(XPINBADPIN);
	pluginID = FSpOpenResFile(&item->pathSpec, fsRdPerm);
	resStat = ResError();
	if (resStat != noErr)
		return(XPINBADPIN);	// Bug out if we don't find it
#endif

	item->state.open = 1;			// otherwise mark it as open
	item->pluginID = pluginID;
/*
  * We look for a function called XPINNAME (defined in XPINCaller.h). It's hardcoded to force
  * consistency. You can, of course, set it up to something different and pass it down at runtime.
  */
#ifdef OS_WIN
	xpinCode = (XPINFunction) GetProcAddress(pluginID, XPINNAME);
	if (xpinCode == NULL) {
		xpinRelease(xpin, index);
	    xpinSetDefault(&oldPath);
		return(XPINBADPIN);
	}
#endif
#ifdef OS_MAC
	xpinCode = Get1IndResource(XPINNAME,1);	
	resStat = ResError();
	if (resStat != noErr) {
		xpinRelease(xpin, index);	// force an unlod
		return(XPINBADPIN);
	}
	HLock(xpinCode);	// lock it down
	StripAddress(xpinCode);
#endif
	item->code = xpinCode;
	item->state.locked = 1;

#ifdef OS_WIN
	xpinSetDefault(&oldPath);
#endif
	return(XPINOK);	// all OK.
}

/*
 * xpinRelease -
 *
 * Releases the memory for an plug-in. It doesn't actually release any index entries, just memory locked
 * for the plug-in code. It makes sure that all associated memory and code handles are purgeable. Also,
 * the resource file associated with the plug-in is closed. It doesn't force a purge itself.
 * To force a purge, use CompactMem or MaxMem.
 *
 * Parameters:
 *		xpin		- The internal xpin structure
 *		index	- index for the plug-in. If it is -1, we release everything.
 *				   Normal value should be between 0 and xpin->count.
 */
void xpinRelease(XPIN *xpin, long index)
{ 
long  		start, end, idx;
XElement		*item;

	if (index > xpin->count) return;		// do nothing if it's invalid

	if (index == -1) {
		start = 0;
		end = xpin->count;
	}
	else {
		start = index;
		end = index + 1;
	}
	for (idx = start; idx < end; idx++) {
		item = xpinListNth(xpin, idx);
	    if (item->state.open == 0) continue; // if it's not loaded go away
#ifdef OS_MAC
		CloseResFile(item->pluginID);
#endif
		item->state.open = 0;
	    if (item->state.locked == 0) continue;
#ifdef OS_WIN
		FreeLibrary(item->pluginID);
#endif
#ifdef OS_MAC
		HUnlock(item->code);
		EmptyHandle(item->code);
#endif
		item->state.locked = 0;
		item->pluginID = 0;
		item->code = 0;
	}
}

/*
  * xpinCallExternal-
  *
  * This is a bit of glue code that actually invokes the external function.
  * We assume it's locked down already. This function should never be called directly.
  * Use XPINCall instead. NOTE: We assume a Pascal calling convention. 
  * Mac NOTE: There's usually some assembler glue here,
  * something along the lines of:
  *
  *		MOVE.L	xblock, -(sp)
  *		MOVEA.L	code, a0
  *		MOVEA.L	(a0), a0
  *		JSR 	(a0)
  *		MOVE.L	(sp)+,d0
  *
  * This chunk of C does the same thing. If your favorite compiler
  * doesn't, replace it with the above assembler code...
  *
  */
XFUNC xpinCallExternal(XPINFunction code, XBlock *xblock)
{
#ifdef OS_WIN
	(*code)(xblock);
#endif
#ifdef OS_MAC
typedef	pascal void (*PascalFuncType)(XBlock *xb); 
register PascalFuncType theFunction;

// Dereference the code handle. We assume it's already locked down...

#ifdef COMPILER_THINK
	theFunction = (PascalFuncType *) *code;
#elif COMPILER_MPW
	theFunction = (PascalFuncType) *((PascalFuncType) *code);
#endif

	(*theFunction)(xblock);
	return;

#endif
}

/*
 *  xpinFindFiles
 *
 * Given a directory name, the main routine in this file
 * (xpinFindFiles), fills up the XPIN structure with file specs
 * of all files that match the criteria. To help locate the proper folder
 * the directory syntax is altered a little.
 *
 * Directory name can be:
 *		- A full path spec: A standard  absolute path. Not really
 *		  recommended unless some general standard is being followed
 *		  (i.e. all plug-ins go into "C:\plug-in"
 *		- Path prefixed by $SYSTEM\ We substitute the path for the "blessed
 *		  folder" and locate the plug-in folder relative to there, for
 *		  example, "$SYSTEM\plug-ins" points at the "plug-ins" folder
 *		  inside the system folder. $SYSTEM can also be used on its own
 *		  to point at the Windows folder.
 *		- Path prefixed by $HOME: The home directory of the current application
 *		  is used. This is the preferred way, since the home directory is
 *		  filled in at runtime, so the program can be moved anywhere and this
 *		  will still work. For example, if the path to the main program is 
 *		  "C:\BigProg", the path "$HOME\plug-in" gets translated
 *		  at runtime to "C:\Bigprog\plug-in". This is the recommended
 *		  scheme.
 *
 * Returns: XPINErr from any of the intervening system calls. noErr if successful.
 *		XPIN->count contains the count of files loaded. XPIN->max is assumed to
 *		be preset to the maximum number of slots in the XPIN structure. XPIN->fileType
 *		is preset to the file type we should be looking for.
 *		XPIN->item[index].pathSpec is the PathSpec for each file.
 *
 * Parameters:
 *		dirName:		Name of directory to search (see above for syntax)
 *		xpin:			XPIN structure (defined in XPINCall.h)
 *						Information is returned inside this structure.
 *
 * Implicit inputs:
 * xpin->max has the maximum number of files the xpin structure can handle.
 * xpin->fileType has the file type to search for.
 * xpin->count is set to the number of actual files found.
 * xpin->item[index].pathSpec is the PathSpec for each file found.
 * Returns:
 *		XPINErr			XPINOK if OK.
 *
 */
XPINErr	xpinFindFiles(char *dirName, XPIN *xpin)
{
XPINErr		status;
#ifdef OS_WIN
PathSpec	path;
#elif OS_MAC
OSErr	err;
FSSpec	path;
#endif

	/* First look for the proper folder */
	status = xpinGetFolder(xpin, dirName, (PathSpec *) &path);

	/* If OK, scan for file matching the type spec. */
	if (status == XPINOK)
	// We make a local copy of the full path to the XPIN directory for later use...
#ifdef OS_WIN
		strcpy(&(xpin->xpinPath), path);
#endif
#ifdef OS_MAC
		memcpy(&xpin->xpinPath, &path, sizeof(PathSpec));
#endif
		status = xpinGetFilesMatchingType(xpin);
	return(status);
}

/*
 * xpinGetFolder
 *
 *  Looks for a given folder and returns the path for it.
 * The folder name can be a complete path name or, one of the following:
 * "$HOME:XXX:" - replaces $HOME with the current application's directory
 * "$SYSTEM:XXX:" - replaces $SYSTEM with the System Folder (see above for
 * more details).
 *
 */
 
XPINErr		xpinGetFolder(XPIN *xpin, char *givenPath, PathSpec *fullPath)
{
#ifdef OS_WIN
char			*fName;
PathSpec		*appPath, *sysPath;
#elif OS_MAC
Str255		fN;
CInfoPBPtr	fDir;
short		vRef;
long			dir, lDir;
char			*fName;
OSErr		err;
PathSpec		localPath;
#endif
int			pathLen;
XPINErr		status;

	/* Do $HOME substitution, if asked for */
	pathLen = strlen(givenPath);
	if (strncmp(givenPath,"$HOME", 5) == 0) {	// got $HOME
#ifdef OS_WIN
		sprintf(fullPath,"%s\\%s", &(xpin->appPath),
			(pathLen == 5) ? "\0" : &(fName[6]));
		xpinCleanPath(fullPath);
#elif OS_MAC
		xpinGetAppDirectory(&localPath);
		if (strlen(fName) > 5) { // $HOME plus more...
			strcpy((char *) &(localPath.name), &(fName[6]));	// get the rest of it
			C2P((char *) &localPath.name);
		        status = xpinFindDirectory(&localPath, fullPath);
		        if (status != XPINOK) return(status);
		   } else  { // $HOME only...
			fullPath->parID = localPath.parID;
			fullPath->vRefNum = localPath.vRefNum;
		}
#endif
	return(XPINOK); // We're done with $HOME.
	}

// Now check for $SYSTEM
	  if (strncmp(fName, "$SYSTEM", 7) == 0)  { // got $SYSTEM
#ifdef OS_WIN
		sprintf(fullPath,"%s\\%s", &(xpin->sysPath),
			(pathLen == 7) ? "\0" : &(fName[8]));
		xpinCleanPath(fullPath);
#elif OS_MAC
		err = FindFolder(kOnSystemDisk, kSystemFolderType, kDontCreateFolder,
				&vRef, &dir);
		if (err != noErr) return(XPINBADPATH);
		if (strlen(fName) > 7) { // $SYSTEM plus more...
			strcpy((char *) &(localPath.name), &(fName[8]));	// get the rest of it
			C2P((char *) &localPath.name);
			status = xpinFindDirectory(&localPath, fullPath);
			if (status != XPINOK) return(status);
		   } else { // It's only $SYSTEM
		     fullPath->parID = dir;
		     fullPath->vRefNum = vRef;
		}
#endif
	return(XPINOK);		// Done with $SYSTEM
	}
	
// No substitution needed, just copy in to out...
#ifdef OS_WIN
	strcpy(fullPath, givenPath);
	xpinCleanPath(fullPath);
#elif OS_MAC
	xpinGetDefault(&localPath);
	strcpy((char *) &localPath.name, fName);
	C2P((char *) localPath.name);
	status = xpinFindDirectory(&localPath, fullPath);
#endif
	 return(XPINOK);
}

/*
 * xpinGetDefault
 *
 * Used to get the current working directory's vRefNum and parID. Only used for absolute paths
 * for xpinGetFolder.
 *
 */
XPINErr xpinGetDefault(PathSpec *path)
{
#ifdef OS_WIN
	getcwd(path, PATHSIZE);
 	xpinCleanPath(path);
#elif OS_MAC
WDPBRec		PB;
	
	memset(&PB, (int) 0, sizeof(CInfoPBRec));
	PBHGetVol(&PB,FALSE);
	path->vRefNum = PB.ioWDVRefNum;
	path->parID = PB.ioWDDirID;
#endif
	return(XPINOK);
}

/*
 * xpinSetDefault
 *
 * Change the default directory to the specified one.
 * On DOS systems, we have to change default disk and
 * directory separately
 */
XPINErr	xpinSetDefault(PathSpec *thePath)
{
XPINErr	status = XPINBADPATH;	// assume the worst
#ifdef OS_WIN
int	drive;
char	*path;
char	letter;

	path = (char *) thePath;
	drive = (int) (path[0] - 'A' + 1);
	if (_chdrive(drive) == 0)
		if (chdir(path) == 0)
			status = XPINOK;
#endif
        return(status);
}

/*
 * xtsFindDirectory
 *
 * Relative path resolution. Locates a directory given its parent's vRefNum and
 * parID. Returns the dirID of the folder, if found. We don't loop around
 * or search too hard. If it's not there, it's not there. To implement wide-ranging searches, use
 * FindFolder.
 *
 */
XPINErr	xpinFindDirectory(PathSpec *thePath, PathSpec *returnPath)
{
#ifdef OS_MAC
CInfoPBRec	PB;
XPINErr		status;
long			foundIt;
	
	memset(&PB, (int) 0, sizeof(CInfoPBRec));
	PB.dirInfo.ioVRefNum = thePath->vRefNum;
	PB.dirInfo.ioNamePtr = (StringPtr) &thePath->name;
	PB.dirInfo.ioDrDirID = thePath->parID;
	PB.dirInfo.ioFDirIndex = 0;
	
	status=PBGetCatInfo(&PB,FALSE);
	if ((status==noErr) &&
		(PB.dirInfo.ioFlAttrib & 0x10)) {
		  returnPath->parID = PB.dirInfo.ioDrDirID;
		  returnPath->vRefNum = thePath->vRefNum;
	}
	return(status);
#endif
}

/*
 * xtsGetAppDirectory
 *
 * Returns the current application's directory vRefNum and parID.
 * Uses the process manager to locate the application. To support pre System 7, you can use the 
 * working directory right at the point of launch.
 *
 */
XPINErr	xpinGetAppDirectory(PathSpec *path) 
{
XPINErr			status = XPINOK;
#ifdef OS_MAC
ProcessSerialNumber psn;
OSErr			err;
ProcessInfoRec		pinfo;
FSSpec			fs;


	memset(&pinfo, (int) 0, sizeof(ProcessInfoRec));
	err = GetCurrentProcess(&psn);
	if (err != noErr) return(XPINBADPATH);
	pinfo.processInfoLength = (long) sizeof(ProcessInfoRec);
	pinfo.processAppSpec = &fs;
	err = GetProcessInformation(&psn, &pinfo);
	if (err != noErr) return(XPINBADPATH);
	path->vRefNum = fs.vRefNum;
	path->parID = fs.parID;
#endif
	return(status);
}

/*
 * xpinGetFilesMatchingType
 *
 * Given a folder, it scans it for all files matching file type, loads the
 * result into the XPIN structure. NOTE: this could easily be modified to look for files matching
 * creator if you want to use creator-based searches.
 * Under DOS, we have to add the "*." to the front of the file type
 * in order to make this a wildcard search...
  *
 * Parameters:
 *		XPIN	- structure in which information is returned.
 *
 * NOTE:
 *	XPIN->xpinPath has the path in which files are looked for...
 *	XPIN->max has the maximum number of files to look for.
 *	XPIN->fileType has the file type to scan for.
 *	XPIN->count returns the count of files actually found.
 *	XPIN(nth item).PathSpec will carry the PathSpec for each file matched.
 *
 * Any error messages from the low-level file manager routines end the
 * search and return.
 * NOTE: We call xpinAllocate to allocate the memory for this
 * xpinDeallocate should be called to deallocate space...
 */ 
XPINErr xpinGetFilesMatchingType(XPIN *xpin)
{
XPINErr		status;
char			*fn;
long			matchCount = 0;
long			kIndex = 0;
XElement		*item;
PathSpec		currentPath;
int			done;
#ifdef OS_WIN
char			fileName[LARGEBUF+1];
#elif OS_MAC
Str255		fileName;
CInfoPBRec	fInfo;
OSErr		err;
long		fileType;
#endif
#ifdef COMPILER_BORLAND
struct		find_t fInfo;
#endif
#ifdef COMPILER_MSC
struct 		_find_t	fInfo;
#endif

#ifdef OS_WIN
	// Save the old path and switch to the XPIN directory
	status = xpinGetDefault(&currentPath);
	if (status != XPINOK) return(status);
	status = xpinSetDefault(&xpin->xpinPath);
	if (status != XPINOK) return(status);

	sprintf(fileName, "*.%s", xpin->fileType);
	done = _dos_findfirst(fileName,_A_NORMAL, &fInfo);
	while (!done)
	{
		status = xpinAllocate(&item);
		if (status != XPINOK) return(status);
		strcpy(&item->pathSpec, &(fInfo.name));
		xpinListAppend(xpin, item);
		if (++matchCount == xpin->max) break;
		done = _dos_findnext(&fInfo);
	}
#elif OS_MAC
	memset(&fInfo, (int) 0, sizeof(CInfoPBRec));
	memset(fileName, (int) 0, sizeof(fileName));
	fInfo.hFileInfo.ioFDirIndex = 0;
	fInfo.hFileInfo.ioNamePtr = (StringPtr) fileName;
	/* NOTE: The filetype is actually stored as a null-terminated string. But
	  * on the Mac, filetypes are 4-byte longs. So we just scrape off the first
	  * four bytes off the string and use that
	  */
	fileType = *((long *) xpin->fileType);
	
	for (;;) {
		fInfo.hFileInfo.ioVRefNum = xpin->xpinPath.vRefNum;
		fInfo.hFileInfo.ioDirID = xpin->xpinPath.parID;
		fInfo.hFileInfo.ioFDirIndex++;
		err = PBHGetFInfo((HParmBlkPtr) &fInfo, FALSE);
		if (err == fnfErr) {
			break;
		}	
		/* Change fdType to fdCreator if you want to search on file creator */
		if (fInfo.hFileInfo.ioFlFndrInfo.fdType == fileType) {
			FSMakeFSSpec(xpin->xpinPath.vRefNum,
					xpin->xpinPath.parID, 
					fileName, &currentPath);
			status = xpinAllocate(&item);
			if (status != XPINOK) return(status);
			xpinListAppend(xpin, item);
			item->pathSpec = currentPath;
			kIndex++;
		}
	if (kIndex == xpin->max) break;
	}
  xpin->count = kIndex;
#endif
  return(XPINOK);
}

/*
 * xpinInitialize
 *
 * This function takes a pre-loaded xpin structure and checks to see if
 * all files are REALLY of the proper plug-in type. It does this by loading the
 * file and calling its entry point. If not found, it's bad. If found,
 * we call it up and get some preliminary information from it which we store
 * away.
 *
 * NOTE: xpinInitialize actually loads the resource into memory. If a resource was
 * marked as "non-purgeable", a side-effect will be that the resource will keep hogging the
 * heap .  We close the resource file down and take it out of the resource chain right after
 * validation.
 * NOTE 2: There's a big assumption here that each plug-in file contains exactly
 * one plug-in code resource of the given type.
 * MAC NOTE 3: We use FSpOpenResFile to access the resource files directly through
 * the cool FSSpec mechanism, rather than mucking about with working directories.
 * This, of course, is very much a System 7'ism...
 *
 * Parameters:
 *		xpin		- the XPIN structure
 *		index	- index into the XPIN structure. has to be between 0 and xpin->count.
 *				   -1 means check everything
 *
 */
XPINErr	xpinInitialize(XPIN *xpin) {

XPINErr		status;
XBlock		xblock;
XElement		*item;
long			start, end, idx;
Bool			retStatus;

	idx = 0;	// start from the beginning
	do {
		item = xpinListNth(xpin, idx);
		if (item == NULL) break;	// bump out...
	/*
	  * Now call the routine with XPIN_INIT to get the startup info from it. If err, it's a corrupted
	  * resource, dump it and go...
	  * ALLOCATE space for each of the params...
	  */
		memset(&xblock, (int) 0, sizeof(struct XBlock));	// clear it out
		XSETp(&xblock, 0, (LPSTR) item->label);
		XSETp(&xblock, 1, (LPSTR) item->desc);
		XSETl(&xblock, 2, item->version.L);

		status = xpinExecute(xpin, idx, &xblock, XOP_INIT); // Call the dang thing
		if (status != XPINOK)  {		// If it failed...
			xpinListDelete(xpin, idx);		// purge it out
			// NOTE: If we purge, we don't bump the index up
			// On the next cycle, we get the last one...
			}
		else {						// but wait! It worked.
			item->state.valid = 1;		// make sure it's valid
			item->state.bad = 0;			// and not bad.
			idx++;					// move onto the next one...
		}
// Continue until we're finished...
// NOTE: we may have had an purge which changes the total count...
// We should check against the count field repeatedly. And it better
// not be cached away...
	} while(idx < xpin->count);

// NOTE: We just return OK. The other option is to return with an error
// as soon as xpinExecute returns a bad value. Our strategy is to
// ignore bad XPIN's. If we return right after xpinExecute it implies
// that someone's going to notify them of the existence of a bad
// XPIN.
	return(XPINOK);
}

/*
 * Utility routine to clean up the path passed by the caller.
 * Under DOS, this takes out the any terminating backslashes in the path.
 */
void	xpinCleanPath(PathSpec *thePath)
{
#ifdef OS_WIN
int		pathLen;
char	*path;

	path = (char *) thePath;
	pathLen = strlen(path) - 1;
   	if (path[pathLen] == '\\')
		path[pathLen] = '\0';
#endif
}


/***********************************************************************
 * LIST management routines
 *
 * These routines all manage an in-memory linked list of type XElement
 *
 ***********************************************************************
 */
void		xpinListAppend(XPIN *xpin, XElement *item)
{
XElement	*last;

	item->next = (void *) 0;
	if (xpin->count == 0)
		xpin->head = xpin->tail = item;
	else {
		last = xpin->tail;
		last->next = item;
		xpin->tail = item;
	}
	xpin->count++;
}


XElement	*xpinListNth(XPIN *xpin, long index)
{
long		i;
XElement	*one;

	if (index > xpin->count) return((void *) 0);
	one = xpin->head;
	for (i = 0; i < index; i++)
	  one = one->next;
	return(one);
}

XElement	*xpinListDelete(XPIN *xpin, long index)
{
long	i;
XElement	*item, *old;

	if (xpin->count == 0 || index >= xpin->count) return;

	// Remove from the head
	if (index == 0) {
		item = xpin->head;
		xpin->head = item->next;
		} else {
	// somewhere else
		item = xpin->head;
		for (i = 0; i < index-1; i++)
		 	item = item->next;	 
	 // look for the one right before the one we want
	 // zero its link. If it's the last one we want,
	 // this one is the "tail" now
		 old = item;
		 item = item->next;
		 old->next = item->next;
		if (index == xpin->count-1)
			xpin->tail = old;
	}
	xpin->count--;
	xpinDeallocate(item);		// reclaim its space...
	return(item);
}

void		xpinListDispose(XPIN *xpin)
{
long	i;
XElement	*item, *old;

	item = old = xpin->head;
	for (i = 0; i < xpin->count; i++) {
	   old = item;
	   item = item->next;
       xpinDeallocate(old);
	 }
	xpin->head = xpin->tail = 0;
	xpin->count = 0;
}
