PROGRAM Seek;

(* SEEK 1.1, Copyright (c) 1994, Jody R. Cairns

   SEEK is a DOS utility which searches for files specified on the command-
   line.  There are many options, including: search files by size; search
   files by attribute; search specific or all drives; run file if found; or
   run a command on any found files. More than one file can be specified on
   the command-line and options can appear in any order.  See procedures
   DisplayOptions and DisplayHelp for more information.

   The source to this program (this file) is free to the public domain with
   the condition that I am notified of any modifications made to this
   program, and the executable of the MODIFIED program is NOT distributed
   without my permission (i.e. Don't pull a fast one like making a few
   changes to the source, compiling it, and saying you wrote it.).  Otherwise,
   the source and executable are free to those who want it.

   It would be very much appreciated if any code used from this program is
   credited to me (bearing in mind that I take absolutely no responsibility
   for the results or misuse of the execution or use of said code).

   NOTE: this program is not very modular due to the speed factor. I wanted
         to make this as fast as possible without adding much assembly code
         and still have sufficient error-checking.  Too many procedures
         would have slowed it down (I timed it). So, please excuse this
         programming oversight; it was done for speed. However, you will
         notice I could have been LESS modular; code size was a factor, too.


   Version changes (for those interested)
   ---------------
   SEEK 1.0 - original public-domain release.
   SEEK 1.1 - replaced system.pos with ChPos, my own faster version.
            - replaced CRT.readkey with my own simpler version.
            - in SEEK 1.0, if the following was executed:
                 SEEK acd:tmp1 tmp2
              then for TMP1, drives A, C and D would be searched, but for
              TMP2, only drives C and D would be searched.  This was a bug.
              Now, TMP2 will be searched for on the current drive using the
              above example. In other words, the drives you specified to be
              search on one file will NOT be carried over onto any other
              files specified on the command-line. Thus, for:
                 SEEK acd:tmp1 c:tmp2
              drives A, C and D for TMP1, and only drive C for TMP2 will be
              searched (not drives A and D, too).
            - with systems that viewed A: and B: as the same logical drive,
              SEEK 1.0 with the /D option (search all drives) would force
              the drive to be assumed as B:.  Now, the drive is forced to be
              drive A: instead (which is logical), skipping over drive B:.
            - began researching into a "search for text" option; that is,
              adding an option to mimic a grep-like utility.  It should be
              implemented in the next version.


   SEEK was written and compiled using Turbo Pascal 6.0 and tested on a variety
   of Personal Computers, from 8086s to 80486s.

   If you find ANY bugs, or have any suggestions or ideas I'd be happy to
   hear from you.  Please don't hesitate to contact me for any information.
   I have more utilites in the works, all DOS-based. Here's my address:

   Jody R. Cairns
   P.O. Box 5991
   St. John's, NF
   Canada
   A1C 5X4

   E-mail: jodyc@cauchy.math.mun.ca


*)


(* Compiler directives follow: *)

{$M 65520, 0, 0}  (* Memory requirements: 64 KB for stack, 0 KB for heap    *)

{$I-}             (* I/O checking off                 *)
{$R-}             (* Range checking off               *)
{$S-}             (* Stack checking off               *)
{$X-}             (* Extended syntax off              *)
{$V-}             (* Strict VAR-string checking off   *)
{$B-}             (* Short-circuit boolean evaluation *)
{$D-}             (* Debug information off            *)
{$L-}             (* Local debug information off      *)
{$E-}             (* 8087 emulation off               *)
{$N-}             (* 8087/80287 code generation off   *)
{$G-}             (* 80286 code generation off        *)
{$F-}             (* Force far calls off              *)
{$A+}             (* Word align data                  *)
{$O-}             (* Overlays not allowed             *)


USES
  CRT, DOS;       (* CRT.TPU and DOS.TPU required *)


LABEL
  Next;           (* Used for goto statements     *)

CONST
  Title        = 'SEEK 1.1, Copyright (c) 1994, Jody R. Cairns';

  MaxFileSize  = maxlongint;   (* maximum size of a file       *)
  Esc          = #27;          (* ASCII code of <Esc> key      *)
  MaxDirLength = 67;           (* maximum length of adirectory *)
  MinFileSize  = 0;            (* minimum size of a file       *)


TYPE

  FileSizeFunc = FUNCTION (FileSize: longint): boolean;

  WriteProc    = PROCEDURE (Str: string);

  DriveRec     = RECORD
		   Number: word;
		   Letter: ARRAY[1..26] OF 'A'..'Z';
	         END;

  NumStr       = string[11];



CONST
  ExecuteCommand: boolean  = FALSE;  (* execute a command of found files  *)
  RunFile       : boolean  = FALSE;  (* run files if found                *)
  ShowInfo      : boolean  = FALSE;  (* show file information             *)
  AllDrives     : boolean  = FALSE;  (* search all drives                 *)
  SeekSize      : boolean  = FALSE;  (* seek files by size                *)
  SeekAttr      : boolean  = FALSE;  (* seek files by their attributes    *)
  At            : boolean  = FALSE;  (* seek Archive files                *)
  RO            : boolean  = FALSE;  (* seek Read-Only files              *)
  Hid           : boolean  = FALSE;  (* seek Hidden files                 *)
  Sys           : boolean  = FALSE;  (* seek System files                 *)
  Dire          : boolean  = FALSE;  (* seek Directories                  *)
  NotInHelp     : boolean  = TRUE;   (* help information is displayed     *)
  TotalFound    : longint  = 0;      (* total # of files found            *)
  FilesFound    : longint  = 0;      (* # of files found                  *)
  NumParams     : byte     = 0;      (* # of command-line parameters      *)
  NumOptions    : byte     = 0;      (* # of options                      *)
  CurrRow       : byte     = 1;      (* current row being output to       *)
  StartDrive    : byte     = 1;      (* assume system has drive a: (1)    *)

  Drive         : DriveRec = (Number: 1;
                              Letter: 'ABCDEFGHIJKLMNOPQRSTUVWXYZ');


VAR
  NumRows       : byte;         (* # of rows on output screen      *)
  PC,                           (* # of parameters on command-line *)
  ColonPos,                     (* used for ChPos function         *)
  Tmp,                          (* temporary storage               *)
  I, J          : word;         (* indices                         *)
  TmpLong,                      (* temporary storage               *)
  MinSize,                      (* minimum size of file to find    *)
  MaxSize       : longint;      (* maximum size of a file to find  *)
  Attr          : string[20];   (* attributes of a file            *)
  Ext           : ExtStr;       (* extension of a file             *)
  Name          : NameStr;      (* name of a file                  *)
  Dir,                          (* directory of a file             *)
  CurrDir       : DirStr;       (* current directory               *)
  TmpStr,                       (* temporary storage               *)
  Param,                        (* parameters to run with files    *)
  SizeStr,                      (* size of files to find           *)
  Command       : ComStr;       (* command to run on found files   *)
  FileDate      : DateTime;     (* file date and time              *)
  ValidFileSize : FileSizeFunc; (* function to determine file size *)
  Writeln       : WriteProc;    (* output procedure                *)
  Reg           : registers;    (* CPU registers                   *)


(***************************************************************************)
(***************************************************************************)


FUNCTION ReadKey: char; assembler;
(* Returns ASCII code of key pressed, unless it's special key. *)
  ASM
    mov ah, 10h     (* Read character pressed on keyboard      *)
    int 16h         (* Call Keyboard function                  *)
  END;

(***************************************************************************)

FUNCTION Num2Str (Num: longint): NumStr;
(* Converts Num to a string type. *)
  VAR
    TmpStr: NumStr;
  BEGIN
    str (Num:1, TmpStr);
    Num2Str := TmpStr
  END;

(*************************************************************************)

PROCEDURE HaltProgram;
(* Halts program execution, displaying total files found.                *)
  BEGIN
    IF NotInHelp THEN
      BEGIN
        Writeln ('');
        Writeln ('# of files found: ' + Num2Str (FilesFound));
        IF (TotalFound > 0) THEN
          Writeln ('Total # of files: ' + Num2Str (TotalFound))
      END;
    halt (0)
  END;

(***************************************************************************)

PROCEDURE NormWrite (Str: string); far;
(* This procedure reduces the size of compiled code and can even (in some
   cases) reduce execution time of program.  Really!                       *)
  BEGIN
    system.writeln (Str)
  END;

(***************************************************************************)

PROCEDURE PauseWrite (Str: string); far;
(* The current row on the screen being displayed is kept track off so
   before the screen scrolls up, the user is prompted to continue.       *)
  VAR
    Ch: char;                    (* character pressed by user            *)
  BEGIN
    inc (CurrRow);               (* increment # of rows displayed so far *)

    NormWrite (Str);             (* output string                        *)

    IF (CurrRow = NumRows) THEN  (* time to prompt user?                 *)
      BEGIN
	write ('Press any key to continue...'#13);
	Ch := ReadKey;
	clreol;
	IF (Ch = Esc) THEN
          HaltProgram
        ELSE IF (Ch = #00) THEN
          Ch := ReadKey;
	CurrRow := 0
      END
  END; (* Wrtln *)

(***************************************************************************)

PROCEDURE DisplayOptions;
(* Display program options. *)
  BEGIN

    Writeln   := PauseWrite;
    NumRows   := hi (crt.windmax);

    (* Need the following statement because if PauseWrite detects that
       <Esc> was pressed, then HaltProgram is called which displays
       the total # of files found, which obviously is zero since help is
       being displayed.  Thus, the statement below is for the benefit of
       the procedure HaltProgram.  Phew.                                   *)

    NotinHelp := FALSE;

    Writeln ('');
    Writeln ('Syntax: SEEK  [options]  [drives:]files...  [options]');
    Writeln ('');
    Writeln ('Options:  /D  -------------- search all Drives for files');
    Writeln ('          /I  -------------- show file size, attribute(s), and time');
    Writeln ('          /P  -------------- Pause after screenful of information');
    Writeln ('          /V  -------------- use DOS routines for Video output (slower)');
    Writeln ('          /R  -------------- Run files (must have COM, EXE, or BAT extension)');
    Writeln ('          /A(AHRSD)  ------- seek files by specified Attribute(s)');
    Writeln ('          /Ccommand  ------- execute specified Command on files');
    Writeln ('          /S[-]size[-]  ---- seek files by Size specified or by range');
    Writeln ('          /?, /H  ---------- for more information and syntax examples')
  END;

(**************************************************************************)

PROCEDURE DisplayHelp;
(* Display detailed help information *)
  BEGIN
    DisplayOptions;

    Writeln ('');
    Writeln ('- SEEK searches for all the files specified on the command-line and displays');
    Writeln ('  their full path name if found.');
    Writeln ('');
    Writeln ('- Options can be combined and can appear in any order on the command-line.');
    Writeln ('');
    Writeln ('- You can search across different drives. Just specify the drives you want');
    Writeln ('  to search before the files seperated by a colon.');
    Writeln ('');
    Writeln ('- NOTE: Pressing <Esc> at any time halts execution of SEEK.');
    Writeln ('');
    Writeln ('Explanation of options (examples follow)');
    Writeln ('======================');
    Writeln ('   /D  ---------- searches all drives on system for files. To search specific');
    Writeln ('                  drives place the drive letters immediately before the files');
    Writeln ('                  to be seeked, seperated by a colon without the /D option.');
    Writeln ('   /I  ---------- display information of found files: size, time, date and');
    Writeln ('                  file attributes.');
    Writeln ('   /P  ---------- pause after each screenful of information.');
    Writeln ('   /V  ---------- use DOS routines for output. This is slower but it allows');
    Writeln ('                  any output to be redirected, if desired.');
    Writeln ('   /R  ---------- runs the files if they are found (they of course must have');
    Writeln ('                  a COM, EXE, or BAT extension).');
    Writeln ('   /A(AHRSD) ---- search files by specified attributes. Attributes can be');
    Writeln ('                  combined in any order as long as the first attribute is');
    Writeln ('                  preceded by /A. The attributes are: A = Archive, H = Hidden');
    Writeln ('                  R = Read-only, S = System, and D = Directory. When');
    Writeln ('                  combined, an OR comparison is performed on the files. For');
    Writeln ('                  example, if the option /AHD was specified, any files with a');
    Writeln ('                  Hidden or a Directory attribute are sought for.');
    Writeln ('   /Ccommand ---- the command specified after this option will be executed');
    Writeln ('                  on files being seeked. The command specified can be any DOS');
    Writeln ('                  command or executable program (which must be in your PATH).');
    Writeln ('   /S[-]size[-] - search files by specified size. There are four search');
    Writeln ('                  possibilities:');
    Writeln ('                   1) /Ssize  ====== find file sizes equal to specified size.');
    Writeln ('                   2) /S-size ====== find file sizes less than or equal to');
    Writeln ('                                     specified size.');
    Writeln ('                   3) /Ssize- ====== find file sizes greater than or equal to');
    Writeln ('                                     specified size.');
    Writeln ('                   4) /Ssize-size1 = find file sizes between the range');
    Writeln ('                                     specified by size and size1.');
    Writeln ('Examples');
    Writeln ('========');
    Writeln ('1) SEEK  ac:*.bak  /vi');
    Writeln ('   - searches for all *.BAK files on drives A and C, displaying all');
    Writeln ('     file details and using DOS output routines.');
    Writeln ('');
    Writeln ('2) SEEK  /dcDEL  *.tmp  *.$$$');
    Writeln ('   - searches all drives for *.TMP and *.$$$ files, executing the DEL');
    Writeln ('     command on any files found.');
    Writeln ('');
    Writeln ('3) SEEK  /p  *.*  /id  /s0');
    Writeln ('   - searches all drives for *.* files which have a size of zero bytes,');
    Writeln ('     displays their file information pausing with each screenful.');
    Writeln ('');
    Writeln ('4) SEEK  /ad  /arh  *.*  /s0-100');
    Writeln ('   - searches current drive for *.* files that have either a directory,');
    Writeln ('     read-only or hidden attribute, and that are between 0 and 100 bytes.');
    Writeln ('');
    Writeln ('5) SEEK  acf:help.exe  /ri  d:*.zip');
    Writeln ('   - searches drives A, C, and F for the file HELP.EXE and only drive D for');
    Writeln ('     *.ZIP, displaying file information and running HELP.EXE if found.');
    Writeln ('');
    Writeln (' Copyright Information');
    Writeln (' =====================');
    Writeln ('- SEEK may be distributed freely in the public domain, although donations');
    Writeln ('  would be most appreciated.');
    Writeln ('');
    Writeln ('- I take no responsibility for any liability, loss, or damage caused or');
    Writeln ('  alleged to be caused directly or indirectly from the use of this program.');
    Writeln ('');
    Writeln ('- SEEK was written in Turbo Pascal 6.0. Source code is available upon request.');
    Writeln ('  Future versions are forthcoming.');
    Writeln ('');
    Writeln ('- If you discover any bugs, or have any ideas or suggestions, please feel');
    Writeln ('  free to contact me:');
    Writeln ('     Jody R. Cairns');
    Writeln ('     P.O. Box 5991');
    Writeln ('     St. John''s, NFLD.');
    Writeln ('     Canada  A1C 5X4');
    Writeln ('     E-mail: jodyc@cauchy.math.mun.ca');
    halt (0)
  END;

(***************************************************************************)

PROCEDURE CheckError (ErrorCode: word);
(* Displays the error associated with ErrorCode.                         *)
  VAR
    Error: string;    (* error to display *)
  BEGIN
    CASE ErrorCode OF
       00: exit;
       03: Error := 'Path not found.';
       15: Error := Drive.Letter[J] + ' is an invalid drive letter.';
       20: Error := 'Invalid command-line option: "' + Param[J] + '"';
       21: Error := 'Invalid command-line. Missing filename(s).';
       23: Error := 'Invalid command-line. Missing command.';
       24: Error := 'Invalid command-line. Missing option(s).';
       25: Error := 'Invalid command-line. Too many drives specified.';
       26: Error := 'Invalid command-line. Missing file attribute(s).';
       27: Error := 'Invalid command-line. Missing file size.';
       28: Error := 'Invalid file attribute for /A option: "' + TmpStr[Tmp] + '"';
       30: Error := 'File size must be >= 0 and <= ' + Num2Str (MaxFileSize) + '.';
       31: Error := 'Invalid DOS version. Requires DOS 3.2 or greater.';
      152: Error := 'Drive ' + Drive.Letter[J] + ': not ready.';
      153: Error := 'Invalid directory path.';
      154: Error := 'CRC error in disk data.';
      155: Error := 'Bad disk request structure length.';
      156: Error := 'Disk seek error.';
      157: Error := 'Drive ' + Drive.Letter[J] + ': has a non-DOS disk.';
      158: Error := 'Disk sector not found.';
      161: Error := 'Device read fault.';
      162: Error := 'Hardware failure.'
      ELSE
	Error := 'Please report to author error #' + Num2Str (ErrorCode) + '.'
    END;

    IF (ErrorCode > 152) THEN
      Error := Error + ' Disk in drive ' + Drive.Letter[J] + ': could be corrupt.';

    Writeln ('ERROR: ' + Error);

    IF (ErrorCode <> 15) AND (ErrorCode <= 30) THEN
      BEGIN
        DisplayOptions;
	halt (ErrorCode)
      END
  END;

(**************************************************************************)

FUNCTION UpCaseStr (Str: ComStr): ComStr; assembler;
(* Convert Str into uppercase letters and return it.  Code adapted from
   Turbo Pascal 6.0 Programmer's Guide by Borland, pages 305-306.     *)
  ASM
    mov   dx, ds            (* Save DS register        *)
    cld                     (* Forward string-ops      *)
    lds   si, Str           (* Load string address     *)
    les   di, @Result       (* Load result address     *)
    lodsb                   (* Load string length      *)
    stosb                   (* copy to result          *)
    xor   ah, ah            (* string length to cx     *)
    mov   cx, ax
    jcxz  @3                (* skip if empty string    *)
  @1:
    lodsb                   (* load character          *)
    cmp   al, 'a'           (* skip if not 'a'..'z'    *)
    jb    @2
    cmp   al, 'z'
    ja    @2
    sub   al, 20h           (* Convert to uppercase    *)
  @2:
    stosb                   (* Store in result         *)
    loop  @1                (* Loop for all characters *)
  @3:
    mov   ds, dx            (* Restore DS register     *)
  END; (* UpCaseStr *)

(**************************************************************************)

(* The following 4 functions are for the /S option: find file by size.    *)

FUNCTION Equal (FileSize: longint): boolean; far;
(* Returns TRUE if the current file size (FileSize) is equal to the
   specified file size.                                                   *)
  BEGIN
    Equal := (FileSize = MinSize)
  END;

FUNCTION GreaterThan (FileSize: longint): boolean; far;
(* Returns TRUE if the current file size (FileSize) is greater than or
   equal to the specified file size.                                      *)
  BEGIN
    GreaterThan := (FileSize >= MinSize)
  END;

FUNCTION LessThan (FileSize: longint): boolean; far;
(* Returns TRUE if the current file size (FileSize) is less than or
   equal to the specified file size.                                      *)
  BEGIN
    LessThan := (FileSize <= MinSize)
  END;

FUNCTION Between (FileSize: longint): boolean; far;
(* Returns TRUE if the current file size (FileSize) is greater than or
   equal to the minimum file specification OR less than or equal to the
   specified file size.                                                   *)
  BEGIN
    Between := (FileSize >= MinSize) AND (FileSize <= MaxSize)
  END;

(***************************************************************************)

FUNCTION Date (N : word) : NumStr;
(* Returns N as a string with leading zeros. *)
  VAR
    S : NumStr;
  BEGIN
    str (N:1,S);
    IF (length (S) = 1) THEN
      S := '0' + S;
    Date := S;
  END;

(***************************************************************************)

FUNCTION LastDrive: char;
(* Returns the last drive on the system. Requires DOS 3.1+ as specified in
   Ralf Brown's Interrupt List. *)
  VAR
    R : registers;
    Ch: char;
  BEGIN
    R.AH := $52;
    msdos (R);
    Ch := chr (ord ('A') + mem[R.ES:R.BX + $0021] - 1);
    IF (Ch < 'A') OR (Ch > 'Z') THEN   (* just to be safe *)
      LastDrive := 'Z'
    ELSE
      LastDrive := Ch
  END;

(*************************************************************************)

PROCEDURE SeekFile (Path: PathStr);
(* This is the main search procedure.  Path determines which directory to
   start searching in.  This is a recursive procedure.                      *)

  LABEL
    Next, Found, Found1;   (* goto labels                                   *)

  VAR
    DirNotShown: boolean;   (* determines if directory was displayed or not *)
    F          : SearchRec; (* used for findfirst and findnext procedures   *)

  BEGIN

    IF KeyPressed AND (ReadKey = Esc) THEN
      HaltProgram;

    (* The following statement is a safeguard. The maximum length of a
       directory is compared to length of Path.  If Path is longer then
       the disk directory structure is probably corrupt in some way.        *)

    IF (length (Path) > MaxDirLength) THEN
      CheckError (153);

    DirNotShown := TRUE;

    (* Find first file. *)
    findfirst (Path + Param, AnyFile, F);

    WHILE (DOSError = 0) DO        (* If found then continue...             *)
      BEGIN

        (* Seek file by size? *)
	IF (SeekSize) THEN
	  IF ValidFileSize (F.Size) AND (F.Attr AND Directory = 0) AND
	     (F.Attr AND VolumeID = 0) THEN
	    goto Found1
	  ELSE
	    goto Next;

	Found1:

        (* Seek file by attribute? *)
	IF (SeekAttr) THEN
	  IF (Dire AND (F.Attr AND Directory <> 0)) OR
	     (RO   AND (F.Attr AND ReadOnly  <> 0)) OR
	     (Hid  AND (F.Attr AND Hidden    <> 0)) OR
	     (Sys  AND (F.Attr AND SysFile   <> 0)) OR
	     (At   AND (F.Attr AND Archive   <> 0)) THEN
	    goto Found
	  ELSE
	    goto Next;

	Found:

        (* Show file information? *)
	IF ShowInfo THEN
	  BEGIN
	    IF DirNotShown THEN
	      BEGIN
		Writeln ('');
		Writeln (Path);
		DirNotShown := FALSE
	      END;

	    Attr := '';
	    IF ((F.Attr AND Archive)  <> 0) THEN
	      Attr := Attr + 'Arc ';
	    IF ((F.Attr AND ReadOnly) <> 0) THEN
	      Attr := Attr + 'R-O ';
	    IF ((F.Attr AND Hidden)   <> 0) THEN
	      Attr := Attr + 'Hid ';
	    IF ((F.Attr AND SysFile)  <> 0) THEN
	      Attr := Attr + 'Sys';

	    unpacktime (F.Time, FileDate);

	    write ('    ', F.Name:13);
	    IF (F.Attr AND Directory <> 0) THEN
	      write ('        <DIR>      ')
	    ELSE IF (F.Attr AND VolumeID <> 0) THEN
	      write ('     <Volume ID>   ')
	    ELSE
	      write (F.Size:10, ' bytes   ');
	    WITH FileDate DO
	      Writeln (Date (Day) + '-' + Date(Month) + '-' + Num2Str (Year) +
                       '  ' + Date(Hour) + ':' + Date(Min) +
                       '  ' + Attr)
	  END (* IF *)

        (* Display without information. *)
	ELSE
	  Writeln (Path + F.Name);

	inc (FilesFound);

        (* Execute file? *)
	IF RunFile THEN
	  BEGIN
	    fsplit (F.Name, Dir, Name, Ext);
	    IF (Ext = '.COM') OR (Ext = '.EXE') OR (Ext = '.BAT') THEN
	      BEGIN
		swapvectors;
		exec (getenv ('COMSPEC'), '/C ' + Path + F.Name);
		swapvectors
	      END
	  END;

        (* Execute command of file? *)
	IF ExecuteCommand THEN
	  BEGIN
	    swapvectors;
	    exec (getenv ('COMSPEC'), '/C ' + Command + ' ' + Path + F.Name);
	    swapvectors
	  END;

	Next:

	findnext (F)
      END; (* WHILE *)

    (* Continue with search... *)

    findfirst (Path + '*.*', AnyFile, F);
    WHILE (DOSError = 0) DO
      BEGIN
	IF ((F.Attr AND Directory) <> 0) AND (F.Name[1] <> '.') THEN
	  SeekFile (Path + F.Name + '\');
	findnext (F)
      END

  END; (* SeekFile *)

(************************************************************************)

FUNCTION ABDrive: boolean; assembler;
(* Returns TRUE if system has one drive acting as A and B. This requires
   DOS 3.2+.                                                             *)
  ASM
    push ds               (* To be safe                                  *)
    mov  ax, 440Eh        (* IOCTL function (44h) and subfunction 0Eh    *)
    mov  bl, 01h          (* Checking drive A                            *)
    int  21h              (* Execute DOS function                        *)
    jc   @Error           (* If Carry Flag is set then there's trouble   *)
    cmp  al, 00h          (* See if A/B are one drive                    *)
    jne  @Okay            (* Yes they are, so jump                       *)
  @Error:
    xor  al, al           (* No they are not, or there was an error      *)
    jmp  @Done
  @Okay:
    mov  al, 01h          (* Set function result to TRUE                 *)
  @Done:
    pop  ds               (* Restore DS register                         *)
  END;

(************************************************************************)

PROCEDURE ForceDriveA; assembler;
(* Force drive B: to be assumed as A:.  This requires DOS 3.2+.         *)
  ASM
    push ds         (* save DS                                          *)
    mov  ax, 440Fh  (* IOCTL function (44h) and set logical drive (0Fh) *)
    mov  bl, 01h    (* Drive A: = 2                                     *)
    int  21h
    pop  ds         (* restore DS                                       *)
  END;

(************************************************************************)

FUNCTION DOSVersion: word; assembler;
(* Returns the DOS version number in register AX. The major version number
   is in AL and the minor version number is in AH.  Sure I could have
   used the function from the DOS unit, but this reduces code.          *)
  ASM
    mov ah, 30h
    int 21h
  END;

(****************************************************************************)

FUNCTION ChPos (C: char; Str : string) : byte; assembler;
(* This is a much faster version of the system.pos function except that
   this function searches only for a character, not a sub-string.    *)
  ASM
    cld                  (* Forward string-ops                       *)
    les   di, Str        (* Load string address                      *)
    mov   cx, es:[di]    (* Load string length                       *)
    jcxz  @1             (* Exit if string length = 0                *)
    mov   bl, cl         (* Save string length                       *)
    xor   ch, ch         (* Only want lower word                     *)
    inc   di             (* Start index at beginning of Str          *)
    mov   al, c          (* Load character to find                   *)
    repne scasb          (* Scan Str for character                   *)
    jne   @1             (* Jump if we didn't find                   *)
    sub   bl, cl         (* Subtract index from Str length for index *)
    xchg  al, bl         (* Return position found                    *)
    jmp   @2             (* Get outta here                           *)
  @1:
    xor   al, al         (* Character wasn't found                   *)
  @2:
END;

(*************************************************************************)
(*************************************************************************)


BEGIN (* main program *)

  Writeln := NormWrite;

  Writeln (Title);

  (* Needs at least DOS 3.2 to run (for ABDrive and ForceDriveB procedures). *)

  IF (lo (DOSVersion) <= 3) AND (hi (DOSVersion) < 20) THEN
    CheckError (31);

  PC := paramcount;   (* Since the parameter count is accessed a lot,
                         this reduces code.                                 *)

  IF (PC = 0) THEN    (* Why not put a halt(0) after this instead of adding *)
    DisplayOptions    (* the ELSE?  This method reduces code.               *)

  ELSE
    BEGIN

      FOR I := 1 TO PC DO    (* Scan command-line for options *)
	BEGIN
	  Param := paramstr (I);
	  IF (Param[1] = '/') THEN
	    BEGIN
	      inc (NumOptions);

	      IF (length (Param) = 1) THEN
		CheckError (24);

	      FOR J := 2 TO length (Param) DO
		CASE upcase (Param[J]) OF
		  'R': RunFile  := TRUE;   (* Run file                      *)
		  'I': ShowInfo := TRUE;   (* Show file info                *)
		  'V': BEGIN               (* Use standard output and input *)
			 assign  (output, '');
			 assign  (input , '');
			 rewrite (output);
			 reset   (input)
		       END;
		  'C': BEGIN               (* Execute a command on file     *)
			 Command := UpCaseStr (copy (Param, J+1, length (Param)) );
			 ExecuteCommand := TRUE;
			 IF (length (Command) = 0) THEN
			   CheckError (23);
                         goto Next
		       END;
		  'P': BEGIN               (* Pause with each screenfull    *)
			 Writeln := PauseWrite;
			 NumRows := hi (CRT.windmax)
		       END;
		  'D': BEGIN               (* Search all drives             *)
			 Drive.Number := (ord (LastDrive) - 64);
			 AllDrives    := TRUE;
                         IF ABDrive THEN
                           BEGIN           (* Force drive B to be A         *)
                             ForceDriveA;
                             Drive.Letter[2] := 'A';
                             StartDrive      := 2
                           END;
		       END;
		  'S': BEGIN               (* Search by file size           *)
			 TmpStr   := copy (Param, J+1, length (Param));
			 SeekSize := TRUE;
			 IF (Length (TmpStr) = 0) THEN
			   CheckError (27);
			 IF (TmpStr[1] = '-') THEN
			   BEGIN
			     delete (TmpStr, 1, 1);
			     ValidFileSize := LessThan
			   END
			 ELSE IF (TmpStr[length(TmpStr)] = '-') THEN
			   BEGIN
			     delete (TmpStr, length (TmpStr), 1);
			     ValidFileSize := GreaterThan
			   END
			 ELSE
			   ValidFileSize := Equal;
			 val (TmpStr, MinSize, Tmp);
			 IF (Tmp <> 0) THEN
			   BEGIN
			     ColonPos := ChPos ('-', TmpStr);
			     val ((copy (TmpStr, 1, ColonPos - 1)), MinSize, Tmp);
			     IF (Tmp <> 0) THEN
			       CheckError (30);
			     val ((copy (TmpStr, ColonPos + 1, length (TmpStr))), MaxSize, Tmp);
			     IF (Tmp <> 0) THEN
			       CheckError (30);
			     IF (MinSize > MaxSize) THEN
			       BEGIN
				 TmpLong := MinSize;
				 MinSize := MaxSize;
				 MaxSize := TmpLong
			       END;
			     ValidFileSize := Between
			   END;
			 IF (MinSize < MinFileSize) THEN
			   CheckError (30);
                         goto Next
		       END;
		  'A': BEGIN               (* Search file by attribute      *)
			 TmpStr   := copy (Param,J+1,length(Param));
			 SeekAttr := TRUE;
			 IF (length (TmpStr) = 0) THEN
			   CheckError (26);
			 FOR Tmp := 1 TO length (TmpStr) DO
			   CASE upcase (TmpStr[Tmp]) OF
			     'A': At   := TRUE;
			     'R': RO   := TRUE;
			     'S': Sys  := TRUE;
			     'H': Hid  := TRUE;
			     'D': Dire := TRUE
			     ELSE
			       CheckError (28)
			   END; (* CASE *)
                         goto Next
		       END;
		  '?',
		  'H': DisplayHelp;
		  ELSE
		    CheckError (20)
		END; (* CASE *)
	    END; (* IF *)
          Next:
	END; (* FOR *)

      (* Get the current directory. *)
      getdir (0, CurrDir);

      FOR I := 1 TO PC DO
	BEGIN
	  Param := UpCaseStr (paramstr (I));

	  IF (Param[1] <> '/') THEN  (* Seek file!!                       *)
	    BEGIN

              (* See if we have to search more than one drive.            *)
	      ColonPos := ChPos (':', Param);

              (* If /D options was specified AND they specify certain
                 drives to search, searching all drives takes precedence. *)

	      IF (ColonPos <> 0) THEN
		BEGIN
                  IF (NOT AllDrives) THEN
                    BEGIN
                      IF (ColonPos = length (Param)) THEN (* No file specified.  *)
		        CheckError (21);

		      IF (ColonPos > 27) THEN    (* Max. of 26 drives: A..Z.     *)
		       CheckError (25);

		      Drive.Number := 0;
		      FOR J := 1 TO (ColonPos-1) DO
		        BEGIN
		          inc (Drive.Number);
		          Drive.Letter[J] := upcase (Param[J]);
		        END
                    END;
		  delete (Param, 1, ColonPos)
		END

	      ELSE IF (NOT AllDrives) THEN  (* else search just current drive *)
                BEGIN
  	          Drive.Letter[1] := currDir[1];
                  Drive.Number    := 1
                END;

	      fsplit (Param, Dir, Name, Ext);
	      IF (length (Dir) = 0) THEN
		Dir := '\'
	      ELSE
		delete (Param, 1, length (Dir));

	      Writeln ('Searching for ' + Param + '...');

              (* Start searching drive(s). *)
	      FOR J := StartDrive TO Drive.Number DO
		BEGIN

                  (* See if drive is valid first. *)
		  chdir (Drive.Letter[J] + ':\');
		  Tmp := IOResult;
		  chdir (CurrDir);
		  CheckError (IOResult);
		  IF (Tmp <> 0) THEN
		    CheckError (Tmp)

                  ELSE     (* valid drive letter *)
	            SeekFile (fexpand (Drive.Letter[J] + ':' + Dir))

		END; (* FOR *)

              IF (FilesFound > 0) THEN
                Writeln ('');

	      Writeln ('# of files found: ' + Num2Str (FilesFound));
              inc (TotalFound, FilesFound);
	      FilesFound := 0
	    END (* IF *)
	END; (* FOR *)

      IF (TotalFound > 0) THEN
        Writeln ('Total # of files: ' + Num2Str (TotalFound));

      IF (NumOptions = PC) THEN   (* If no files specified, then error. *)
	CheckError (21)

    END (* ELSE *)
END. (* main program *)