/***************************************************************

	sfs_gt.c        Ground Track Routines for SFS

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "ctype.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "sfs.h"

#ifdef  __STDC__
#include "stdlib.h"
#endif

#define GT_DIVIDEND     4       /* Size of gt display on x should be this */
#define GT_DIVISOR      2       /* ...divided by this (5/2 = 2.5 * y size */

static int gt_dready;           /* Data ready */
static int gt_xsize;            /* Size of ground track map, x */
static int gt_ysize;            /* Size of ground track map, y */
static int gt_mapbase;          /* Bottom of gt map */
static int gt_mapedge;          /* Left edge of groundtrack map */

/***************************************************************

   gt_init()

   This function provides all necessary initialization
   for the ground track system.

***************************************************************/

gt_init( sorbit_array, orbit )
   struct sfs_orbit **sorbit_array;
   int orbit;
   {
   sorbit_array[ orbit ]->gt_bufstart
      = sorbit_array[ orbit ]->gt_bufpos = 0;
   gt_dready = FALSE;
   }

/***************************************************************

   gt_update()

   This function updates the ground track circular buffer
   by adding a new pair of latitude and longitude
   coordinates to the buffer.

***************************************************************/

gt_update( newtime, sorbit_array, n_orbits )
   long newtime;
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   {
   register int orbit;
   static double lat, lon;
   static long r, n;

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {

#ifdef	OLD_DEBUG
      sprintf( bw_ebuf, "DEBUG: update orbit %d", orbit );
      bw_message( bw_ebuf );
#endif

      if ( sorbit_array[ orbit ] != NULL )
	 {
	 or_ssp( sorbit_array[ orbit ]->aorbit, newtime, &lat, &lon, &r, &n );
	 sorbit_array[ orbit ]->gt_buffer[ sorbit_array[ orbit ]->gt_bufpos ][ 0 ] = lat;
	 sorbit_array[ orbit ]->gt_buffer[ sorbit_array[ orbit ]->gt_bufpos ][ 1 ] = lon;
	 ++sorbit_array[ orbit ]->gt_bufpos;
	 if ( sorbit_array[ orbit ]->gt_bufpos == GT_POINTS )
	    {
	    sorbit_array[ orbit ]->gt_bufpos = 0;
	    }
	 if ( sorbit_array[ orbit ]->gt_bufpos == sorbit_array[ orbit ]->gt_bufstart )
	    {
	    ++sorbit_array[ orbit ]->gt_bufstart;
	    if ( sorbit_array[ orbit ]->gt_bufstart == GT_POINTS )
	       {
	       sorbit_array[ orbit ]->gt_bufstart = 0;
	       }
	    }
	 }              /* end if orbit != NULL */
      }                 /* end for orbits loop */
   gt_dready = TRUE;

#ifdef	OLD_DEBUG
   bw_debug( "Orbits updated" );
#endif
   }

/***************************************************************

   gt_draw()

   This function draws the ground track screen.
   It is called by fo_draw().

***************************************************************/

gt_draw( display, uiwind, focus, elements, redraw_screen,
   s_start, s_end, sorbit_array, n_orbits )
   struct sfs_display *display;
   struct uiwindow *uiwind;
   struct as_focus *focus;      /* orbital focus to display */
   unsigned int elements;
   int redraw_screen;
   struct spj_pt *s_start, *s_end;
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   {
   register int orbit;
   int x, y;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "gt_draw(): %d %d %d %d", uiwind->u_x1, uiwind->u_y1,
      uiwind->u_x2, uiwind->u_y2 );
   bw_debug( bw_ebuf );
#endif

   bw_message( GT_DRAW );

   if ( redraw_screen == TRUE )
      {

      /* Prepare the screen */

      gt_prep( uiwind );
      display->gt_xsize   = gt_xsize;
      display->gt_ysize   = gt_ysize;
      display->gt_mapbase = gt_mapbase;
      display->gt_mapedge = gt_mapedge;

#ifdef  OLD_DEBUG
      bw_debug( "Ready to draw grid" );
#endif

      /* Draw the surface features */

      gt_plot( s_start, s_end, cl_surface, SOLID );

      /* Display the point data if toggled */

#ifdef  USEPOINTS
      if ( ( elements & VI_POINTS ) > 0 )
	 {
	 gt_plot( p_start, p_end, cl_grid, SOLID );
	 }
#endif

      /* Display the point titles if toggled */

#ifdef  USEPOINTS
      if ( ( elements & VI_PTITLES ) > 0 )
	 {
	 gt_titles( uiwind, p_start, p_end, cl_mback, cl_mfore );
	 }
#endif

      /* Show toggle status */

      el_show( GR_PRIMARY, uiwind, elements );

      /* Title the screen */

      sprintf( bw_ebuf, GT_WTITLE,
	 focus->adjective );
      ui_wtitle( uiwind, bw_ebuf );

     }

   else
      {
      gt_xsize = display->gt_xsize;
      gt_ysize = display->gt_ysize;
      gt_mapbase = display->gt_mapbase;
      gt_mapedge = display->gt_mapedge;
      }

   /* Show the ground track */

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( ( sorbit_array[ orbit ] != NULL )
         && ( sorbit_array[ orbit ]->aorbit->focus == focus ))
	 {
	 gt_track( sorbit_array, orbit, cl_orbits[ orbit % 6 ] );
	 }                              /* end if orbit != NULL */
      }                                 /* end for orbit loop */

   }

/***************************************************************

   gt_prep()

   This routine draws the panel at the bottom of the screen
   which contains various parameters and is used when the
   ground track display is in effect.  The routine is called
   by gt_draw().

***************************************************************/

gt_prep( uiwind )
   struct uiwindow *uiwind;
   {
   register int c;
   int inc;
   double x;
   int x_xmax, x_ymax;

#ifdef  OLD_DEBUG
   bw_debug( "gt_prep(): enter" );
#endif

   /***  Blank the entire window area */

   ui_fbox( uiwind->u_x1, uiwind->u_y1,
      uiwind->u_x2, uiwind->u_y2, BLACK, SOLID );

   /***  First determine if x or y axis must be scaled */

   x_xmax = uiwind->u_x2 - uiwind->u_x1;
   x_ymax = uiwind->u_y2 - uiwind->u_y1;

   if ( ( x_xmax / 2 ) >
	( ( x_ymax * gr_pysize ) / gr_pxsize )
      )

      {                              /* TRUE = scale x */

#ifdef  OLD_DEBUG
      bw_debug( "Scaling x" );
#endif

      gt_ysize = x_ymax;
      gt_xsize = 2 * (( gt_ysize * gr_pysize ) / gr_pxsize );

      /***  Calculate gt_mapedge */

      gt_mapedge = uiwind->u_x1 +
	 ( ( x_xmax - gt_xsize ) / 2 );

      /***  Calculate gt_mapbase */

      gt_mapbase = uiwind->u_y1;

      /***  Draw lines demarcating the display within the window */

      gr_line( GR_PRIMARY, gt_mapedge, gt_mapbase, gt_mapedge,
	 uiwind->u_y2, cl_grid, SOLID );

      gr_line( GR_PRIMARY, gt_mapedge + gt_xsize, gt_mapbase,
	 gt_mapedge + gt_xsize, uiwind->u_y2,
	 cl_grid, SOLID );

      }

   else                             /* FALSE = scale y */
      {

#ifdef  OLD_DEBUG
      bw_debug( "Scaling y" );
#endif

      gt_xsize = x_xmax;
      gt_ysize = (( gt_xsize * gr_pxsize ) / gr_pysize ) / 2;

      /***  Calculate gt_mapedge */

      gt_mapedge = uiwind->u_x1;

      /***  Calculate gt_mapbase */

      gt_mapbase = uiwind->u_y1 +
	 ( ( x_ymax - gt_ysize ) / 2 );

      /***  Draw lines demarcating the display within the window */

      gr_line( GR_PRIMARY, uiwind->u_x1, gt_mapbase, uiwind->u_x2,
	 gt_mapbase, cl_grid, SOLID );

      gr_line( GR_PRIMARY, uiwind->u_x1, gt_mapbase + gt_ysize,
	 uiwind->u_x2, gt_mapbase + gt_ysize,
	 cl_grid, SOLID );

      }

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "x_size %d, y_size %d, mapedge %d, mapbase %d",
      gt_xsize, gt_ysize, gt_mapedge, gt_mapbase );
   bw_debug( bw_ebuf );
#endif

   /***  Now draw the latitude and longitude lines. We must draw
	 only a quarter of a line at a time, or gt_line will
	 reject the request. */

   for ( x = -75.0; x < 90.0; x += 15.0 )               /* latitudes */
      {
      gt_line( x, -180.0, x, -90.0, cl_grid, GRID );
      gt_line( x,  -90.0, x,   0.0, cl_grid, GRID );
      gt_line( x,    0.0, x,  90.0, cl_grid, GRID );
      gt_line( x,   90.0, x, 180.0, cl_grid, GRID );
      }

   for ( x = -165.0; x < 180.0; x += 15.0 )             /* longitudes */
      {
      gt_line( -90.0, x, -45.0, x, cl_grid, GRID );
      gt_line( -45.0, x,   0.0, x, cl_grid, GRID );
      gt_line(   0.0, x,  45.0, x, cl_grid, GRID );
      gt_line(  45.0, x,  90.0, x, cl_grid, GRID );
      }

#ifdef  OLD_DEBUG
   bw_debug( "end of gt_prep() " );
#endif

   }

/***************************************************************

   gt_plot()

   This routine plots a set of coordinates on the ground
   track map.  It is called by gt_draw().

***************************************************************/

gt_plot( start, end, color, style )
   struct spj_pt *start, *end;
   int color, style;
   {
   struct spj_pt *current;
   double prevx, prevy;

   current = start->next;

#ifdef  OLD_DEBUG
   bw_debug( "DEBUG:  Plotting" );
#endif

   prevx = current->longitude;
   prevy = current->latitude;
   while ( current != end )
      {
      if ( current->code < 1000 )
	 {
	 gt_line( current->latitude, current->longitude,
	    prevy, prevx, color, style );
	 }
      prevx = current->longitude;
      prevy = current->latitude;
      current = current->next;
      }

#ifdef  OLD_DEBUG
   bw_message( " " );
#endif
   }

gt_titles( uiwind, start, end, background, foreground )
   struct uiwindow *uiwind;
   struct spj_pt *start, *end;
   int background, foreground;
   {
   struct spj_pt *current;
   int x, y;

   /* Turn clipping on */

   gr_clip( GR_PRIMARY, TRUE, gt_mapedge, gt_mapbase,
      gt_mapedge + gt_xsize, uiwind->u_y2 );

   current = start;
   while( current != end )
      {
      x = ( current->longitude + 180.0 ) * ( gt_xsize / 360.0 );
      y = ( current->latitude + 90.0 ) * ( gt_ysize / 180.0 );
      x = gt_mapedge + x;

      gr_text( GR_PRIMARY,
	 x + 3,
	 y - ( ui_grwind->fysize / 2 ),
	 current->name, foreground, background );

      current = current->next;
      }

   /* Turn clipping off */

   gr_clip( GR_PRIMARY, FALSE, 0, 0, 0, 0 );

   }

/***************************************************************

   gt_track()

   This routine draws the spacecraft ground-track path on
   the ground track map.  It is called by gt_draw().

***************************************************************/

gt_track( sorbit_array, orbit, color )
   struct sfs_orbit **sorbit_array;
   int orbit, color;
   {
   int current;

   if ( gt_dready != TRUE )
      {
      bw_error( GTERR_NODATA );
      return BW_ERROR;
      }

   current = sorbit_array[ orbit ]->gt_bufstart + 1;
   if ( current == GT_POINTS )
      {
      current = 0;
      }

#ifdef  OLD_DEBUG
   bw_message( "DEBUG:  Tracking " );
#endif

   while ( current != sorbit_array[ orbit ]->gt_bufpos )
      {
      if ( current == 0 )
	 {
#ifdef  OLD_DEBUG
	 sprintf( bw_ebuf, "gt_track from x = %.2lf y = %.2lf, to x = %.2lf y = %.2lf, color = %d, style = %d,",
	    sorbit_array[ orbit ]->gt_buffer[ GT_POINTS - 1 ][ 0 ],
	    sorbit_array[ orbit ]->gt_buffer[ GT_POINTS - 1 ][ 1 ],
	    sorbit_array[ orbit ]->gt_buffer[ current       ][ 0 ],
	    sorbit_array[ orbit ]->gt_buffer[ current       ][ 1 ],
	    color, SOLID );
	 bw_debug( bw_ebuf );
#endif
	 gt_line( sorbit_array[ orbit ]->gt_buffer[ GT_POINTS - 1 ][ 0 ],
		  sorbit_array[ orbit ]->gt_buffer[ GT_POINTS - 1 ][ 1 ],
		  sorbit_array[ orbit ]->gt_buffer[ current       ][ 0 ],
		  sorbit_array[ orbit ]->gt_buffer[ current       ][ 1 ],
		  color, SOLID );
	 }
      else
	 {
#ifdef  OLD_DEBUG
	 sprintf( bw_ebuf, "gt_track from x = %.2lf y = %.2lf, to x = %.2lf y = %.2lf, color = %d, style = %d,",
	    sorbit_array[ orbit ]->gt_buffer[ current - 1   ][ 0 ],
	    sorbit_array[ orbit ]->gt_buffer[ current - 1   ][ 1 ],
	    sorbit_array[ orbit ]->gt_buffer[ current       ][ 0 ],
	    sorbit_array[ orbit ]->gt_buffer[ current       ][ 1 ],
	    color, SOLID );
	 bw_debug( bw_ebuf );
#endif
	 gt_line( sorbit_array[ orbit ]->gt_buffer[ current - 1 ][ 0 ],
		  sorbit_array[ orbit ]->gt_buffer[ current - 1 ][ 1 ],
		  sorbit_array[ orbit ]->gt_buffer[ current     ][ 0 ],
		  sorbit_array[ orbit ]->gt_buffer[ current     ][ 1 ],
		  color, SOLID );
	 }
      ++current;
      if ( current == GT_POINTS )
	 {
	 current = 0;
	 }
      }

   /* Now plot the forward track */

   ft_plot( sorbit_array, orbit );

   }

/***************************************************************

   gt_line()

   This routine draws a single line on the ground
   track map.

***************************************************************/

gt_line( fromlat, fromlon, tolat, tolon, color, style )
   double fromlat, fromlon, tolat, tolon;
   int color, style;
   {
   int x1, y1, x2, y2;

#ifdef  DEBUG
   if ( ( fromlat < -90.0 ) || ( fromlat > 90.0 ))
      {
      sprintf( bw_ebuf, "gt_line received from latitude %.2lf", fromlat );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( tolat < -90.0 ) || ( tolat > 90.0 ))
      {
      sprintf( bw_ebuf, "gt_line received to latitude %.2lf", tolat );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( fromlon < -180.0 ) || ( fromlon > 180.0 ))
      {
      sprintf( bw_ebuf, "gt_line received from longitude %.2lf", fromlon );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( tolon < -180.0 ) || ( tolon > 180.0 ))
      {
      sprintf( bw_ebuf, "gt_line received to longitude %.2lf", tolon );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "DEBUG from x = %.2lf, from y = %.2lf, to x = %.2lf, to y = %.2lf, color = %d, style = %d,",
      fromlon,
      fromlat,
      tolon,
      tolat,
      color, style );
   bw_debug( bw_ebuf );
#endif

   x1 = (( fromlon + 180.0 ) * ( gt_xsize / 360.0 ));
   x2 = ((   tolon + 180.0 ) * ( gt_xsize / 360.0 ));

   if ( abs( x1 - x2 ) > ( gt_xsize / 3 ) )
      {
#ifdef  OLD_DEBUG
      sprintf( sfs_tbuf, "DEBUG: gt_line() failed: x1 = %d, x2 = %d",
	 x1, x2 );
      bw_debug( sfs_tbuf );
      sprintf( sfs_tbuf, "DEBUG: gt_line() #2: abs() = %d, gt_xsize = %d, gt_xsize / 3 = %d",
	 abs( x1 - x2), gt_xsize, (gt_xsize / 3) );
      bw_debug( sfs_tbuf );
#endif
      return;
      }

   y1 = (( fromlat + 90.0 ) * ( gt_ysize / 180.0 ));
   y2 = ((   tolat + 90.0 ) * ( gt_ysize / 180.0 ));

   if ( abs( y1 - y2 ) > ( gt_ysize / 3 ) )
      {
#ifdef  OLD_DEBUG
      sprintf( sfs_tbuf, "gt_line() failed: y1 = %d, y2 = %d",
	 y1, y2 );
      bw_debug( sfs_tbuf );
#endif
      return;
      }

#ifdef  OLD_DEBUG
   sprintf( sfs_tbuf, "DEBUG x1 = %d, y1 = %d, x2 = %d, y2 = %d, color = %d, style = %d,",
      gt_mapedge + x1,
      gt_mapbase + y1,
      gt_mapedge + x2,
      gt_mapbase + y2,
      color, style );
   bw_debug( sfs_tbuf );
#endif

   gr_line( GR_PRIMARY,
      gt_mapedge + x1,
      gt_mapbase + y1,
      gt_mapedge + x2,
      gt_mapbase + y2,
      color, style );
   }
