/**********************************************************************
 *  
 *  NAME:           getstr.cpp
 *  
 *  DESCRIPTION:    getstr() etc functions for window lib
 *  
 *  copyright (c) 1987-90 J. Alan Eldridge
 * 
 *  M O D I F I C A T I O N   H I S T O R Y
 *
 *  when        who                 what
 *  -------------------------------------------------------------------
 *  09/05/90    J. Alan Eldridge    created from wgetstr.c (UWin v. 3.10)
 *  
 *  09/06/90    JAE                 limited to one line on screen
 *  
 *  11/29/90    JAE                 modified from UWin++ 1.0
 *
 *********************************************************************/

#include    "w.h"

//  local defines

#define ONE_LINE_ONLY   1       //  string input doesn't line wrap
#define GETSTR_MAX      512     //  buffer size for egetstr()

//  default character validation function

static int
ok(int c)
{
    return TRUE;
}

//  replace one char with another

static void
replace(
    uchar   *pStr,
    uchar   srch,
    uchar   rep)
{
    while (pStr = strchr(pStr, srch))
        *pStr++ = rep;
}            

int
basewin::editfld(
    uchar   *pBuf,
    int     bMax,
    int     *pTermKeys,
    int     eFlags,
    int     (*OKFunc)(int c),
    int     fillChr)
{
    int     c;
    int     bLen;
    int     ins     =   TRUE;
    int     bPos    =   0;
    int     y, x;
    int     errBeep =   !(eFlags & noErrBeep);
    uchar   *cp;
    
    static int  nlESC[] = { K_NL, K_ESC, K_ILLEGAL };

#if ONE_LINE_ONLY
    //  limit to one line of window
    
    if (bMax > xLR - xCur + 1) {
        bMax = xLR - xCur + 1;
        pBuf[bMax] = 0;
    }
#endif

    cp = pBuf + (bLen = strlen(pBuf));

    //  check for defaults
    
    if (!OKFunc) OKFunc = ok;
    if (!pTermKeys) pTermKeys = nlESC;
    
    //  pad or trim string as necessary

    if ((eFlags & trimStr) || fillChr != ' ') {
        while (bLen > 0 && cp[-1] == ' ') {
            bLen--; 
            *--cp = 0;
        }
    }

    if ((eFlags & expandStr) && bLen < bMax) {
        while (bLen < bMax) {
            bLen++;
            *cp++ = fillChr;
        }
        *cp = 0;
    }

    getpos(y, x);
    put(pBuf);
    setpos(y, x);
    refresh();
    
    while (!findkey(pTermKeys, c = getkey())) {

        //  check for field auto-exit (data entry forms...)

        if ((eFlags & autoTab) == autoTab) {
            if (((c == K_RIGHT || c == ' ') && bPos == bMax) || c == K_TAB
                    || c == K_NL)
                return K_TAB;
            else if (((c == K_LEFT || c == K_BACK) && !bPos) || c == K_BACKTAB)
                return K_BACKTAB;
        } 

        //  try to process a self-insert character
        
        if (isascii(c) && isprint(c) && OKFunc(c)) {
            if (eFlags & zapField) {
                int     y, x, f;
                
                while (bPos > 0) {
                    bPos--;
                    BS();
                }
                getpos(y, x);
                f = (eFlags & expandStr) ? fillChr : 0;
                while (bLen > 0) {
                    pBuf[--bLen] = f;
                    put(fillChr);
                }
                if (eFlags & expandStr)
                    bLen = bMax;
                setpos(y, x);
            }
            if (bPos >= bLen) {
                if (bLen < bMax) {
                    pBuf[bLen++] = c;
                    pBuf[++bPos] = 0;
                    put(c);
                } else if (errBeep) {
                    beep();
                }
            } else if (ins) {
                if (bLen < bMax || (bLen == bMax && pBuf[bLen-1] == fillChr)) {
                    int     n, y, x;

                    for (n = bLen - (bLen >= bMax); n > bPos; n--)
                        pBuf[n] = pBuf[n - 1];
                    put(pBuf[bPos] = c);
                    if (bLen < bMax)
                        pBuf[++bLen] = 0;
                    getpos(y, x);
                    put(pBuf + (++bPos));
                    setpos(y, x);
                } else if (errBeep) {
                    beep();
                }
            } else {
                put(pBuf[bPos++] = c);
            }
            refresh();
            eFlags &= ~zapField;
            continue;   //  done with self-insert char
        }

        //  it's either an editing char or illegal
        
        switch (c) {
        case K_LEFT:
            if (bPos > 0) {
                bPos--;
                BS();
            } else if (errBeep) {
                beep();
            }
            break;
        case K_RIGHT:
            if (bPos < bLen) {
                bPos++;
                put(get());
            } else if (errBeep) {
                beep();
            }
            break;
        case K_HOME:
            while (bPos > 0) {
                bPos--;
                BS();
            }
            break;
        case K_END:
            while (bPos < bLen) {
                bPos++;
                put(get());
            }
            break;
        case K_INS:
            ins = !ins;
            break;
        case K_CTLX:
            //  ctl-x cancels the entire line (unix)
            {
                int     y, x, f;

                while (bPos > 0) {
                    bPos--;
                    BS();
                }
                f = (eFlags & expandStr) ? fillChr : 0;
                getpos(y, x);
                while (bLen > 0) {
                    pBuf[--bLen] = f;
                    put(fillChr);
                }
                if (eFlags & expandStr)
                    bLen = bMax;
                setpos(y, x);
            }
            break;
        case K_BS:
        case K_DEL:
            if (c == K_BS)
                if (bPos > 0) {
                    bPos--;
                    BS();
                } else {
                    if (errBeep)
                        beep();
                    break;
                }
            if (bPos < bLen && bLen > 0) {
                int     p, y, x;

                for (p = bPos; p < bLen-1; p++)
                    pBuf[p] = pBuf[p+1];
                pBuf[p] = fillChr;
                getpos(y,x);
                put(pBuf+bPos);
                setpos(y,x);
                if (eFlags & trimStr) {
                    pBuf[p] = 0;
                    bLen--;
                }
            } else if (errBeep) {
                beep();
            }
            break;
        default:
            if (errBeep)
                beep();
            break;
        }

        refresh();
        eFlags &= ~zapField;
    }

    if (fillChr != ' ')
        replace(pBuf, fillChr, ' ');
        
    if (eFlags & trimStr) {
        while (bLen > 0 && pBuf[bLen-1] == ' ')
            pBuf[--bLen] = 0;
    }

    setpos(y, x);
    put(pBuf);

    return c;
}

int
basewin::egetstr(
    uchar   *pBuf)
{
    static int  nl[] = { K_NL, K_ILLEGAL };

    int     retVal;
    char    bufLocal[GETSTR_MAX];
    
    strncpy(bufLocal, pBuf, GETSTR_MAX);
    bufLocal[GETSTR_MAX-1] = 0;
    retVal = editfld(bufLocal, GETSTR_MAX-1, nl, trimStr, ok);
    strcpy(pBuf, bufLocal);
    if (retVal == K_NL) {
        NL();
        refresh();
    }
    return retVal;
}
    
int
basewin::getstr(
    uchar   *pBuf)
{
    pBuf[0] = 0;
    return egetstr(pBuf);
}            
