/*
 *========================================================================== 
 * Copyright 1991 Avinash Chopde, All Rights Reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Avinash Chopde not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * Avinash Chopde makes no representations about the suitability of this
 * software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *
 * AVINASH CHOPDE DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL AVINASH CHOPDE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 *
 * Author:  Avinash Chopde, 1991
 *	    C2 Colonial Drive #4, Andover, MA 01810, USA.
 *
 */

#include "itrans.h"
#include "ifm.h"

static char S_RCSID[] = "$Header: e:/itrans/src/rcs/font.c 1.11 91/10/14 00:40:13 avinash Exp $";

/* definitions used by decode_name() */
#define	ALL_CHARS	-1
	/* ALL_CHARS must be zero-1, see decode_name() */
#define	NO_CHAR		-2

/* =================================================================== */
static int S_ccadd(FILE* ifmfp);
static void S_clear_ccadds();
static int S_ccs(FILE* ifmfp, font_t *font);
static int S_cc(FILE* ifmfp, font_t *font);
static int S_prop(FILE* ifmfp, font_t *font);
static int S_fillup_ps(FILE* afmfp, font_t* font);
static int S_fillup_tfm(font_t* font);

/* The extra_map[] stores user defined khadi names --
 * these are only used in "same-as" (CCS) IFM tag
 * Cannot be generated directly by the user.
 * See also the definition of G_ifm_map ahead in this file.....
 */
static ifm_enc_t S_ifm_extra_map[NUMEXTRA];

/* =================================================================== */
/* initialise font with empty values */
void init_font(font_t* font)
{
    int i, j;

    font->prop = UNDEF_FONT;
    font->name[0] = '\0';
    font->fname[0] = '\0';
    font->use_ligatures = TRUE;

    for (i = 0; i < NUMKHADI; i ++) {
	font->khadi[i].cus = NULL;
	font->khadi[i].same_as = NULL;
    }

    for (i = 0; i < NUMCHARS; i ++) {
        for (j = 0; j < NUMCHARS; j ++) {
	    font->ligatures[i][j].cus = NULL;
	    font->ligatures[i][j].same_as = NULL;
        }
    }
    for (i = 0; i < 10; i ++) {
	font->digits[i].cus = NULL;
	font->digits[i].same_as = NULL;
    }

    for (i = 0; i < NUMPSCHARS; i ++) {
	font->psfm[i].w = 0;
	font->psfm[i].llx = 0;
	font->psfm[i].lly = 0;
	font->psfm[i].urx = 0;
	font->psfm[i].ury = 0;
    }
} /* init_font() */

/* =================================================================== */
font_t*	find_font(allfonts_t af, char fname[])
{
    font_t* f;
    int i;

#ifdef DEBUG
    fprintf(stderr, "looking for font %s\n", fname);
#endif

    f = NULL;
    for (i = 0; i < FONTS_MAX;  i ++) {

#ifdef DEBUG
	if (af[i]) fprintf(stderr, "found font %s\n", af[i]->fname);
#endif

	if (af[i] && !strcmp(fname, af[i]->fname)) {
	   f = af[i];
	   break;
	}
    }
    return f;
}
/* =================================================================== */
/* fillup the font data by reading in the IFM file supplied */
/* assumed that init_font() has been called before.... */

int fillup_font(font_t* font, char ifmfname[])
{
    int		i;
    char	word[256];
    char	ifmword[256];
    int		ifmtoken;
    char	fmfname[NAMELEN];
    FILE*	fmfp;
    char*	ienv;
    char*	dp;
    FILE*	ifmfp;
    int		errflg= 0;
    
    /* get the search path... */
    ienv = getenv(ITRANS_PATH);
    if (!ienv) {
	putenv(ITRANS_PATH_DEF);
        ienv = getenv(ITRANS_PATH);
	if (!ienv) { /* STUPID ERROR, SYSTEM PROBLEM.. */
	    ienv = strchr(ITRANS_PATH_DEF, '=') + 1;
	}
    }

    if ( !font ) {
	fprintf(stderr, "Program error (malloc failed ?): fillup_font() got NULL font\n");
	return FALSE;
    }

    ifmfp = search_fopen(ienv, ifmfname, "r");
    if (!ifmfp) {
	 fprintf(stderr, "Error: could not open %s for reading\n", ifmfname);
	 return FALSE;
    }

	/* -------- FILLUP khadi */
	/* FILL UP THE DEVNAGARI CHARS */

    reset_pifm(); /* reset the IFM parser, set line count to zero etc */

    while ( !errflg && (ifmtoken = get_ifm_token(ifmfp, ifmword)) != 0) {
	/* line format is:
	   Comment -I- CC <ifm> n ; PCC <pscharnum> <deltax> <deltay> ;
	   Comment -I- CCADD <newifmname> ;
	   Comment -I- CCS <ifmname> <otherifmname> ;
	 */

	switch (ifmtoken) {
	case FONT_IFMTAG:

		/* example: Comment -I- FONT marathi devnac.afm */
	    ifmtoken = get_ifm_token(ifmfp, ifmword);
	    if (ifmtoken != DNAME_IFMTAG) {
		fprintf(stderr, "Error in parsing ifm font metrics - expect a fontname after FONT\n");
	        errflg++; break;
	    }
	    strcpy(word, ifmword);
	    ifmtoken = get_ifm_token(ifmfp, ifmword);
	    if (ifmtoken != DNAME_IFMTAG) {
		fprintf(stderr, "Error in parsing ifm font metrics - expect a filename after FONT %s\n", word);
	        errflg++; break;
	    }
	    strcpy(fmfname, ifmword);

	    /* strip of path name stuff, and copy */
	    dp = strrchr(ifmfname, DIRSEP);
	    if (!dp) dp = ifmfname;
	    else dp++;
	    strcpy(font->fname, dp);

	    strcpy(font->name, word);
	        /* -------- FILLUP ENCODING NAMES */
	        /* XXX - fill up font with G_ifm_map - maybe different
		 * fonts will use different names for ka, kha, etc!
		 * I hope not.
		 */
            for (i = 0; i < NUMCHARS; i ++) {
	        font->enc[i] = G_ifm_map[i];
            }

		/* fill in font Metrics */
	    strcpy(word, &ifmword[strlen(ifmword) - 3]); /* get last 3 chars */
	    if (!strcmp(word, "afm")) {
		font->prop = TEX_PS_FONT; /* use font for both PostScript
					   * and TeX output
					   */
	    } else if (!strcmp(word, "tfm")) {
		font->prop = TEX_FONT; /* use font for TeX output only */
	    } else  {
		fprintf(stderr, "font.c::Error: illegal metric file name %s (must end in .afm or .tfm)\n", ifmword);
	        errflg++; break;
	    }

	    if (font->prop == TEX_FONT) {

		/* dont read in any file, all TeX chars are non-zero
		 * widths anyway.
		 */
		fmfp = NULL;
		S_fillup_tfm(font);

	    }  else {

		/* fill in PostScript Metrics */
	        fmfp = search_fopen(ienv, fmfname, "r");
	        if (!fmfp) {
		    fprintf(stderr, "font.c::Error: could not open %s for reading (afm)\n", fmfname);
		    errflg++; break;
	        } 
		S_fillup_ps(fmfp, font);
	        fclose(fmfp);
	    }

	    break;

	case PROP_IFMTAG:
	    if (!S_prop(ifmfp, font)) {
		fprintf(stderr,"**** line %d - illegal PROP line\n", G_ifm_lineno);
	        errflg++;
	    }
	    break;

	case CCADD_IFMTAG:
	    if (!S_ccadd(ifmfp)) {
		fprintf(stderr,"**** line %d - illegal CCADD line (last token %s)\n",
			G_ifm_lineno, ifmword);
	        errflg++;
	    }
	    break;

	case CCS_IFMTAG:
	    if (!S_ccs(ifmfp, font)) {
		fprintf(stderr,"**** line %d - illegal CCS line (last token %s)\n",
			G_ifm_lineno, ifmword);
	        errflg++;
	    }
	    break;
	case CC_IFMTAG:
	    if (!S_cc(ifmfp, font)) {
		fprintf(stderr,"**** line %d - illegal CC line (last token %s)\n",
			G_ifm_lineno, ifmword);
	        errflg++;
	    }
	    break;
	case COMMENT_IFMTAG:
	    break;
	default:
		fprintf(stderr,"**** around line %d (+-1) - warning: unrecognized IFM token %s\n",
			G_ifm_lineno, ifmword);
		if (ifmtoken == PCC_IFMTAG)
		    fprintf(stderr,"**** around line %d (+-1) - warning: maybe the CC count is incorrect -- too many PCC's\n",
			G_ifm_lineno);
	    break;
	} /* switch */

    } /* while get_ifm_token(ifmfp, ifmword) - reading in IFM statements */

    /* clear any CCADD's this file may have added, so that each IFM
     * file can use upto NUMEXTRA CCADD codenames.
     * Without this clearing, the TOTAL number of CCADD statements in
     * ALL the IFM files is limited to NUMEXTRA, this way, EACH file
     * can use upto NUMEXTRA codenames...
     */
    S_clear_ccadds();

    if (!errflg) return TRUE;
    else return FALSE;
} /* fillup_font() */
    
/* ==================================================================== */
static int S_fillup_ps(FILE* afmfp, font_t* font)
{
    int		i;
    pschar_t*	psch;
    char	line_in[LINELEN], *inc;

    /* fill up the postscript font metrics */

    psch = font->psfm;
    while (fgets(line_in, LINELEN, afmfp)) {
	/* line format is: C 33; WX 301; N exclam; B 93 -9 207 685; */

	if (line_in[0] != 'C' || line_in[1] != ' ') continue;
	inc = &line_in[1];

	    /* read in character code */
	if (sscanf(inc, "%d", &i) != 1 || i < 0 || i >= NUMPSCHARS) {
	    fprintf(stderr, "Warning: illegal line <%s> in AFM file\n", line_in);
	    continue;
	}

	inc = strchr(inc, 'W'); /* look for W.... */
	inc = strchr(inc, ' '); /* look for <space> after W.... */

	sscanf(inc, "%d", &psch[i].w); /* read in width */

	inc = strrchr(inc, 'B'); /* look for ....B */
	inc = strchr(inc, ' '); /* look for <space> after ..B */

	    /* read in bounding box */
	if (sscanf(inc, "%d%d%d%d", &psch[i].llx, &psch[i].lly, &psch[i].urx, &psch[i].urx) != 4) {
	    fprintf(stderr, "Warning: illegal B values - line <%s> in AFM file\n", line_in);
	}

    } /* while fgets() */

    return TRUE;
}
/* ==================================================================== */
/* this func does nothing, TeX never has zero width fonts (??) */
static int S_fillup_tfm(font_t* font)
{
    pschar_t*	psch;
    int i;

    /* fill up the TeX font metrics */

    psch = font->psfm;
    for (i = 0 ; i < NUMPSCHARS; i ++ ) {

	psch[i].w = -1; /* just some non-zero value, unused really..*/
			/* make it non-zero so that ichar.c::S_add_cus()
			 * is fooled into thinking that all chars have
			 * non-zero width, and it does not have to apply
			 * a reverse delta to back up
			 * Of course, this implies that until I get code
			 * here to actually read in the TFM's, the IFM
			 * file should have data to do all the backing up
			 * that may be necessary.
			 * 13 Aug Note: I did get the code, but
			 * all fonts seemed to have non-zero widths
			 * anyway..
			 * ...
			 * Maybe this backup on zero width chars
			 * was a bad idea anyway..
			 */
	psch[i].llx = psch[i].lly = psch[i].urx = psch[i].urx = 0;

    }
    return TRUE;
}
/* ==================================================================== */
/* line format: Comment -I- CCADD <newifmname> ; */
static int S_icurrextra = 0;
static int S_ccadd(FILE* ifmfp)
{

    int ifmtoken;
    char word[256];
    int cctype, ccl0, ccl1, ccform;
    char	ifmword[256];

    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != DNAME_IFMTAG) {
	fprintf(stderr, "Error in parsing indian font metrics - expect a word after CCADD\n");
        return FALSE;
    }
    strcpy(word, ifmword);

    /* decode ifm name */
    cctype = decode_name(word, &ccl0, &ccl1, &ccform);
    if (cctype) {
	/* some other ifm file also used this codename, so no need to add
	 * it again....
	 */
#ifdef DEBUG
	fprintf(stderr, "Note: new codename :%s: - already exists in khadi\n", word);
#endif /* DEBUG */
	return TRUE;
    }

    /* check for overflow ... */
    if (S_icurrextra >= NUMEXTRA) {
	fprintf(stderr, "(line %d) Fatal Error in parsing indian font metrics - too many user-defined codenames\n", G_ifm_lineno);
	exit(10);
    }

    S_ifm_extra_map[S_icurrextra].token = LASTCHAR+1 + S_icurrextra;
    strncpy(S_ifm_extra_map[S_icurrextra].codename, word, CNAMELEN);
    S_ifm_extra_map[S_icurrextra].codename[CNAMELEN-1] = '\0';
    S_icurrextra++;
#ifdef DEBUG
fprintf(stderr, "added CCADD tag, word is %s, tok %d (num %d)\n", S_ifm_extra_map[S_icurrextra-1].codename,S_ifm_extra_map[S_icurrextra-1].token, S_icurrextra);
#endif /* DEBUG */
    return TRUE;

} /* S_ccadd */

/* clear all the codenames added here (to prepare for the next IFM
 * file, this function called at the end of fillup_font()
 */
static void S_clear_ccadds()
{
    int i;

    S_icurrextra = 0;
    for (i = 0; i < NUMEXTRA; i ++) {
	S_ifm_extra_map[i].codename[0] = '\0';
	S_ifm_extra_map[i].token = -999;
    }
}
/* ==================================================================== */
/* line format:
Comment -I- CC <ifmname> n ; PCC <pscharnum> <deltax> <deltay> ;
 */
static int S_cc(FILE* ifmfp, font_t *font)
{
    int i, j, ifmtoken;
    char word[256];
    int cctype, ccl0, ccl1, ccform;
    int isconsd;
    comp_unit_t*	pcu;
    comp_unit_t**	ppcu;
    int numpcc, pcode, dx, dy;
    char	ifmword[256];

    isconsd = FALSE;
    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != DNAME_IFMTAG) {
	fprintf(stderr, "(line %d, token %d) Error in parsing indian font metrics - expect a word after CC\n", G_ifm_lineno, ifmtoken);
        return FALSE;
    }
    strcpy(word, ifmword);

    cctype = decode_name(word, &ccl0, &ccl1, &ccform);
    if (!cctype) {
	fprintf(stderr, "Error: CC: illegal codename :%s:\n", word);
	return FALSE;
    }

    if (cctype == CONSONANT_DOUBLE_TYPE) isconsd = TRUE;

#ifdef DEBUG
    fprintf(stderr, "name: %s, decoded %d %d %d type %d\n", word, ccl0, ccl1, ccform,cctype);
#endif /* DEBUG */

	/* look for n - number of PCC segments */
    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != NUMBER_IFMTAG) {
	fprintf(stderr, "(line %d, token %d yytesxt %s) Error in parsing indian font metrics - expect a number after CC word.. \n", G_ifm_lineno, ifmtoken, ifmword);
        return FALSE;
    }
    word[0] = 0; strcpy(word, ifmword);
    if (sscanf(word, "%d", &numpcc) != 1) {
	fprintf(stderr, "Warning: (line %d) illegal value in IFM file (looking for number of PCC segments)\n", G_ifm_lineno);
	return FALSE;
    }

    if (isconsd) {
	ppcu = font->ligatures[ccl0][ccl1].cus;
    } else {
	ppcu = font->khadi[ccl0].cus;
    }
    if (!ppcu) { /* first time, allocate */
	ppcu = (comp_unit_t**) malloc(NUMFORMS*sizeof(*ppcu));
	if (isconsd) {
	    font->ligatures[ccl0][ccl1].cus = ppcu;
	} else  {
	    font->khadi[ccl0].cus = ppcu;
	}
	for (i = 0; i < NUMFORMS; i ++) {
	    ppcu[i] = (comp_unit_t*) NULL;
	}
    }
    pcu = (comp_unit_t*) malloc(numpcc*sizeof(*pcu));
    if (isconsd) {
	font->ligatures[ccl0][ccl1].cus[ccform] = pcu;
    /*
    fprintf(stderr, "fillip: filling ligature %d %d %d\n", ccl0, ccl1, ccform);
    */
    } else {
	font->khadi[ccl0].cus[ccform] = pcu;
    }

#ifdef DEBUG
    fprintf(stderr, "filling up PCC's for ccl0 %d l1 %d form %d type %d\n",
    ccl0, ccl1, ccform, cctype);
#endif /* DEBUG */

    for (i = 0; i < numpcc; i ++) {

	/* look for " ; PCC" */
        ifmtoken = get_ifm_token(ifmfp, ifmword);
        if (ifmtoken != PCC_IFMTAG) {
	    fprintf(stderr, "(line %d, token %d ifmword %s)Error in parsing indian font metrics - expect a PCC tag\n", G_ifm_lineno, ifmtoken, ifmword);
            return FALSE;
        }

	    /* defaults */
	pcu->u_pschar = NO_PSCHAR;
	pcu->dorg = DORG_CLR;
	pcu->deltax = 0;
	pcu->deltay = 0;
	if (i == (numpcc - 1)) pcu->next = NULL; /* last comp_unit ... */
	else pcu->next = pcu + 1; /* point to next comp_unit ... */

        ifmtoken = get_ifm_token(ifmfp, ifmword);
        word[0] = 0; strcpy(word, ifmword);
	if (ifmtoken == IMPLICIT_IFMTAG) {
	    pcode = IMPLICIT_PSCHAR;
	} else if (ifmtoken == NONE_IFMTAG) {
	    pcode = NO_PSCHAR;
	} else if (ifmtoken == NUMBER_IFMTAG) {
	    j = sscanf(word, "%d", &pcode);
	} else {
	    fprintf(stderr, "(line %d, token %d %s)Error in parsing indian font metrics - expect a number after PCC tag\n", G_ifm_lineno, ifmtoken, ifmword);
	    return FALSE;
        }

	    /* point to dx, could be %d or s%d */
	ifmtoken = get_ifm_token(ifmfp, ifmword);
	if (ifmtoken == DELTAS_IFMTAG) {
	    pcu->dorg = DORG_LL;
	    fprintf(stderr, "Error: s+/- option not supported in Devn Metrics - ignoring it\n");
	    ifmword[0] = ' ';  /* replace "s" with space, number is read
				* by scanf
				*/
	} else if (ifmtoken != NUMBER_IFMTAG) {
	    fprintf(stderr, "Error in parsing indian font metrics - expect a number after PCC ...\n");
            return FALSE;
        }
        word[0] = 0; strcpy(word, ifmword);
	j = sscanf(word, "%d", &dx);

	ifmtoken = get_ifm_token(ifmfp, ifmword);
	    /* point to dy, has to be %d (s%d not supported yet) */
	    /* no need to support s+70 or s-70 etc since height
	     * of char is constant - cannot be modified for 
	     * roman/devnagari/like scripts.
	     */
        if (ifmtoken != NUMBER_IFMTAG) {
	    fprintf(stderr, "Error in parsing indian font metrics - expect a number after PCC ... dx ..\n");
            return FALSE;
        }
        word[0] = 0; strcpy(word, ifmword);
	j += sscanf(word, "%d", &dy);

	if (j != 2) {
	    fprintf(stderr, "Warning: IFM file - illegal dx/dy values-\n");
	} else {
	    pcu->u_pschar = pcode;
	    pcu->deltax = dx;
	    pcu->deltay = dy;
	}

	/* check for errors: if NO_PSCHAR, then dy must be zero */
	if (pcode == NO_PSCHAR && dy != 0) {
	    fprintf(stderr, "(line %d)Warning PCC none .. must have dy == 0, will make it 0\n",
			G_ifm_lineno);
	    pcu->deltay = 0;
	}

#ifdef DEBUG
    fprintf(stderr, "pcode %d dx %d dy %d\n", pcode, dx, dy);
#endif /* DEBUG */

	pcu++;
    } /* for i numpcc */

    return TRUE;
} /* S_cc */
/* ==================================================================== */
/* line format: Comment -I- CCS <ifmname> <otherifmname> ; */
static int S_ccs(FILE* ifmfp, font_t *font)
{
    int ifmtoken;
    char word[256];
    int cctype, ccl0, ccl1, ccform;
    int pcctype, pccl0, pccl1, pccform;
    dchar_t*		dptr;
    int			i;
    char	ifmword[256];

    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != DNAME_IFMTAG) {
	fprintf(stderr, "(line %d, token %d) Error in parsing indian font metrics - expect a word after CCS\n", G_ifm_lineno, ifmtoken);
        return FALSE;
    }
    strcpy(word, ifmword);

    cctype = decode_name(word, &ccl0, &ccl1, &ccform);
    if (!cctype) {
	fprintf(stderr, "Error: CCS:cc: illegal codename :%s:\n", word);
	return FALSE;
    }
#ifdef DEBUG
fprintf(stderr, "got CCS tag, word is %s type %d\n", word, cctype);
#endif /* DEBUG */

    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != DNAME_IFMTAG) {
	fprintf(stderr, "(line %d, token %d) Error in parsing indian font metrics - expect a second word after CCS\n", G_ifm_lineno, ifmtoken);
        return FALSE;
    }
    strcpy(word, ifmword);

    pcctype = decode_name(word, &pccl0, &pccl1, &pccform);
    if (!pcctype) {
	fprintf(stderr, "Error: CCS: pcc: illegal codename :%s:\n", word);
	return FALSE;
    }
#ifdef DEBUG
fprintf(stderr, "otherifmname is %s \n", word);
#endif /* DEBUG */
    dptr = NULL;
    if (pcctype == CONSONANT_DOUBLE_TYPE) {
	if (pccl0 == ALL_CHARS || pccl1 == ALL_CHARS) {
	    fprintf(stderr, "Error: <otherifmname> (line %d) in CCS should not contain *'s\n", G_ifm_lineno);
	    return FALSE;
	} else {
	    dptr = &font->ligatures[pccl0][pccl1];
	}
    } else {
	if (cctype == CONSONANT_SINGLE_TYPE && pccl0 == ALL_CHARS) {
	    fprintf(stderr, "Error: <otherifmname> (line %d) in CCS must be a single khadi cannot be *.\n", G_ifm_lineno);
	    return FALSE;
	} else {
	    if (pccl0 != ALL_CHARS) dptr = &font->khadi[pccl0];
	}
    }
    if (cctype == CONSONANT_DOUBLE_TYPE) {
#ifdef DEBUG
fprintf(stderr, "ligatures %d %d %d same as %d %d\n", ccl0, ccl1, ccform, pccl0, pccl1);
#endif /* DEBUG */
	if (ccl0 == ALL_CHARS) {
	    if (pccl0 == ALL_CHARS) {
		for (i = FIRSTCONS; i < NUMCHARS; i ++)
		    font->ligatures[i][ccl1].same_as = &font->khadi[i];
	    } else {
		for (i = FIRSTCONS; i < NUMCHARS; i ++)
		    font->ligatures[i][ccl1].same_as = dptr;
	    }
	} else if (ccl1 == ALL_CHARS) {
	    if (pccl0 == ALL_CHARS) {
		for (i = FIRSTCONS; i < NUMCHARS; i ++)
		    font->ligatures[ccl0][i].same_as = &font->khadi[i];
	    } else {
		for (i = FIRSTCONS; i < NUMCHARS; i ++)
		    font->ligatures[ccl0][i].same_as = dptr;
	    }
	} else {
	    font->ligatures[ccl0][ccl1].same_as = dptr;
	}
    } else {
#ifdef DEBUG
fprintf(stderr, "khadi %d %d same as %d %d\n", ccl0, ccl1, pccl0, pccl1);
#endif /* DEBUG */
	font->khadi[ccl0].same_as = dptr;
    }

    return TRUE;
} /* S_ccs() */

/* ==================================================================== */
/* Given a word referring to a full indian language character, decode
 * it into its letter and form parts.
 * For ex:
 * ka-i		is letter "ka" (KA_TOK) and form "i" (I_FORM)
 * a		is letter "a" (A_TOK) and form "a" (A_FORM)
 * am		is letter "am" (AM_TOK) and form "a" (A_FORM)
 * ksha-half	is letter "ksha" (KSHA_TOK) and form (HALF_FORM)
 * da-dha	is (half) letter "da" (DA_TOK) and full letter dha (DHA_TOK)
 * etc
 * Function returns type: VOWEL_TYPE, CONSONANT_SINGLE_TYPE, 
 * CONSONANT_DOUBLE_TYPE or 0 for error
 */

int decode_name(char inword[], int* firstc, int* secondc, int* form)
{
    char word[256], word1[256], word2[256], word3[256];
    char *start;
    int i, l0, l1, f, t;

    word1[0] = '\0'; word2[0] = '\0'; word3[0] = '\0';
    l0 = l1 = f = NO_CHAR;
    t = 0;
    *firstc = *secondc = *form = NO_CHAR;

    strcpy(word, inword);
    start = strtok(word, "-");
    if (!start) return 0; /* ERROR */
    strcpy(word1, start); /* first word is in here now */

    if ((start = strtok(NULL, "-")) != NULL) { /* look for second word */
        strcpy(word2, start); /* second word is in here now */
        if ((start = strtok(NULL, "-")) != NULL) { /* look for third word */
            strcpy(word3, start); /* third word is in here now */
        }
    }

#ifdef DEBUG
fprintf(stderr, "decoding: word 1 :%s:, 2 :%s:, 3 :%s:\n", word1, word2,
word3);
#endif /* DEBUG */

    /* look for l0, could be vowel, cons, or user-defined cons (extra_map) */
    for (i = 0; i < NUMCHARS; i ++) {
	if (!strcmp(G_ifm_map[i].codename, word1)) {
	    l0 = i; break;
	}
    }
    if (l0 < 0) {
        for (i = 0; i < NUMEXTRA; i ++) {
	    if (!strcmp(S_ifm_extra_map[i].codename, word1)) {
	        l0 = S_ifm_extra_map[i].token; break;
	    }
        }
    }
    if (l0 < 0) {
	if (!strcmp("*", word1)) {
	    l0 = ALL_CHARS;
	}
    }

    /* look for l1, must be vowel or cons or allchars */
    if (word2[0]) {
        if (!strcmp("half", word2)) {
	    l1 = HALF_FORM;
        } else if (!strcmp("implicit", word2)) {
	    l1 = IMPLICIT_FORM;
        } else {
            for (i = 0; i < NUMCHARS; i ++) {
	        if (!strcmp(G_ifm_map[i].codename, word2)) {
	            l1 = i; break;
	        }
            }
	}
    }
    if (l1 < 0) {
	if (!strcmp("*", word2)) {
	    l1 = ALL_CHARS;
	}
    }


    /* look for form, must be vowel */
    if (word3[0]) {
        if (!strcmp("half", word3)) {
	    f = HALF_FORM;
        } else if (!strcmp("implicit", word3)) {
	    f = IMPLICIT_FORM;
        } else {
            for (i = 0; i <= AHA_FORM; i ++) {
	        if (!strcmp(G_ifm_map[i].codename, word3)) {
	            f = i; break;
	        }
            }
	}
    }

    /* DECODE IT */
    if (l0 >= ALL_CHARS && (l1 >= FIRSTCONS || l1 == ALL_CHARS) &&
	    strcmp(word2, "half") && strcmp(word2, "implicit")) {
	/* is a double consonant with possible form (may be absent) */
	/* only if the second word was not half or implicit */
	    t = CONSONANT_DOUBLE_TYPE;
#ifdef DEBUG
   fprintf(stderr, "defined as DOUBLE: t is %d\n", t);
#endif /* DEBUG */
	    if (f < A_FORM || f > IMPLICIT_FORM) {
#ifdef DEBUG
		fprintf(stderr, "Warning: Double Consonants (ligatures) must have a valid form (Unless this is a CCS statement!) - assuming IMPLICIT here (%s)\n", inword);
#endif /* DEBUG */
		f = IMPLICIT_FORM;
	    }
    } else if (l0 >= FIRSTVOW && l0 <= LASTVOW) {
        /* check if l0 is a vowel */
	f = A_FORM; /* vowels are always one word only */
	l1 = NO_CHAR;
	t = VOWEL_TYPE;
    } else if (l0 >= 0) {
        /* must be single cons type, code is two words only */
        f = l1;
	l1 = NO_CHAR;
        if (l0 > LASTCONS && l0 <= LASTCHAR) {
	    t = SPECIAL_TYPE;
	    if (f != IMPLICIT_FORM) fprintf(stderr, "Warning: All Specials must have IMPLICIT_FORM only (%s)\n", inword);
	    f = IMPLICIT_FORM;
        } else {
	    t = CONSONANT_SINGLE_TYPE;
        }
    } else if (l0 == ALL_CHARS) {
	f = NO_CHAR; /* vowels are always one word only */
	l1 = NO_CHAR;
	t = CONSONANT_SINGLE_TYPE;
#ifdef DEBUG
   fprintf(stderr, "l0 is ALL_CHARS, defined as SINGLE: t is %d\n", t);
#endif /* DEBUG */
    } /* else is a new userdefined char */

    *firstc = l0;
    *secondc = l1;
    *form = f;


#ifdef DEBUG
    fprintf(stderr, "decode l0 %d l1 %d f %d returning %d, f %d s %d form %d\n",
l0, l1, f, t, *firstc, *secondc, *form);
    if (*form < 0) {
	fprintf(stderr, "WARNING: form is NEGATIVE: %d (%s)\n", *form, inword);
    }
#endif /* DEBUG */
    return t;
}

/* ==================================================================== */
/* The encoding map: BE CAREFUL of changing the size of this
 * array - MUST always reflect the NUMCHARS defined value in imap.h
 * this array is GLOBAL, error handling in ichar.c (missing chars)
 * uses this data
 * NOTE:
 * All names must be distinct (see imap.h) and the order of
 * tokens here MUST be the same as in y.tab.h
 */
ifm_enc_t G_ifm_map[NUMCHARS] = {
    /* 0 */	A_TOK,		A_CNAME,
    /* 1 */	AA_TOK,		AA_CNAME,
    /* 2 */	I_TOK,		I_CNAME,
    /* 3 */	II_TOK,		II_CNAME,
    /* 4 */	U_TOK,		U_CNAME,
    /* 5 */	UU_TOK,		UU_CNAME,
    /* 6 */	RI_TOK,		RI_CNAME,
    /* 7 */	RII_TOK,	RII_CNAME,
    /* 8 */	LI_TOK,		LI_CNAME,
    /* 9 */	LII_TOK,	LII_CNAME,
    /* 10 */	AY_TOK,		AY_CNAME,
    /* 10 */	AAY_TOK,	AAY_CNAME,
    /* 11 */	AI_TOK,		AI_CNAME,
    /* 12 */	O_TOK,		O_CNAME,
    /* 12 */	OO_TOK,		OO_CNAME,
    /* 13 */	AU_TOK,		AU_CNAME,
    /* 14 */	AM_TOK,		AM_CNAME,
    /* 15 */	AHA_TOK,	AHA_CNAME,
    /* 16 */	KA_TOK,		KA_CNAME,
    /* 17 */	KHA_TOK,	KHA_CNAME,
    /* 18 */	GA_TOK,		GA_CNAME,
    /* 19 */	GHA_TOK,	GHA_CNAME,
    /* 20 */	NGA_TOK,	NGA_CNAME,
    /* 21 */	CHA_TOK,	CHA_CNAME,
    /* 22 */	CHHA_TOK,	CHHA_CNAME,
    /* 23 */	JA_TOK,		JA_CNAME,
    /* 24 */	JHA_TOK,	JHA_CNAME,
    /* 25 */	JNH_TOK,	JNH_CNAME,
    /* 26 */	TTA_TOK,	TTA_CNAME,
    /* 27 */	TTHA_TOK,	TTHA_CNAME,
    /* 28 */	DDA_TOK,	DDA_CNAME,
    /* 29 */	DDHA_TOK,	DDHA_CNAME,
    /* 30 */	NNA_TOK,	NNA_CNAME,
    /* 31 */	TA_TOK,		TA_CNAME,
    /* 32 */	THA_TOK,	THA_CNAME,
    /* 33 */	DA_TOK,		DA_CNAME,
    /* 34 */	DHA_TOK,	DHA_CNAME,
    /* 35 */	NA_TOK,		NA_CNAME,
    /* 36 */	PA_TOK,		PA_CNAME,
    /* 37 */	PHA_TOK,	PHA_CNAME,
    /* 38 */	BA_TOK,		BA_CNAME,
    /* 39 */	BHA_TOK,	BHA_CNAME,
    /* 40 */	MA_TOK,		MA_CNAME,
    /* 41 */	YA_TOK,		YA_CNAME,
    /* 42 */	RA_TOK,		RA_CNAME,
    /* 43 */	LA_TOK,		LA_CNAME,
    /* 44 */	VA_TOK,		VA_CNAME,
    /* 45 */	SHA_TOK,	SHA_CNAME,
    /* 46 */	SHHA_TOK,	SHHA_CNAME,
    /* 47 */	SA_TOK,		SA_CNAME,
    /* 48 */	HA_TOK,		HA_CNAME,
    /* 49 */	LDA_TOK,	LDA_CNAME,
    /* 50 */	KSHA_TOK,	KSHA_CNAME,
    /* 51 */	GYA_TOK,	GYA_CNAME,
    /* 52 */	NNX_TOK,	NNX_CNAME,
    /* 53 */	NYA_TOK,	NYA_CNAME,
    /* 54 */	RRA_TOK,	RRA_CNAME,
    /* 55 */	KADOT_TOK,	KADOT_CNAME,
    /* 56 */	KHADOT_TOK,	KHADOT_CNAME,
    /* 57 */	GADOT_TOK,	GADOT_CNAME,
    /* 58 */	DDADOT_TOK,	DDADOT_CNAME,
    /* 59 */	DDHADOT_TOK,	DDHADOT_CNAME,
    /* 50 */	JADOT_TOK,	JADOT_CNAME,
    /* 61 */	PHADOT_TOK,	PHADOT_CNAME,
    /* 62 */	RA_HALF_TOK,	RA_HALF_CNAME,
    /* 63 */	ANUSVARA_TOK,	ANUSVARA_CNAME,
    /* 64 */	CHANDRA_TOK, 	CHANDRA_CNAME,
    /* 65 */	CHANDRA_BN_TOK, CHANDRA_BN_CNAME,
    /* 66 */	VIRAAM_TOK,	VIRAAM_CNAME,
    /* 67 */	AVAGRAHA_TOK,	AVAGRAHA_CNAME,
    /* 68 */	SRI_TOK, 	SRI_CNAME,
    /* 69 */	AUM_TOK, 	AUM_CNAME,
};

/* =================================================================== */
/* print out font data (for debugging) */
void dump_font(font_t* font, FILE* outfp)
{
    int i, j;
    comp_unit_t* cus;

    fprintf(outfp, "font prop: %d\n", font->prop);
    fprintf(outfp, "font name: %s\n", font->name);

    for (i = 0; i < NUMKHADI; i ++) {
	if (font->khadi[i].cus) {
	    cus = font->khadi[i].cus[IMPLICIT_FORM];
	    if (!cus) cus = font->khadi[i].cus[A_FORM];
	}
	if (!cus && font->khadi[i].same_as && !font->khadi[i].same_as->cus[IMPLICIT_FORM])
	    continue;

	fprintf(outfp, "Form a/implicit: Character %d, ", i);
	fprintf(outfp, "name: %s\n", font->enc[i].codename);

	while (cus) { /* NULL test */
	    fprintf(stderr, " cus: pschar %d, delta %d %d; ",
			(cus)->u_pschar, (cus)->deltax, (cus)->deltay);
	    cus = cus->next;
	}
	cus = NULL; 
	if (font->khadi[i].same_as) cus = font->khadi[i].same_as->cus[IMPLICIT_FORM];
	while (cus) { /* NULL test */
	    fprintf(outfp, " cus: pschar %d, delta %d %d; ",
			(cus)->u_pschar, (cus)->deltax, (cus)->deltay);
	    cus = cus->next;
	}
	fprintf(outfp, "\n");
    }

    for (i = 0; i < NUMCHARS; i ++) {
        for (j = 0; j < NUMCHARS; j ++) {
	    /* font->ligatures[i][j].cus = NULL;
	    font->ligatures[i][j].same_as = NULL;
	    */
        }
    }
    for (i = 0; i < 10; i ++) {
	/*
	font->digits[i].cus = NULL;
	font->digits[i].same_as = NULL;
	*/
    }

    for (i = 0; i < NUMPSCHARS; i ++) {
	if (font->psfm[i].w > 0) {
	fprintf(outfp, "char %d:: width %d ", i, font->psfm[i].w);
	fprintf(outfp, "llx %d ", font->psfm[i].llx);
	fprintf(outfp, "lly %d ", font->psfm[i].lly);
	fprintf(outfp, "urx %d ", font->psfm[i].urx);
	fprintf(outfp, "ury %d\n", font->psfm[i].ury);
	}
    }
} /* print_font() */
/* ==================================================================== */
/* line format: Comment -I- PROP %d ligatures ; ... */
/* or Comment -I- PROP %d no_ligatures ; ... */
static int S_prop(FILE* ifmfp, font_t* ft)
{
    int  i, n, ifmtoken;
    char word[256];
    char ifmword[256];

	/* look for n - number of property names */
    ifmtoken = get_ifm_token(ifmfp, ifmword);
    if (ifmtoken != NUMBER_IFMTAG) {
	fprintf(stderr, "(line %d, token %d yytext %s) Error in parsing indian font metrics - expect a number after PROP word.. \n", G_ifm_lineno, ifmtoken, ifmword);
        return FALSE;
    }
    word[0] = 0; strcpy(word, ifmword);
    if (sscanf(word, "%d", &n) != 1) {
	fprintf(stderr, "Warning: (line %d) illegal value in IFM file (looking for number of PROPerties)\n", G_ifm_lineno);
	return FALSE;
    }

    for (i = 0; i < n; i ++) {
        ifmtoken = get_ifm_token(ifmfp, ifmword);
        if (ifmtoken != DNAME_IFMTAG) {
	    fprintf(stderr, "Error in parsing indian font metrics - expect a word after PROP (line %d, i %d)\n", G_ifm_lineno, i);
	    return FALSE;
        }
        strcpy(word, ifmword);

        if (!strcmp(word, "no_ligatures")) {
	    ft->use_ligatures = FALSE;
        } else if (!strcmp(word, "ligatures")) {
	    ft->use_ligatures = TRUE;
        } else {
	    fprintf(stderr, "Error in parsing indian font metrics: PROP keyword (%s)  unrecognized (line %d)\n", word, G_ifm_lineno);
        }
    }

    return TRUE;
}
/* ===========================^ font.c ^ ============================== */
