{$F+} { Compiler Directive: Force far calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  B-Tree.
    Version 1.9

    This unit holds all the needed procedures to manage a simple b-tree file
      structure on a disk.  This version only supports one key to a file.

  Purpose:
    To allow easy implementation of a single key database in a permanent
    disk file.

  How it works!
    First, the program will open the tree file.  If the file doesn't exist
      yet, the file is created and initialized.
    Next, records can be entered into the database.
    After that, they can be accessed, updated, inserted and deleted.
    Closing the tree file saves the data for later.

  Features:
    Uses the B-tree data structure in the file to ensure fastest possible
      data access.
    B-tree is automatically managed.  No need to worry about processing.
    Deleted records are tracked to ensure tightest file management.

  Versions:
    1.1 - ?
    1.2 - Revamped the code in an attempt to remove the file sharing crashes.
    1.3 - ?
    1.4 - ?
    1.5 - Stripped out all of the file locking and unlocking code from the
            unit because of too many system hang ups.
    1.6 - Introduced the alternate code to increase performance.
    1.7 - Updated code to automatically produce optimal record spacing.
    1.8 - Updated several routines to work faster.  Repaired minor bugs.
    1.9 - Added updated code to compile and work with Speed Pascal/2.

  Copyright 1990, 1995,  All rights reserved.
    P. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2.

*****************************************************************************)

Unit BTree;

  Interface

    Uses
      DOS;

    Const
     { Maximum size of the b-tree unit -  Optimally this should match the disk's sector size }
      Node_Maximum_Length = 512;
     { Maximum size of the b-tree record }
      Maximum_Record_Size = 500;
     { Maximum amount of records per node }
      Maximum_Records_Per_Node = 255;
     { Minimum amount of records per node }
      Minimum_Records_Per_Node = 4;
     { Minimum size of key }
      Minimum_Key_Size = 5;

(**********************************************************

  These types are defined to allow the program to allocate
    a tree type.  They are automatically handled by the
    routines and should not be directly manipulated.

**********************************************************)

    Type
      Data_Type = packed array[ 1 .. Maximum_Record_Size ] of Char;
      Point_Type = LongInt;
      Position = 0 .. Maximum_Records_Per_Node;
      Node_Type = Record
                    Previous: Point_Type;
                    Count: Position;
                    Case Boolean of
                      True: ( Branch: array[ Position ] of Point_Type );
                      False: ( Data: array[ 1 .. Node_Maximum_Length ] of Byte );
                  End;
      Info_Type = Record
                    Top,
                    Free: Point_Type;
                    Start,
                    Finish,
                    Identity,
                    Node_Size,
                    Record_Size: Word;
                    Maximum,
                    Minimum: Byte;
                  End;
     {$IFNDEF VER40}
      Compare_Function = Function( Var Data1, Data2; Word1, Word2: Word ): Boolean;
     {$ENDIF}
      Tree_Type = Record
                    Default_Drive: Byte;
                    The_File: File;
                    Where1,
                    Pointer: Point_Type;
                    Where2: Position;
                    Info: Info_Type;
                    Buffer: Node_Type;
                   {$IFNDEF VER40}
                    Equal,
                    Less_Than: Compare_Function;
                   {$ENDIF}
                  End;

(***********************************************************

  Procedure: Open tree file.

    This procedure opens the file and sets up the
    information in the tree record for use.  If the tree
    file already exists, the information will be compared to
    make sure the right tree file is being used.  Thus, the
    sizes and key offsets must match exactly. The data
    structure is set up for maximum efficiency.

***********************************************************)

    Procedure Open_Tree_File( Var Tree: Tree_Type; File_Name: String; Record_Size, Key_Offset, Key_Length: Word );

(***********************************************************

  Procedure: Close tree file.

    This procedure closes the tree file so that the data
    input will be permanently saved.  If the file is not
    closed, the data may be lost.  Therefore, always
    remember to close the tree.

***********************************************************)

    Procedure Close_Tree_File( Var Tree: Tree_Type );

(***********************************************************

  Function: Insert data into tree.

    This function returns true if the new data was inserted
    into the b-tree.

***********************************************************)

    Function Insert_In_Tree( Var Tree: Tree_Type; Var New_Data ): Boolean;

(***********************************************************

  Function: Update data in the B-tree.

    This function returns true If the old data is found in
    the b-tree and if the new data was placed in the b-tree.
    The old data must match exactly what is found in the
    tree to prevent same key records from being replaced.

***********************************************************)

    Function Update_In_Tree( Var Tree: Tree_Type; Var Old_Data, New_Data ): Boolean;

(***********************************************************

  Function: Delete data from tree.

    This function returns true If the old data record was
    found in the b-tree and was removed from it.  The old
    data record must match exactly with what is found in the
    tree.

***********************************************************)

    Function Delete_From_Tree( Var Tree: Tree_Type; Var Old_Data ): Boolean;

(***********************************************************

  Function: Find data in tree.

    This function looks in the tree for the key given in
    data.  Then the information is returned in the data
    variable.

***********************************************************)

    Function Find_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find first in tree.

    This function gets the very first element in the logical
    order in the b-tree.  It is very useful for ordered
    listings.

***********************************************************)

    Function Find_First_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find next in tree.

    This function will find the next element in the tree
    relative to the last one found.  Useful for ordered
    listings.

***********************************************************)

    Function Find_Next_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find last in tree.

    This function will find the last element in logical
    order in the tree.  It is very useful for ordered
    listings.

***********************************************************)

    Function Find_Last_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find previous in tree.

    This function will find the logical element previous to
    the last one found in the b-tree.  Useful for ordered
    listings.

***********************************************************)

    Function Find_Previous_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find first in file.

    This function finds the first element in the file.  It
    does not use any key and is best for when doing a
    process on the whole b-tree.

***********************************************************)

    Function Find_First_In_File( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find next in file.

    This function finds the next element in the file.  It
    does not use any order and is best for whole file
    processing.

***********************************************************)

    Function Find_Next_In_File( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find last in file.

    This function finds the last element in the file.  It
    does not use any key and is best for use in whole file
    processing.

***********************************************************)

    Function Find_Last_In_File( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find previous in file.

    This function finds the previous element in the file.
    It does not use any key and is best for whole file
    processing.

***********************************************************)

    Function Find_Previous_In_File( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Procedure: Change key routines.

    This procedure will substitute your own comparing
    routines in place of the default routines.  This
    procedure is intended for advanced programming only and
    is not supported with Pascal version 4.0.

***********************************************************)

   {$IFNDEF VER40}
    Procedure Change_Key_Routines( Var Tree: Tree_Type; Var Equal, Less_Than: Compare_Function );
   {$ENDIF}

{----------------------------------------------------------------------------}

  Implementation

    {$DEFINE NoDebug}
    {$DEFINE Quick}  { Allows use of alternate code to increase speed. }

    Const
      Null = 0;        { Must be zero }
      Identity = 1259; { Allows For identification of tree file version }

{////////////////////////////////////////////////////////////////////////////}

  {$I BTree2.Pas}

{////////////////////////////////////////////////////////////////////////////}

(*************************************************

  Procedure: Scan the tree.
    This procedure scans for the previous pointer
    in the tree.

*************************************************)

    Procedure Scan( Var Tree: Tree_Type; Var Temporary: Point_Type; Var Point1: Point_Type; Var Point2: Position );
      Var
        Count: Position;
      Begin
        For Count := 0 to Get_Count( Tree, Temporary ) do
          If ( Point1 = Get_Pointer( Tree, Temporary, Count ) )
            then
              Begin
                Point1 := Temporary;
                Point2 := Count;
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move previous.
    This procedure moves the pointer to the
    previous record.

*************************************************)

    Procedure Move_Previous( Var Tree: Tree_Type; Var Point1: Point_Type; Var Point2: Position );
      Var
        Temporary: Point_Type;
      Begin
       {$IFDEF Debug}
        If ( Point2 < 0 )
          then
            Point2 := 0;
        If ( Point1 = Null )
          then
            Exit;
       {$ENDIF}
        Point1 := Get_Pointer( Tree, Point1, Point2 );
       {$IFDEF Debug}
        If ( Point1 = Null )
          then
            Exit;
       {$ENDIF}
        Point2 := Get_Count( Tree, Point1 );
        Temporary := Get_Pointer( Tree, Point1, Point2 );
        While ( Temporary <> Null ) do
          Begin
            Point1 := Temporary;
            Point2 := Get_Count( Tree, Point1 );
            Temporary := Get_Pointer( Tree, Point1, Point2 );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get previous in tree.
    This procedure tries to find the location of
    the previous record in the tree.  It it does
    not exist, the it returns the null pointer.

*************************************************)

    Procedure Get_Previous_In_Tree( Var Tree: Tree_Type; Var Point1: Point_Type; Var Point2: Position; Var Found: Boolean );
      Var
        Done: Boolean;
        Temporary: Point_Type;
      Begin
        If ( Point1 = Null )
          then
            Found := False
          else
            Begin
              Dec( Point2 );
              If ( Point2 <= 0 )
                then
                  Begin
                    If ( Get_Pointer( Tree, Point1, Point2 ) = Null )
                      then
                        Begin
                          Repeat
                            Done := True;
                            Temporary := Get_Previous( Tree, Point1 );
                            If ( Temporary = Null )
                              then
                                Point1 := Null
                              else
                                Begin
                                  Scan( Tree, Temporary, Point1, Point2 );
                                  Done := ( Point2 <> 0 );
                                End;
                          Until Done;
                        End
                      else
                        Move_Previous( Tree, Point1, Point2 );
                  End
                else
                  If ( Get_Pointer( Tree, Point1, Point2 ) <> Null )
                    then
                      Move_Previous( Tree, Point1, Point2 );
              Found := ( ( Point1 <> Null ) and ( Point2 <> 0 ) );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move next.
    This procedure moves the pointer to the next
    record.

*************************************************)

    Procedure Move_Next( Var Tree: Tree_Type; Var Point1: Point_Type; Var Point2: Position );
      Var
        Temporary: Point_Type;
      Begin
       {$IFDEF Debug}
        If ( Point1 = Null )
          then
            Exit;
        If ( Point2 > Get_Count( Tree, Point1 ) )
          then
            Point2 := Get_Count( Tree, Point1 );
       {$ENDIF}
        Point1 := Get_Pointer( Tree, Point1, Point2 );
       {$IFDEF Debug}
        If ( Point1 = Null )
          then
            Exit;
       {$ENDIF}
        Temporary := Get_Pointer( Tree, Point1, 0 );
        While ( Temporary <> Null ) do
          Begin
            Point1 := Temporary;
            Temporary := Get_Pointer( Tree, Point1, 0 );
          End;
        Point2 := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get next in tree.
    This procedure tries to find the location of
    the next piece of data stored in the tree.
    If it doesn't find it, it returns the nul
    pointer.

*************************************************)

    Procedure Get_Next_In_Tree( Var Tree: Tree_Type; Var Point1: Point_Type; Var Point2: Position; Var Found: Boolean );
      Var
        Done: Boolean;
        Temporary: Point_Type;
      Begin
        If ( Point1 = Null )
          then
            Found := False
          else
            Begin
              If ( Point2 >= Get_Count( Tree, Point1 ) )
                then
                  Begin
                    If ( Get_Pointer( Tree, Point1, Point2 ) = Null )
                      then
                        Begin
                          Repeat
                            Done := True;
                            Temporary := Get_Previous( Tree, Point1 );
                            If ( Temporary = Null )
                              then
                                Point1 := Null
                              else
                                Begin
                                  Scan( Tree, Temporary, Point1, Point2 );
                                  Done := ( Point2 <> Get_Count( Tree, Point1 ) );
                                End;
                          Until Done;
                        End
                      else
                        Move_Next( Tree, Point1, Point2 );
                  End
                else
                  If ( Get_Pointer( Tree, Point1, Point2 ) <> Null )
                    then
                      Move_Next( Tree, Point1, Point2 );
              Inc( Point2 );
              Found :=  ( ( Point1 <> Null ) and ( Point2 <= Get_Count( Tree, Point1 ) ) );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Look.
    This procedure searches keys in the given
    node by using a Binary search.  It returns
    the location of Target in Result if it's
    found, otherwise Result will point to a branch
    on which to continue the search.

*************************************************)

    Procedure Look( Var Tree: Tree_Type; Where: Point_Type; Var Result: Position; Var Target, Temporary: Data_Type );
      Var
        Top,
        Bottom: Position;
      Begin
        { Start a binary search through keys }
        Top := 1;
        Bottom := Get_Count( Tree, Where );
        Repeat
          Result := ( Top + Bottom ) div 2;
          Get_Data( Tree, Where, Result, Temporary );
         {$IFDEF VER40}
          If Less_Than_Default( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
         {$ELSE}
          If Tree.Less_Than( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
         {$ENDIF}
            then
              Bottom := Result
            else
              Top := Result
        Until ( Succ( Top ) >= Bottom );
        If ( Result = Top )
          then
           Begin
             Result := Bottom;
             Get_Data( Tree, Where, Result, Temporary );
           End;
       {$IFDEF VER40}
        If ( Less_Than_Default( Target, Temporary, Tree.Info.Start, Tree.Info.Finish ) and ( Result > Top ) )
       {$ELSE}
        If ( Tree.Less_Than( Target, Temporary, Tree.Info.Start, Tree.Info.Finish ) and ( Result > Top ) )
       {$ENDIF}
          then
            Begin
              Dec( Result );
              Get_Data( Tree, Where, Result, Temporary );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search the node.
    This procedure searches keys in the given
    node: Where^ for the target.  The location is
    returned in Result if it's found, otherwise
    the branch should be checked.

*************************************************)

  Procedure Search_Node( Var Tree: Tree_Type; Var Target: Data_Type; Where: Point_Type; Var Found: Boolean;
                         Var Result: Position );
    Var
      Temporary: Data_Type;
    Begin
      Get_Data( Tree, Where, 1, Temporary );
     {$IFDEF VER40}
      If Less_Than_Default( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
     {$ELSE}
      If Tree.Less_Than( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
     {$ENDIF}
        then
          Begin
            Found := False;
            Result := 0;
          End
        else
          Begin
            Look( Tree, Where, Result, Target, Temporary );
           {$IFDEF VER40}
            Found := Equal_Default( Target, Temporary, Tree.Info.Start, Tree.Info.Finish );
           {$ELSE}
            Found := Tree.Equal( Target, Temporary, Tree.Info.Start, Tree.Info.Finish );
           {$ENDIF}
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search for node exactly.
    This procedure searches for the exact record
    as specified by Target.  It returns the
    location of Target in Result if it's found,
    otherwise it returns the branch on which to
    continue the search.

*************************************************)

  Procedure Search_Node_Exact( Var Tree: Tree_Type; Var Target: Data_Type; Where: Point_Type; Var Found: Boolean;
                               Var Result: Position );
    Var
      Temporary: Data_Type;
    Begin
      Get_Data( Tree, Where, 1, Temporary );
     {$IFDEF VER40}
      If Less_Than_Default( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
     {$ELSE}
      If Tree.Less_Than( Target, Temporary, Tree.Info.Start, Tree.Info.Finish )
     {$ENDIF}
        then
          Begin
            Found := False;
            Result := 0;
          End
        else
          Begin
            Look( Tree, Where, Result, Target, Temporary );
           {$IFDEF VER40}
            Found := Equal_Default( Target, Temporary, 1, Tree.Info.Record_Size );
           {$ELSE}
            Found := Tree.Equal( Target, Temporary, 1, Tree.Info.Record_Size );
           {$ENDIF}
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search.
    This is the main procedure that handles the
    search for the target data.  When it finds it,
    it then searches for the first record with the
    same key name that is stored in the file.

*************************************************)

  Procedure Search( Var Tree: Tree_Type; Var Target: Data_Type; Root: Point_Type; Var Found: Boolean;
                    Var Target_Node: Point_Type; Var Target_Position: Position );
    Var
      Done,
      Extra_Found: Boolean;
      Temporary,
      Point1: Point_Type;
      Data: Data_Type;
      Point2: Position;
    Begin
      If ( Root = Null )
        then
          Found := False
        else
          Begin
            Search_Node( Tree, Target, Root, Found, Target_Position );
            If Found
              then
                Begin
                  Target_Node := Root;
                  Point1 := Target_Node;
                  Point2 := Target_Position;
                  Repeat
                    Get_Previous_In_Tree( Tree, Point1, Point2, Extra_Found );
                    If Extra_Found
                      then
                        Begin
                          Get_Data( Tree, Point1, Point2, Data );
                         {$IFDEF VER40}
                          If Equal_Default( Target, Data, Tree.Info.Start, Tree.Info.Finish )
                         {$ELSE}
                          If Tree.Equal( Target, Data, Tree.Info.Start, Tree.Info.Finish )
                         {$ENDIF}
                            then
                              Begin
                                Target_Node := Point1;
                                Target_Position := Point2;
                                Done := False;
                              End
                            else
                              Done := True;
                        End
                      else
                        Done := True;
                  Until Done;
                End
              else
                Begin
                  Temporary := Get_Pointer( Tree, Root, Target_Position );
                  Search( Tree, Target, Temporary, Found, Target_Node, Target_Position );
                End;
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Push it in.
    This procedure inserts Data and Right_Pointer
    into the node: Where^ at the location
    Position_Point.  Then it performs the overhead
    to keep the b-tree in optimum shape.

*************************************************)

  Procedure Push_In( Var Tree: Tree_Type; Data: Data_Type; Right_Pointer, Where: Point_Type; Position_Point: Position );
    Begin
      Scoot_Up( Tree, Where, Succ( Position_Point ), Right_Pointer, Data );
      If ( Right_Pointer <> Null )
        then
          Put_Previous( Tree, Right_Pointer, Where );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Split.
    This procedure  splits the node: Where^ with
    Data and Right_Pointer at the position
    specified by Position_Point into two nodes
    Where^ and Result_Right^ with the median key
    as a guide.

*************************************************)

  Procedure Split( Var Tree: Tree_Type; Data: Data_Type; Right_Pointer, Where: Point_Type; Position_Point: Position;
                   Var Result: Data_Type; Var Result_Right: Point_Type );
    Var
      Index,
      Median,
      Temporary1: Position;
      Temporary2: Data_Type;
      Temporary3: Point_Type;
    Begin
      If ( Position_Point <= Tree.Info.Minimum )
        then
          { New data goes to left half }
          Median := Tree.Info.Minimum
        else
          Median := Succ( Tree.Info.Minimum );
      New_Pointer( Tree, Result_Right );
      { Get new node and put it on the right }
      Put_Previous( Tree, Result_Right, Null );
      For Index := Succ( Median ) to Tree.Info.Maximum do
        Begin
          Get_Data_And_Pointer( Tree, Where, Index, Temporary2, Temporary3 );
          Put_Data_And_Pointer( Tree, Result_Right, ( Index - Median ), Temporary2, Temporary3 );
          If ( Temporary3 <> Null )
            then
              Put_Previous( Tree, Temporary3, Result_Right );
        End;
      Put_Count( Tree, Result_Right, ( Tree.Info.Maximum - Median ) );
      Put_Count( Tree, Where, Median );
      If ( Position_Point <= Tree.Info.Minimum )
        then
          Push_In( Tree, Data, Right_Pointer, Where, Position_Point )
        else
          Push_In( Tree, Data, Right_Pointer, Result_Right, ( Position_Point - Median ) );
      Temporary1 := Get_Count( Tree, Where );
      Get_Data_And_Pointer( Tree, Where, Temporary1, Result, Temporary3 );
      Put_Pointer( Tree, Result_Right, 0 , Temporary3 );
      If ( Temporary3 <> Null )
        then
          Put_Previous( Tree, Temporary3, Result_Right );
      Put_Count( Tree, Where, Pred( Temporary1 ) );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Push down.
    This procedure pushes the data around so that
    the new data can be inserted.  It performs
    using recursion, but takes care not to use
    up too much stack space.  A stack overflow,
    however can occur.

*************************************************)

  Procedure Push_Down( Var Tree: Tree_Type; New_Key: Data_Type; Where: Point_Type; Var Push_Up: Boolean; Var Data: Data_Type;
                       Var Right_Pointer: Point_Type );
    Var
      Position_Point: Position;
      Found: Boolean;
    Begin
      If ( Where = Null )
        then
          Begin
            { Cannot insert into empty tree; Recursion terminates }
            Push_Up := True;
            Data := New_Key;
            Right_Pointer := Null;
          End
        else
          Begin
            { Search current node }
            Search_Node( Tree, New_Key, Where, Found, Position_Point );
            Push_Down( Tree, New_Key, Get_Pointer( Tree, Where, Position_Point ), Push_Up, Data, Right_Pointer );
            If Push_Up
              then
                { Reinsert median key }
                If ( Get_Count( Tree, Where ) < Tree.Info.Maximum )
                  then
                    Begin
                      Push_Up := False;
                      Push_In( Tree, Data, Right_Pointer, Where, Position_Point );
                    End
                  else
                    Begin
                      Push_Up := True;
                      Split( Tree, Data, Right_Pointer, Where, Position_Point, Data, Right_Pointer );
                    End;
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert.
    This is the core routine that inserts New_Key
    into the b-tree with the given root.
    Normally it requires that New_Key is not
    already present, but had been revised to
    handle it.

*************************************************)

  Procedure Insert( Var Tree: Tree_Type; New_Key: Data_Type; Var Root: Point_Type );
    Var
      Push_Up: Boolean;
      Data: Data_Type;
      Where,
      Right_Pointer: Point_Type;
    Begin
      Push_Down( Tree, New_Key, Root, Push_Up, Data, Right_Pointer );
      If Push_Up
        then
          Begin
            New_Pointer( Tree, Where );
            Put_Previous( Tree, Where, Null );
            Put_Count( Tree, Where, 1 );
            Put_Pointer( Tree, Where, 0, Root );
            Put_Data_And_Pointer( Tree, Where, 1, Data, Right_Pointer );
            If ( Root <> Null )
              then
                Put_Previous( Tree, Root, Where );
            If ( Right_Pointer <> Null )
              then
                { Move the subtree on right of data }
                Put_Previous( Tree, Right_Pointer, Where );
            Root := Where;
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Successor.
    This procedure replaces Where^.Key[
    Position_Point ] by its immediate successor
    under natural order.

*************************************************)

  Procedure Successor( Var Tree: Tree_Type; Where: Point_Type; Position_Point: Position );
    Var
      Other_Point: Point_Type;
      Temporary: Data_Type;
    Begin
      Other_Point := Get_Pointer( Tree, Where, Position_Point );
      While ( Get_Pointer( Tree, Other_Point, 0 ) <> Null ) do
        Other_Point := Get_Pointer( Tree, Other_Point, 0 );
      Get_Data( Tree, Other_Point, 1, Temporary );
      Put_Data( Tree, Where, Position_Point, Temporary );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move everything right.
    This procedure moves everything right in the
    node for inserting records.

*************************************************)

  Procedure Move_Right( Var Tree: Tree_Type; Where: Point_Type; Position_Point: Position );
    Var
      Temporary1,
      Temporary2,
      Temporary3: Point_Type;
      Temporary4: Data_Type;
    Begin
      Get_Data_And_Pointer( Tree, Where, Position_Point, Temporary4, Temporary1 );
      Temporary3 := Get_Pointer( Tree, Temporary1, 0 );
      Scoot_Up( Tree, Temporary1, 1, Temporary3, Temporary4 );
      If ( Temporary3 <> Null )
        then
          Put_Previous( Tree, Temporary3, Temporary1 );
      Temporary1 := Get_Pointer( Tree, Where, Pred( Position_Point ) );
      Get_Data_And_Pointer( Tree, Temporary1, Get_Count( Tree, Temporary1 ), Temporary4, Temporary3 );
      Put_Data( Tree, Where, Position_Point, Temporary4 );
      Temporary2 := Get_Pointer( Tree, Where, Position_Point );
      Put_Pointer( Tree, Temporary2, 0, Temporary3 );
      If ( Temporary3 <> Null )
        then
          Put_Previous( Tree, Temporary3, Temporary2 );
      Decrement_Count( Tree, Temporary1 );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move everything left.
    This procedure moves everything in the node
    left for inserting records.

*************************************************)

  Procedure Move_Left( Var Tree: Tree_Type; Where: Point_Type; Position_Point: Position );
    Var
      Temporary1,
      Temporary2: Point_Type;
      Temporary3: Data_Type;
    Begin
      Temporary1 := Get_Pointer( Tree, Where, Pred( Position_Point ) );
      Get_Data_And_Pointer( Tree, Where, Position_Point, Temporary3, Temporary2 );
      Temporary2 := Get_Pointer( Tree, Temporary2, 0 );
      Increment_Count( Tree, Temporary1 );
      Put_Data_And_Pointer( Tree, Temporary1, Get_Count( Tree, Temporary1 ), Temporary3, Temporary2 );
      If ( Temporary2 <> Null )
        then
          Put_Previous( Tree, Temporary2, Temporary1 );
      Temporary1 := Get_Pointer( Tree, Where, Position_Point );
      Get_Data_And_Pointer( Tree, Temporary1, 1, Temporary3, Temporary2 );
      Put_Pointer( Tree, Temporary1, 0, Temporary2 );
      Put_Data( Tree, Where, Position_Point, Temporary3 );
      If ( Temporary2 <> Null )
        then
          Put_Previous( Tree, Temporary2, Temporary1 );
      Scoot_Back( Tree, Temporary1, 1 );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Combine nodes.
    This procedure combines two nodes together
    into one node.

*************************************************)

  Procedure Combine( Var Tree: Tree_Type; Where: Point_Type; Position_Point: Position );
    Var
      Other_Position: Position;
      Other_Point,
      Temporary1,
      Temporary2: Point_Type;
      Temporary3: Data_Type;
    Begin
      Get_Data_And_Pointer( Tree, Where, Position_Point, Temporary3, Other_Point );
      Temporary1 := Get_Pointer( Tree, Where, Pred( Position_Point ) );
      Temporary2 := Get_Pointer( Tree, Other_Point, 0 );
      Increment_Count( Tree, Temporary1 );
      Put_Data_And_Pointer( Tree, Temporary1, Get_Count( Tree, Temporary1 ), Temporary3, Temporary2 );
      If ( Temporary2 <> Null )
        then
          Put_Previous( Tree, Temporary2, Temporary1 );
      For Other_Position := 1 to Get_Count( Tree, Other_Point ) do
        Begin
          Increment_Count( Tree, Temporary1 );
          Get_Data_And_Pointer( Tree, Other_Point, Other_Position, Temporary3, Temporary2 );
          Put_Data_And_Pointer( Tree, Temporary1, Get_Count( Tree, Temporary1 ), Temporary3, Temporary2 );
          If ( Temporary2 <> Null )
            then
              Put_Previous( Tree, Temporary2, Temporary1 );
        End;
      Scoot_Back( Tree, Where, Position_Point );
      Dispose_Pointer( Tree, Other_Point );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore node.
    This procedure finds a key and inserts it into
    Where^.Branch[ Position_Point ]^ so as to
    restore minimum for balance.

*************************************************)

  Procedure Restore( Var Tree: Tree_Type; Where: Point_Type; Position_Point: Position );
    Begin
      If ( Position_Point > 0 )
        then
          If ( Get_Count( Tree, Get_Pointer( Tree, Where, Pred( Position_Point ) ) ) > Tree.Info.Minimum )
            then
              Move_Right( Tree, Where, Position_Point )
            else
              Combine( Tree, Where, Position_Point )
        else
          If ( Get_Count( Tree, Get_Pointer( Tree, Where, 1 ) ) > Tree.Info.Minimum )
            then
              Move_Left( Tree, Where, 1 )
            else
              Combine( Tree, Where, 1 );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Record delete.
    This procedure tries to delete the given
    target record from the b-tree.

*************************************************)

  Procedure Record_Delete( Var Tree: Tree_Type; Target: Data_Type; Where: Point_Type; Var Found: Boolean );
    Var
      Position_Point: Position;
      Temporary: Data_Type;
    Begin
      Found := ( Where <> Null );
      If Found
        then
          Begin
            Search_Node_Exact( Tree, Target, Where, Found, Position_Point );
            If Found
              then
                If ( Get_Pointer( Tree, Where, Pred( Position_Point ) ) = Null )
                  then
                    { Where^ is a leaf, remove key from Position_Point of Where^ }
                    Scoot_Back( Tree, Where, Position_Point )
                  else
                    Begin
                      { Replace Key[ Position_Point ] by its successor }
                      Successor( Tree, Where, Position_Point );
                      Get_Data( Tree, Where, Position_Point, Temporary );
                      Record_Delete( Tree, Temporary, Get_Pointer( Tree, Where, Position_Point ), Found );
                      If not Found
                        then
                          Write_Error( 215, 'Record_Delete: Record disappeared' );
                          { We know that new Key[ Position_Point ] was in the leaf }
                    End
              else
                { Target was not found in current node. }
                Record_Delete( Tree, Target, Get_Pointer( Tree, Where, Position_Point ), Found );
            { By now, Procedure has returned from recursive call. }
            If ( Get_Pointer( Tree, Where, Position_Point ) <> Null )
              then
                If ( Get_Count( Tree, Get_Pointer( Tree, Where, Position_Point ) ) < Tree.Info.Minimum )
                  then
                    Restore( Tree, Where, Position_Point );
          End;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete.
    This procedure is the core procedure for
    deleting a record from the b-tree.

*************************************************)

  Function Delete( Var Tree: Tree_Type; Target: Data_Type; Var Root: Point_Type ): Boolean;
    Var
      Found: Boolean;
      Where: Point_Type;
    Begin
      Record_Delete( Tree, Target, Root, Found );
      If Found
        then
          Begin
            If ( Get_Count( Tree, Root ) = 0 )
              then
                Begin
                  Where := Root;
                  Root := Get_Pointer( Tree, Root, 0 );
                  If ( Root <> Null )
                    then
                      Put_Previous( Tree, Root, Get_Previous( Tree, Where ) );
                  Dispose_Pointer( Tree, Where );
                End;
            Delete := True;
          End
        else
          Delete := False; { Target was not in the b-tree }
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find data in tree.
    As previously defined.

*************************************************)

    Function Find_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Found: Boolean;
        The_Data: Data_Type absolute Data;
      Begin
        Search( Tree, The_Data, Tree.Info.Top, Found, Tree.Where1, Tree.Where2 );
        If Found
          then
            Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
        Find_In_Tree := Found;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find previous in tree.
    As previously defined.

*************************************************)

    Function Find_Previous_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Save1: Point_Type;
        Save2: Position;
        Found: Boolean;
        The_Data: Data_Type absolute Data;
      Begin
        Save1 := Tree.Where1;
        Save2 := Tree.Where2;
        Get_Previous_In_Tree( Tree, Save1, Save2, Found );
        Find_Previous_In_Tree := Found;
        If Found
          then
            Begin
              Tree.Where1 := Save1;
              Tree.Where2 := Save2;
              Get_Data( Tree, Save1, Save2, The_Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find next in tree.
    As previously defined.

*************************************************)

    Function Find_Next_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Save1: Point_Type;
        Save2: Position;
        Found: Boolean;
        The_Data: Data_Type absolute Data;
      Begin
        Save1 := Tree.Where1;
        Save2 := Tree.Where2;
        Get_Next_In_Tree( Tree, Save1, Save2, Found );
        Find_Next_In_Tree := Found;
        If Found
          then
            Begin
              Tree.Where1 := Save1;
              Tree.Where2 := Save2;
              Get_Data( Tree, Save1, Save2, The_Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find first in tree.
    As previously defined.

*************************************************)

    Function Find_First_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        The_Data: Data_Type absolute Data;
        Temporary: Point_Type;
      Begin
        Tree.Where2 := 0;
        Tree.Where1 := Tree.Info.Top;
        If ( Tree.Where1 <> Null )
          then
            Begin
              Temporary := Get_Pointer( Tree, Tree.Where1, Tree.Where2 );
              While ( Temporary <> Null ) do
                Begin
                  Tree.Where1 := Temporary;
                  Temporary := Get_Pointer( Tree, Tree.Where1, Tree.Where2 );
                End;
              Get_Data( Tree, Tree.Where1, 1, The_Data );
              Tree.Where2 := 1;
              Find_First_In_Tree := True;
            End
          else
            Find_First_In_Tree := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find last in tree.
    As previously defined.

*************************************************)

    Function Find_Last_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        The_Data: Data_Type absolute Data;
        Temporary: Point_Type;
      Begin
        Tree.Where1 := Tree.Info.Top;
        If ( Tree.Where1 <> Null )
          then
            Begin
              Tree.Where2 := Get_Count( Tree, Tree.Where1 );
              Temporary := Get_Pointer( Tree, Tree.Where1, Tree.Where2 );
              While ( Temporary <> Null ) do
                Begin
                  Tree.Where1 := Temporary;
                  Tree.Where2 := Get_Count( Tree, Tree.Where1 );
                  Temporary := Get_Pointer( Tree, Tree.Where1, Tree.Where2 );
                End;
              Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
              Find_Last_In_Tree := True;
            End
          else
            Find_Last_In_Tree := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Update data in the B-tree.
    As previously defined.

*************************************************)

    Function Update_In_Tree( Var Tree: Tree_Type; Var Old_Data, New_Data ): Boolean;
      Var
        The_Old_Data: Data_Type absolute Old_Data;
        The_New_Data: Data_Type absolute New_Data;
      Begin
        Update_In_Tree := False;
        If Delete( Tree, The_Old_Data, Tree.Info.Top )
          then
            Begin
              Insert( Tree, The_New_Data, Tree.Info.Top );
              Update_In_Tree := True;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Delete data from tree.
    As previously defined.

*************************************************)

    Function Delete_From_Tree( Var Tree: Tree_Type; Var Old_Data ): Boolean;
      Var
        The_Data: Data_Type absolute Old_Data;
      Begin
        Delete_From_Tree := Delete( Tree, The_Data, Tree.Info.Top );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Insert data into tree.
    As previously defined.

*************************************************)

    Function Insert_In_Tree( Var Tree: Tree_Type; Var New_Data ): Boolean;
      Var
        The_Data: Data_Type absolute New_Data;
      Begin
        Insert( Tree, The_Data, Tree.Info.Top );
        Insert_In_Tree := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find first in file.
    As previously defined.

*************************************************)

    Function Find_First_In_File( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        The_Data: Data_Type absolute Data;
      Begin
        Tree.Where1 := 1;
        Tree.Where2 := 1;
        While ( ( Tree.Where1 < FileSize( Tree.The_File ) ) and ( Get_Count( Tree, Tree.Where1 ) = 0 ) ) do
          Inc( Tree.Where1 );
        If ( Tree.Where1 >= FileSize( Tree.The_File ) )
          then
            Find_First_In_File := False
          else
            Begin
              Find_First_In_File := True;
              Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find next in file.
    As previously defined.

*************************************************)

    Function Find_Next_In_File( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Save: Point_Type;
        The_Data: Data_Type absolute Data;
      Begin
        Save := Tree.Where1;
        If ( Tree.Where2 < Get_Count( Tree, Tree.Where1 ) )
          then
            Begin
              Inc( Tree.Where2 );
              Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
              Find_Next_In_File := True;
            End
          else
            Begin
              Repeat
                Inc( Tree.Where1 )
              Until ( ( Tree.Where1 >= FileSize( Tree.The_File ) ) or ( Get_Count( Tree, Tree.Where1 ) <> 0 ) );
              If ( Tree.Where1 < FileSize( Tree.The_File ) )
                then
                  Begin
                    Tree.Where2 := 1;
                    Find_Next_In_File := True;
                    Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
                  End
                else
                  Begin
                    Find_Next_In_File := False;
                    Tree.Where1 := Save;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find last in file.
    As previously defined.

*************************************************)

    Function Find_Last_In_File( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        The_Data: Data_Type absolute Data;
      Begin
        Tree.Where1 := Pred( FileSize( Tree.The_File ) );
        Tree.Where2 := Get_Count( Tree, Tree.Where1 );
        While ( ( Tree.Where1 > 0 ) and ( Get_Count( Tree, Tree.Where1 ) = 0 ) ) do
          Dec( Tree.Where1 );
        If ( Tree.Where1 = 0 )
          then
            Find_Last_In_File := False
          else
            Begin
              Find_Last_In_File := True;
              Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find previous in file.
    As previously defined.

*************************************************)

    Function Find_Previous_In_File( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Save: Point_Type;
        The_Data: Data_Type absolute Data;
      Begin
        Save := Tree.Where1;
        If ( Tree.Where2 > 1 )
          then
            Begin
              Dec( Tree.Where2 );
              Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
              Find_Previous_In_File := True;
            End
          else
            Begin
              Repeat
                Dec( Tree.Where1 )
              Until ( ( Tree.Where1 <= 0 ) or ( Get_Count( Tree, Tree.Where1 ) <> 0 ) );
              If ( Tree.Where1 > 0 )
                then
                  Begin
                    Tree.Where2 := Get_Count( Tree, Tree.Where1 );
                    Find_Previous_In_File := True;
                    Get_Data( Tree, Tree.Where1, Tree.Where2, The_Data );
                  End
                else
                  Begin
                    Find_Previous_In_File := False;
                    Tree.Where1 := Save;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(***********************************************************

  Procedure Change key routines.
    As previously defined.

***********************************************************)

   {$IFNDEF VER40}
    Procedure Change_Key_Routines( Var Tree: Tree_Type; Var Equal, Less_Than: Compare_Function );
      Begin
        Tree.Equal := Equal;
        Tree.Less_Than := Less_Than;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

  End.
