/*------------------------------------------------------------------------*
 * tzset(): POSIX-compatible timezone handling.                           *
 * ---------------------------------------------------------------------- *
 * This version of tzset() provides more correct analysis of the          *
 * environment variable TZ and DST rules than the function from standard  *
 * libraries of Borland C and Microsoft C.                                *
 *                                                                        *
 * Also provides _isDST() and _isindst() for internal use in the          *
 * Borland C or Microsoft C standart libraries.                           *
 *                                                                        *
 * Note: 'M' format of DST rules supported only, but time > 24 hours      *
 *       allowed, thus "Sun after fourth Sat" or "Sun >= 23"              *
 *       as in GB will be: "GMT0BST,M3.5.0/1,M10.4.6/25"                  *
 * BUG: 2100 year assuming leap wrongly!                                  *
 *                                                                        *
 *                                         (c) 1995 by Timofei Bondatenko *
 *------------------------------------------------------------------------*/
#include <time.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef   _MSC_VER
#define  tzset     _tzset
#define  tzname    _tzname
#define  timezone  _timezone
#define  daylight  _daylight
#define MONTH_DAY(x) (_days[x]+1)
extern int _days[13];
#else
#define MONTH_DAY(x)  (_monthDay[x])
const /*unsigned*/ int _monthDay[ ] =
{   0,   31,  59,  90, 120, 151, 181, 212, 243, 273, 304, 334, 365 };
#ifndef     tzname
static char _stdZone[4] = "GMT",
            _dstZone[4] = "   ";
char *const tzname[2] = { _stdZone, _dstZone };
#endif
#ifndef timezone
long    timezone; 
#endif
#ifndef daylight
 int    daylight; /*  Apply daylight savings */
#endif
#endif /*_MSC_VER*/
#ifndef altzone
long    altzone; /*   timezone for daylight */
#endif

/******************* Startup setting, Compiler-dependent! *****************/
#if   defined(_MSC_VER)
void __tzset(void) { tzset(); }
#elif defined(__BORLANDC__)
#pragma startup tzset 30
#endif
/**************************************************************************/

struct _TimeRule { int dt[3]; /* month, week, wday; */
                   int  hour;
                 };

static struct _TimeRule summer, winter;
    /* Russia: last Sun of Mar 02:00 / last Sun of Oct 03:00
          USA: first Sun of Apr / last Sun of Oct 02:00a */
static const struct _TimeRule summDef = { {  4, 1, 0 }, 2 },
                              wintDef = { { 10, 5, 0 }, 2 };

static long _parse_time(char **str)   /* time string: [+|-]00[:00[:00]] */
{                                     /* Returns: in seconds            */
 long tim = 0, mul = 60 * 60;
 char *s = *str;

 if (*s == '-') mul = -mul, s++;
 else if (*s == '+') s++;

 while(isdigit(*s))
   {
    tim += mul * atoi(s); mul /= 60;
    while(isdigit(*(++s)));
    if (mul && *s == ':') s++;
    else break;
   }
 *str = s;
 return tim;
}

static int _parse_date(struct _TimeRule *xd, char **str)
{                            /* Date format: Mmm.nn.dd[/hour] */
 int i = 0;                  /* Return:  0 if Ok              */
 char *s = *str;

 if (*s++ != ',' || *s++ != 'M') return -1;
 for(;;)
   if (isdigit(*s))
     {
      xd->dt[i] = atoi(s);
      while(isdigit(*(++s)));
      if (++i == 3) break;
      if (*s == '.') s++;
      else return -1;
     }
   else return -1;

 xd->hour = *s == '/'? (s++, (int)(_parse_time(&s) / (60 * 60))): 2;
 *str = s;
 return 0; /* Ok */
}

/*--------------------------------------------------------------------------*
  tzset() sets local timezone info base on the "TZ" environment string:
  TZ=<STD><hour1>[<DST>[<hour2>]',M'<day1>['/'<hour3>]',M'<day2>['/'<hour4>]]
  hour  :: [+|-]hh[:mm[:ss]] time, offset of local timezone, relativety GMT:
           0 < on the west, 0 > on the east;
  day   :: mm.nn.ww date of change to DST/STD, where mm- month (1...12);
           ww- day of week (0=Sun...6); nn- week in the month (1...5=last).
           for example: M3.5.0 = last Sun in Mar;
  STD   ::= 3-letter's name of STD timezone (GMT, UTC, MSK, EST etc...);
  hour1 ::= offset of standard (astronomical) time - STD;
  DST   ::= 3-letter's name of DST (daylight saving) timezone (MSD, EDT ...)
            if acceptable;
  hour2 ::= offset of DST - optional, assuming (hour1 - 01:00:00) by default;
  day1,2 ::= date of change to DST/STD;
  hour3,4 ::= time of change to DST/STD (by default = 02:00am)
              can be >= 24:00, for example in GB: "M10.4.6/25" mean
                                 September, Sun after fourth Sat 1:00am.
*---------------------------------------------------------------------------*/

void tzset(void)
{
 char *env;
#ifndef _MSC_VER
 (char*)tzname[0] = _stdZone;
 (char*)tzname[1] = _dstZone;
#endif

 if (!(env = getenv("TZ"))) return;
#if   1  /* Same TZ name ? */
 if (!strncmp(env, tzname[0], 3)) return; /* TimeZone changed? */
#elif 1  /* Hashing for skip parse */
{
 static unsigned hash;
 unsigned char *sps = (unsigned char*)env;
 unsigned hsh, sf; hsh = sf = 0;
 while(*sps) hsh ^= ((unsigned)*sps++) << ((sizeof(int)-1)*8-1 & sf++);
 if (hash == ~hsh) return;
 hash = ~hsh;
}
#endif   /* Hashing */

 if (!isalpha(env[0]) || !isalpha(env[1]) || !isalpha(env[2])) return;
 memcpy(tzname[0], env, 3); env += 3;
 timezone = _parse_time(&env);
 memcpy(tzname[1], "   ", 3);
 daylight = 0; summer = summDef; winter = wintDef;
 if (!isalpha(env[0]) || !isalpha(env[1]) || !isalpha(env[2])) return;
 memcpy(tzname[1], env, 3); env += 3;
 daylight++;
 altzone = isdigit(*env) ||
            '-' == *env ||
            '+' == *env ? _parse_time(&env): timezone - 60 * 60;
 if (_parse_date(&summer, &env)) return;
 if (_parse_date(&winter, &env)) return;
}

/*--------------------------------------------------------------------------*
 __mday() - helper for _isDST(), which determinates 'week's 'wday',
            as in 'M'-format.

 FDAY,LDAY = 0...365 first and last days of the requested month;
 YEAR= 1970...  year;
 WEEK= 1=first...5=last week (or wday) of the month, which defined by month's
       first and last days (fday, lday);
 WDAY= 0=Sun...6;
 Returns: div_t::quot = the day-of-year + hour/24, (0...365),
          div_t::rem = hour%24.
*---------------------------------------------------------------------------*/

static div_t __mday(int fday, int lday, int year,
                    int week, int wday, int hour)
{
 div_t dv = div(hour, 24);   /* 365 % 7 == 1 */
 wday -= ((year - 1970)/* *365*/ + fday + 4 + /* 01-01-70 was Thursday */
         ((year - 1968 - 1) >> 2)) % 7; /* leap days, 1968 was leap */
 fday += wday < 0 ? 7 + wday: wday;
 for(year = 0, lday -= 7; ++year < week && fday <= lday; fday += 7);
 dv.quot += fday;
 return dv;
}

/*--------------------------------------------------------------------------*
 _isDST() - determines whether daylight savings is in effect (for Borland C).
 If month is 0, yday is the day of the year (0...365), otherwise yday is the
 day of the month (0...30), year = 0 --> 1970.
 Returns 1 if daylight savings is in effect for the given date.
*---------------------------------------------------------------------------*/
#if defined(__BORLANDC__) && !defined(__FLAT__)
#define _isDST  __isDST
int pascal near
#else
int
#endif
    _isDST(int hour, int yday, int month, int year)
{
 int fday, lday = 0;
                       /* leap year < 2100 !! */
 year += 1970;         /* ??!! No required for Borland C++ */
 if (!daylight /*|| year <= 1980*/) return 0;

 if (month)             /* if month+day of month given */
   {
    fday = MONTH_DAY(month-1);
    if (!(year & 3) && month >= 2)
      {
       lday++;
       if (month > 2) fday++;
      }
    yday += fday;
   }
 else                   /* if only day of year given */
   {
    fday = yday;        /* if Mar 1 or later and leap year, sub day */
    if (!(year & 3) && yday >= 31+28) lday++, fday--;
    while(fday >= MONTH_DAY(++month));
    fday = MONTH_DAY(month-1);
    if (month > 2) fday += lday;
   }
 lday += MONTH_DAY(month)-1;
/*
 wday = ((year - 1970) * 365 + yday + 4 + // 01-01-70 was Thursday
        ((year - 1968 - 1) >> 2)) % 7; // leap days, 1968 was leap
*/
 if (summer.dt[0] > winter.dt[0])  /* Ausralia & Argentina */
   { if (month > summer.dt[0] || month < winter.dt[0]) return 1; }
 else { if (month > summer.dt[0] && month < winter.dt[0]) return 1; }

 if (month == summer.dt[0])
   {
    div_t dv = __mday(fday, lday, year,
                      summer.dt[1], summer.dt[2], summer.hour);
    if (yday > dv.quot || yday == dv.quot && hour >= dv.rem) return 1;
   }
 else if (month == winter.dt[0])
   {
    div_t dv = __mday(fday, lday, year,
                      winter.dt[1], winter.dt[2], winter.hour);
    if (yday < dv.quot) return 1;
    if (yday == dv.quot && hour < dv.rem)
      return /*hour < dv.rem - (timezone - altzone)/(60*60)? 1: -*/1;
   }
 return 0;
}

/*--------------------------------------------------------------------------*
 _isindst() - same as _isDST() for Microsoft C.
              determines whether daylight savings is in effect.
 *--------------------------------------------------------------------------*/

int _isindst(struct tm *pt)
{
 return _isDST(pt->tm_hour, pt->tm_yday, 0, pt->tm_year + 70);
}
/* end of tzset.c ---------------------------------------------------------*/