#include <dos.h>
#include <dpmi.h>
#include <bios.h>
#include <pc.h>
#include "gccint70.h"

#define IRQ8 0x70
#define SET_EVENT 0x8300
#define CANCEL_EVENT 0x8301
#define LARGEST 0xffff
#define PIT0 0x40
#define PIT2 0x42
#define PITMODE 0x43
#define PITCONST 1193180L
#define KBCTRL 0x61
#define NEW70H 1

static float    tick_per_ms = 1.024;
static float    ms_per_tick = 0.9765625;
static char    *event;
static unsigned char flag70h = 0;
static _go32_dpmi_seginfo rm_old_handler,
                rm_new_handler,
                pm_old_handler,
                pm_new_handler,
                event_handler;
static _go32_dpmi_registers r,
                r1;

volatile unsigned long int ticks_70h;

void            init70h(void)
{
    union REGS      regs;
    struct SREGS    sregs;

    unsigned int    event_seg,
                    event_off;

    if (flag70h != NEW70H) {

        _go32_dpmi_get_protected_mode_interrupt_vector(IRQ8, &pm_old_handler);
        pm_new_handler.pm_offset = (int) pm_new70h;
        pm_new_handler.pm_selector = _go32_my_cs();
        _go32_dpmi_chain_protected_mode_interrupt_vector(IRQ8, &pm_new_handler);

        _go32_dpmi_get_real_mode_interrupt_vector(IRQ8, &rm_old_handler);
        rm_new_handler.pm_offset = (int) rm_new70h;
        _go32_dpmi_allocate_real_mode_callback_iret(&rm_new_handler, &r1);
        _go32_dpmi_set_real_mode_interrupt_vector(IRQ8, &rm_new_handler);

        event_handler.pm_offset = (int) &event;

        event_seg = event_handler.rm_segment;
        event_off = event_handler.rm_offset;

        regs.x.ax = SET_EVENT;
        sregs.es = event_seg;
        regs.x.bx = event_off;
        regs.x.cx = LARGEST;
        regs.x.dx = LARGEST;
        int86x(0x15, &regs, &regs, &sregs);

        flag70h = NEW70H;
    }
}

void            quit70h(void)
{
    union REGS      regs;

    if (flag70h == NEW70H) {
        regs.x.ax = CANCEL_EVENT;
        int86(0x15, &regs, &regs);

        disable();

        _go32_dpmi_set_real_mode_interrupt_vector(IRQ8, &rm_old_handler);
        _go32_dpmi_set_protected_mode_interrupt_vector(IRQ8, &pm_old_handler);
        _go32_dpmi_free_real_mode_callback(&rm_new_handler);

        enable();

        flag70h = 0;
    }
}

void            rm_new70h(void)
{
    disable();
    ticks_70h++;
    memset(&r, 0, sizeof(r));
    r.x.cs = rm_old_handler.rm_segment;
    r.x.ip = rm_old_handler.rm_offset;
    r.x.ss = r.x.sp = 0;
    enable();
    _go32_dpmi_simulate_fcall_iret(&r);
}

void            pm_new70h(void)
{
    disable();
    ticks_70h++;
    enable();
}

unsigned long   time70h(unsigned long start, unsigned long stop)
{
    unsigned long   duration,
                    millisec;

    if (stop < start)
        return 0;
    else {
        duration = stop - start;
        millisec = duration * ms_per_tick;
        return millisec;
    }
}

void            delay70h(unsigned int delayms)
{
    unsigned long int delaybegin = 0;
    unsigned long int delayend = 0;
    unsigned int    delaytick;

    if (flag70h == NEW70H) {
        delaytick = delayms * tick_per_ms;
        delaybegin = ticks_70h;
        do {
            delayend = ticks_70h;
        } while ((delayend - delaybegin) < delaytick);
    } else {
        delaytick = delayms * 0.0182065;
        biostime(0, (long) &delaybegin);
        do {
            biostime(0, (long) &delayend);
        } while ((delayend - delaybegin) < delaytick);
    }
}

void            sound70h(int freq, int duration)
{
    int             byte;
    unsigned int    freq1;

    freq1 = PITCONST / freq;
    outportb(PITMODE, 0xb6);
    byte = (freq1 & 0xff);
    outportb(PIT2, byte);
    byte = (freq1 >> 8);
    outportb(PIT2, byte);
    byte = inportb(KBCTRL);
    outportb(KBCTRL, (byte | 3));

    delay70h(duration);
    outportb(KBCTRL, (byte & 0xfc));
}
