;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                         ;
;       Forms Dictionary utility - report printing                        ;
;                                                                         ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; --------------------------------------------------------------------------
; Procedure   : FormsReport()
; Description : produce printed reports
;    Argument : menu option specifying required report
; --------------------------------------------------------------------------

proc FormsReport(action)

switch
   case action = "Report/Images":
      message "Initialising report layout"
      ReportSetup("Frmimage", 1, controls["Printer"])
      message "Sending report to the printer"
      Report "Frmimage" 1
   case action = "Report/Fields":
      message "Initialising report layout"
      ReportSetup("Frmfield", 1, controls["Printer"])
      message "Sending report to the printer"
      Report "Frmfield" 1
   case action = "Report/Where":
      message "Initialising report layout"
      ReportSetup("Frmfield", 2, controls["Printer"])
      message "Sending report to the printer"
      Report "Frmfield" 2
   case action = "Report/Links":
      message "Initialising report layout"
      ReportSetup("Frmlinks", 1, controls["Printer"])
      message "Sending report to the printer"
      Report "Frmlinks" 1
   otherwise:
      CombinedReport(action)
endswitch
message ""

endproc
WriteLib AppLib FormsReport
Release procs FormsReport

; --------------------------------------------------------------------------
; Procedure   : CombinedReport()
; Description : produce combined screen image and field report
;   Arguments : option Report/Single (one form), Report/All (all forms)
; --------------------------------------------------------------------------

proc CombinedReport(action)
   private retcode, pagelen, resetstring, formtot

switch
   case action = "Report/Single":
      DisplayMenu("Select")
      view "Formlist"
      rwin = getwindow()
      window setattributes rwin from stdwindow
      rpwin = SetPrompt(
      "Move the cursor to the required form, pick option Select or press F2",
         "Pick option Cancel or press Escape to cancel")
      window select rwin
      echo normal
      echo off
      wait workspace
         proc "PickFormWait"
         key "Esc",  "F2"
         message "Menuselect"
      endwait
      if retcode = False then
         ClearPrompt(rpwin)
         clearimage
         echo normal echo off
         return
      endif
      ClearPrompt(rpwin)
      window move rwin to -100, -100
      echo normal echo off
      message "Initialising report output"
      InitCombinedReport()
      message "Printing report for table " + [Table] + ", form " + [Form]
      PrintCombinedReport([Table], [Form], [Form name])
      message ""
      clearimage
      echo normal echo off
   case action = "Report/All":
      InitCombinedReport()
      formtot = strval(nrecords("Formlist"))
      view "Formlist"
      scan
         message "Printing details of form " + strval(recno()) + " of " +
            formtot
         PrintCombinedReport([Table], [Form], [Form name])
      endscan
      clearimage
      echo normal echo off
endswitch
print CrPrintControl(resetstring)
message ""

endproc
WriteLib AppLib CombinedReport
Release procs CombinedReport

; --------------------------------------------------------------------------
; Procedure   : PickFormWait()
; Description : Actions triggered by user when defining a query
;   Arguments - trigger, event data, cycle number
; --------------------------------------------------------------------------

proc PickFormWait(ttype, edata, cnum)

      switch
         case edata["type"] = "MESSAGE" and edata["Menutag"] = "Select" or
            edata["type"] = "KEY" and edata["Keycode"] = -60:             ;F2
            retcode = True
            return 2
         case edata["type"] = "MESSAGE" and edata["Menutag"] = "Cancel" or
            edata["type"] = "KEY" and edata["Keycode"] = 27:              ;Esc
            retcode = False
            return 2
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;     case edata["type"] = "MESSAGE" and edata["Menutag"] = "Help" or      ;
    ;        edata["type"] = "KEY" and edata["Keycode"] = -59:             ;F1 ;
    ;        ShowHelp("CeditHelp", 2)                                          ;
    ;        DisplayMenu("Edit")                                               ;
    ;        return 1                                                          ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   endswitch

endproc
WriteLib AppLib PickFormWait
Release procs PickFormWait

; --------------------------------------------------------------------------
; Procedure   : InitCombinedReport()
; Description : initialise combined report output
; --------------------------------------------------------------------------

proc InitCombinedReport()

if controls["Printer"] = "None" then
   warning("No printer setup defined. Current printer setting will be used")
   return
endif
view "Printers"
moveto [Printer]
locate controls["Printer"]
if retval then
   pagelen = [Lines]
   resetstring = [Reset]
   print CrPrintControl([Setup])
else
   warning("Unknown printer setup. Current printer setting will be used")
endif
clearimage

endproc
WriteLib AppLib InitCombinedReport
Release procs InitCombinedReport

; --------------------------------------------------------------------------
; Procedure   : PrintCombinedReport()
; Description : produce combined screen image and field report for 1 form
;   Arguments : table name, form number (in Fnn format), form name
; --------------------------------------------------------------------------

proc PrintCombinedReport(tabname, fnum, fname)
   private fcode, pagenum, linect

fcode = iif(fnum = "F", "F", substr(fnum, 2, 2))
CrHeader(tabname, fcode, fname)

linect = linect + CrLinks(tabname, fnum, fcode)

pagenum = 1
while True
   if CrImage(tabname, fnum, pagenum) = False then
      quitloop
   endif
   linect = linect + 28
   CrFields(tabname, fnum, pagenum)
   pagenum = pagenum + 1
endwhile

endproc
WriteLib AppLib PrintCombinedReport
Release procs PrintCombinedReport

; --------------------------------------------------------------------------
; Procedure   : CrHeader()
; Description : print combined report embedded form data
;   Arguments : table name, form number in nn format, form name
;     Returns : number of lines printed
; --------------------------------------------------------------------------

proc CrHeader(tabname, fcode, fname)

print "Table: " + controls["Workdir"] + upper(tabname)
   + "\n\n"
print "Form: " + fcode + "  " + fname + "\n\n"
linect = 4

endproc
WriteLib AppLib CrHeader
Release procs CrHeader

; --------------------------------------------------------------------------
; Procedure   : CrPrintControl()
; Description : output combined report printer control codes
;   Arguments : setup/reset string as used by Paradox reports
;     Returns : string to be sent to the printer
; --------------------------------------------------------------------------

proc CrPrintControl(pcs)
   private i, n, newpcs

if upper(substr(pcs, 1, 5)) = "FILE=" then
   Run NoRefresh "copy " + substr(pcs, 6, 25) + " prn >nul"
   return ""
endif

n = len(pcs)
newpcs = ""
for i from 1 to n
   if substr(pcs, i, 1) = "\\" then
      newpcs = newpcs + chr(numval(substr(pcs, i + 1, 3)))
      i = i + 3
   else
      newpcs = newpcs + substr(pcs, i, 1)
   endif
endfor
return newpcs

endproc
WriteLib AppLib CrPrintControl
Release procs CrPrintControl

; --------------------------------------------------------------------------
; Procedure   : CrLinks()
; Description : print combined report embedded form data
;   Arguments : table name, form number (in Fnn format) and nn format
;     Returns : number of lines printed
; --------------------------------------------------------------------------

proc CrLinks(tabname, fnum, fcode)
   private nlines, foundone

foundone = False
nlines = 0
view "Frmlinks"
scan for [Table] = tabname and [Form] = fnum
   if foundone = False then
      print "Embedded forms: \n"
      nlines = 1
      foundone = True
   endif
   print "Page " + format("W2", strval([Page])) + "    Table: " +
   [Linktable] + "\n        Form: " + format("w2", fcode) + "   from " +
      format("w5",strval([Startrow]) + "," + strval([Startcol]))
      + " to " + format("w5", strval([Endrow]) + "," + strval([Endcol])) +
      " - " + [Linktype] + "\n"
   nlines = nlines + 2
endscan
print "\n"
clearimage
return nlines

endproc
WriteLib AppLib CrLinks
Release procs CrLinks

; --------------------------------------------------------------------------
; Procedure   : CrImage()
; Description : print combined report screen image data
;   Arguments : table name, form number (in Fnn format), page number
;     Returns : True if image printed, False if not
; --------------------------------------------------------------------------

proc CrImage(tabname, fnum, pagenum)

view "Frmimage"
locate tabname, fnum, pagenum
if retval = False then
   clearimage
   return False
endif
if (linect + 28 > pagelen) or (pagenum > 1) then
   CrHeader(tabname, fcode, fname)
endif
print "Page: " + strval(pagenum) + "\n"
print "            1         2         3         4" +
                             "         5         6         7         8\n" +
                            "   ----5----0----5----0----5----0----5----0" +
                               "----5----0----5----0----5----0----5----0\n"

scan for [Table] = tabname and [Form] = fnum and [Page] = pagenum
   print format("w2, ar", strval([Row])) + " " +
      [Picture] + "\n"
endscan
print "\n"
clearimage
return True

endproc
WriteLib AppLib CrImage
Release procs CrImage

; --------------------------------------------------------------------------
; Procedure   : CrFields()
; Description : print combined report field data
;   Arguments : table name, form number (in Fnn format), page number
; --------------------------------------------------------------------------

proc CrFields(tabname, fnum, pagenum)
   private nlines

if (linect + 6 > pagelen) then
   CrHeader(tabname, fcode, fname)
endif
print "Fields:\n" +
" From     To       Type            Wrap    Field Name/Expression\n"
nlines = 2

view "Frmfield"
scan for [Table] = tabname and [Form] = fnum and [Page] = pagenum
   print format("w2,ar", strval([Startrow])) + format("w3,ar", [Startcol]) +
      "  " + format("w3,ar", [Endrow]) + format("w3,ar", [Endcol]) +
      "      " + format("w13,al", [Type]) + "   " +
      format("w3,ar", [Wrap]) + "     " + [Name/Formula] + "\n"
   nlines = nlines + 1
   if linect + nlines + 4 >= pagelen then
      print "\f"
      CrHeader(tabname, fcode, fname)
      print "Fields:\n" +
      " From     To       Type              FieldName/Expression\n"
      nlines = 2
   endif
endscan

print "\f"
clearimage

endproc
WriteLib AppLib CrFields
Release procs CrFields
