/**********************************************************************
 *  
 *  NAME:           video.cpp
 *  
 *  DESCRIPTION:    video functions for Turbo C video library
 *  
 *  copyright (c) 1990 J. Alan Eldridge
 * 
 *  M O D I F I C A T I O N   H I S T O R Y
 *
 *  when        who                 what
 *  -------------------------------------------------------------------
 *  08/05/90    J. Alan Eldridge    created
 *
 *  08/12/90    JAE                 removed some error checking,
 *                                  made some functions void
 *  
 *  12/21/90    JAE                 un-object-oriented it, because
 *                                  it just didn't seem to fit
 *
 *********************************************************************/

#include    "aedef.h"
#include    "vidchr.h"
#include    "video.h"

#if defined(__TURBOC__)
#include    "envutils.h"
#endif

//  static storage for video driver

static  struct text_info    tInfo;  //  so we can put the world back

static  uchar   mode;           //  mode we're in
static  uchar   nRows,          //  size of physical display
                nCols;
static  uchar   cRow,           //  current cursor position
                cCol;
static  uchar   fOpen,          //  device has been opened
                fColor,         //  device has color support
                fSaved;         //  image has been saved
static  vidchr  *pSvImg;        //  ptr to saved image

static  uchar   curscnt,        //  count for show/hide cursor
                curstype;       //  type cursor would be if shown

//  global for default video attribute

uchar   vid_defaultatt = vid_mkatt(LIGHTGRAY,BLACK);

//  globals to avoid function call overhead

int     vid_ROWS, vid_COLS, vid_ISCOLOR;

//  flags handling

static void
zflags()    
{
    vid_ROWS = vid_COLS = 0;
    fOpen = fColor = fSaved = vid_ISCOLOR = FALSE;
} 

int
vid_isopen()
{
    return fOpen;
}

int
vid_iscolor()
{
    return fColor;
}

//  write (up to) one row of screen memory

int
vid_write(
    uchar   atRow,
    uchar   atCol,
    vidchr   *pBuf,
    uchar   nChrs)
{
    ++atRow, ++atCol;
    return puttext(atCol, atRow, atCol + nChrs - 1, atRow, pBuf) ? OK : ERR;
}

//  read (up to) one row of screen memory

int
vid_read(
    uchar   atRow,
    uchar   atCol,
    vidchr   *pBuf,
    uchar   nChrs)
{
    ++atRow, ++atCol;
    return gettext(atCol, atRow, atCol + nChrs - 1, atRow, pBuf) ? OK : ERR;
}

//  set physical cursor position

int
vid_setcpos(
    uchar   atRow,
    uchar   atCol)
{
    if (atRow < nRows && atCol < nCols) {
        cCol = atCol++;
        cRow = atRow++;
        gotoxy(atCol, atRow);
        return OK;
    } else
        return ERR;
}

//  get physical cursor position

void
vid_getcpos(
    uchar   *pRC)
{
    pRC[0] = cRow;
    pRC[1] = cCol;
}

//  clear the screen

void
vid_clear(
    uchar   att)
{
    textattr(att);
    clrscr();
}

//  initialize the video mode & try to save the image

void
vid_open(
    int     newMode,
    char    *evar)
{
    struct text_info    newInfo;
    
    if (fOpen) return;
    
    zflags();
    newMode = vid_envmode(newMode, evar);
    gettextinfo(&tInfo);
    if (newMode != LASTMODE) {
        textmode(newMode);
        gettextinfo(&newInfo);
    } else
        newInfo = tInfo;
    mode = newInfo.currmode;
    vid_ROWS = nRows = newInfo.screenheight;
    vid_COLS = nCols = newInfo.screenwidth;
    if (newInfo.currmode != MONO && newInfo.currmode != BW80 &&
        newInfo.currmode != BW40)
        vid_ISCOLOR = fColor = TRUE;
    pSvImg = new vidchr [ tInfo.screenwidth * tInfo.screenheight ];
    if (pSvImg) {
        gettext(1, 1, tInfo.screenwidth, tInfo.screenheight, pSvImg);
        fSaved = TRUE;
    }
    curscnt = 0;
    _setcursortype(curstype = _NORMALCURSOR);
    fOpen = TRUE;
    atexit(vid_close);
}

//  restore saved video mode & image

void
vid_close()
{
    if (fOpen) {
        textmode(tInfo.currmode);
        textattr(tInfo.normattr);
        if (fSaved) {
            puttext(1, 1, tInfo.screenwidth, tInfo.screenheight, pSvImg);
            delete pSvImg;
            gotoxy(tInfo.curx, tInfo.cury);
        } else {
            clrscr();
            gotoxy(1, 1);
        }
    }
    _setcursortype(_NORMALCURSOR);
    zflags();
}
        
//  parse environment variable & return desired video mode

int
vid_envmode(
    int     mode,
    char    *varName)
{
#if defined(__TURBOC__)
    static struct {
        char    *name;
        int     mode;
    } eVals[] = {
        "MONO",     MONO,
        "CO40",     C40,
        "BW40",     BW40,
        "CO80",     C80,
        "BW80",     BW80,
        "C4350",    C4350,
        "",         LASTMODE
    };
    
    int     nVals;    
    char    *vals[2];
    
    if (!varName || !varName[0])
        return mode;
        
    if (nVals = envParse(varName, ";, \t", vals, 2, 0)) {
        for (int n = 0; eVals[n].name[0]; n++) {
            if (!stricmp(vals[0], eVals[n].name) 
                || (nVals > 1 && !stricmp(vals[1], eVals[n].name))) {
                mode = eVals[n].mode;
                break;
            }
        }
        if (!stricmp(vals[0], "BIOS") || (nVals > 1 && !stricmp(vals[1], "BIOS")))
            directvideo = 0;
    }
    return mode;
#else
    return LASTMODE;
#endif
}

int
vid_rows()
{
    return nRows;
}

int
vid_cols()
{
    return nCols;
}

//  high level cursor handling

void
vid_hidecursor(int newcnt)
{
    int musthide = !curscnt;
    
    if (newcnt)
        curscnt = newcnt;
    else
        curscnt--;
        
    if (musthide)
        _setcursortype(_NOCURSOR);
}

void
vid_showcursor()
{
    if (curscnt < 0)
        ++curscnt;
    if (curscnt == 0)
        _setcursortype(curstype);
}

void
vid_forcecursor(int &oldcnt)
{
    oldcnt = curscnt;
    curscnt = 0;
    _setcursortype(curstype);
}
    
int
vid_cursortype(int type)
{
    int oldtype = curstype;
    
    curstype = type;
    if (curscnt == 0)
        _setcursortype(curstype);
    return oldtype;
}

int
vid_cursortype()
{
    return curstype;
}

#if NOT_NEEDED

//  this class exists just to make sure the screen 
//  gets put back when the program exits: the destructor
//  for the static variable Video calls vid_close()

class video {
public:
    ~video()
        { vid_close(); }
};

static video Video;

#endif



