/* xgettext - extracts strings from C source file to Uniforum style .po file
   Copyright (C) 1995 Free Software Foundation, Inc.
   Written by Ulrich Drepper <drepper@gnu.ai.mit.edu>, April 1995.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <pwd.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
#endif

#ifdef HAVE_LOCALE_H
# include <locale.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifndef errno
extern int errno;
#endif

#include "error.h"
#include "hash.h"
#include "getline.h"
#include "system.h"
#include "po.h"
#include "message.h"
#include "xget-lex.h"

#include "gettext.h"
#include "domain.h"
#include <libintl.h>

#ifndef _POSIX_VERSION
struct passwd *getpwuid ();
#endif

/* A convenience macro.  I don't like writing gettext() every time.  */
#define _(str) gettext (str)


/* If nonzero add all comments immediately preceding one of the keywords. */
static int add_all_comments;

/* If nonzero add comments for file name and line number for each msgid.  */
static int line_comment;

/* Tag used in comment of prevailing domain.  */
static char *comment_tag;

/* Name of default domain file.  If not set defaults to messages.po.  */
static char *default_domain;

/* Content of .po files with symbols to be excluded.  */
static message_list_ty *exclude;

/* If nonzero extract all strings.  */
static int extract_all;

/* String used as prefix for msgstr.  */
static char *msgstr_prefix;

/* String used as suffix for msgstr.  */
static char *msgstr_suffix;

/* Directory in which output files are created.  */
static char *output_dir;

/* If nonzero omit header with information about this run.  */
static int omit_header;

/* String containing name the program is called with.  */
const char *program_name;

/* If nonzero be verbose in giving warnings.  */
static int verbose;

/* String length from with on warning are given for possible problem
   while exceeding tools limits.  */
static size_t warn_id_len;

/* Long options.  */
static const struct option long_options[] =
{
  { "add-comments", optional_argument, NULL, 'c' },
  { "add-location", no_argument, &line_comment, 1 },
  { "c++", no_argument, NULL, 'C' },
  { "default-domain", required_argument, NULL, 'd' },
  { "directory", required_argument, NULL, 'D' },
  { "escape", no_argument, NULL, 'E' },
  { "exclude-file", required_argument, NULL, 'x' },
  { "extract-all", no_argument, &extract_all, 1 },
  { "files-from", required_argument, NULL, 'f' },
  { "help", no_argument, NULL, 'h' },
  { "indent", no_argument, NULL, 'i' },
  { "join-existing", no_argument, NULL, 'j' },
  { "keyword", optional_argument, NULL, 'k' },
  { "msgstr-prefix", optional_argument, NULL, 'm' },
  { "msgstr-suffix", optional_argument, NULL, 'M' },
  { "no-escape", no_argument, NULL, 'e' },
  { "no-location", no_argument, &line_comment, 0 },
  { "omit-header", no_argument, &omit_header, 1 },
  { "output-dir", required_argument, NULL, 'p' },
  { "sort-by-file", no_argument, NULL, 'F' },
  { "sort-output", no_argument, NULL, 's' },
  { "strict", no_argument, NULL, 'S' },
  { "string-limit", required_argument, NULL, 'l' },
  { "trigraphs", no_argument, NULL, 'T' },
  { "verbose", no_argument, NULL, 'v' },
  { "version", no_argument, NULL, 'V' },
  { NULL, 0, NULL, 0 }
};


/* Prototypes for local functions.  */
static void usage __P ((int status))
#if defined __GNUC__ && ((__GNUC__ == 2 && __GNUC_MINOR__ > 4) || __GNUC__ > 2)
	__attribute__ ((noreturn))
#endif
;
static void error_print __P ((void));
static string_list_ty *read_name_from_file __P ((const char *__file_name));
static void exclude_directive_domain __P ((po_ty *__pop, char *__name));
static void exclude_directive_message __P ((po_ty *__pop, char *__msgid,
					    lex_pos_ty *__msgid_pos,
					    char *__msgstr,
					    lex_pos_ty *__msgstr_pos));
static void read_exclusion_file __P ((char *__file_name));
static void remember_a_message __P ((message_list_ty *__mlp,
				     xgettext_token_ty *__tp));
static void scan_c_file __P ((const char *__file_name,
			      message_list_ty *__mlp));
static void extract_constructor __P ((po_ty *__that));
static void extract_directive_domain __P ((po_ty *__that, char *__name));
static void extract_directive_message __P ((po_ty *__that, char *__msgid,
					    lex_pos_ty *__msgid_pos,
					    char *__msgstr,
					    lex_pos_ty *__msgstr_pos));
static void extract_parse_brief __P ((po_ty *__that));
static void extract_comment __P ((po_ty *__that, const char *__s));
static void extract_comment_dot __P ((po_ty *__that, const char *__s));
static void extract_comment_filepos __P ((po_ty *__that, const char *__name,
					  int __line));
static void read_po_file __P ((const char *__file_name,
			       message_list_ty *__mlp));
static long difftm __P ((const struct tm *__a, const struct tm *__b));
static message_ty *construct_header __P ((char *option_list, char *file_list));


int
main (argc, argv)
     int argc;
     char *argv[];
{
  int cnt;
  int optchar;
  int do_help = 0;
  int do_version = 0;
  message_list_ty *mlp;
  int join_existing = 0;
  int sort_output = 0;
  int sort_by_file = 0;
  char *file_name;
  const char *directory = NULL;
  const char *old_wd = NULL;
  const char *files_from = NULL;
  string_list_ty *file_list;

  /* Set program name for messages.  */
  program_name = argv[0];
  error_print_progname = error_print;
  warn_id_len = WARN_ID_LEN;

#ifdef HAVE_SETLOCALE
  /* Set locale via LC_ALL.  */
  setlocale (LC_ALL, "");
#endif

  /* Set the text message domain.  */
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* Set initial value of variables.  */
  line_comment = -1;
  default_domain = MESSAGE_DOMAIN_DEFAULT;

  while ((optchar = getopt_long (argc, argv,
				 "ac::Cd:D:eEf:Fhijk::l:m::M::np:sTvVx:",
				 long_options, NULL)) != EOF)
    switch (optchar)
      {
      case '\0':		/* Long option.  */
	break;
      case 'a':
	extract_all = 1;
	break;
      case 'c':
	if (optarg == NULL)
	  {
	    add_all_comments = 1;
	    comment_tag = NULL;
	  }
	else
	  {
	    add_all_comments = 0;
	    comment_tag = optarg;
	    /* We ignore leading white space.  */
	    while (isspace (*comment_tag))
	      ++comment_tag;
	  }
	break;
      case 'C':
	xgettext_lex_cplusplus ();
	break;
      case 'd':
	default_domain = optarg;
	break;
      case 'D':
	directory = optarg;
	break;
      case 'e':
	message_print_style_escape (0);
	break;
      case 'E':
	message_print_style_escape (1);
	break;
      case 'f':
	files_from = optarg;
	break;
      case 'F':
	sort_by_file = 1;
        break;
      case 'h':
	do_help = 1;
	break;
      case 'i':
	message_print_style_indent ();
	break;
      case 'j':
	join_existing = 1;
	break;
      case 'k':
	if (optarg == NULL || *optarg != '\0')
	  xgettext_lex_keyword (optarg);
	break;
      case 'l':
	{
	  char *endp;
	  size_t tmp_val = strtoul (optarg, &endp, 0);
	  if (endp[0] == '\0')
	    warn_id_len = tmp_val;
	}
	break;
      case 'm':
	/* -m takes an optional argument.  If none is given "" is assumed. */
	msgstr_prefix = optarg == NULL ? "" : optarg;
	break;
      case 'M':
	/* -M takes an optional argument.  If none is given "" is assumed. */
	msgstr_suffix = optarg == NULL ? "" : optarg;
	break;
      case 'n':
	line_comment = 1;
	break;
      case 'p':
	{
	  size_t len = strlen (optarg);

	  if (output_dir != NULL)
	    free (output_dir);

	  if (optarg[len - 1] == '/')
	    output_dir = xstrdup (optarg);
	  else
	    {
	      asprintf (&output_dir, "%s/", optarg);
	      if (output_dir == NULL)
		/* We are about to construct the absolute path to the
		   directory for the output files but asprintf failed.  */
		error (EXIT_FAILURE, errno, _("while preparing output"));
	    }
	}
	break;
      case 's':
	sort_output = 1;
	break;
      case 'S':
	message_print_style_uniforum ();
	break;
      case 'T':
	xgettext_lex_trigraphs ();
	break;
      case 'v':
	verbose = 1;
	break;
      case 'V':
	do_version = 1;
	break;
      case 'x':
	read_exclusion_file (optarg);
	break;
      default:
	usage (EXIT_FAILURE);
	/* NOTREACHED */
      }

  /* Normalize selected options.  */
  if (omit_header != 0 && line_comment < 0)
    line_comment = 0;

  if (!line_comment && sort_by_file)
    {
      error (EXIT_FAILURE, 0, _("%s and %s are mutually exclusive"),
	     "--no-location", "--sort-by-file");
    }

  if (sort_output && sort_by_file)
    {
      error (EXIT_FAILURE, 0, _("%s and %s are mutually exclusive"),
	     "--sort-output", "--sort-by-file");
    }

  /* Version information requested.  */
  if (do_version)
    {
      fprintf (stderr, "%s - GNU %s %s\n", program_name, PACKAGE, VERSION);
      exit (EXIT_SUCCESS);
    }

  /* Help is requested.  */
  if (do_help)
    usage (EXIT_SUCCESS);

  /* Test whether we have some input files given.  */
  if (files_from == NULL && optind >= argc)
    {
      error (EXIT_SUCCESS, 0, _("no input file given"));
      usage (EXIT_FAILURE);
    }

  /* Canonize msgstr prefix/suffix.  */
  if (msgstr_prefix != NULL && msgstr_suffix == NULL)
    msgstr_suffix = "";
  else if (msgstr_prefix == NULL && msgstr_suffix != NULL)
    msgstr_prefix = NULL;

  /* Default output directory is the current directory.  */
  if (output_dir == NULL)
    output_dir = "./";

  /* Construct the name of the ouput file.  */
  file_name = (char *) xmalloc (strlen (output_dir)+ strlen (default_domain)
				+ sizeof (".po"));
  stpcpy (stpcpy (stpcpy (file_name, output_dir), default_domain), ".po");

  /* Determine list of files we have to process.  */
  if (files_from != NULL)
    file_list = read_name_from_file (files_from);
  else
    file_list = string_list_alloc ();
  /* Append names from command line.  */
  for (cnt = optind; cnt < argc; ++cnt)
    string_list_append_unique (file_list, argv[cnt]);

  /* Allocate a message list to remember all the messages.  */
  mlp = message_list_alloc ();

  /* Generate a header, so that we know how and when this PO file was
     created.  */
  if (!omit_header)
    {
      string_list_ty *slp;
      char *opt;
      char *fil;

      /* Compose a string with all command line options.  */
      slp = string_list_alloc ();
      for (cnt = 1; cnt < optind; ++cnt)
	string_list_append (slp, argv[cnt]);

      /* A make a single string from the list.  */
      opt = string_list_join (slp);

      string_list_free (slp);

      /* We need a string with the names of all .c files.  */
      fil = string_list_join (file_list);

      message_list_append (mlp, construct_header (opt, fil));
      free (opt);
      free (fil);
    }

  /* Read in the old messages, so that we can add to them.  */
  if (join_existing)
    read_po_file (file_name, mlp);

  /* Change to given directory before reading input files.  */
  if (directory != NULL)
    {
      old_wd = xgetcwd ();
      if (old_wd == NULL || chdir (directory) < 0)
	error (EXIT_FAILURE, errno, _("cannot change to directory \"%s\""),
	       directory);
    }

  /* Process all input files.  */
  for (cnt = 0; cnt < file_list->nitems; ++cnt)
    {
      /* Locate input file and open it.  */
      scan_c_file (file_list->item[cnt], mlp);
    }
  string_list_free (file_list);

  /* Change back to initial directory.  */
  if (old_wd != NULL && chdir (old_wd) < 0)
    error (EXIT_FAILURE, errno, _("cannot change back to directory \"%s\""),
	   old_wd);

  /* Sorting the list of messages.  */
  if (sort_by_file)
    message_list_sort_by_filepos (mlp);
  else if (sort_output)
    message_list_sort_by_msgid (mlp);

  /* Write the PO file.  */
  message_list_print (mlp, file_name);

  exit (EXIT_SUCCESS);
}


/* Display usage information and exit.  */
static void
usage (status)
     int status;
{
  if (status != EXIT_SUCCESS)
    fprintf (stderr, _("Try `%s --help' for more information\n"),
	     program_name);
  else
    {
      printf (_("\
Usage: %s [OPTION] INPUTFILE ...\n\
Mandatory arguments to long options are mandatory for short options too.\n\
  -a, --extract-all              extract all strings\n\
  -c, --add-comments[=TAG]       place comment block with TAG (or those\n\
                                 preceding keyword lines) in output file\n\
  -C, --c++                      recognize C++ style comments\n\
  -d, --default-domain=NAME      use NAME.po for output (instead of messages.po)\n\
  -D, --directory=DIRECTORY      change to DIRECTORY before processing\n\
  -e, --no-escape                do not use C escapes in output (default)\n\
  -E, --escape                   use C escapes in output, no extended chars\n\
  -f, --files-from=FILE          get list of input files from FILE\n\
  -F, --sort-by-file             sort output by file location\n\
  -h, --help                     display this help and exit\n\
  -i, --indent                   write the .po file using indented style\n\
  -j, --join-existing            join messages with existing file\n"),
	      program_name);
      printf (_("\
  -k, --keyword[=WORD]           additonal keyword to be looked for (without\n\
                                 WORD means not to use default keywords)\n\
  -l, --string-limit=NUMBER      set string length limit to NUMBER instead %u\n\
  -m, --msgstr-prefix[=STRING]   use STRING or \"\" as prefix for msgstr entries\n\
  -M, --msgstr-suffix[=STRING]   use STRING or \"\" as suffix for msgstr entries\n\
      --no-location              do not write '#: filename:line' lines\n\
  -n, --add-location             generate '#: filename:line' lines (default)\n\
      --omit-header              don't write header with `msgid \"\"' entry\n"),
	      WARN_ID_LEN);
      fputs (_("\
  -p, --output-dir=DIR           output files will be placed in directory DIR\n\
  -s, --sort-output              generate sorted output and remove duplicates\n\
      --strict                   write out strict Uniforum conforming .po file\n\
  -T, --trigraphs                understand ANSI C trigraphs for input\n\
  -v, --verbose                  be verbose when giving warnings\n\
  -V, --version                  output version information and exit\n\
  -x, --exclude-file=FILE        entries from FILE are not extracted\n\
\n\
If INPUTFILE is -, standard input is read.\n"),
	     stdout);
    }

  exit (status);
}


/* The address of this function will be assigned to the hook in the error
   functions.  */
static void
error_print ()
{
  /* We don't want the program name to be printed in messages.  */
}


/* Read list of files to process from file.  */
static string_list_ty *
read_name_from_file (file_name)
     const char *file_name;
{
  size_t line_len = 0;
  char *line_buf = NULL;
  FILE *fp;
  string_list_ty *result;

  if (strcmp (file_name, "-") == 0)
    fp = stdin;
  else
    {
      fp = fopen (file_name, "r");
      if (fp == NULL)
	error (EXIT_FAILURE, errno,
	       _("error while opening \"%s\" for reading"), file_name);
    }

  result = string_list_alloc ();

  while (!feof (fp))
    {
      /* Read next line from file.  */
      int len = getline (&line_buf, &line_len, fp);

      /* In case of an error leave loop.  */
      if (len < 0)
	break;

      /* Remove trailing '\n'.  */
      if (len > 0 && line_buf[len - 1] == '\n')
	line_buf[--len] = '\0';

      /* Test if we have to ignore the line.  */
      if (*line_buf == '\0' || *line_buf == '#')
	continue;

      string_list_append_unique (result, line_buf);
    }

  /* Free buffer allocated through getline.  */
  if (line_buf != NULL)
    free (line_buf);

  /* Close input stream.  */
  if (fp != stdin)
    fclose (fp);

  return result;
}


static void
exclude_directive_domain (pop, name)
     po_ty *pop;
     char *name;
{
  po_gram_error (_("this file may not contain domain directives"));
}


static void
exclude_directive_message (pop, msgid, msgid_pos, msgstr, msgstr_pos)
     po_ty *pop;
     char *msgid;
     lex_pos_ty *msgid_pos;
     char *msgstr;
     lex_pos_ty *msgstr_pos;
{
  message_ty *mp;

  /* See if this message ID has been seen before.  */
  if (exclude == NULL)
    exclude = message_list_alloc ();
  mp = message_list_search (exclude, msgid);
  if (mp != NULL)
    free (msgid);
  else
    {
      mp = message_alloc (msgid);
      /* Do not free msgid.  */
      message_list_append (exclude, mp);
    }

  /* All we care about is the msgid.  Throw the msgstr away.
     Don't even check for duplicate msgids.  */
  free (msgstr);
}


/* So that the one parer can be used for multiple programs, and also
   use good data hiding and encapsulation practices, an object
   oriented approach has been taken.  An object instance is allocated,
   and all actions resulting from the parse will be through
   invocations of method functions of that object.  */

static po_method_ty exclude_methods =
{
  sizeof (po_ty),
  NULL, /* constructor */
  NULL, /* destructor */
  exclude_directive_domain,
  exclude_directive_message,
  NULL, /* parse_brief */
  NULL, /* parse_debrief */
  NULL, /* comment */
  NULL, /* comment_dot */
  NULL, /* comment_filepos */
};


static void
read_exclusion_file (file_name)
     char *file_name;
{
  po_ty *pop;

  pop = po_alloc (&exclude_methods);
  po_scan (pop, file_name);
  po_free (pop);
}


static void
remember_a_message (mlp, tp)
     message_list_ty *mlp;
     xgettext_token_ty *tp;
{
  char *msgid;
  message_ty *mp;
  char *msgstr;

  msgid = tp->string;

  /* See whether we shall exclude this message.  */
  if (exclude != NULL && message_list_search (exclude, msgid) != NULL)
    {
      /* Tell the lexer to reset its comment buffer, so that the next
	 message gets the correct comments.  */
      xgettext_lex_comment_reset ();

      return;
    }

  /* See if we have seen this message before.  */
  mp = message_list_search (mlp, msgid);
  if (mp != NULL)
    free (msgid);
  else
    {
      static lex_pos_ty pos = { __FILE__, __LINE__ };

      /* Allocate a new message and append the message to the list.  */
      mp = message_alloc (msgid);
      /* Do not free msgid.  */
      message_list_append (mlp, mp);

      /* Construct the msgstr from the prefix and suffix, otherwise use the
	 empty string.  */
      if (msgstr_prefix)
	{
	  msgstr = (char *) xmalloc (strlen (msgstr_prefix)
				     + strlen (msgid)
				     + strlen(msgstr_suffix) + 1);
	  stpcpy (stpcpy (stpcpy (msgstr, msgstr_prefix), msgid),
		  msgstr_suffix);
	}
      else
	msgstr = "";
      message_variant_append (mp, MESSAGE_DOMAIN_DEFAULT, msgstr, &pos);
    }

  /* Ask the lexer for the comments it has seen.  Only do this for the
     first instance, otherwise there could be problems; especially if
     the same comment appears before each.  */
  /* FIXME I think we should handle multiple comments.  Doublicated
     comments could be filtered out. --drepper  */
  if ((add_all_comments || comment_tag) && !mp->comment_dot)
    {
      int j;

      for (j = 0; ; ++j)
	{
	  const char *s = xgettext_lex_comment (j);
	  if (s == NULL)
	    break;
	  if (comment_tag == NULL
	      || (comment_tag != NULL && strncmp (s, comment_tag,
						  strlen (comment_tag)) == 0))
	    message_comment_dot_append (mp, s);
	}
    }

  /* Remember where we saw this msgid.  */
  if (line_comment)
    message_comment_filepos (mp, tp->file_name, tp->line_number);

  /* Tell the lexer to reset its comment buffer, so that the next
     message gets the correct comments.  */
  xgettext_lex_comment_reset ();
}


static void
scan_c_file(filename, mlp)
     const char *filename;
     message_list_ty *mlp;
{
  int state;

  /* The file is broken into tokens.  Scan the token stream, looking for
     a keyword, followed by a left paren, followed by a string.  When we
     see this sequence, we have something to remember.  We assume we are
     looking at a valid C or C++ program, and leave the complains about
     the grammar to the compiler.  */
  xgettext_lex_open(filename);

  /* Start state is 0.  */
  state = 0;

  while (1)
   {
     xgettext_token_ty token;

     /* A simple state machine is used to do the recognising:
        State 0 = waiting for something to happen
        State 1 = seen one of our keywords with string in first parameter
        State 2 = was in state 1 and now saw a left paren
	State 3 = seen one of our keywords with string in second parameter
	State 4 = was in state 3 and now saw a left paren
	State 5 = waiting for comma after being in state 4
	State 6 = saw comma after being in state 5  */
     xgettext_lex (&token);
     switch (token.type)
       {
       case xgettext_token_type_keyword1:
	 state = 1;
	 continue;

       case xgettext_token_type_keyword2:
	 state = 3;
	 continue;

       case xgettext_token_type_lp:
	 switch (state)
	   {
	   case 1:
	     state = 2;
	     break;
	   case 3:
	     state = 4;
	     break;
	   default:
	     state = 0;
	   }
	 continue;

       case xgettext_token_type_comma:
	 state = state == 5 ? 6 : 0;
	 continue;

       case xgettext_token_type_string_literal:
	 if (state == 2 || state == 6)
	   {
	     remember_a_message (mlp, &token);
	     state = 0;
	   }
	 else
	   {
	     free (token.string);
	     state = (state == 4 || state == 5) ? 5 : 0;
	   }
	 continue;

       case xgettext_token_type_symbol:
	 state = (state == 4 || state == 5) ? 5 : 0;
	 continue;

       default:
	 state = 0;
	 continue;

       case xgettext_token_type_eof:
	 break;
       }
     break;
   }

  /* Close scanner.  */
  xgettext_lex_close ();
}


typedef struct extract_class_ty extract_class_ty;
struct extract_class_ty
{
  /* Inherited instance variables and methods.  */
  PO_BASE_TY

  /* Cumulative list of messages.  */
  message_list_ty *mlp;

  /* Cumulative comments for next message.  */
  string_list_ty *comment;
  string_list_ty *comment_dot;
  int filepos_count;
  lex_pos_ty *filepos;
};


static void
extract_constructor (that)
     po_ty *that;
{
  extract_class_ty *this = (extract_class_ty *) that;

  this->mlp = NULL; /* actually set in read_po_file, below */
  this->comment = NULL;
  this->comment_dot = NULL;
  this->filepos_count = 0;
  this->filepos = NULL;
}


static void
extract_directive_domain (that, name)
     po_ty *that;
     char *name;
{
  po_gram_error (_("this file may not contain domain directives"));
}


static void
extract_directive_message (that, msgid, msgid_pos, msgstr, msgstr_pos)
     po_ty *that;
     char *msgid;
     lex_pos_ty *msgid_pos;
     char *msgstr;
     lex_pos_ty *msgstr_pos;
{
  extract_class_ty *this = (extract_class_ty *)that;
  message_ty *mp;
  message_variant_ty *mvp;
  int j;

  /* If the msgid is the empty string, it is the old header.
     Throw it away, we have constructed a new one.  */
  if (*msgid == '\0')
    {
      free (msgid);
      free (msgstr);
      return;
    }

  /* See if this message ID has been seen before.  */
  mp = message_list_search (this->mlp, msgid);
  if (mp)
    free (msgid);
  else
    {
      mp = message_alloc (msgid);
      message_list_append (this->mlp, mp);
    }

  /* Add the accumulated comments to the message.  Clear the
     accumulation in preparation for the next message. */
  if (this->comment != NULL)
    {
      for (j = 0; j < this->comment->nitems; ++j)
	message_comment_append (mp, this->comment->item[j]);
      string_list_free (this->comment);
      this->comment = NULL;
    }
  if (this->comment_dot != NULL)
    {
      for (j = 0; j < this->comment_dot->nitems; ++j)
	message_comment_dot_append (mp, this->comment_dot->item[j]);
      string_list_free (this->comment_dot);
      this->comment_dot = NULL;
    }
  for (j = 0; j < this->filepos_count; ++j)
    {
      lex_pos_ty *pp;

      pp = &this->filepos[j];
      message_comment_filepos (mp, pp->file_name, pp->line_number);
      free (pp->file_name);
    }
  if (this->filepos != NULL)
    free (this->filepos);
  this->filepos_count = 0;
  this->filepos = NULL;

  /* See if this domain has been seen for this message ID.  */
  mvp = message_variant_search (mp, MESSAGE_DOMAIN_DEFAULT);
  if (mvp != NULL)
    {
      gram_error_with_loc (msgid_pos, _("duplicate message definition"));
      gram_error_with_loc (&mvp->pos, _("\
...this is the location of the first definition"));
      free (msgstr);
    }
  else
    message_variant_append (mp, MESSAGE_DOMAIN_DEFAULT, msgstr, msgstr_pos);
}


static void
extract_parse_brief (that)
     po_ty *that;
{
  po_lex_pass_comments (1);
}


static void
extract_comment (that, s)
     po_ty *that;
     const char *s;
{
  extract_class_ty *this = (extract_class_ty *) that;

  if (this->comment == NULL)
    this->comment = string_list_alloc ();
  string_list_append (this->comment, s);
}


static void
extract_comment_dot (that, s)
     po_ty *that;
     const char *s;
{
  extract_class_ty *this = (extract_class_ty *) that;

  if (this->comment_dot == NULL)
    this->comment_dot = string_list_alloc ();
  string_list_append (this->comment_dot, s);
}


static void
extract_comment_filepos (that, name, line)
     po_ty *that;
     const char *name;
     int line;
{
  extract_class_ty *this = (extract_class_ty *) that;
  size_t nbytes;
  lex_pos_ty *pp;

  nbytes = (this->filepos_count + 1) * sizeof (this->filepos[0]);
  this->filepos = xrealloc (this->filepos, nbytes);
  pp = &this->filepos[this->filepos_count++];
  pp->file_name = xstrdup (name);
  pp->line_number = line;
}


/* So that the one parser can be used for multiple programs, and also
   use good data hiding and encapsulation practices, an object
   oriented approach has been taken.  An object instance is allocated,
   and all actions resulting from the parse will be through
   invocations of method functions of that object.  */

static po_method_ty extract_methods =
{
  sizeof (extract_class_ty),
  extract_constructor,
  NULL, /* destructor */
  extract_directive_domain,
  extract_directive_message,
  extract_parse_brief,
  NULL, /* parse_debrief */
  extract_comment,
  extract_comment_dot,
  extract_comment_filepos,
};


/* Read the contents of the specified .po file into a message list.  */

static void
read_po_file (file_name, mlp)
     const char *file_name;
     message_list_ty *mlp;
{
  po_ty *pop = po_alloc (&extract_methods);
  ((extract_class_ty *) pop)->mlp = mlp;
  po_scan (pop, file_name);
  po_free (pop);
}



#define TM_YEAR_ORIGIN 1900

/* Yield A - B, measured in seconds.  */
static long
difftm (a, b)
     const struct tm *a;
     const struct tm *b;
{
  int ay = a->tm_year + (TM_YEAR_ORIGIN - 1);
  int by = b->tm_year + (TM_YEAR_ORIGIN - 1);
  /* Some compilers cannot handle this as a single return statement.  */
  long days = (
	       /* difference in day of year  */
	       a->tm_yday - b->tm_yday
	       /* + intervening leap days  */
	       + ((ay >> 2) - (by >> 2))
	       - (ay / 100 - by / 100)
	       + ((ay / 100 >> 2) - (by / 100 >> 2))
	       /* + difference in years * 365  */
	       + (long) (ay - by) * 365l);

  return 60l * (60l * (24l * days + (a->tm_hour - b->tm_hour))
		+ (a->tm_min - b->tm_min))
	 + (a->tm_sec - b->tm_sec);
}


static message_ty *
construct_header (option_list, file_list)
     char *option_list;
     char *file_list;
{
  time_t now;
  struct tm local_time;
  struct passwd *pwd;
  const char *full_user_name;
  char tz_sign;
  long tz_min;
  message_ty *mp;
  char *msgstr;
  static lex_pos_ty pos = { __FILE__, __LINE__, };

  mp = message_alloc("");

  time (&now);
  local_time = *localtime (&now);
  tz_sign = '+';
  tz_min = difftm (&local_time, gmtime (&now)) / 60;
  if (tz_min < 0)
    {
      tz_min = -tz_min;
      tz_sign = '-';
    }
  pwd = getpwuid (getuid ());
#if defined (__DJGPP__)
  if (pwd == NULL)
    full_user_name = "DOSUSER";
#else
  if (pwd == NULL || pwd->pw_gecos == NULL)
    full_user_name = "???";
  else
    full_user_name = pwd->pw_gecos;
#endif
  asprintf (&msgstr, "\
Date: %d-%02d-%02d %02d:%02d:%02d%c%02ld%02ld\n\
From: %s <%s>\n\
Content-Type: text/plain; charset=\n\
Xgettext-Options: %s\n\
Files: %s\n",
	    local_time.tm_year + TM_YEAR_ORIGIN,
	    local_time.tm_mon + 1,
	    local_time.tm_mday,
	    local_time.tm_hour,
	    local_time.tm_min,
	    local_time.tm_sec,
	    tz_sign,
	    tz_min / 60l,
	    tz_min % 60l,
	    full_user_name, get_submitter (NULL),
	    option_list,
	    file_list);

  if (msgstr == NULL)
    error (EXIT_FAILURE, errno, _("while preparing output"));

  message_variant_append (mp, MESSAGE_DOMAIN_DEFAULT, msgstr, &pos);

  return mp;
}
