{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add character.
    This procedure adds the given character to the
    editing row buffer and updates the screen.

*************************************************)

    Procedure Add_Character( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        If ( Data.Where.Column < Data.String_Length )
          then
            Begin
              If Insert_Mode
                then
                  Begin
                    Insert_Line( Line, Data.Where.Column, Data.Character );
                    Data.String_Length := Line.Size;
                    Increment_Block( Data.Where, Start, Finish );
                  End
                else
                  Line.Data[ Data.Where.Column ] := Data.Character;
              Data.Line_Altered := True;
            End
          else
            If Insert_At_EoLn
              then
                Begin
                  Append_Line( Line, ' ' );
                  Data.String_Length := Line.Size;
                  Line.Data[ Data.Where.Column ] := Data.Character;
                  Data.Line_Altered := True;
                End;
        If ( Automatic_Paragraph_Reform and ( Data.String_Length > Right_Margin ) and ( Data.Character = ' ' ) )
          then
            Command := Press_Automatic_Reform;
        If ( Data.Where.Column < Data.String_Length )
          then
            Inc( Data.Where.Column );
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move right.
    This procedure moves the editing row buffer
    cursor right and updates the screen.

*************************************************)

    Procedure Move_Right( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt; Amount: Byte );
      Var
        Count: Byte;
        Changed: Boolean;
      Begin
        Changed := False;
        For Count := 1 to Amount do
          If ( Data.Where.Column < Data.String_Length )
            then
              Begin
                Inc( Data.Where.Column );
                Changed := True;
              End
            else
              If Space_Right
                then
                  Begin
                    Append_Line( Line, ' ' );
                    Data.String_Length := Line.Size;
                    If ( Data.Where.Column < Data.String_Length )
                      then
                        Inc( Data.Where.Column );
                    Data.Line_Altered := True;
                    Changed := True;
                  End
                else
                  If Automatic_Return
                    then
                      Begin
                        Data.Where.Column := 1;
                        Command := Press_Down_Arrow;
                      End;
        If Changed
          then
            Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move left.
    This procedure moves the editing row buffer
    cursor left and updates the screen.

*************************************************)

    Procedure Move_Left( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt; Amount: Byte );
      Var
        Count: Byte;
        Changed: Boolean;
      Begin
        Changed := False;
        For Count := 1 to Amount do
          If ( Data.Where.Column > 1 )
            then
              Begin
                Dec( Data.Where.Column );
                Changed := True;
              End
            else
              If Automatic_Return
                then
                  Begin
                    Data.Where.Column := Line_Limit;
                    Command := Press_Up_Arrow;
                  End;
        If Changed
          then
            Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move horizontal.
    This procedure moves the editing row buffer
    cursor to the given position and updates the
    screen.

*************************************************)

    Procedure Move_Horizontal( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish, Where: LongInt );
      Begin
        Data.Where.Column := Where;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Erase character.
    This procedure deletes the character under the
    editing row buffer cursor and updates the
    screen.

*************************************************)

    Procedure Erase_Character( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        Delete_Line( Line, Data.Where.Column, 1 );
        Data.String_Length := Line.Size;
        Decrement_Block( Data.Where, Start, Finish );
        Write_Truncated( Row, Line, Data, Start, Finish );
        Command := Press_Lower_Letters;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete character.
    This procedure removes the character under the
    editing row buffer cursor and updates the
    screen.

*************************************************)

    Procedure Delete_Character( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        If ( Data.Where.Column < Data.String_Length )
          then
            Erase_Character( Line, Data, Row, Start, Finish )
          else
            Line.Data[ Data.Where.Column ] := ' ';
        Data.Line_Altered := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Arrow delete.
    This procedure removes the character to the
    left of the editing row buffer cursor and
    updates the screen.

*************************************************)

    Procedure Arrow_Delete( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        If ( Data.Where.Column > 1 )
          then
            Begin
              Dec( Data.Where.Column );
              Erase_Character( Line, Data, Row, Start, Finish );
              Data.Line_Altered := True;
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move word right.
    This procedure moves the editing row buffer
    cursor to the right-most word and updates the
    screen.

*************************************************)

    Procedure Move_Word_Right( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Begin
        If ( Data.Where.Column < Data.String_Length )
          then
            Begin
              While ( ( Data.Where.Column < Data.String_Length ) and
                      ( not ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) ) do
                Inc( Data.Where.Column );
              While ( ( Data.Where.Column < Data.String_Length ) and
                      ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) do
                Inc( Data.Where.Column );
              Write_Truncated( Row, Line, Data, Start, Finish );
            End
          else
            If Automatic_Return
              then
                Begin
                  Data.Where.Column := 1;
                  Command := Press_Down_Arrow;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move word left.
    This procedure moves the editing row buffer
    cursor to the left-most word and updates the
    screen.

*************************************************)

    Procedure Move_Word_Left( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Begin
        If ( Data.Where.Column > 1 )
          then
            Begin
              While ( ( Data.Where.Column > 1 ) and ( not ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) ) do
                Dec( Data.Where.Column );
              While ( ( Data.Where.Column > 1 ) and ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) do
                Dec( Data.Where.Column );
              While ( ( Data.Where.Column > 1 ) and ( not ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) ) do
                Dec( Data.Where.Column );
              If ( ( Data.Where.Column < Data.String_Length ) and
                   ( not ( Line.Data[ Succ( Data.Where.Column ) ] in Word_Separators ) ) )
                then
                  Inc( Data.Where.Column );
              Write_Truncated( Row, Line, Data, Start, Finish );
            End
          else
            If Automatic_Return
              then
                Begin
                  Data.Where.Column := Line_Limit;
                  Command := Press_Up_Arrow;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move delete line.
    This procedure erases the entire line, moves
    the editing row buffer cursor to the left-most
    position and updates the screen.

*************************************************)

    Procedure Move_Delete_Line( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Begin
        FillChar( Line.Data[ 1 ], Data.String_Length, ' ' );
        Data.Where.Column := 1;
        Data.Line_Altered := True;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete end.
    This procedure erases the remainder of the line
    from the editing row buffer cursor and updates
    the screen.

*************************************************)

    Procedure Delete_End( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        FillChar( Line.Data[ Data.Where.Column ], Succ( Data.String_Length - Data.Where.Column ), ' ' );
        Truncate_Block( Data.Where, Start, Finish );
        Data.Line_Altered := True;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete word right.
    This procedure erases the word to the right of
    the editing row buffer cursor and updates the
    screen.

*************************************************)

    Procedure Delete_Word_Right( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Procedure Adjust_Word;
        Begin
          Delete_Line( Line, Data.Where.Column, 1 );
          Dec( Data.String_Length );
          Decrement_Block( Data.Where, Start, Finish );
        End;
      Begin
        While ( ( Data.Where.Column < Data.String_Length ) and ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) do
          Adjust_Word;
        While ( ( Data.Where.Column < Data.String_Length ) and
                ( not ( Line.Data[ Data.Where.Column ] in Word_Separators ) ) ) do
          Adjust_Word;
        Data.Line_Altered := True;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Tab.
    This procedure moves the editing row buffer
    cursor to the right one tab space and updates
    the screen.

*************************************************)

    Procedure Tab( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Var
        Count,
        Tab_To: Word;
      Begin
        If ( Insert_Tab and Insert_Mode )
          then
            Begin
              Tab_To := Succ( Data.Where.Column div Tab_Amount ) * Tab_Amount;
              For Count := Succ( Data.Where.Column ) to Tab_To do
                Begin
                  Insert_Line( Line, Data.Where.Column, ' ' );
                  Inc( Data.Where.Column );
                  Inc( Data.String_Length );
                End;
              Data.Line_Altered := True;
            End
          else
            Begin
              Data.Where.Column := Succ( Data.Where.Column div Tab_Amount ) * Tab_Amount;
              If ( Data.Where.Column > Data.String_Length )
                then
                  Data.Where.Column := Data.String_Length;
            End;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Inverse tab.
    This procedure moves the editing row buffer
    cursor to the left one tab space and updates
    the screen.

*************************************************)

    Procedure Inverse_Tab( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Var
        Hold: LongInt;
      Begin
        Hold := ( Pred( Data.Where.Column ) Div Tab_Amount ) * Tab_Amount;
        If ( Hold > 0 )
          then
            Data.Where.Column := Hold
          else
            Data.Where.Column := 1;
        Write_Truncated( Row, Line, Data, Start, Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Mark single word.
    This procedure move the buffer selection text
    markers to the current word or the next one,
    then updates the screen.

*************************************************)

    Procedure Mark_Single_Word( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Var Start, Finish: LongInt );
      Begin
        Wave.Block_Start := Data.Where;
        If ( Line.Data[ Wave.Block_Start.Column ] in Word_Separators )
          then
            While ( ( Wave.Block_Start.Column < Data.String_Length ) and
                    ( Line.Data[ Wave.Block_Start.Column ] in Word_Separators ) ) do
              Inc( Wave.Block_Start.Column )
          else
            Begin
              While ( ( Wave.Block_Start.Column > 1 ) and
                      ( not ( Line.Data[ Wave.Block_Start.Column ] in Word_Separators ) ) ) do
                Dec( Wave.Block_Start.Column );
              If ( Wave.Block_Start.Column < Data.String_Length )
                then
                  Inc( Wave.Block_Start.Column );
            End;
        Wave.Block_Finish := Wave.Block_Start;
        While ( ( Wave.Block_Finish.Column < Data.String_Length ) and
                ( not ( Line.Data[ Wave.Block_Finish.Column ] in Word_Separators ) ) ) do
          Inc( Wave.Block_Finish.Column );
        If ( Wave.Block_Start.Column = 2 ) and ( not ( Line.Data[ 1 ] in Word_Separators ) )
          then
            Wave.Block_Start.Column := 1;
        Start := Wave.Block_Start.Column;
        Finish := Wave.Block_Finish.Column;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust value.
    This procedure adjusts the column value to fit
    in the current screen area.

*************************************************)

    Procedure Adjust_Value( Wave: Point_Type; Row: LongInt; Var Value: LongInt );
      Begin
       If ( Wave.Row < Row )
         then
           Value := 0
         else
           If ( Wave.Row = Row )
             then
               Value := Wave.Column
             else
               Value := Line_Limit;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Center the text.
    This procedure attempts to center the current
    text line using the current margin settings.

*************************************************)

    Procedure Center_The_Text( Var Line: Line_Type; Var Data: Cluster_Type; Row: Byte; Start, Finish: LongInt );
      Var
        Count,
        Width: Word;
      Begin
        While ( Line.Data[ 1 ] = ' ' ) and ( Line.Size > 1 ) do
          Delete_Line( Line, 1, 1 );
        While ( Line.Data[ Line.Size ] = ' ' ) and ( Line.Size > 1 ) do
          Dec( Line.Size );
        Width := Abs( Right_Margin - Left_Margin );
        If ( Line.Size < Width )
          then
            Begin
              For Count := 1 to ( ( Width - Line.Size ) div 2 ) do
                Insert_Line( Line, 1, ' ' );
              Write_Truncated( Row, Line, Data, Start, Finish );
              Data.Line_Altered := True;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Ripple;
    This routine is responsible for taking care of
    all the screen updating during the editing
    session.  Linked into the KeyBoard wait
    routine, this simple procedure is called
    repeatedly while the system is waiting for
    input.  Each time it is called, and if the
    screen needs it, the routine updates only one
    line of text on the screen in a rippling
    fashion from the line closest to the working
    line until it reaches the end.

*************************************************)

   {$IFDEF VER60}
    Procedure Ripple; Far;
   {$ELSE}
    {$F+}
    Procedure Ripple;
   {$ENDIF}
      Var
        Start,
        Finish: LongInt;
      Begin
        If ( Wave.Screen_Up > 0 )
          then
            Begin
              Get_Text_Line( Wave.Buffer^, Wave.Ripple_Up, Work_Space^ );
              Adjust_Value( Wave.Block_Start, Wave.Ripple_Up, Start );
              Adjust_Value( Wave.Block_Finish, Wave.Ripple_Up, Finish );
              Display_Truncated( Wave.Screen_Up, Work_Space^, Start, Finish );
              Dec( Wave.Ripple_Up );
              Dec( Wave.Screen_Up );
              Rippling := True;
            End
          else
            If ( Wave.Screen_Down <= Wave.Window_Bottom )
              then
                Begin
                  Get_Text_Line( Wave.Buffer^, Wave.Ripple_Down, Work_Space^ );
                  Adjust_Value( Wave.Block_Start, Wave.Ripple_Down, Start );
                  Adjust_Value( Wave.Block_Finish, Wave.Ripple_Down, Finish );
                  Display_Truncated( Wave.Screen_Down, Work_Space^, Start, Finish );
                  Inc( Wave.Ripple_Down );
                  Inc( Wave.Screen_Down );
                  Rippling := True;
                End
              else
                Begin
                  Rippling := False;
                  Hold_Routine;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Finish screen.
    This procedure is responsible for allowing
    the screen to finish updating before it may
    leave.  This is important before allowing
    another routine which uses KeyBoard to take
    control because the other routines screen data
    may be unintentionally overwritten by the
    rippler.

*************************************************)

    Procedure Finish_Screen;
      Begin
        While Rippling do
          Ripple;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process function keys.
    This procedure is allows the function keys to
    be given special functions.

*************************************************)

    Procedure Process_Function_Keys( Var Command: Byte; Var Character: Char );
      Begin
        Rippling := True;
        Finish_Screen;
        Function_Key( Command, Character );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Edit row.
    This is a single row editing buffer routine
    that makes it possible to edit a single line
    of text.  In the text editor, this routine
    holds most of the control for most of the time
    until a command is intercepted which it can
    not handle.  When this happens, control is
    transfered back to the calling routine where
    the specific task is performed.  If the line
    was not altered, this routine returns false.

*************************************************)

    Function Edit_Row( Row: Byte; Var Line: Line_Type; Var Cursor: Point_Type; Var Start, Finish: LongInt ): Boolean;
      Const
        Remain_Values: Set of Byte = [ Press_Numbers, Press_Capital_Letters, Press_Lower_Letters, Press_Extra_Characters,
                                       Press_Right_Arrow, Press_Left_Arrow, Press_Home, Press_End, Press_Insert,
                                       Press_Word_Right, Press_Control_Right_Arrow, Press_Word_Left, Press_Control_Left_Arrow,
                                       Press_Delete_End, Press_Control_T, Press_Tab, Press_Shift_Tab, Press_Left_Margin,
                                       Press_Right_Margin, Press_Center_Text, Press_Other_Characters,
                                       Pointer_Left, Pointer_Right ];
      Var
        Data: Cluster_Type;
      Begin
        Cursor_Row := Row;
        Append_Line( Line, ' ' );
        Data.Start := Start;
        Data.Finish := Finish;
        Data.String_Length := Line.Size;
        Data.Where := Cursor;
        Data.Line_Altered := False;
        If ( Data.Where.Column > Data.String_Length )
          then
            Begin
              Data.Where.Column := Data.String_Length;
              Data.Start := 1;
              Data.Finish := ( Right_Of_Window^ - Left_Of_Window^ );
            End;
        Adjust_Value( Wave.Block_Start, Wave.Where, Start );
        Adjust_Value( Wave.Block_Finish, Wave.Where, Finish );
        Write_Truncated( Row, Line, Data, Start, Finish );
        Repeat
          Wave.Start := Data.Start;
          Wave.Finish := Data.Finish;
          Get_Command( Character, Command );
          If ( Command in Function_Keys )
            then
              Process_Function_Keys( Command, Character );
          Data.Character := Character;
          Case Command of
            Pointer_Right:
              Move_Right( Line, Data, Row, Start, Finish, Adjust_Amount );
            Pointer_Left:
              Move_Left( Line, Data, Row, Start, Finish, Adjust_Amount );
            Press_Numbers,
            Press_Capital_Letters,
            Press_Lower_Letters,
            Press_Other_Characters,
            Press_Extra_Characters:
              Add_Character( Line, Data, Row, Start, Finish );
            Press_Right_Arrow:
              Move_Right( Line, Data, Row, Start, Finish, 1 );
            Press_Left_Arrow:
              Move_Left( Line, Data, Row, Start, Finish, 1 );
            Press_Home:
              Move_Horizontal( Line, Data, Row, Start, Finish, 1 );
            Press_End:
              Move_Horizontal( Line, Data, Row, Start, Finish, Data.String_Length );
            Press_Insert:
              Insert_Mode := ( not Insert_Mode );
            Press_Delete:
              Delete_Character( Line, Data, Row, Start, Finish );
            Press_Delete_Arrow:
              Arrow_Delete( Line, Data, Row, Start, Finish );
            Press_Word_Right,
            Press_Control_Right_Arrow:
              Move_Word_Right( Line, Data, Row, Start, Finish );
            Press_Word_Left,
            Press_Control_Left_Arrow:
              Move_Word_Left( Line, Data, Row, Start, Finish );
            Press_Delete_Line:
              Move_Delete_Line( Line, Data, Row, Start, Finish );
            Press_Delete_End:
              Delete_End( Line, Data, Row, Start, Finish );
            Press_Control_T:
              Delete_Word_Right( Line, Data, Row, Start, Finish );
            Press_Tab:
              Tab( Line, Data, Row, Start, Finish );
            Press_Shift_Tab:
              Inverse_Tab( Line, Data, Row, Start, Finish );
            Press_Left_Margin:
              Left_Margin := Data.Where.Column;
            Press_Right_Margin:
              Right_Margin := Data.Where.Column;
            Press_Center_Text:
              Center_The_Text( Line, Data, Row, Start, Finish );
            Press_Control_K:
              If ( Character = 'T' )
                then
                  Mark_Single_Word( Line, Data, Row, Start, Finish );
            Pointer_Button1_Double:
              Mark_Single_Word( Line, Data, Row, Start, Finish );
            Press_Control_P:
              If Character_Menu( Data.Character )
                then
                  Add_Character( Line, Data, Row, Start, Finish );
          End; { Case }
        Until not ( Command in Remain_Values );
        Cursor := Data.Where;
        Start := Data.Start;
        Finish := Data.Finish;
        Edit_Row := Data.Line_Altered;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Change.
    This function takes the given info in search
    data type and performs the change to the line.

*************************************************)

    Function Change( Var Text: Text_Type; Var Info: Search_Data_Type ): Boolean;
      Begin
        Get_Text_Line( Text, Info.Find_Position.Row, Working_Buffer^ );
        Change_Line( Working_Buffer^, Info.Find_Position.Column, Length( Info.Find_String ), Info.Replace_String );
        Change := Put_Text_Line( Text, Info.Find_Position.Row, Working_Buffer^, True )
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set start.
    This procedure sets up the search starting
    position.

*************************************************)

    Procedure Set_Start( Var Start_Position, Cursor: Point_Type );
      Begin
        If Start_Search_At_Cursor
          then
            Start_Position := Cursor
          else
            If Search_Selected_Text
              then
                Start_Position := Wave.Block_Start
              else
                Begin
                  Start_Position.Row := 1;
                  Start_Position.Column := 0;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Inside the block?
    This function tells if the given position is
    inside the given block or not.

*************************************************)

    Function Inside_Block( Start, Finish, Position: Point_Type ): Boolean;
      Var
        Okay1,
        Okay2: Boolean;
      Begin
        Okay1 := ( Position.Row > Start.Row ) or ( ( Position.Row = Start.Row ) and ( Position.Column >= Start.Row ) );
        Okay2 := ( Position.Row < Finish.Row ) or ( ( Position.Row = Finish.Row ) and ( Position.Column < Finish.Column ) );
        Inside_Block := Okay1 and Okay2;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Search core.
    This function searches Text for the given info
    If it's found, then the screen is displayed
    and the system waits for the next keystroke.

*************************************************)

    Function Search_Core( Var All: All_Type; Var Info: Search_Data_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Write_Wait;
        Okay := Scan_Text( All.Text, Info.Find_Position, Info.Find_String );
        If ( Okay and Search_Selected_Text )
          then
            Okay := Inside_Block( Wave.Block_Start, Wave.Block_Finish, Info.Find_Position );
        Write_Complete;
        If Okay
          then
            Begin
              Draw_Screen( All.Text, Info.Find_Position, ( Wave.Window_Bottom div 2 ), Length( Info.Find_String ),
                           Wave.Window_Bottom );
              Repeat
              Until Data_Ready;
              All.Data.Cursor := Info.Find_Position;
              All.Data.Screen_Row := ( Wave.Window_Bottom div 2 );
            End;
        Search_Core := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search engine.
    This procedure searches for the given data.
    If it's not found, it writes an error message.

*************************************************)

    Procedure Search_Engine( Var All: All_Type; Var Info: Search_Data_Type );
      Var
        Temporary_Holder: Procedure;
      Begin
        If not Search_Core( All, Info )
          then
            Begin
              Temporary_Holder := KeyBoard.Wait_Routine;
              KeyBoard.Wait_Routine := Procedure_Default;
              Write_Search_Failure;
              KeyBoard.Wait_Routine := Temporary_Holder;
            End;
        Info.Last_Operation := Find;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Replace core.
    This function performs the find and replace
    routines.  If the data is found and if it's
    permitted, it displays the screen.

*************************************************)

    Function Replace_Core( Var All: All_Type; Var Info: Search_Data_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Write_Wait;
        Okay := Scan_Text( All.Text, Info.Find_Position, Info.Find_String );
        If ( Okay and Search_Selected_Text )
          then
            Okay := Inside_Block( Wave.Block_Start, Wave.Block_Finish, Info.Find_Position );
        Write_Complete;
        If ( Okay and Search_Show_Line )
          then
            Draw_Screen( All.Text, Info.Find_Position, ( Wave.Window_Bottom div 2 ), Length( Info.Find_String ),
                         Wave.Window_Bottom );
        Replace_Core := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Background search.
    During a normal search and replace operation,
    the system may sometimes have to stop and wait
    for the user to decide what action to take.
    In order to speed up the searches, this
    routine takes control when permitted and
    continues the search bit by bit in the back-
    ground.

*************************************************)

   {$IFDEF VER60}
    Procedure Background_Search; Far;
   {$ELSE}
    {$F+}
    Procedure BackGround_Search;
   {$ENDIF}
      Var
        Okay: Boolean;
      Begin
        If BackGround_Searching
          then
            Begin
              If ( BackGround_Row <= BackGround_Text^.FileSize )
                then
                  Okay := Look_Text( BackGround_Text^, BackGround_Info.Find_Position, BackGround_Info.Find_String,
                                     BackGround_Row )
                else
                  Okay := True;
              BackGround_Searching := ( not Okay );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Prompt core.
    This procedure prompts the user of input and
    changes the data when instructed to do so.

*************************************************)

    Procedure Prompt_Core( Var All: All_Type; Var Info: Search_Data_Type; Var Changes: Word; Var Done,
                           BackGround_Find: Boolean );
      Begin
        If Search_Prompt_On_Replace
          then
            Begin
              BackGround_Searching := ( Allow_BackGround_Searching and Search_Change_All );
              If Background_Searching
                then
                  Begin
                    BackGround_Info := Info;
                    BackGround_Info.Find_Position.Column := Info.Find_Position.Column + Length( Info.Find_String );
                    BackGround_Text := @All.Text;
                    BackGround_Row := Info.Find_Position.Row;
                  End;
              Case Get_Confirmation_of_Change of
                'Y': If Change( All.Text, Info )
                       then
                         Inc( Changes )
                       else
                         Write_Error( 400 );
                'E': Done := True;
              End; { Case }
              If ( Allow_Background_Searching and Search_Change_All )
                then
                  Begin
                    BackGround_Searching := False;
                    If ( BackGround_Info.Find_Position.Row <> Info.Find_Position.Row )
                      then
                        Begin
                          BackGround_Find := True;
                          Dec( BackGround_Info.Find_Position.Column );
                        End;
                  End;
            End
          else
            If Change( All.Text, Info )
              then
                Inc( Changes )
              else
                Write_Error( 400 );
        All.Data.Cursor := Info.Find_Position;
        All.Data.Screen_Row := ( Wave.Window_Bottom div 2 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Replace engine.
    This procedure handles the bulk of the search
    and replace functions.  It prompts for user
    input occasionally and handles most of the
    text searching and replacing.

*************************************************)

    Procedure Replace_Engine( Var All: All_Type; Var Info: Search_Data_Type; Var Changed: Boolean );
      Var
        Done,
        Show_Done,
        BackGround_Find: Boolean;
        Changes: Word;
        Temporary_Holder: Procedure;
      Begin
        Show_Done := False;
        Changes := 0;
        If ( Allow_BackGround_Searching and Search_Prompt_On_Replace and Search_Change_All )
          then
            Begin
              Temporary_Holder := KeyBoard.Wait_Routine;
              KeyBoard.Wait_Routine := BackGround_Search;
            End;
        If ( Search_Change_All and ( not Search_Prompt_on_Replace) )
          then
            Search_Show_Line := False
          else
            Search_Show_Line := True;
        Done := not Search_Change_All;
        Repeat
          BackGround_Find := False;
          If Replace_Core( All, Info )
            then
              Prompt_Core( All, Info, Changes, Done, BackGround_Find )
            else
              Begin
                Done := True;
                If ( Changes = 0 )
                  then
                    Show_Done := True
                  else
                    Begin
                      Draw_Screen( All.Text, All.Data.Cursor, All.Data.Screen_Row, Length( Info.Find_String ),
                                   Wave.Window_Bottom );
                      Confirm_Changes( Changes );
                      Changed := True;
                    End;
              End;
          If not Done
            then
              If BackGround_Find
                then
                  Info.Find_Position := BackGround_Info.Find_Position
                else
                  Info.Find_Position.Column := ( Info.Find_Position.Column + Length( Info.Replace_String ) );
        Until Done;
        Info.Last_Operation := Replace;
        If ( Allow_BackGround_Searching and Search_Prompt_On_Replace and Search_Change_All )
          then
            KeyBoard.Wait_Routine := Temporary_Holder;
        If Show_Done
          then
            Begin
              Temporary_Holder := KeyBoard.Wait_Routine;
              KeyBoard.Wait_Routine := Procedure_Default;
              Write_Search_Failure;
              KeyBoard.Wait_Routine := Temporary_Holder;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Scan and reduce the line.
    This function is designed to remove all the
    multiple spaces on the given line, except for
    those at the beginning, defining the paragraph
    tab.  If the line is invalid, or nothing was
    done, this function should return false.

*************************************************)

    Function Scan_Reduce_Line( Var Line: Line_Type; Var Line_Point: Word; Point: Word ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := False;
       { First skip the initial tab space. }
        While ( ( Point <= Line.Size ) and ( Line.Data[ Point ] = ' ' ) ) do
          Inc( Point );
       { If we haven't reached the end of the line then start the process. }
        If ( Point <= Line.Size )
          then
            Begin
              Repeat
               { Here we skip the word. }
                While ( ( Point <= Line.Size ) and ( Line.Data[ Point ] <> ' ' ) ) do
                  Inc( Point );
               { Here we skip an extra space for punctuation. }
                If ( Point > 1 ) and ( Point <= Line.Size ) and ( Line.Data[ Pred( Point ) ] in Punctuation )
                  then
                    Inc( Point );
               { Here we skip a space. }
                If ( ( Point <= Line.Size ) and ( Line.Data[ Point ] = ' ' ) )
                  then
                    Inc( Point );
               { And here we delete the extra spaces. }
                While ( ( Point < Line.Size ) and ( Line.Data[ Point ] = ' ' ) ) do
                  Begin
                    Delete_Line( Line, Point, 1 );
                    If ( Line_Point > Point )
                      then
                        Dec( Line_Point );
                    Okay := True;
                  End;
              Until ( Point > Line.Size );
            End;
        Scan_Reduce_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Shift word forward.
    This procedure splits the text line by moving
    the last word on the line forward to the next
    line.

*************************************************)

    Procedure Shift_Word_Forward( Var Line, Line2: Line_Type );
      Begin
        If ( Line.Size > 1 )
          then
            Begin
              Repeat
                Insert_Line( Line2, 1, Line.Data[ Line.Size ] );
                Dec( Line.Size );
              Until ( ( Line.Size = 0 ) or ( Line.Data[ Line.Size ] in Split_Points ) );
            End;
      End;
{----------------------------------------------------------------------------}

(*************************************************

  Function: Try backwards.
    This function checks to see if the first word
    on the next line can be moved to the end of
    the current line.  If it can, it does the
    operation.  If it succeeds, it returns true.

*************************************************)

    Function Try_Backwards( Var Line, Line2: Line_Type ): Boolean;
      Var
        Look,
        Count: Word;
      Begin
        Try_Backwards := False;
        Look := 1;
        While ( ( Look < Line2.Size ) and ( not ( Line2.Data[ Look ] in Split_Points ) ) ) do
          Inc( Look );
        If ( ( Line.Size + Look ) <= Right_Margin )
          then
            Begin
              For Count := 1 to Look do
                Begin
                  Append_Line( Line, Line2.Data[ 1 ] );
                  Delete_Line( Line2, 1, 1 );
                End;
              Try_Backwards := True;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Adjust line.
    This function adjusts the size of the first
    line to the margin.  If the first line is
    too small, it adds words from the second line
    until it has nothing more to add, in which
    case it returns true.  If the first line is
    larger than the margin, the extra words are
    added to the beginning of the second line.

*************************************************)

    Function Adjust_Line( Var Line, Line2: Line_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        If ( Line.Size > Succ( Right_Margin ) )
          then
            While ( Line.Size > Succ( Right_Margin ) ) do
              Shift_Word_Forward( Line, Line2 )
          else
            Repeat
              Okay := Try_Backwards( Line, Line2 );
            Until ( not Okay ) or ( Line2.Size = 1 );
        Adjust_Line := ( Line2.Size = 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find split point.
    This function tries to find the optimal
    location to split the line from the right
    margin.  If there is only one word on that
    line, then the result is moved to the end of
    the word.

*************************************************)

    Function Find_Split_Point( Var Line: Line_Type ): Word;
      Var
        Count: Word;
      Begin
        Count := Right_Margin;
        While ( ( Count > 1 ) and ( not ( Line.Data[ Count ] in Split_Points ) ) ) do
          Dec( Count );
        If ( Count > 1 )
          then
            Find_Split_Point := Succ( Count )
          else
            Begin
              While ( ( Count < Line.Size ) and ( not ( Line.Data[ Count ] in Split_Points ) ) ) do
                Inc( Count );
              Find_Split_Point := Succ( Count );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reform paragraph.
    This procedure attempts to rearrange the words
    in a paragraph to the margins.  Since it is
    working only with pure ASCII, the only way it
    can decide what makes up a paragraph is by
    using default assumptions.  It assumes that
    the first, and only the first line will have
    spaces preceding it.  After that, all the
    succeeding lines that start at the beginning
    of the line, { A character in column 1 } are
    considered part of the paragraph.

*************************************************)

    Procedure Reform_Paragraph( Var All: All_Type; Var Line, Line2: Line_Type; Var Changed: Boolean );
      Var
        Start,
        Finish,
        Index1,
        Index2: LongInt;
        Extra: Word;
        Point: Point_Type;
        Redo,
        Okay: Boolean;
      Begin
       { Exit if the margin is too small. }
        If ( Right_Margin >= 20 )
          then
            Begin
             { First we remove the extra space. }
              Start := All.Data.Cursor.Row;
              Finish := Start;
              Get_Text_Line( All.Text, Finish, Line );
              If ( Line.Size > 0 )
                then
                  Begin
                    Okay := True;
                    Repeat
                      Append_Line( Line, ' ' );
                      Extra := 1;
                      If Scan_Reduce_Line( Line, Extra, 1 )
                        then
                          Okay := Put_Text_Line( All.Text, Finish, Line, True );
                      Inc( Finish );
                      Get_Text_Line( All.Text, Finish, Line );
                    Until ( ( not Okay ) or ( Line.Size = 0 ) or ( Line.Data[ 1 ] = ' ' ) );
                    Dec( Finish );
                   { Now we try to restructure the lines. }
                    Index1 := Start;
                    Index2 := Succ( Start );
                    While ( Index2 <= Finish ) do
                      Begin
                        Get_Text_Line( All.Text, Index1, Line );
                        If ( Line.Size > 0 )
                          then
                            Append_Line( Line, ' ' );
                        Get_Text_Line( All.Text, Index2, Line2 );
                        If ( Line2.Size > 0 )
                          then
                            Append_Line( Line2, ' ' );
                        Redo := Adjust_Line( Line, Line2 );
                        Okay := Put_Text_Line( All.Text, Index1, Line, True );
                        Okay := Put_Text_Line( All.Text, Index2, Line2, True );
                        If Redo
                          then
                            Begin
                              Okay := Delete_Text_Line( All.Text, Index2 );
                              If ( Wave.Block_Finish.Row >= Finish )
                                then
                                  Dec( Wave.Block_Finish.Row );
                              If ( Wave.Block_Start.Row >= Finish )
                                then
                                  Dec( Wave.Block_Start.Row );
                              Dec( Finish );
                            End
                          else
                            Begin
                              Inc( Index1 );
                              Inc( Index2 );
                            End;
                      End;
                   { Here we have to add more lines when necessary. }
                    If ( Start = Finish )
                      then
                        Get_Text_Line( All.Text, Finish, Line2 );
                    While ( Line2.Size > Succ( Right_Margin ) ) do
                      Begin
                        Point.Row := Finish;
                        Point.Column := Find_Split_Point( Line2 );
                        Okay := Split_Text( All.Text, Line2, Point, True );
                        If ( Wave.Block_Finish.Row >= Finish )
                          then
                            Inc( Wave.Block_Finish.Row );
                        If ( Wave.Block_Start.Row >= Finish )
                          then
                            Inc( Wave.Block_Start.Row );
                        Inc( Finish );
                        Get_Text_Line( All.Text, Finish, Line2 );
                        If ( not Okay )
                          then
                            Exit;
                      End;
                  End;
             { Here we update the cursor to the new line. }
              All.Data.Cursor.Row := Succ( Finish );
              All.Data.Cursor.Column := 1;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reform block.
    This procedure is called to reform all the
    paragraphs that begin in the given block.

*************************************************)

    Procedure Reform_Block( Var All: All_Type; Var Line, Line2: Line_Type; Var Changed: Boolean );
      Var
        Count,
        Limit: LongInt;
        Continue: Boolean;
      Begin
       { Exit if the margin is too small. }
        If ( ( Right_Margin >= 20 ) and ( Wave.Block_Start.Row <= Wave.Block_Finish.Row ) )
          then
            Begin
              Write_Wait;
              All.Data.Cursor := Wave.Block_Start;
              Continue := True;
              Repeat
                Reform_Paragraph( All, Line, Line2, Changed );
                Limit := Succ( Wave.Block_Finish.Row - Wave.Block_Start.Row );
                Count := Succ( All.Data.Cursor.Row - Wave.Block_Start.Row );
                Process_Status( Count, Limit, Continue );
              Until ( ( All.Data.Cursor.Row >= Wave.Block_Finish.Row ) or ( not Continue ) );
              Write_Complete;
              Changed := True;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reform text.
    This procedure is called to reform all the
    paragraphs in the text file.

*************************************************)

    Procedure Reform_Text( Var All: All_Type; Var Line, Line2: Line_Type; Var Changed: Boolean );
      Var
        Count,
        Limit: LongInt;
        Continue: Boolean;
      Begin
       { Exit if the margin is too small. }
        If ( Right_Margin >= 20 )
          then
            Begin
              Write_Wait;
              All.Data.Cursor.Row := 1;
              All.Data.Cursor.Column := 1;
              Continue := True;
              Repeat
                Reform_Paragraph( All, Line, Line2, Changed );
                Count := All.Data.Cursor.Row;
                Limit := Get_Text_Size( All.Text );
                Process_Status( Count, Limit, Continue );
              Until ( ( Count >= Limit ) or ( not Continue ) );
              Write_Complete;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find true size.
    This function returns the true size of the
    current line.  That is the size of the text
    excluding any appended blank spaces.

*************************************************)

    Function Find_True_Size( Var Line: Line_Type ): Word;
      Var
        Count: Word;
      Begin
        Count := Line.Size;
        While ( Count > 0 ) and ( Line.Data[ Count ] = ' ' ) do
          Dec( Count );
        Find_True_Size := Count;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Automatic reform.
    This procedure is called when the editor is
    functioning in the autoform mode.  It is
    designed to reformat the current line and move
    all the remaining text into the next line.
    This function is only to be called when the
    cursor inserts a space character.

*************************************************)

    Procedure Automatic_Reform( Var All: All_Type; Var Line, Line2: Line_Type; Var Changed: Boolean );
      Var
        Okay: Boolean;
        Adjustment,
        Temporary_Cursor: Point_Type;
      Begin
       { Exit if the margin is too small. }
        If ( Right_Margin < 20 )
          then
            Exit;
        Temporary_Cursor := All.Data.Cursor;
        Okay := Scan_Reduce_Line( Line, All.Data.Cursor.Column, All.Data.Cursor.Column );
        If ( All.Data.Cursor.Column >= Line.Size )
          then
            Begin
              Reduce_Line( Line );
              Adjustment.Row := All.Data.Cursor.Row;
              Adjustment.Column := Find_Split_Point( Line );
              Temporary_Cursor.Row := Succ( All.Data.Cursor.Row );
              Okay := Put_Text_Line( All.Text, All.Data.Cursor.Row, Line, True );
              If ( Adjustment.Column >= All.Data.Cursor.Column ) or ( Line.Size = Right_Margin )
                then
                  Temporary_Cursor.Column := 1
                else
                  Begin
                    Temporary_Cursor.Column := Succ( All.Data.Cursor.Column - Adjustment.Column );
                    Okay := Split_Text( All.Text, Line, Adjustment, True );
                  End;
              All.Data.Cursor := Temporary_Cursor;
              If ( All.Data.Screen_Row < Wave.Window_Bottom )
                then
                  Inc( All.Data.Screen_Row );
              Get_Text_Line( All.Text, Temporary_Cursor.Row, Line );
              Append_Line( Line, ' ' );
              Okay := Put_Text_Line( All.Text, Temporary_Cursor.Row, Line, False );
            End
          else
            Begin
              Okay := Put_Text_Line( All.Text, All.Data.Cursor.Row, Line, False );
              If ( All.Data.Cursor.Column < Right_Margin )
                then
                  Begin
                    Adjustment.Row := All.Data.Cursor.Row;
                    Adjustment.Column := Find_Split_Point( Line );
                  End
                else
                  Adjustment := All.Data.Cursor;
              If ( Adjustment.Column < Find_True_Size( Line ) )
                then
                  Begin
                    While ( Adjustment.Column < Line.Size ) and ( Line.Data[ Adjustment.Column ] = ' ' ) do
                      Inc( Adjustment.Column );
                    Okay := Split_Text( All.Text, Line, Adjustment, False );
                  End;
              Temporary_Cursor := All.Data.Cursor;
              Inc( All.Data.Cursor.Row );
              Reform_Paragraph( All, Line, Line2, Changed );
              All.Data.Cursor := Temporary_Cursor;
            End;
      End;

{----------------------------------------------------------------------------}
