//-------------------------------------------------------------------------
// Copyright (c) 1993 by Lawrence W. Salomon, Jr.  All rights reserved.
//
// This file may not be modified in any form without the expressed,
// written consent of the author.  See also the licensing agreement
// included with this package for more information.
//-------------------------------------------------------------------------
#ifndef COMMONLIB_INCLUDED
#define COMMONLIB_INCLUDED

#ifndef OS2_INCLUDED
#error You must include <os2.h> prior to including <common.h>
#endif

//-------------------------------------------------------------------------
// By defining the following symbols, the corresponding functions are
// defined:
//
// INCL_CMNBUF  -       Buffer routines
// INCL_CMNCOM  -       Communication routines
// INCL_CMNDBG  -       Debugging routines
// INCL_CMNFIL  -       File routines
// INCL_CMNLST  -       Linked list routines
// INCL_CMNMEM  -       Memory routines
// INCL_CMNOBJ  -       Object routines
// INCL_CMNSET  -       Set routines
// INCL_CMNSIG  -       Signal routines
// INCL_CMNSPR  -       Sprite routines
// INCL_CMNSTR  -       String routines
// INCL_CMNVIO  -       VIO routines
// INCL_CMNWIN  -       PM routines
//-------------------------------------------------------------------------
#ifdef INCL_COMMONALL
#define INCL_CMNBUF
#define INCL_CMNCOM
#define INCL_CMNDBG
#define INCL_CMNFIL
#define INCL_CMNLST
#define INCL_CMNMEM
#define INCL_CMNOBJ
#define INCL_CMNSET
#define INCL_CMNSIG
#define INCL_CMNSPR
#define INCL_CMNSTR
#define INCL_CMNVIO
#define INCL_CMNWIN
#endif                           // #ifdef INCL_COMMONALL

#include <cmndefs.h>

#pragma pack(1)

#ifdef __cplusplus
extern "C" {
#endif                           // #ifdef __cplusplus

USHORT EXPENTRY CmnQueryHandle(LHANDLE lhHandle);
//-------------------------------------------------------------------------
// This function returns the type of the specified handle
//
// Input:  lhHandle - specifies the handle to query
// Returns:  QH_* constant
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnQueryHandleInfo(LHANDLE lhHandle,PVOID pvInfo);
//-------------------------------------------------------------------------
// This function returns information about the handle specified.
//
// Input:  lhHandle - specifies the handle to query
// Output:  pvInfo - points to the buffer which received the result.  The
//                   size of this buffer depends on the handle type.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetHandleInfo(LHANDLE lhHandle,PVOID pvInfo);
//-------------------------------------------------------------------------
// This function sets certain attributes about the handle specified.
//
// Input:  lhHandle - specifies the handle to query
//         pvInfo - points to the buffer which received the result.  The
//                  size and type of this buffer depends on the handle type.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

VOID EXPENTRY CmnQueryVersion(PUSHORT pusMajor,
                              PUSHORT pusMinor,
                              PUSHORT pusRev);
//-------------------------------------------------------------------------
// This function returns the major, minor, and revision numbers.
//
// Output:  pusMajor - points to the variable that received the major
//                     version number
//          pusMinor - points to the variable that received the minor
//                     version number
//          pusRev - points to the variable that received the revision
//                   version number
//-------------------------------------------------------------------------

#ifdef INCL_CMNBUF
//-------------------------------------------------------------------------
// Buffer routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnBufCompressBytes(PBYTE pbInBuf,
                                  ULONG ulSzInBuf,
                                  PBYTE pbOutBuf,
                                  PULONG pulSzOutBuf);
//-------------------------------------------------------------------------
// This function uses a run-length encoding algorithm to attempt to
// compress the contents of a specified buffer.
//
// Input:   pbInBuf - points to the buffer to compress
//          ulSzInBuf - specifies the size of the buffer pointed to by
//                      pbInBuf
//          pbOutBuf - points to the buffer to hold the result.  If NULL,
//                     no results are written.
// Output:  pulSzOutBuf - points to the variable containing the size of
//                        the compressed buffer.
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnBufDecompressBytes(PBYTE pbInBuf,
                                    ULONG ulSzInBuf,
                                    PBYTE pbOutBuf,
                                    PULONG pulSzOutBuf);
//-------------------------------------------------------------------------
// This function uses a run-length encoding algorithm to decompress
// the contents of a specified buffer that was compressed using
// CmnBufCompressBytes().
//
// Input:   pbInBuf - points to the buffer to decompress
//          ulSzInBuf - specifies the size of the buffer pointed to by
//                      pbInBuf
//          pbOutBuf - points to the buffer to hold the result.  If NULL,
//                     no results are written.
// Output:  pulSzOutBuf - points to the variable containing the size of
//                        the decompressed buffer.
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnBufEncryptBytes(PBYTE pbInBuf,ULONG ulSzBuf,PBYTE pbOutBuf);
//-------------------------------------------------------------------------
// This function encrypts the contents of a buffer using a not-too-secure
// algorithm based on the XOR operation.  The resulting buffer is always
// of the same size as the input.
//
// Input:  pbInBuf - points to the buffer to be encrypted
//         ulSzBuf - specifies the size of the buffer pointed to by pbInBuf
//         pbOutBuf - points to the buffer to hold the encrypted result
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnBufDecryptBytes(PBYTE pbInBuf,ULONG ulSzBuf,PBYTE pbOutBuf);
//-------------------------------------------------------------------------
// This function decrypts the contents of a buffer encrypted using
// CmnBufEncryptBytes().  The resulting buffer is always of the same size
// as the input.
//
// Input:  pbInBuf - points to the buffer to be decrypted
//         ulSzBuf - specifies the size of the buffer pointed to by pbInBuf
//         pbOutBuf - points to the buffer to hold the decrypted result
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNBUF

#ifdef INCL_CMNCOM
//-------------------------------------------------------------------------
// Communication routines
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComOpenConnection(PCCOPENINFO pcoiInfo,
                                       PHCCCONNECT phccConnect);
//-------------------------------------------------------------------------
// This function opens a connection with the desired attributes and
// returns the connection handle.
//
// Input:  pcoiInfo - points to the CCOPENINFO structure describing the
//                    type of connection desired
// Output:  phccConnect - points to the variable which receives the result
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComCloseConnection(PHCCCONNECT phccConnect);
//-------------------------------------------------------------------------
// This function closes the specified connection and destroys the connection
// handle.
//
// Input:  phccConnect - points to the variable containing the handle to
//                       destroy.  On exit, this variable contains NULL.
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComWaitConnection(HCCCONNECT hccServer,
                                       PHCCCONNECT phccConnect);
//-------------------------------------------------------------------------
// For server connections, this function waits until a client connects to
// the server connection, then creates a new connection for private
// communication to the client.  This leaves the server connection free
// to service other requests.  If a client connection is specified, this
// function returns COM_ERR_NOTSERVER.
//
// Input:  hccServer - handle to the server connection
// Output:  phccConnect - points to the variable which receives the new
//                        connection once a client connects to the
//                        connection specified by hccServer
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComSendControl(HCCCONNECT hccConnect,ULONG ulCtrlCode);
//-------------------------------------------------------------------------
// This function sends a control packet to the other end of the connection.
// The semantics of the various control codes are as follows:
//
//    CSC_CTRL_CLOSE - indicates that the connection should be closed.
//    CSC_CTRL_RESET - indicates that the connection should be reset.
//                     "Reset" is an abstract term that should be defined
//                     by the application.
//
// Specifying any other value for ulCtrlCode returns COM_ERR_BADCONTROL.
//
// Input:  hccConnect - handle to the connection
//         ulCtrlCode - specifies a CSC_CTRL_* constant
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComReadData(HCCCONNECT hccConnect,
                                 PVOID pvBuf,
                                 PULONG pulSzBuf);
//-------------------------------------------------------------------------
// This function reads data from the connection.
//
// Input:  hccConnect - handle to the connection
//         pulSzBuf - points to the variable containing the size of the
//                    buffer pointed to by pvBuf.  On exit, this variable
//                    contains the number of bytes read.
// Output:  pvBuf - points to the buffer to receive the data
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComWriteData(HCCCONNECT hccConnect,
                                  PVOID pvBuf,
                                  ULONG ulSzBuf);
//-------------------------------------------------------------------------
// This function writes data to a connection.
//
// Input:  hccConnect - handle to the connection
//         pvBuf - points to the buffer containing the data
//         ulSzBuf - specifies the size of the buffer pointed to by pvBuf
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------

COMERROR EXPENTRY CmnComQueryData(HCCCONNECT hccConnect,PULONG pulData);
//-------------------------------------------------------------------------
// This function returns the size of the data waiting to be read by
// the specified connection.  Since this does not return any data that
// was written by the connection, the number returned does *not* necessarily
// equal the number of bytes in the connection.
//
// Input:  hccConnect - handle to the connection
// Output:  pulData - points to the variable that receives the result
// Returns:  COM_ERR_* constant
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNCOM

#ifdef INCL_CMNDBG
//-------------------------------------------------------------------------
// Debugging routines
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnDbgWriteRecord(PVOID pvFile,
                                 USHORT usFileType,
                                 struct _CSLINEDESC *pcFormat,
                                 USHORT usNumRecs,
                                 PBYTE pbBuf);
//-------------------------------------------------------------------------
// This function writes the contents of the specified buffer to the specified
// file.  The format of the buffer is specified in acFormat.
//
// Input:  pvFile - point to the file name or FILE structure
//         usFileType - DWT_TYPE_* constant describing the contents of pvFile
//         pcFormat - points to an array of CSLINEDESC describing the line
//         usNumRecs - specifies the number of CSLINEDESC records are
//                     pointed to by acFormat
//         pbBuf - points to the buffer to print
// Returns:  the number of fields printed is successful, DWT_ERROR otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnDbgWriteText(PVOID pvFile,USHORT usFileType,PCHAR pchFmt,...);
//-------------------------------------------------------------------------
// This function writes debug info to the file described by pvFile.  pvFile
// can be a pointer to a file name or a (FILE *).  In the former case, the
// file is opened upon entry and closed upon exit.
//
// Input:  pvFile - points to the file name or FILE structure
//         usFileType - DWT_TYPE_* constant describing the contents of pvFile
//         pchFmt - point to the format string (as in printf)
// Returns:  number of characters written if successful, DWT_ERROR otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnDbgWriteBinary(PVOID pvFile,
                                 USHORT usFileType,
                                 PVOID pvBuf,
                                 USHORT usSzBuf);
//-------------------------------------------------------------------------
// This function writes the binary data in hexadecimal/ASCII format (a la
// Norton's disk editor) to the specified file.
//
// Input:  pvFile - points to the file name or FILE structure
//         usFileType - DWT_TYPE_* constant describing the contents of pvFile
//         pvBuf - points to the data to be written
//         usSzBuf - specifies the size of the buffer pointed to by pvBuf
// Returns:  number of characters written if successful, DWT_ERROR otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNDBG

#ifdef INCL_CMNFIL
//-------------------------------------------------------------------------
// File routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilCreateSearch(PCHAR pchMask,
                                 ULONG ulAttr,
                                 PHCFSEARCH phcSearch);
//-------------------------------------------------------------------------
// This function creates a search handle
//
// Input:  pchMask - points to the file mask to search with
//         ulAttr - points to the file attributes to retrieve
// Output:  phcSearch - points to the search handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

LONG EXPENTRY CmnFilSearchFiles(HCFSEARCH hcSearch,
                                PCHAR pchBuf,
                                ULONG ulSzBuf);
//-------------------------------------------------------------------------
// This function searches for the files matching the mask specified in the
// CmnFilCreateSearch call.
//
// Input:  hcSearch - search handle
//         ulSzBuf - contains the size of the buffer pointed to by pchBuf
// Output:  pchBuf - points to the next file that matches the mask
// Returns:  attribute of the file found, FSF_NOFILES if none found, or
//           FSF_ERROR if an error occurred.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilDestroySearch(PHCFSEARCH phcSearch);
//-------------------------------------------------------------------------
// This function destroys the specified search handle
//
// Input:  phcSearch - points to the search handle to destroy.
// Output:  phcSearch - points to the value NULL.
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilQueryExtAttribute(PCHAR pchFile,
                                      PCHAR pchVendor,
                                      PCHAR pchAppl,
                                      PCHAR pchName,
                                      PUSHORT pusAttr,
                                      PBYTE pbValue,
                                      PUSHORT pusSzValue);
//-------------------------------------------------------------------------
// This function queries the value of the specified extended attribute
// from the specified file.  This function cannot be used to query
// EAs of type EAT_MVMT, EAT_MVST, or EAT_ASN1.
//
// Input:  pchFile - points to the name of the file to query
//         pchVendor - points to the name of the company, or is one of
//                     the FEA_* constants.  If an FEA_* constant is
//                     specified, pchAppl and pchName are ignored.
//         pchAppl - points to the name of the application
//         pchName - points to the name of the extended attribute
//         pusAttr - points to the variable which receives the EAT_*
//                   constant for the data
//         pbValue - points to the buffer which receives the data
//         pusSzValue - points to the variable containing the size of the
//                      buffer pointed to by pbValue
// Output:  pusAttr - points to the variable containing the EAT_* type
//          pbValue - points to the extended attribute data
//          pusSzValue - points to the variable containing the size of the
//                       data returned
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilSetExtAttribute(PCHAR pchFile,
                                    USHORT usAttr,
                                    PCHAR pchVendor,
                                    PCHAR pchAppl,
                                    PCHAR pchName,
                                    PBYTE pbValue,
                                    USHORT usSzValue);
//-------------------------------------------------------------------------
// This function adds or changes the specified extended attribute for the
// specified file.  This function cannot be used to set EAs of type
// EAT_MVMT, EAT_MVST, or EAT_ASN1.
//
// Input:  pchFile - points to the name of the file to query
//         usAttr - specifies the EAT_* constant for the data
//         pchVendor - points to the name of the company, or is one of
//                     the FEA_* constants.  If an FEA_* constant is
//                     specified, pchAppl and pchName are ignored.
//         pchAppl - points to the name of the application
//         pchName - points to the name of the extended attribute
//         pbValue - points to the buffer which contains the data
//         usSzValue - specifies the size of the buffer pointed to by
//                     pbValue
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilSplitFilename(PCHAR pchFile,
                                  PCHAR pchDrive,
                                  PCHAR pchPath,
                                  PCHAR pchName);
//-------------------------------------------------------------------------
// This function splits a filename into the drive, path, and name components.
//
// Input:  pchFile - points to the filename to split
// Output:  pchDrive - if not NULL, points to the buffer which receives
//                     the drive portion of the filename
//          pchPath - if not NULL, points to the buffer which receives
//                    the path portion of the filename
//          pchName - if not NULL, points to the buffer which receives
//                    the name portion of the filename
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilFormatDiskette(CHAR chDrive,
                                   PCFFORMATINFO pfiInfo,
                                   PFNDISKIO pfnCallback,
                                   PVOID pvUser);
//-------------------------------------------------------------------------
// This function formats a diskette according to the specified parameters.
//
// For the notification FCD_MSG_WANTFORMATDISK and the type
// FCD_TYPE_PROGESS, pfnCallback should return TRUE for the operation to
// continue or FALSE to halt the operation.
//
// Input:  chDrive - specifies the drive to be formatted.
//         pfiInfo - points to the format parameters.  If NULL, the
//                   drive defaults are used.
//         pfnCallback - pointer to a callback function
//         pvUser - user-data passed to pfnCallback
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilCopyDiskette(CHAR chSrc,
                                 CHAR chDest,
                                 ULONG ulOptions,
                                 PFNDISKIO pfnCallback,
                                 PVOID pvUser);
//-------------------------------------------------------------------------
// This function copies the contents of one diskette to another.  If the
// destination diskette is already formatted, its capacity must match that
// of the source diskette or this function will fail.
//
// For the notification type FCD_TYPE_PROGRESS and for the following
// FCD_MSG_* notifications, pfnCallback should return TRUE for the operation
// to continue/retry or FALSE to halt/abort the operation.
//
// Input:  chSrc - specifies the source drive
//         chDest - specifies the destination drive
//         ulOptions - specifies one or mor FCD_OPT_* constants
//         pfnCallback - pointer to a callback function
//         pvUser - user-data passed to pfnCallback
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnFilQueryLabel(CHAR chDrive,PCHAR pchLabel);
//-------------------------------------------------------------------------
// This function returns the label of the diskette in the drive specified.
//
// Input:  chDrive - specifies the drive to check
//         pchLabel - points to the buffer to copy the label into
// Output:  pchLabel - points to the diskette label
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNFIL

#ifdef INCL_CMNLST
//-------------------------------------------------------------------------
// Link-list routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstAddRecord(HCLLIST hclList,
                              PVOID pvRecord,
                              PFNRECCOMP pfnSort,
                              PVOID *ppvResult);
//-------------------------------------------------------------------------
// This function adds a "record" into the linked list in sorted order as
// defined by the user-defined pfnSort function.
//
// Input:  hclList - handle to the list to add the record to
//         pvRecord - points to the record to add
//         pfnSort - points to the comparison function.  If NULL, the function
//                   inserts at the end of the list.  See above for a
//                   description of its parameters.
// Output:  ppvResult - if not NULL on entry, points to the variable pointing
//                      to the new entry
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnLstAddUniqueRecord(HCLLIST hclList,
                                      PVOID pvRecord,
                                      PFNRECCOMP pfnSearch,
                                      PFNRECCOMP pfnSort,
                                      PVOID *ppvResult);
//-------------------------------------------------------------------------
// This function adds a "record" into the linked list only if the record
// does not already exist.  The record is added in sorted order using the
// pfnSort function.
//
// Input:  hclList - handle to the list to add the record to
//         pvRecord - points to the record to add
//         pfnSearch - points to the search function.  See above for a
//                     description of its parameters.
//         pfnSort - points to the comparison function.  If NULL, the function
//                   inserts at the end of the list.  See above for a
//                   description of its parameters.
// Output:  ppvResult - if not NULL on entry, points to the variable pointing
//                      to the old entry if one already existed or the newly
//                      created entry.
// Returns:  LAUR_NOERROR if successful, LAUR_EXISTS if the record already
//           exists, or LAUR_ERROR otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstMoveRecord(HCLLIST hclList,
                               PVOID pvMove,
                               PVOID pvWhere,
                               SHORT sRelative);
//-------------------------------------------------------------------------
// This function moves a record to a position relative to another record in
// the specified list.
//
// Input:  hclList - handle to the list
//         pvMove - pointer to the record to move
//         pvWhere - pointer to the record specifying the new location
//         sRelative - either LQR_PREVIOUS or LQR_NEXT specifying
//                     immediately before or after pvWhere, respectively.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstCopyList(HCLLIST hclList,
                             HCLLIST hclCopy,
                             PFNRECCOMP pfnSort);
//-------------------------------------------------------------------------
// This function copies one list into another list in sorted order.  Note
// that if a call to CmnLstAddRecord fails, the state of the list being
// modified is undefined.
//
// Input:  hclList - handle to the list to copy into
//         hclCopy - handle to the list to copy from
//         pfnSort - points to the comparison function.  If NULL, the function
//                   will append hclCopy to hclList.  See above for a
//                   description of its parameters.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstCreateList(USHORT usSzRecord,PHCLLIST phclList);
//-------------------------------------------------------------------------
// This function creates a linked list of records, and initializes it to
// "empty".
//
// Input:  usSzRecord - the size of each record of the linked list
// Output:  phclList - points to the handle to the new list
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstDeleteList(HCLLIST hclList);
//-------------------------------------------------------------------------
// This function deletes the entire list but does not destroy it
//
// Input:  hclList - handle to the list to delete
// Output:  ppvHead - points to the head of the list==NULL
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstDeleteRecord(HCLLIST hclList,PVOID pvRecord);
//-------------------------------------------------------------------------
// This function deletes a record from the linked list.
//
// Input:  hclList - handle to the list to delete the record from
//         pvRecord - points to the record to delete
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstDestroyList(PHCLLIST phclList);
//-------------------------------------------------------------------------
// This function deletes and then destroys the specified list.
//
// Input:  phclList - points to the handle to the list
// Output:  pchList - points to a variable containing NULL
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstPruneList(HCLLIST hclList,
                              PVOID pvRecord,
                              PFNRECCOMP pfnSearch);
//-------------------------------------------------------------------------
// This function deletes records that match the search criteria from the
// specified list.
//
// Input:  hclList - handle to the list to prune
//         pvSearch - points to the application defined search data.  It is
//                    passed to pfnSearch as the second parameter.
//         pfnSearch - points to the comparison function.  See above for a
//                     description of its parameters.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

#define LQR_END                  -1

PVOID EXPENTRY CmnLstQueryRecord(HCLLIST hclList,SHORT sNumRecord);
//-------------------------------------------------------------------------
// This function returns the nth record in the specified linked list.
//
// Input:  hclList - handle to the list to query
//         sNumRecord - specifies the 0-based number of the record to
//                      retrieve or a LQR_* constant
// Returns:  pointer to the record if successful, NULL otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnLstQueryRecordCount(HCLLIST hclList);
//-------------------------------------------------------------------------
// This function returns the number of records in the linked list.
//
// Input:  hclList - handle to the list to query
// Returns:  the number of records if successful, LQRC_ERROR otherwise
//-------------------------------------------------------------------------

PVOID EXPENTRY CmnLstQueryRelative(PVOID pvRecord,SHORT sWhich);
//-------------------------------------------------------------------------
// This function returns either the previous or next record relative to the
// specified record.
//
// Input:  pvRecord - points to the record to query
//         sNumRecord - specifies either LQR_PREVIOUS or LQR_NEXT
// Returns:  pointer to the record if successfull, or NULL if pvRecord is the
//           head of the list and LQR_PREVIOUS was specified, pvRecord is the
//           tail of the list and LQR_NEXT was specified or an error occurred.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstQuickSortList(HCLLIST hclList,
                                  PFNRECMED pfnMedian,
                                  PFNRECCOMP pfnCompare);
//-------------------------------------------------------------------------
// This function sorts the specified list using a version of the quick sort
// algorithm that has been slightly modified to conserve memory.
//
// Input:  hclList - handle to the list to sort
//         pfnMedian - points to a median function.  See above for a
//                     description of its parameters.
//         pfnCompare - points to a comparison function.  See above for a
//                      description of its parameters.
// Output:  hclList - handle to the sorted list
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

PVOID EXPENTRY CmnLstSearchRecord(PVOID pvRecord,
                                  PVOID pvSearch,
                                  PFNRECCOMP pfnSearch);
//-------------------------------------------------------------------------
// This function searches the linked list for a record using the pfnSearch
// function.
//
// Input:  pvRecord - points to the first record of the linked list where
//                    searching is to begin
//         pvSearch - points to the application defined search data.  It is
//                    passed to pfnSearch as the second parameter.
//         pfnSearch - points to the comparison function.  See above for a
//                     description of its parameters.
// Returns:  pointer to record if found, NULL otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstSortList(HCLLIST hclList,PFNRECCOMP pfnSort);
//-------------------------------------------------------------------------
// This function performs a bubble sort on the specified list.
//
// Input:  hclList - handle to the list to sort
//         pfnSort - points to the comparison function.  See above for a
//                   description of its parameters.
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnLstTraverseList(HCLLIST hclList,PFNRECFUNC pfnFunc);
//-------------------------------------------------------------------------
// This function traverses the specified list, calling the specified function
// for each record in the list.  This can be used, for example, to print
// out each record in the list for debugging purposes.
//
// Input:  hclList - handle to the list to traverse
//         pfnFunc - callback function.  See above for a description of its
//                   parameters.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNLST

#ifdef INCL_CMNMEM
//-------------------------------------------------------------------------
// Memory allocation routines
//-------------------------------------------------------------------------

MEMERROR EXPENTRY CmnMemAllocate(HCMMEM hcmMem,ULONG ulSzBuf,PVOID *ppvBuf);
//-------------------------------------------------------------------------
// This function allocates a block of memory from a memory manager instance.
//
// Input:  hcmMem - handle to the memory manager instance.
//         ulSzBuf - specifies the requested size of the memory block.
// Output:  ppvBuf - points to the variable which receives the pointer
//                   to the allocated memory.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise.
//-------------------------------------------------------------------------

MEMERROR EXPENTRY CmnMemFree(HCMMEM hcmMem,PVOID pvBuf);
//-------------------------------------------------------------------------
// This function returns a block of memory to the memory manager.
//
// Input:  hcmMem - handle to the memory manager instance.
//         pvBuf - points to the block of memory to free.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

MEMERROR EXPENTRY CmnMemInitialize(PCMMEMINFO pcmiInfo,PHCMMEM phcmMem);
//-------------------------------------------------------------------------
// This function creates an instance of the memory manager.
//
// Input:  pcmiInfo - points to a CMMEMINFO structure which specifies the
//                    desired attributes of the instance.  If NULL, the
//                    defaults are used.
// Output:  phcmMem - points to the variable containing the memory handle.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise.
//-------------------------------------------------------------------------

ULONG EXPENTRY CmnMemQueryMemSize(PVOID pvBuf);
//-------------------------------------------------------------------------
// This function returns the size of a block of memory.  No validation is
// done to determine whether or not the memory was allocated with
// CmnMemAllocate.
//
// Input:  pvBuf - points to the block of memory to query.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

MEMERROR EXPENTRY CmnMemReset(HCMMEM hcmMem);
//-------------------------------------------------------------------------
// This function deallocates *all* memory associated with a memory manager
// instance and resets the heaps to "empty."
//
// Input:  hcmMem - handle to the memory manager instance.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

MEMERROR EXPENTRY CmnMemTerminate(PHCMMEM phcmMem);
//-------------------------------------------------------------------------
// This function destroys an instance of the memory manager.
//
// Input:  phcmMem - points to the variable containing the memory handle
//                   to destroy.
// Output:  phcmMem - points to the value NULL.
// Returns:  MEM_ERR_NOERROR if successful, error code otherwise.
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNMEM

#ifdef INCL_CMNOBJ
//-------------------------------------------------------------------------
// Object routines
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjInitialize(PCOROOTINFO pcoriInfo,PHCOOBJECT phcoRoot);
//-------------------------------------------------------------------------
// This function initializes the object manager and returns the handle
// to the root object.
//
// Input:  pcoriInfo - points to the COROOTINFO structure containing
//                     the desired characteristics of the instance
// Output:  phcoRoot - points to the variable that received the created
//                     root object handle
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjTerminate(PHCOOBJECT phcoRoot);
//-------------------------------------------------------------------------
// This function destroys all children of the specified root object
// (via a call to CmnObjReset()) and terminates the object manager
// instance.
//
// Input:  phcoRoot - points to the variable specified the root object
//                    handle.  On return, this variable contains NULL.
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjReset(HCOOBJECT hcoRoot);
//-------------------------------------------------------------------------
// This function deletes all child objects from the specified root object.
//
// Input:  hcoRoot - handle of the root object
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjQueryObject(HCOOBJECT hcoObject,
                                    SHORT sQuery,
                                    PHCOOBJECT phcoResult);
//-------------------------------------------------------------------------
// This function returns the handle of the object related to the specified
// object in the specified way.
//
// Input:  hcoObject - handle to the object to query the relationship of
//         sQuery - an OQO_* constant specifying the relationship
// Output:  phcoResult - points to the variable which receives the result
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjCreateObject(HCOOBJECT hcoParent,
                                     PCHAR pchName,
                                     PHCOOBJECT phcoResult);
//-------------------------------------------------------------------------
// This function creates a new object whose parent is specified in
// hcoParent.
//
// Input:  hcoParent - handle to the parent object
//         pchName - points to the variable specifying the subname.  This
//                   cannot contain a backslash
// Output:  phcoResult - points to the variable which receives the handle
//                       to the newly created object
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjDestroyObject(PHCOOBJECT phcoObject);
//-------------------------------------------------------------------------
// This function destroys the specified object.  It is an error to call
// this function with the root object handle.
//
// Input:  phcoObject - points to the variable specifying the handle of
//                      the object to destroy.  On return, this variable
//                      contains NULL.
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjSetObjectData(HCOOBJECT hcoObject,
                                      PVOID pvBuffer,
                                      ULONG ulSzBuf);
//-------------------------------------------------------------------------
// This function sets the object-specific data.  Any data that was
// previously set into the object is disposed of an the memory is returned
// to the system.
//
// Intput:  hcoObject - handle to the object
//          pvBuffer - points to the new object data
//          ulSzBuf - specifies the size of the buffer pointed to by
//                    pvBuffer
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjQueryObjectData(HCOOBJECT hcoObject,
                                        PVOID pvBuffer,
                                        PULONG pulSzBuf);
//-------------------------------------------------------------------------
// This function returns the object-specific data set by CmnObjSetData().
//
// Intput:  hcoObject - handle to the object
// Output:  pvBuffer - points to the receiving buffer.  If NULL, pulSzBuf
//                     contains the size required to completely hold the
//                     data
//          pulSzBuf - points to the variable that receives the number of
//                     of bytes copied if pvBuffer!=NULL.  If pvBuffer==NULL,
//                     the variable instead contains the size in bytes of
//                     the data
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------

OBJERROR EXPENTRY CmnObjObjectFromName(HCOOBJECT hcoObject,
                                       PCHAR pchName,
                                       PHCOOBJECT phcoResult);
//-------------------------------------------------------------------------
// This function returns the object with the given relative or absolute
// name.  The name to search for consists of one or more subnames separated
// by backslashes.  If the name begins with a backslash, searching begins
// at the root object.  As with filenames, the subnames "." and ".." have
// special meanings analagous to the file system interpretations.
//
// Input:  hcoObject - handle to the object
//         pchName - points to the name
// Output:  phcoResult - points to the variable which receives the result
// Returns:  OBJ_ERR_NOERROR if successful, error code otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNOBJ

#ifdef INCL_CMNSET
//-------------------------------------------------------------------------
// Set routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetCreateSet(ULONG ulMaxElems,PHCSSET phcsSet);
//-------------------------------------------------------------------------
// This function creates a set with the specified number of elements.
//
// Input:  ulMaxElems - specifies the maximum number of elements.
// Output:  phcsSet - points to the variable containing the set handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetDestroySet(PHCSSET phcsSet);
//-------------------------------------------------------------------------
// This function destroys the set handle pointed to by phcsSet.
//
// Input:  phcsSet - points to the variable specifying the set handle to
//                  destroy
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetSetElement(HCSSET hcsSet,ULONG ulElement);
//-------------------------------------------------------------------------
// This function sets the specified element in the specified set.
//
// Input:  hcsSet - specifies the set handle
//         ulElement - specifies the element to set
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetClearElement(HCSSET hcsSet,ULONG ulElement);
//-------------------------------------------------------------------------
// This function clears the specified element in the specified set.
//
// Input:  hcsSet - specifies the set handle
//         ulElement - specifies the element to clear
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetInvertElement(HCSSET hcsSet,ULONG ulElement);
//-------------------------------------------------------------------------
// This function inverts the state of the specified element.
//
// Input:  hcsSet - specifies the set handle
//         ulElement - specifies the element to invert
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

LONG EXPENTRY CmnSetFindElement(HCSSET hcsSet,ULONG ulNumber,USHORT usType);
//-------------------------------------------------------------------------
// This function finds the nth element that is either set or clear.
//
// Input:  hcsSet - specifies the set handle
//         ulNumber - specifies the number of the element desired
//         usType - specifies either SFE_SET if the nth element set is
//                  desired or SFE_CLEAR if the nth element clear is
//                  desired
// Returns:  Element number if found, SFE_NOTFOUND if not found, or
//           SFE_ERROR if an error occurred
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetSetSet(HCSSET hcsSet);
//-------------------------------------------------------------------------
// This function sets all of the elements of the specified set.
//
// Input:  hcsSet - specifies the set handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetClearSet(HCSSET hcsSet);
//-------------------------------------------------------------------------
// This function clears all of the elements of the specified set.
//
// Input:  hcsSet - specifies the set handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSetInvertSet(HCSSET hcsSet);
//-------------------------------------------------------------------------
// This function inverts all of the elements of the specified set.
//
// Input:  hcsSet - specifies the set handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnSetQueryElement(HCSSET hcsSet,ULONG ulElement);
//-------------------------------------------------------------------------
// This function returns the state of the specified element.
//
// Input:  hcsSet - specifies the set handle
//         ulElement - specifies the element to query
// Returns:  SQE_SET if the element is set, SQE_CLEAR if the element is not
//           set, or SQE_ERROR if an error occurs
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnSetQueryEqualSets(HCSSET hcsSet1,HCSSET hcsSet2);
//-------------------------------------------------------------------------
// This function compares two sets for equality.
//
// Input:  hcsSet1 - specifies the first set handle
//         hcsSet2 - specifies the second set handle
// Returns:  SQES_EQUAL if the sets are equal, SQES_NOTEQUAL if the sets
//           are not equal, or SQES_ERROR if an error occurs
//-------------------------------------------------------------------------

HCSSET EXPENTRY CmnSetCombineSets(HCSSET hcsSet1,HCSSET hcsSet2);
//-------------------------------------------------------------------------
// This function calculates the union of the specified sets and returns
// the result.
//
// Input:  hcsSet1 - specifies the first set handle
//         hcsSet2 - specifies the second set handle
// Returns:  a new set handle containing the union if successful, NULL
//           otherwise
//-------------------------------------------------------------------------

HCSSET EXPENTRY CmnSetIntersectSets(HCSSET hcsSet1,HCSSET hcsSet2);
//-------------------------------------------------------------------------
// This function calculates the intersection of the specified sets and returns
// the result.  If the sets are not equal in size, the function pads the
// smaller set with "set" elements.
//
// Input:  hcsSet1 - specifies the first set handle
//         hcsSet2 - specifies the second set handle
// Returns:  a new set handle containing the intersection if successful,
//           NULL otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNSET

#ifdef INCL_CMNSIG
//-------------------------------------------------------------------------
// Signal routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigCreateSignalList(PHCSSIGNALLIST phcsList);
//-------------------------------------------------------------------------
// This function creates a signal list which is used in subsequent calls to
// the CmnSig functions.  Typically, only one per application is needed,
// although this might not always be true.
//
// Output:  phcsList - points to the variable specifying the list handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigDestroySignalList(PHCSSIGNALLIST phcsList);
//-------------------------------------------------------------------------
// This function destroys the specified signal list.
//
// Input:  phcsList - points to the variable specifying the list handle
// Output:  phcsList - points to the variable specifying NULL
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigCreateSignal(HCSSIGNALLIST hcsList,
                                 ULONG ulId,
                                 PHCSSIGNAL phcSignal);
//-------------------------------------------------------------------------
// This function creates a signal handle in the specified signal list.
// The returned signal handle is used in subsequent calls to the signal
// functions.  Note that ulId can be any id unique to the list.
//
// Input:  hcsList - specifies the list handle to add the signal to
//         ulId - specifies the id of the handle to be created
// Output:  phcSignal - points to the variable containing the created
//                      signal handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigDestroySignal(PHCSSIGNAL phcSignal);
//-------------------------------------------------------------------------
// This function destroys the specified signal from the owner list.
//
// Input:  phcSignal - points to the variable specifying the signal handle
//                     to delete
// Output:  phcSignal - points to the variable specifying NULL
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigQuerySignal(HCSSIGNALLIST hcsList,
                                ULONG ulId,
                                PHCSSIGNAL phcSignal);
//-------------------------------------------------------------------------
// This function searches the specified list for the signal handle with the
// specified id.
//
// Input:  hcsList - specifies the list to search
//         ulId - specifies the signal id to search for
// Output:  phcSignal - points to the variable specifying the signal handle
//                      if found, NULL otherwise
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigSetSignalValue(HCSSIGNAL hcSignal,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function sets the signal value for the specified signal handle to
// the specified value.  This is useful for unique signal values.  Note
// that since 0 could possibly be a valid signal value, you must call
// CmnSigSetSignalValue() with the "empty" value (this is application-specific)
// to clear this value.
//
// For signal values whose individual bits have specific interpretations,
// use CmnSigSetSignalBit().
//
// Input:  hcSignal - specifies the signal handle to set
//         ulSignal - specifies the new signal value
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigSetSignalBit(HCSSIGNAL hcSignal,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function sets the specified signal value bits for the specified
// signal handle.
//
// Input:  hcSignal - specifies the signal handle to set
//         ulSignal - specifies the signal bits to set
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigClearSignalBit(HCSSIGNAL hcSignal,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function clears the specified signal value bits for the specified
// signal handle.
//
// Input:  hcSignal - specifies the signal handle to clear
//         ulSignal - specifies the signal bits to clear
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigSetAllSignalValues(HCSSIGNALLIST hcsList,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function sets all of the signals in the specified list to a specific
// value.
//
// Input:  hcsList - specifies the list to set
//         ulSignal - specifies the new value of the list
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigSetAllSignalBits(HCSSIGNALLIST hcsList,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function sets the specified bits of all of the signals in the
// specified list.
//
// Input:  hcsList - specifies the list to set
//         ulSignal - specifies the new value of the list
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigClearAllSignalBits(HCSSIGNALLIST hcsList,ULONG ulSignal);
//-------------------------------------------------------------------------
// This function clears the specified bits of all of the signals in the
// specified list.
//
// Input:  hcsList - specifies the list to set
//         ulSignal - specifies the new value of the list
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigQuerySignalId(HCSSIGNAL hcSignal,PULONG pulId);
//-------------------------------------------------------------------------
// This function returns the id of the specified signal handle.
//
// Input:  hcSignal - specifies the signal handle to query
// Output:  pulId - points the variable specifying the signal id
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnSigQuerySignalValue(HCSSIGNAL hcSignal,PULONG pulSignal);
//-------------------------------------------------------------------------
// This function returns the current signal value for the specified signal.
//
// Input:  hcSignal - specifies the signal handle to query
// Output:  pulSignal - points the variable specifying the signal value
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnSigWaitSignalValue(HCSSIGNAL hcSignal,
                                      ULONG ulSignal,
                                      LONG lTimeout);
//-------------------------------------------------------------------------
// This function waits for a signal handle's signal value to equal a
// specified value, subject to a timeout.
//
// Input:  hcSignal - specifies the signal handle to wait for
//         ulSignal - specifies the signal value to wait for
//         lTimeout - specifies the timeout in milliseconds, or an
//                    SWS_TIMEOUT_* value
// Returns:  SWS_RC_SIGNALSET if the signal was set, SWS_RC_TIMEOUT if
//           a timeout occurred, or SWS_RC_ERROR if an error occurred
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnSigWaitSignalBit(HCSSIGNAL hcSignal,
                                    ULONG ulSignal,
                                    LONG lTimeout);
//-------------------------------------------------------------------------
// This function waits for a signal handle's signal value to have a
// specific set of bits set, subject to a timeout.
//
// Note that this is different than CmnSigWaitSignal() in that the other
// bits can be any value, whereas the said function checks for equality
// for *all* bits.
//
// Input:  hcSignal - specifies the signal handle to wait for
//         ulSignal - specifies the signal value bits to wait for
//         lTimeout - specifies the timeout in milliseconds, or an
//                    SWS_TIMEOUT_* value
// Returns:  SWS_RC_SIGNALSET if the signal was set, SWS_RC_TIMEOUT if
//           a timeout occurred, or SWS_RC_ERROR if an error occurred
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnSigWaitAllSignalValues(HCSSIGNALLIST hcsList,
                                          ULONG ulSignal,
                                          LONG lTimeout);
//-------------------------------------------------------------------------
// This function waits for *all* signal handles in the specified list to have
// the signal value specified, subject to a timeout.
//
// Input:  hcsList - specifies the signal list to wait for
//         ulSignal - specifies the signal value to wait for
//         lTimeout - specifies the timeout in milliseconds, or an
//                    SWS_TIMEOUT_* value
// Returns:  SWS_RC_SIGNALSET if the signal was set, SWS_RC_TIMEOUT if
//           a timeout occurred, or SWS_RC_ERROR if an error occurred
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnSigWaitAllSignalBits(HCSSIGNALLIST hcsList,
                                        ULONG ulSignal,
                                        LONG lTimeout);
//-------------------------------------------------------------------------
// This function waits for *all* signal handles in the specified list to have
// the set of signal value bits set, subject to a timeout.
//
// Note that this is different than CmnSigWaitSignal() in that the other
// bits can be any value, whereas the said function checks for equality
// for *all* bits.
//
// Input:  hcsList - specifies the signal list to wait for
//         ulSignal - specifies the signal value to wait for
//         lTimeout - specifies the timeout in milliseconds, or an
//                    SWS_TIMEOUT_* value
// Returns:  SWS_RC_SIGNALSET if the signal was set, SWS_RC_TIMEOUT if
//           a timeout occurred, or SWS_RC_ERROR if an error occurred
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNSIG

#ifdef INCL_CMNSPR
SPRERROR EXPENTRY CmnSprCreatePlayground(HAB habAnchor,
                                         PHCSPLAYGROUND phpgPlay);
//-------------------------------------------------------------------------
// This function creates a playground to which sprites can be added.
//
// Input:  habAnchor - anchor block of the calling thread.
// Output:  phpgPlay - points to the variable with the HCSPLAYGROUND handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprDestroyPlayground(HCSPLAYGROUND hpgPlay);
//-------------------------------------------------------------------------
// This function destroys the playground including any sprites that are
// still members of it.  All resources consumed by the playground,
// including the back bitmap, are returned to the system.
//
// Input:  hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprAddSprite(HCSPLAYGROUND hpgPlay,HCSSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function labels a sprite as a "member" of the specified playground.
// Doing so allows the application to control the sprite's position,
// visibility, etc. on a drawing surface.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to add
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprRemoveSprite(HCSPLAYGROUND hpgPlay,HCSSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function removes the sprite from the membership list of the
// specified playground.  The sprite can then be added to another
// playground, or this one at a later time.
//
// Since there is a limited number of sprites that can be members of
// a playground, this function can be used to temporarily remove unused
// sprites from a playground so that others can be used.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to remove
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetPlaygroundBack(HCSPLAYGROUND hpgPlay,
                                          HBITMAP hbmNew,
                                          HBITMAP *phbmOld);
//-------------------------------------------------------------------------
// This function sets the background bitmap of the playground.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application unless the bitmap is "unset" from the playground (by
// calling this function again with a different handle).
//
// Input:  hpgPlay - handle to the playground
//         hbmNew - handle to the new bitmap to used as the background
// Output:  phbmOld - points to the handle to the old background bitmap.
//          This can be NULL, meaning that the application isn't interested
//          in receiving this value.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQueryPlaygroundBack(HCSPLAYGROUND hpgPlay,
                                            HBITMAP *phbmBack);
//-------------------------------------------------------------------------
// This function returns the handle of the background bitmap currently in
// use.
//
// Input:  hpgPlay - handle to the playground
// Output:  phbmBack - points to the handle to the background bitmap.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetPlaygroundSize(HCSPLAYGROUND hpgPlay,
                                          PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function sets the playground size for playgrounds that do not have
// a bitmap set as the background.
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the size of the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQueryPlaygroundSize(HCSPLAYGROUND hpgPlay,
                                            PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function returns the size of the playground.  For playgrounds with
// bitmaps set as the background, the returned value is the size of the
// bitmap.  Otherwise, the returned value is that which was specified on
// the last call to CmnSprSetPlaygroundSize().
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the variable to receive the size of the
//                    playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetPlaygroundColor(HCSPLAYGROUND hpgPlay,
                                           LONG lBackColor);
//-------------------------------------------------------------------------
// This function sets the new background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         lBackColor - specifies the new background color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQueryPlaygroundColor(HCSPLAYGROUND hpgPlay,
                                             PLONG plBackColor);
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetUpdateFlag(HCSPLAYGROUND hpgPlay,BOOL bUpdate);
//-------------------------------------------------------------------------
// This function sets the update flag for the playground.  If FALSE, no
// drawing actually takes place in any of the functions requiring an HPS,
// and the value of the HPS handle may be NULLHANDLE.  If TRUE, updating
// is reenabled, but you should still call CmnSprDrawPlayground() to refresh
// the screen with the current contents.
//
// Input:  hpgPlay - handle to the playground
//         bUpdate - specifies the new update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQueryUpdateFlag(HCSPLAYGROUND hpgPlay,PBOOL pbUpdate);
//-------------------------------------------------------------------------
// This function returns the setting of the update flag.  See the notes
// for CmnSprSetUpdateFlag() for more information about this setting.
//
// Input:  hpgPlay - handle to the playground
//         pbUpdate - points to the variable to receive the update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprDrawPlayground(HPS hpsDraw,HCSPLAYGROUND hpgPlay);
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprCreateSprite(HAB habAnchor,
                                     HBITMAP hbmBitmap,
                                     PHCSSPRITE phsSprite);
//-------------------------------------------------------------------------
// This function creates a sprite from the specified bitmap.  The sprite
// cannot be moved, shown, etc., however, until it is associated with a
// playground.
//
// The color black is used as the transparency color.  If you need to use
// black in the bitmap without it becoming transparent, use the next
// closest color.  <grin>
//
// New sprites are initialized as being at position (0,0) and hidden.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application or else unpredictable results will occur.
//
// Input:  habAnchor - anchor block of the calling thread.
//         hbmBitmap - handle to the bitmap
// Output:  phsSprite - points to the sprite handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprDestroySprite(HCSSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function destroys the sprite and returns all resources to the
// system.
//
// Input:  hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetSpritePosition(HPS hpsDraw,
                                          HCSSPRITE hsSprite,
                                          PPOINTL pptlNew);
//-------------------------------------------------------------------------
// This function changes the position of the sprite.  This function is
// optimized so that, if the rectangle bounding the sprite at the new
// position overlaps the old, only one "bit blit" to the specified HPS
// is done, eliminating flicker.
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in once it is
//                   moved
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQuerySpritePosition(HCSSPRITE hsSprite,
                                            PPOINTL pptlPos);
//-------------------------------------------------------------------------
// This function returns the current position of the sprite.  Note that
// a sprite has a current position even if it is hidden.
//
// Input:  hsSprite - handle to the sprite
// Output:  pptlPos - points to the current position
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQuerySpriteSize(HCSSPRITE hsSprite,PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function returns the current size of the sprite.
//
// Input:  hsSprite - handle to the sprite
// Output:  pszlSize - points to the current size
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQuerySpriteRect(HCSSPRITE hsSprite,PRECTL prclRect);
//-------------------------------------------------------------------------
// This function returns the bounding rectangle of the sprite at its
// current position.
//
// Input:  hsSprite - handle to the sprite
// Output:  prclRect - points to the current bounding rectangle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprSetSpriteVisibility(HPS hpsDraw,
                                            HCSSPRITE hsSprite,
                                            BOOL bVisible);
//-------------------------------------------------------------------------
// This function shows or hides a sprite.
//
// Input:  hpsDraw - handle to the HPS to draw in once the sprite is
//                   shown or hidden
//         hsSprite - handle to the sprite
//         bVisible - new visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprQuerySpriteVisibility(HCSSPRITE hsSprite,
                                              PBOOL pbVisible);
//-------------------------------------------------------------------------
// This function returns the visibility state of the sprite
//
// Input:  hsSprite - handle to the sprite
// Output:  pbVisible - points to the visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY CmnSprDrawSprite(HPS hpsDraw,HCSSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function draws a sprite
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNSPR

#ifdef INCL_CMNSTR
//-------------------------------------------------------------------------
// String routines
//-------------------------------------------------------------------------

USHORT EXPENTRY CmnStrQueryWordCount(PCHAR pchLine);
//-------------------------------------------------------------------------
// This function returns the number of words in the specified string
//
// Input:  pchLine - points to the string to query
// Returns:  number of words in the string
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnStrQueryWordPosition(PCHAR pchLine,SHORT sWord);
//-------------------------------------------------------------------------
// This function finds the 0-based position of the specified word in the
// specified string.
//
// Input:  pchLine - points to the string to search
//         sWord - specifies the word number
// Returns:  position of the word if successful, SQWP_ERROR otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnStrQueryWordLength(PCHAR pchLine,SHORT sWord);
//-------------------------------------------------------------------------
// This function returns the length of the word specified.
//
// Input:  pchLine - points to the string to search
//         sWord - specifies the word number
// Returns:  length of the word if successful, SQWP_ERROR otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnStrQueryWord(PCHAR pchLine,
                              SHORT sWord,
                              PCHAR pchWord,
                              USHORT usSzWord);
//-------------------------------------------------------------------------
// This function extracts a word non-destructively from the specified line
// and copies it to the specified buffer.
//
// Input:  pchLine - points to the line to parse
//         sWord - contains the 0-based word number to extract
//         usSzWord - contains the size of the buffer pointed to by pchWord
// Output:  pchWord - points to the extracted word
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnStrConvertToNumber(PCHAR pchString,
                                    PVOID pvResult,
                                    USHORT usType);
//-------------------------------------------------------------------------
// This function converts a string to a number.  This is more robust than the
// ato*() family since this positively indicates whether or not an error
// occurred.
//
// Input:  pchString - points to the string to convert
//         usType - a combination of one SCTN_TYPE_* and one SCTN_ATTR_*
//                  constants.
// Output:  pvResult - points to the resulting number
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnStrParseLine(PCHAR pchLine,
                               PCSLINEDESC pcFormat,
                               USHORT usNumRecs,
                               PBYTE pbBuf);
//-------------------------------------------------------------------------
// This function parses the specified line, whose format is described by an
// array of CSLINEDESC structures (acFormat).  If the usType field of a
// CSLINEDESC record is blank, then usLength bytes are skipped in the
// result buffer, allowing the caller to have other fields that are not to
// be filled in by this function.
//
// Upon exit, the usType field is either SPL_FOUND or SPL_NOTFOUND, indicating
// whether or not a non-whitespace character was found in the usOffset<->
// usOffset+usLength-1 positions of the line.
//
// For each field descriptor, the corresponding field is extracted, and
// CmnStrConvertToNumber is called if the destination type is numeric.
//
// Notes:
// For numeric types, an SPL_ATTR_* constant may be or'd to specify
//    whether or not the number is in decimal or hexidecimal
// For SPL_TYPE_STRING fields, usLength also specifies the width of the
//    placeholder in pbBuf and does not include the trailing '\0'.
// For SPL_TYPE_RESERVED fields, usType always equals SPL_FOUND on return.
//
// Input:  pchLine - points to the line to parse
//         pcFormat - points to an array of CSLINEDESC describing the line
//         usNumRecs - specifies the number of CSLINEDESC records are
//                     pointed to by acFormat
// Output:  pcFormat - usType field indicates found/not found status of the
//                     field
//          pbBuf - points to the parsed line
// Returns:  the number of fields found is successful, SPL_ERROR otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnStrParseCommandLine(PCHAR *ppchArgs,
                                     USHORT usNumArgs,
                                     PCSCMDDESC pccdFormat,
                                     USHORT usNumSwitches,
                                     PFNCMDERR pfnError);
//-------------------------------------------------------------------------
// This function parses the command line as described by the pccdFormat
// switch description array.
//
// Input:  ppchArgs - points to an array of pointers to the command-line
//                    arguments
//         usNumArgs - specifies the number of command-line arguments,
//                     including the program name (ppchArgs[0])
//         pccdFormat - points to an array of CSCMDDESC structs describing
//                      the valid command-line arguments:
//
//                      usId - specifies a unique id assigned to the switch.
//                             This is used by the pfnCallback function.
//                      achSwitch - points to the switch string, without the
//                                  leading "-" or "/".
//                      usNumArgs - specifies the number of arguments the
//                                  switch requires.
//                      usFlags - specifies a combination of SPCL_FLG_*
//                                constants describing the switch's attributes:
//
//                                SPCL_CASESENSITIVE specifies that achSwitch
//                                                   is case-sensitive.
//                                SPCL_SUBSTRING     specifies that achSwitch
//                                                   is a substring of what
//                                                   will be encountered on
//                                                   the command line (e.g
//                                                   "-FPa" where "FP" is the
//                                                   switch).
//                                SPCL_ARGCONCAT     (valid only with
//                                                   SPCL_SUBSTRING)
//                                                   specifies that the first
//                                                   argument is concatenated
//                                                   to the switch name (e.g.
//                                                   "-FPa" where "a" is the
//                                                   argument)
//                      pfnCallback - points to a function which is called
//                                    whenever a switch match is found.  It
//                                    takes the following arguments:
//
//                                    USHORT - the identifier of the switch as
//                                             specified in the CSCMDDESC
//                                             structure.
//                                    PCHAR * - the switch arguments.
//                                    USHORT - the number of arguments.
//                                    Returns - TRUE if successful, FALSE
//                                              otherwise.
//         usNumSwitches - specifies the number of CSCMDDESC structures are
//                         pointed to by pccdFormat.
//         pfnError - points to an error function which is called whenever
//                    CmnStrParseCommandLine encounters an error.  It takes
//                    a single USHORT - specifying a SPCL_ERR_* error id -
//                    as its argument.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnStrPadString(PCHAR pchString,
                              USHORT usSide,
                              USHORT usLength,
                              CHAR chPad);
//-------------------------------------------------------------------------
// This function pads a string to the specified length with the specified
// character.  It is assumed that the string is large enough to contain
// the result.
//
// Input:  pchString - points to the string to pad.  On exit, the string
//                     is padded
//         usSide - specifies the side on which to pad.  It can be SPS_LEFT
//                  or SPS_RIGHT.
//         usLength - specifies the desired length of the string.
//         chPad - specifies the character to pad with.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnStrStripSpace(PCHAR pchString,USHORT usOptions);
//-------------------------------------------------------------------------
// This function removes the whitespace from the beginning or the end of the
// specified string.
//
// Input:  pchString - points to the string to strip
//         usOptions - SPS_LEFT, SPS_RIGHT, or a combination of both
//                     to specify from which side(s) the whitespace should
//                     be removed
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNSTR

#ifdef INCL_CMNVIO
//-------------------------------------------------------------------------
// Screen I/O routines
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnVioLoadMessage(PCHAR pchMsgFile,
                                ULONG ulId,
                                PCHAR pchBuf,
                                ULONG ulSzBuf);
//-------------------------------------------------------------------------
// This function loads a message from the specified message file.
//
// Input:  pchMsgFile - pointer to the message file name to load from
//         ulId - id of the message to load
//         pchBuf - pointer to the receiving buffer
//         ulSzBuf - size of the buffer pointed to by pchBuf
// Output:  pchBuf - points to the message text
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnVioDisplayMessage(PCHAR pchMsgFile,ULONG ulId,...);
//-------------------------------------------------------------------------
// This function puts a message to the screen in character mode.
//
// Input:  pchMsgFile - pointer to the message file name to load from
//         ulId - id of the message to load
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL CmnVioGetString(PCHAR pchBuf,ULONG ulSzBuf,ULONG ulOptions);
//-------------------------------------------------------------------------
// This function gets a string from the user.  The behavior of this
// function can be influenced by the options specified:
//
//    VGS_UNREADABLE    - the string should be unreadable.
//
// This function requires that the ANSI device driver is loaded.
//
// Input:  pchBuf - points to the buffer to hold the result
//         ulSzBuf - specifies the size of the buffer pointed to by pchBuf
//         ulOptions - one or more VGS_* constants
// Output:  pchBuf - points to the buffer containing the result
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNVIO

#ifdef INCL_CMNWIN
//-------------------------------------------------------------------------
// Windowing routines
//-------------------------------------------------------------------------

SHORT EXPENTRY CmnWinDisplayMessage(HWND hwndParent,
                                    HWND hwndOwner,
                                    ULONG ulStyle,
                                    HMODULE hmDll,
                                    USHORT usId,
                                    USHORT usHelpId,...);
//-------------------------------------------------------------------------
// This function puts a message to the screen in PM mode (using WinMessageBox).
// The title of the message box is assumed to have the message id usId|0x8000.
//
// Input:  hwndParent - handle of the parent window
//         hwndOwner - handle of the owning window
//         ulStyle - specifies the WinMessageBox styles
//         hmDll - handle of the DLL containing the message
//         usId - specifies the id of the message to load
//         usHelpId - specifies the id of the corresponding help panel
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

VOID EXPENTRY CmnWinCenterWindow(HWND hwndCenter);
//-------------------------------------------------------------------------
// This function centers the window within its parent
//
// Input:  hwndCenter - handle of the window to center
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnWinSavePosition(HWND hwndWindow,
                                 HINI hiniProfile,
                                 PCHAR pchAppl,
                                 PCHAR pchKey);
//-------------------------------------------------------------------------
// This function saves the current position of the specified window in the
// specified .INI file
//
// Input:  hwndWindow - specifies the window to save
//         hiniProfile - specifies the profile to save to
//         pchAppl - specifies the application name to save as
//         pchKey - specifies the key name to save as
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnWinRestorePosition(HWND hwndWindow,
                                    HINI hiniProfile,
                                    PCHAR pchAppl,
                                    PCHAR pchKey);
//-------------------------------------------------------------------------
// This function restores the current position of the specified window from the
// specified .INI file
//
// Input:  hwndWindow - specifies the window to restore
//         hiniProfile - specifies the profile to restore from
//         pchAppl - specifies the application name to restore from
//         pchKey - specifies the key name to restore from
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnWinSavePosToBuffer(HWND hwndWindow,PCWPOSITION pcpPosition);
//-------------------------------------------------------------------------
// This function saves the current position of the specified window in the
// specified buffer
//
// Input:  hwndWindow - specifies the window to save
// Output:  pcpPosition - points to the variable containing the size and
//                        position data
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------

BOOL EXPENTRY CmnWinRestorePosFromBuffer(HWND hwndWindow,
                                         PCWPOSITION pcpPosition);
//-------------------------------------------------------------------------
// This function restores the current position of the specified window from the
// specified buffer
//
// Input:  hwndWindow - specifies the window to restore
//         pcpPosition - points to the variable containing the size and
//                       position data
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
#endif                           // #ifdef INCL_CMNWIN

#ifdef __cplusplus
}
#endif                           // #ifdef __cplusplus

#pragma pack()

#endif                           // #ifndef COMMONLIB_INCLUDED
