///////////////////////////////////////////////////////////////////////////////
// Copyright (c) 1994 Paul Long All rights reserved.
// CompuServe: 72607,1506
//
// cyclo.prg v1.2
//
// This Clipper program displays the following metrics for each module in a
// file along with file-wide totals, averages, minimums, and maximums:
//
//    McCabe's cyclomatic complexity
//    Lines
//    Commented lines
//    Lines of code
//
// What the stats mean
// -------------------
// A cyclomatic complexity index (CCI) greater than 10 generally indicates
// that a module is too complex.  The number of lines in a module is from the
// module introducer, e.g., "function," to the last line with code on it
// before the next module introducer.  The same goes for the number of lines
// of code in a module, except that it ignores blank lines and lines with only
// comments on them.  All comments within and before a module are counted for
// that module.  Any comments after the last module are included in the total
// but do not contribute to the module average because they don't "belong" to
// any module.
//
// Executing it
// ------------
// The name of the file is passed on the command line along with an optional
// /n that suppresses processing of the procedure at the beginning of the file
// with the same name as the file.
//
// .PRG versus .PPO: you decide
// ----------------------------
// cyclo can process a .PRG file, which may contain preprocessor directives and
// comments.  However, it is safest to run it against the output of the Clipper
// preprocessor (.PPO files).  This is because macros could confuse cyclo,
// causing it to calculate the wrong complexities.  For example, cyclo
// would incorrectly count the "for" in "locate for" as a decision point.
// The Clipper preprocessor converts this and other macros to "real" code.
// Also, because cyclo skips all forms of comments, the "*" multiplicative
// operator could be mistaken for the "*" comment introducer, causing
// cyclo to miss some code.  Consider the following:
//   a := 3 ;
//     * b; if c == 0; doIt(); endif
// In this case, cyclo would miss the if statement because it looked like it
// was part of a comment.  If you run this code through the preprocessor,
// cyclo would see this:
//   a := 3 * b; if c == 0; doIt(); endif
// and not miss the if statement (nice try, though <g>).
//
// In general, it's okay to run cyclo against your .PRG files if you know that
// they don't contain the multiplicative operator, "*", at the beginning of a
// line and they don't contain references to any macros that may erroneously
// look like a decision-point keyword, e.g. "for" in "locate for".  Otherwise,
// run the preprocessor and cyclo like this:
//   clipper my.prg /p
//   cyclo my.ppo
//
// However, running against the output of the preprocessor isn't ideal, either.
// The problem occurs with macros that generate a different number of
// decision points than the macro represents to the user.  For example, I've
// seen a repeat-until macro that should represent one decision point; however,
// it generates a do-while and a nested if, which would be two decision points.
// In the .PRG, cyclo wouldn't recognize the construct at all because it's not
// normal Clipper syntax; in the .PPO, cyclo would incorrectly count two
// decision points; in reality, cyclo should count it as one decision point.
// There is no general solution because with macros one can extend and modify
// the language.  cyclo would have to recognize any macro that a user might
// define and understand how many decision points it represents--an impossible
// task.  Like any tool, use it with a great deal of common sense.
//
// Table-driven
// ------------
// If you have any other keywords that you want counted as decision points,
// add them to the acDecisionKeywords array and rebuild.  Likewise, if you for
// some reason have other module-introducer keywords, add them to
// acModKeywords.
//
// S p  e   e    d
// ---------------
// Note that cyclo is rather slow.  This is in part because only one character
// at a time is read from the file.  I did this to simplify the implementation
// and because I didn't want to require the use of a particular third-party
// library to read the file.  To speed it up, you could, for example, use
// Funcky's freadline() or do your own input buffering.  Clipper is not a great
// language for writing software-development tools.
//
// Building cyclo
// --------------
// I built cyclo with the following commands:
//   clipper cyclo /l
//   blinker fi cyclo incremental blinker off
//
// History
// -------
// (no version #) Original version
// v1.1           Parse comments so that preprocessing may not be necessary.
//                Use round() instead of int() for the average.
//                Make keyword comparisons table driven so easier to add more.
// v1.2           Display number of lines, commented lines, and lines of code
//                   for each module, minimum, maximum, average, and total.
//                /n option suppresses processing of first, implicit procedure.
//                   Like the Clipper option.
//
// If you find any bugs or have any questions or suggestions, please send
// e-mail to the address at the top of this file.
//
//
// Disclaimer
// ----------
// THE SOURCE AND EXECUTABLE FILES ("THE SOFTWARE") ARE NOT WARRANTED IN
// ANY WAY TO BE SUITABLE FOR YOUR SITUATION. YOU USE THE SOFTWARE ENTIRELY
// AT YOUR OWN RISK.
//
//
// Licensing agreement
// -------------------
// The executable and included accessory files ("the software") are the
// copyrighted work of their author, Paul Long. All descendant
// manifestations of the software are also the copyright of Paul Long. All
// rights under US and international copyright law are reserved. You are
// hereby granted a limited license at no charge to use the software and to
// make copies of and distribute said copies of the software, as long as:
//
// 1. Such use is not primarily for profit.  "For profit" use is defined as
// primary use of the program whereby the user accrues financial benefit or
// gain directly from use of the software; for example, a consultant
// performing code analysis for a client with the software or a product
// that is descendant from the software. "Primary use" is defined as the
// major use of the program, or the primary reason for acquiring and using
// the program.
//
// 2. Such copies maintain the complete set of files as provided in the
// original distribution set, with no changes, deletions or additions. The
// archive storage format may be changed as long as the rest of this
// condition is met.
//
// 3. Copies are not distributed by any person, group or business that has
// as its primary purpose the distribution of free and "shareware" software
// by any means magnetic, electronic or in print, for profit. BBS
// distribution is allowed as long as no fee is charged specifically for
// this software. Bona fide non-profit user's groups, clubs and other
// organizations may copy and distribute the software as long as no charge
// is made for such service beyond a nominal disk/duplication fee not to
// exceed $5.00. For-profit organizations or businesses wishing to
// distribute the software must contact the author for licensing
// agreements.

#include "common.ch"
#include "fileio.ch"
#include "inkey.ch"

// (Ungodly large, isn't it?)
#define MAX_NUMERIC  999999999999999999999999999999999999999999

parameter cFile, cOpt1

// Statistics for all modules (functions and procedures).
public nMods               // Number of modules.
public nTotCCI             // Total, minimum, maximum CCI.
public nMinCCI
public nMaxCCI
public nTotLines           // Total, minimum, maximum number of lines.
public nMinLines
public nMaxLines
public nTotCmntdLines      // Total, minimum, maximum number of commented lines.
public nMinCmntdLines
public nMaxCmntdLines
public nTotCodeLines       // Total, minimum, maximum number of lines of code.
public nMinCodeLines
public nMaxCodeLines

// Statistics for the file.  Some comments and code appear outside of a module.
public nFileCmntdLines     // Number of commented lines.
public nFileCodeLines      // Number of lines of code.

// Working counters.  These are bumped at the end of lines, summed into
// other variables then reset.
public nCmntdLines         // Commented-lines counter.
public nCodeLines          // Lines-of-code counter.

// Whether any comments or code was found on a line.  Cleared at end-of-line.
public lComment, lCode

// Where module starts (e.g., "function") and ends (last line of code).
public nModStartsOn, nModEndsOn

public nLastLexOn          // Where last (code) lexeme was encountered.
public nThisModCmntdLines  // Number of commented lines in current module.

public cModName            // Module name.
public nDecisions          // Number of binary decision points in a module.
public cLexBuf             // Where lexeme is accumulated in by input().
public cPutBack            // Put-back buffer where unput() puts characters.
public nHandle             // Handle of input file.
public lStartOfLine        // Whether first character of a line. Set by input().
public nLineNo             // Current line number.
public lSuppressFileProc   // Suppress processing implicit file procedure?


?? "cyclo v1.2  Copyright (c) 1994 Paul Long All rights reserved."

// Validate comand line.  File name required.  "/n" optional.
if valtype(cFile) != "C" .or. (valtype(cOpt1) != "U" .and. ;
      (valtype(cOpt1) != "C" .or. lower(cOpt1) != "/n"))
   ? "Usage: cyclo <fileName> [/n]"
else
   // Attempt to open input file.
   if ferror(nHandle := fopen(cFile, FO_READ)) != 0
      ? "Cannot open file, " + cFile
   else
      // Initialize public variables then analyze input file and print stats.
      startOfFile()
      analyzeFile()
   endif
endif


////////////////////////////////////////////////////////////////////////////////
// This function scans the input file, collecting statistics and printing them
// out along the way.
static function analyzeFile()

static cLex                      // Lexeme returned into here from nextLex()
static cLowerLex                 // Lower-case version stored in here.
static lEatCase := FALSE         // Whether to consume "case", i.e, after "do".
static lModNameNext := FALSE     // Whether module name is next lexeme.
// Array of decision-point keywords that don't require special handling.
static acDecisionKeywords := { ;
      "while", "whil", "for", "if", "elseif", "elsei", "iif" }
// Array of module-introducer keywords and their abbreviations.
static acModKeywords := { ;
      "function", "functio", "functi", "funct", "func", ;
      "procedure", "procedur", "procedu", "proced", "proce", "proc" }

set(_SET_EXACT, TRUE)            // Need this for exact comparisons by ascan().

startOfMod()                     // First module starts at beginning of file.

?

// Loop for each lexeme, e.g., do_it, for, "[", while, "4", a, asc2bin.
// input() quits when EOF is encountered.
while TRUE
   cLex := nextLex()             // Get next lexeme.
   cLowerLex := lower(cLex)      // Use lower-case version for tests.

   if cLowerLex == "do"
      lEatCase := TRUE           // If "case" follows, don't count as decision.
   else
      // If "function" or "procedure" previous, this must be module name.
      if lModNameNext
         cModName := cLex        // Save module name for report.
         lModNameNext := FALSE   // Stop looking for module name.

      // Decision points and their abbreviations.
      elseif ascan(acDecisionKeywords, cLowerLex) != 0
         ++nDecisions
      // Special handling to distinguish between "do case" and "case".
      elseif cLowerLex == "case"
         // Only count "case" as decision point if not part of "do case".
         if !lEatCase
            ++nDecisions
         endif

      // Various module introducers.
      elseif ascan(acModKeywords, cLowerLex) != 0
         // End preceding module, start this one, and look for module name.
         endOfMod()
         startOfMod()
         lModNameNext := TRUE

      else
         // The last time this is set before the next module starts is where
         // this module ends.  (xBase has no clean end-of-module lexeme like
         // C.  A module ends only when another starts or the end of file is
         // encountered.)
         nLastLexOn := nLineNo

      endif

      lEatCase := FALSE          // Stop looking for "case" after "do".
   endif
enddo

return


////////////////////////////////////////////////////////////////////////////////
// This function returns the next lexeme from the file.  When it reaches
// end-of-file, input() displays the statistics for the last module and then
// quits. If you're curious, this is a Deterministic Finite Automaton (DFA).
// input() is called to read in the next character and automatically build up
// the lexeme in cLexBuf.  (This function should be partitioned further, but
// since this is the bottle-neck of the program, I decided not to so as not
// to slow things down even more with more function calls.)
static function nextLex()

local cChar

// Loop until a (non-whitespace) lexeme is scanned into cLexBuf.  Comments are
// considered whitespace.
while TRUE
   cLexBuf := ""                 // Start accumulating new lexeme
   // If first character of identifier, scan in the rest.
   if (isalpha(cChar := input())) .or. cChar == "_"
      while isalpha(cChar := input()) .or. isdigit(cChar) .or. cChar == "_"
      enddo
      unput(cChar)               // Put back character terminating identifier.
      lCode := TRUE              // Indicate that code found on this line.
      exit                       // Leave loop to return identifier.

   // Look for literal strings so we don't accidentally find identifiers in
   // them, such as "Do function while for is true."  Since we know that the
   // caller doesn't need the entire literal-string lexeme, don't accumulate the
   // rest of the string.  Just pass back the introducer character, e.g., '"' by
   // itself.  This should speed things up a little.  God knows we need it.
   elseif cChar == '"'
      while input(FALSE) != '"'; enddo
      lCode := TRUE              // Indicate that code found on this line.
      exit
   elseif cChar == "'"
      while input(FALSE) != "'"; enddo
      lCode := TRUE              // Indicate that code found on this line.
      exit
   elseif cChar == "["
      while input(FALSE) != "]"; enddo
      lCode := TRUE  // Indicate that code found on this line.
      exit

   // Look for comments.  Don't build them up in cLexBuf because they are not
   // returned as a lexeme.
   elseif cChar == "/"
      // Good, old-fashioned C comment?
      if (cChar := input(FALSE)) == "*"
         lComment := TRUE        // Indicate that comment found on this line.
         while TRUE
            // Possible end of comment?
            if (cChar := input(FALSE)) == "*"
               if (cChar := input(FALSE)) == "/"
                  // Yes, it is!
                  exit
               else
                  // Oh, sorry, not end of comment. Need to put back because
                  // of the following situation: "**/". Think about it.
                  unput(cChar)
               endif
            endif
         enddo
      // Fashionable, C++ comment?
      elseif cChar == "/"
         lComment := TRUE        // Indicate that comment found on this line.
         // Consume all until end-of-line.
         while (cChar := input(FALSE)) != chr(K_CTRL_J); enddo
      else
         // Oops, wasn't a comment after all. Put back the last two characters.
         unput(cChar)
         lCode := TRUE           // Indicate that code found on this line.
      endif
   // Old-fashioned Clipper comment?
   elseif cChar == "&"
      if (cChar := input(FALSE)) == "&"
         lComment := TRUE        // Indicate that comment found on this line.
         // Consume all until end-of-line.
         while input(FALSE) != chr(K_CTRL_J); enddo
      else
         // Oops, wasn't a comment after all. Put both "&"s back.
         unput(cChar)
         lCode := TRUE           // Indicate that code found on this line.
      endif
   // Nasty, old-fashioned Clipper comment?  Look for optional whitespace at
   // beginning of line then "*".
   elseif lStartOfLine
      while cChar == " " .or. cChar == chr(K_TAB); cChar := input(FALSE); enddo
      if cChar == "*"
         lComment := TRUE        // Indicate that comment found on this line.
         // Consume all until end-of-line.
         while input(FALSE) != chr(K_CTRL_J); enddo
      else
         // Wrong again--wasn't a comment after all.
         unput(cChar)
      endif

   // Throw away non-graphic characters, e.g., spaces, tabs, carriage returns,
   // and line feeds.
   elseif asc(cChar) < 33 .or. asc(cChar) > 126

   // Everything else is considered a single-character lexeme, so return it.
   else
      lCode := TRUE              // Indicate that code found on this line.
      exit

   endif
enddo

return cLexBuf


////////////////////////////////////////////////////////////////////////////////
// This function returns the next character from the file pointed to by nHandle.
// However, if the put-back buffer has something in it, this function returns
// the character that was most recently placed in the buffer instead.
static function input(lBuildLexeme)

local cRetChar                      // Returned character.
static cPrevChar := chr(K_CTRL_J)   // Act like line already read at start.

default lBuildLexeme to TRUE        // Default is accumulate lexeme in buffer.

// 0 string indicates previous end-of-file.  Display complexity for last module
// and quit, returning back to operating system.
if cPrevChar == chr(0)
   // Count comments at end of file, but not as part of the last module.
   nFileCmntdLines += nCmntdLines
   nCmntdLines := 0

   endOfMod()
   endOfFile()
endif

// Anything in put-back buffer? (Would have been placed there by unput().)
if len(cPutBack) > 0
   // Get character most recently placed in put-back buffer.
   cRetChar := substr(cPutBack, 1, 1)
   cPutBack := substr(cPutBack, 2)
else
   // Get 1 character from the file at a time.  (Slow, but straightforward.)
   cRetChar := freadstr(nHandle, 1)
endif

if cRetChar == chr(K_CTRL_J)     // End of a line?
   ++nLineNo                     // Bump the line counter.
   if lComment                   // See whether comment was on this line.
      ++nCmntdLines              // Bump the commented-lines counter.
      lComment := FALSE
   endif
   if lCode                      // Was there any code on this line?
      ++nCodeLines               // Bump lines-of-code counter.

      // Count commented lines since last code encountered.
      nThisModCmntdLines += nCmntdLines
      nCmntdLines := 0

      nModEndsOn := nLastLexOn   // This could be last line of module, so save.

      lCode := FALSE
   endif
elseif cRetChar == ""            // End of file?  freadstr() returns this if so.
   // Pass back a dummy character this one, last time in case a lexeme is in
   // progress.  We will catch this next time at beginning of function.
   cRetChar := chr(0)
endif

// Indicate whether this is the first character of a line (last character was
// a line feed).
lStartOfLine := cPrevChar == chr(K_CTRL_J)
cPrevChar := cRetChar

// Append new character to cLexBuf unless caller explicitly indicated not to.
if lBuildLexeme
   cLexBuf += cRetChar
endif

return cRetChar


////////////////////////////////////////////////////////////////////////////////
// This function puts a character into the put-back buffer and removes the last
// character from cLexBuf.  One could put back a character other than the one
// last scanned in by input(), but that isn't very common.
static function unput(cChar)

cPutBack := cChar + cPutBack
cLexBuf := substr(cLexBuf, 1, len(cLexBuf) - 1)

return


////////////////////////////////////////////////////////////////////////////////
// This function resets decision-point, number-of-commented-lines, and
// code-lines counts, and remembers what line number this module started on.
static function startOfMod()

nDecisions := 0
nThisModCmntdLines := 0
nCodeLines := 0
nModStartsOn := nLineNo          // (First module starts on first line.)

return


////////////////////////////////////////////////////////////////////////////////
// This function displays the name of the current module along with the
// accumulated statistics.  It also maintains file-wide statistics for later
// display by endOfFile().
static function endOfMod()

local nLines, nCCI

// If /n option specified on command line and this is the first module,
// don't print or use statistics for this module.
if !lSuppressFileProc .or. nMods > 0

   if nModEndsOn < nModStartsOn     // Just to be safe.
      nModEndsOn := nModStartsOn
   endif

   // Calculate number of lines in module.
   nLines := nModEndsOn - nModStartsOn + 1
   nCCI := nDecisions + 1

   ? cModName
   ? "  CCI:        " + ltrimStr(nCCI)
   ? "  Lines:      " + ltrimStr(nLines)
   ? "  w/code:     " + ltrimStr(nCodeLines)
   ? "  w/comments: " + ltrimStr(nThisModCmntdLines)

   // Accumulate statistics for all modules.
   nTotCCI += nCCI
   nTotLines += nLines
   nTotCmntdLines += nThisModCmntdLines
   nTotCodeLines += nCodeLines

   // Calculate minimums and maximums.
   if nCCI > nMaxCCI
      nMaxCCI := nCCI
   endif
   if nCCI < nMinCCI
      nMinCCI := nCCI
   endif

   if nLines > nMaxLines
      nMaxLines := nLines
   endif
   if nLines < nMinLines
      nMinLines := nLines
   endif

   if nThisModCmntdLines > nMaxCmntdLines
      nMaxCmntdLines := nThisModCmntdLines
   endif
   if nThisModCmntdLines < nMinCmntdLines
      nMinCmntdLines := nThisModCmntdLines
   endif

   if nCodeLines > nMaxCodeLines
      nMaxCodeLines := nCodeLines
   endif
   if nCodeLines < nMinCodeLines
      nMinCodeLines := nCodeLines
   endif
endif

// Accumulate statistics for file.
nFileCmntdLines += nThisModCmntdLines
nFileCodeLines += nCodeLines

++nMods

return


////////////////////////////////////////////////////////////////////////////////
// This function initializes the public variables and is called immediately
// before the file is processed.
static function startOfFile()

cModName := cFile          // First module is always name of input file.
cPutBack := ""
nTotCCI := 0
nMods := 0
nTotLines := 0
nModEndsOn := nLastLexOn := nLineNo := 1
lComment := FALSE
lCode := FALSE
nCmntdLines := 0
nTotCmntdLines := 0
nTotCodeLines := 0
nFileCmntdLines := 0
nFileCodeLines := 0
lSuppressFileProc := valtype(cOpt1) == "C" .and. lower(cOpt1) == "/n"

// Initialize minimums and maximums.
nMinCCI := nMinLines := ;
      nMinCmntdLines := nMinCodeLines := MAX_NUMERIC
nMaxCCI := nMaxLines := ;
      nMaxCmntdLines := nMaxCodeLines := -1

return


////////////////////////////////////////////////////////////////////////////////
// This function displays the statistics for the file then quits, returning
// back to DOS.
static function endOfFile()

// Correct number-of-modules if the first module was suppressed.
if lSuppressFileProc
   --nMods
endif

// If some statistics never set, give values that won't look funny.
if nMinCCI == MAX_NUMERIC
   nMinCCI := 0
endif
if nMaxCCI == -1
   nMaxCCI := 0
endif
if nMinLines == MAX_NUMERIC
   nMinLines := 0
endif
if nMaxLines == -1
   nMaxLines := 0
endif
if nMinCmntdLines == MAX_NUMERIC
   nMinCmntdLines := 0
endif
if nMaxCmntdLines == -1
   nMaxCmntdLines := 0
endif
if nMinCodeLines == MAX_NUMERIC
   nMinCodeLines := 0
endif
if nMaxCodeLines == -1
   nMaxCodeLines := 0
endif

?
? "File Summary  (<min, >max, ~avg, total)"
? "  CCI:        <" + ltrimStr(nMinCCI) + ;
      " >" + ltrimStr(nMaxCCI) + ;
      " ~" + ltrimStr(round(nTotCCI / nMods, 0))
? "  Lines:      <" + ltrimStr(nMinLines) + ;
      " >" + ltrimStr(nMaxLines) + ;
      " ~" + ltrimStr(round(nTotLines / nMods, 0)) + ;
      " " + ltrimStr(nLineNo - 1)
? "  w/code:     <" + ltrimStr(nMinCodeLines) + ;
      " >" + ltrimStr(nMaxCodeLines) + ;
      " ~" + ltrimStr(round(nTotCodeLines / nMods, 0)) + ;
      " " + ltrimStr(nFileCodeLines)
? "  w/comments: <" + ltrimStr(nMinCmntdLines) + ;
      " >" + ltrimStr(nMaxCmntdLines) + ;
      " ~" + ltrimStr(round(nTotCmntdLines / nMods, 0)) + ;
      " " + ltrimStr(nFileCmntdLines)
? "  Modules:    " + ltrimStr(nMods)

quit

return


////////////////////////////////////////////////////////////////////////////////
// This function returns a numeric value as a left-trimmed string.
static function ltrimStr(nValue)
return ltrim(str(nValue))
