/*
 *  CDPlayer uses Windows's Media Control Interface (MCI) to play audio
 *  CDs in CD-ROM drives.
 */

#include <windows.h>
#include <mmsystem.h>
#include <stdlib.h>
#include "cdplayer.h"

long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL AboutDlgProc (HWND, WORD, WORD, LONG);

void BeginPlay (HWND, BYTE);
void UpdateTrackButtons (HWND);
char *IntToMinSec (int, char *);
char *TMSFtoMinSec (DWORD, char *);
void InitProgram (HWND);
void InitFrameRect (HWND, int, RECT *);
void DrawFrame (HDC, RECT *, HBRUSH);
void SaveSettings (HWND);

MCI_STATUS_PARMS MCIStatus;                 // MCI_STATUS structure
MCI_OPEN_PARMS MCIOpen;                     // MCI_OPEN structure
MCI_PLAY_PARMS MCIPlay;                     // MCI_PLAY structure
MCI_SET_PARMS MCISet;                       // MCI_SET structure

BOOL bMediaPresent = FALSE;                 // Media status flag
BOOL bScrolling = FALSE;                    // Scroll flag
BOOL bPlaying = FALSE;                      // Playing flag
BOOL bPaused = FALSE;                       // Pause flag

int nTrackCount;                            // Number of tracks
int nCurrentTrack = 0;                      // Current track number
int nTrackLength[99];                       // Track lengths (in seconds)
int nPageNumber = 0;                        // Current page number
int nPageCount = 1;                         // Number of pages

UINT nTimerID;                              // Timer ID
HWND hwndTrack, hwndLength, hwndStatus;     // Window handles
HWND hwndTime, hwndScrollBar;               // More window handles
RECT rectFrame[4];                          // Frame coordinates
HMENU hSysMenu;                             // System menu handle

BOOL bContinuousPlay = FALSE;               // Continuous play flag
BOOL bSleepWhileMin = FALSE;                // Sleep while minimized flag
BOOL bAlwaysOnTop = FALSE;                  // Always on Top flag

int nStatus = 255;                          // Current status
char *szStatusMsg[] = { "Empty",            // nStatus == 0
                        "Ready",            // nStatus == 1
                        "Playing",          // nStatus == 2
                        "Paused"  };        // nStatus == 3

char szIniFile[128];                        // INI file name
char szSection[] = "Settings";              // INI file section name
char *szEntry[] = { "SleepWhileMin",        // INI file entry names
                    "AlwaysOnTop",
                    "ContinuousPlay",
                    "WindowPos" };

char *szError[] = { "Error",
                    "Unable to open the CD Audio device",
                    "Unable to allocate a timer" };

/*
 *  Function WinMain.
 */

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    static char szAppName[] = "CDPlayer";
    WNDCLASS wndclass;
    HWND hwnd;
    MSG msg;

    if (!hPrevInstance) {
        wndclass.style = 0;
        wndclass.lpfnWndProc = (WNDPROC) WndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = DLGWINDOWEXTRA;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon (hInstance, szAppName);
        wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
        wndclass.hbrBackground = NULL;
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szAppName;

        RegisterClass (&wndclass);
    }

    hwnd = CreateDialog (hInstance, szAppName, 0 , NULL);

    InitProgram (hwnd);

    ShowWindow (hwnd, nCmdShow == SW_SHOWMAXIMIZED ? SW_SHOW : nCmdShow);
    UpdateWindow (hwnd);

    while (GetMessage (&msg, NULL, 0, 0))
        if ((hwnd == 0) || (!IsDialogMessage (hwnd, &msg)))
            DispatchMessage (&msg);

    return msg.wParam;
}

/*
 *  WndProc processes messages to the main window.
 */

long FAR PASCAL WndProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    char szBuffer[32];
    static HINSTANCE hInstance;
    static HBRUSH hPatternBrush, hGrayBrush;
    FARPROC lpfnAboutDlgProc;
    int nThumbPos, nCtrlID;
    HBITMAP hBitmap;
    PAINTSTRUCT ps;
    POINT point;
    RECT rect;
    HDC hdc;
    WORD i;

    switch (message) {

    case WM_CREATE:
        //
        // Formulate the path to CDPLAYER.INI, create the GDI brushes used
        // to paint the window background, open and initialize the CD Audio
        // device, allocate a timer, and customize the system menu.
        //
        hInstance = ((LPCREATESTRUCT) lParam)->hInstance;
        i = GetModuleFileName (hInstance, szIniFile, sizeof (szIniFile));
        while (szIniFile[i] != 0x5C)
            i--;
        lstrcpy (&szIniFile[i+1], "CDPLAYER.INI");

        MCIOpen.lpstrDeviceType = (LPCSTR) MCI_DEVTYPE_CD_AUDIO;
        if (mciSendCommand (NULL, MCI_OPEN, MCI_OPEN_TYPE | MCI_OPEN_TYPE_ID,
            (DWORD) (LPVOID) &MCIOpen))

            if (mciSendCommand (NULL, MCI_OPEN, MCI_OPEN_TYPE |
                MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE,
                (DWORD) (LPVOID) &MCIOpen)) {

                MessageBeep (MB_ICONEXCLAMATION);
                MessageBox (hwnd, szError[1], szError[0],
                    MB_ICONEXCLAMATION | MB_OK);
                return -1;
            }

        MCISet.dwTimeFormat = MCI_FORMAT_TMSF;
        mciSendCommand (MCIOpen.wDeviceID, MCI_SET, MCI_SET_TIME_FORMAT,
            (DWORD) (LPVOID) &MCISet);

        if ((nTimerID = SetTimer (hwnd, 1, 1000, NULL)) == 0) {
            MessageBeep (MB_ICONEXCLAMATION);
            MessageBox (hwnd, szError[2], szError[0],
                MB_ICONEXCLAMATION | MB_OK);
            mciSendCommand (MCIOpen.wDeviceID, MCI_CLOSE, NULL, NULL);
            return -1;
        }

        hGrayBrush = CreateSolidBrush (RGB (192, 192, 192));
        hPatternBrush = CreatePatternBrush (hBitmap = LoadBitmap (hInstance,
            "CDPlayer"));
        DeleteObject (hBitmap);

        hSysMenu = GetSystemMenu (hwnd, FALSE);
        DeleteMenu (hSysMenu, SC_SIZE, MF_BYCOMMAND);
        DeleteMenu (hSysMenu, SC_MAXIMIZE, MF_BYCOMMAND);
        AppendMenu (hSysMenu, MF_SEPARATOR, 0, NULL);
        AppendMenu (hSysMenu, MF_STRING, IDM_SLEEP, "&Sleep While Minimized");
        AppendMenu (hSysMenu, MF_STRING, IDM_ALWAYSONTOP, "Always on &Top");
        AppendMenu (hSysMenu, MF_SEPARATOR, 0, NULL);
        AppendMenu (hSysMenu, MF_STRING, IDM_ABOUT, "&About CDPlayer...");
        return 0;

    case WM_PAINT:
        //
        // Draw the frames surrounding the control groups.
        //
        hdc = BeginPaint (hwnd, &ps);
        for (i=0; i<4; i++)
            DrawFrame (hdc, &rectFrame[i], hGrayBrush);
        EndPaint (hwnd, &ps);
        return 0;

    case WM_ERASEBKGND:
        //
        // Fill the window with the pattern stored in CDPLAYER.BMP.
        //
        GetClientRect (hwnd, &rect);
        UnrealizeObject (hPatternBrush);
        point.x = point.y = 0;
        ClientToScreen (hwnd, &point);
        SetBrushOrg ((HDC) wParam, point.x, point.y); 
        FillRect ((HDC) wParam, &rect, hPatternBrush);
        return 1;

    case WM_CTLCOLOR:
        //
        // Customize the control colors.
        //
        nCtrlID = GetDlgCtrlID ((HWND) LOWORD (lParam));

        if (nCtrlID == IDD_SCROLLBAR)
            return NULL;

        if (((nCtrlID >= IDD_TRACK) && (nCtrlID <= IDD_TIME)) ||
            (nCtrlID == IDD_STATUS))
            SetTextColor ((HDC) wParam, RGB (192, 0, 0));

        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        return hGrayBrush;

    case WM_COMMAND:
        //
        // Begin play at the specified track when a track button is clicked.
        //
        if ((wParam >= IDD_1) && (wParam <= IDD_10)) {
            BeginPlay (hwnd, (BYTE) ((wParam - IDD_1 + 1) +
                (nPageNumber * 10)));
            return 0;
        }

        switch (wParam) {

        case IDD_PLAY:
            //
            // Begin play at track 1 when the Play button is clicked.
            //
            if (bMediaPresent)
                BeginPlay (hwnd, 1);
            return 0;

        case IDD_STOP:
            //
            // Stop play when the Stop button is clicked.
            //
            if (bMediaPresent) {
                bPaused = FALSE;
                nCurrentTrack = 0;
                mciSendCommand (MCIOpen.wDeviceID, MCI_STOP, NULL, NULL);
            }
            return 0;

        case IDD_PAUSE:
            //
            // Pause play if the Pause button is clicked and a CD is playing.
            // Resume play if the CD is already paused.
            //
            if (bMediaPresent) {
                if (bPaused) {
                    bPaused = FALSE;
                    MCIPlay.dwCallback = (DWORD) hwnd;
                    mciSendCommand (MCIOpen.wDeviceID, MCI_PLAY, MCI_NOTIFY,
                        (DWORD) (LPVOID) &MCIPlay);
                }
                else {
                    MCIStatus.dwItem = MCI_STATUS_MODE;
                    mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS,
                        MCI_STATUS_ITEM | MCI_WAIT, (DWORD) (LPVOID)
                        &MCIStatus);

                    if (MCIStatus.dwReturn == MCI_MODE_PLAY) {
                        bPaused = TRUE;
                        SetWindowText (hwndStatus, szStatusMsg[3]);
                        mciSendCommand (MCIOpen.wDeviceID, MCI_PAUSE, NULL,
                            NULL);
                    }
                }
            }
            return 0;

        case IDD_EJECT:
            //
            // Eject the CD.
            //
            mciSendCommand (MCIOpen.wDeviceID, MCI_SET, MCI_SET_DOOR_OPEN,
                NULL);
            return 0;

        case IDD_PREVTRACK:
            //
            // Jump to the previous track.
            //
            if (bMediaPresent)
                BeginPlay (hwnd, (BYTE) ((nCurrentTrack <= 1) ? nTrackCount :
                    nCurrentTrack - 1));
            return 0;

        case IDD_NEXTTRACK:
            //
            // Jump to the next track.
            //
            if (bMediaPresent)
                BeginPlay (hwnd, (BYTE) ((nCurrentTrack == nTrackCount) ? 1 :
                    nCurrentTrack + 1));
            return 0;

        case IDD_10PLUS:
            //
            // Add 10 to the numbers displayed on the track buttons, or reset
            // them to 1 through 10.
            //
            nPageNumber = (nPageNumber == nPageCount - 1) ? 0 : nPageNumber++;
            UpdateTrackButtons (hwnd);
            return 0;

        case IDD_REPEAT:
            //
            // Toggle the continuous play flag.
            //
            bContinuousPlay = bContinuousPlay ? FALSE: TRUE;
            return 0;

        case IDD_CLOSE:
            //
            // Terminate the program when the Close button is pressed.
            //
            SendMessage (hwnd, WM_CLOSE, 0, 0L);
            return 0;

        case IDCANCEL:
            //
            // Minimize the window when the Esc key is pressed.
            //
            ShowWindow (hwnd, SW_MINIMIZE);
            return 0;
        }

    case WM_TIMER:
        //
        // Ignore this message if the window is minimized and the sleep while
        // minimized flag is set.
        //
        if (IsIconic (hwnd) && bSleepWhileMin)
            return 0;

        //
        // Determine the program's current status and update the on-screen
        // status indicator if the status has changed since the last timer
        // message.
        //
        MCIStatus.dwItem = MCI_STATUS_MODE;
        mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS, MCI_STATUS_ITEM |
            MCI_WAIT, (DWORD) (LPVOID) &MCIStatus);

        if ((MCIStatus.dwReturn == MCI_MODE_NOT_READY) ||
            (MCIStatus.dwReturn == MCI_MODE_OPEN))
            i = 0;
        else if (MCIStatus.dwReturn == MCI_MODE_PLAY)
            i = 2;
        else if ((MCIStatus.dwReturn == MCI_MODE_STOP) && bPaused)
            i = 3;
        else
            i = 1;

        if (nStatus != i) {
            nStatus = i;
            SetWindowText (hwndStatus, szStatusMsg[nStatus]);
        }

        //
        // Determine whether there is a CD in the drive.
        //
        MCIStatus.dwItem = MCI_STATUS_MEDIA_PRESENT;
        mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS, MCI_STATUS_ITEM |
            MCI_WAIT, (DWORD) (LPVOID) &MCIStatus);

        //
        // Reinitialize the program if a CD was inserted since the last
        // timer message (bMediaPresent is FALSE but there is now a CD in
        // the drive).
        //
        if ((!bMediaPresent) && MCIStatus.dwReturn) {
            bMediaPresent = TRUE;
            bScrolling = FALSE;
            bPaused = FALSE;
            nCurrentTrack = 0;
            nPageNumber = 0;

            //
            // Get the number of tracks and recompute the page count.
            //
            MCIStatus.dwItem = MCI_STATUS_NUMBER_OF_TRACKS;
            mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS, MCI_STATUS_ITEM |
                MCI_WAIT, (DWORD) (LPVOID) &MCIStatus);

            nTrackCount = min ((int) MCIStatus.dwReturn, 99);
            nPageCount = ((nTrackCount - 1) / 10) + 1;

            //
            // Enable the track buttons and, if the CD contains more than 10
            // tracks, the 10+ button.
            //
            if (nTrackCount <= 10)
                for (i=0; i<nTrackCount; i++)
                    EnableWindow (GetDlgItem (hwnd, IDD_1 + i), TRUE);
            else {
                for (i=0; i<10; i++)
                    EnableWindow (GetDlgItem (hwnd, IDD_1 + i), TRUE);
                EnableWindow (GetDlgItem (hwnd, IDD_10PLUS), TRUE);
            }

            //
            // Get the track lengths and store them in the nTrackLength array.
            //
            for (i=0; i<nTrackCount; i++) {
                MCIStatus.dwItem = MCI_STATUS_LENGTH;
                MCIStatus.dwTrack = (DWORD) i + 1;
                mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS,
                    MCI_TRACK | MCI_STATUS_ITEM | MCI_WAIT,
                    (DWORD) (LPVOID) &MCIStatus);

                nTrackLength[i] =
                    ((int) MCI_MSF_MINUTE (MCIStatus.dwReturn) * 60) +
                    (int) MCI_MSF_SECOND (MCIStatus.dwReturn);
            }
        }

        //
        // Reset the track position indicators and disable the track buttons
        // if a CD was removed since the last timer message (bMediaPresent is
        // TRUE but there is no longer a CD in the drive). Then return.
        //
        else if ((bMediaPresent) && !MCIStatus.dwReturn) {
            bMediaPresent = FALSE;
            bPaused = FALSE;

            for (i=0; i<10; i++)
                EnableWindow (GetDlgItem (hwnd, IDD_1 + i), FALSE);
            EnableWindow (GetDlgItem (hwnd, IDD_10PLUS), FALSE);

            if (nPageNumber != 0)
                for (i=0; i<10; i++)
                    SetDlgItemText (hwnd, IDD_1 + i,
                        _itoa (i + 1, szBuffer, 10));

            SetWindowText (hwndTrack, "");
            SetWindowText (hwndLength, "");
            SetWindowText (hwndTime, "");
            SetScrollPos (hwndScrollBar, SB_CTL, 0, TRUE);
            return 0;
        }

        //
        // If a CD is playing, update the track position indicators.
        //
        if (nStatus == 2) {
            MCIStatus.dwItem = MCI_STATUS_POSITION;
            mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS, MCI_STATUS_ITEM |
                MCI_WAIT, (DWORD) (LPVOID) &MCIStatus);

            //
            // Update the track number and track length indicators if this is
            // a new track.
            //
            if (nCurrentTrack != (int) MCI_TMSF_TRACK (MCIStatus.dwReturn)) {
                nCurrentTrack = (int) MCI_TMSF_TRACK (MCIStatus.dwReturn);
                SetWindowText (hwndTrack, _itoa (nCurrentTrack, szBuffer, 10));
                SetWindowText (hwndLength, IntToMinSec
                    (nTrackLength[nCurrentTrack - 1], szBuffer));
                SetScrollRange (hwndScrollBar, SB_CTL, 0,
                    nTrackLength[nCurrentTrack - 1], FALSE);
            }

            //
            // Update the time indicator and the scroll bar position if the
            // user is not using the scroll bar.
            //
            if (!bScrolling) {
                SetWindowText (hwndTime, TMSFtoMinSec (MCIStatus.dwReturn,
                    szBuffer));
                SetScrollPos (hwndScrollBar, SB_CTL,
                    (((int) MCI_TMSF_MINUTE (MCIStatus.dwReturn)) * 60) +
                    (int) MCI_TMSF_SECOND (MCIStatus.dwReturn), TRUE);
            }
        }

        //
        // If a CD is not playing and play is not paused, reset the track
        // position indicators if they're not already reset.
        //
        else if (!bPaused) {
            GetWindowText (hwndTrack, szBuffer, sizeof (szBuffer));
            if (szBuffer[0] != 0) {
                SetWindowText (hwndTrack, "");
                SetWindowText (hwndLength, "");
                SetWindowText (hwndTime, "");
                SetScrollPos (hwndScrollBar, SB_CTL, 0, TRUE);
                nCurrentTrack = 0;
            }
        }
        return 0;

    case WM_HSCROLL:
        //
        // Ignore scroll bar messages if a CD is not playing.
        //
        MCIStatus.dwItem = MCI_STATUS_MODE;
        mciSendCommand (MCIOpen.wDeviceID, MCI_STATUS,
            MCI_STATUS_ITEM | MCI_WAIT, (DWORD) (LPVOID) &MCIStatus);

        if (MCIStatus.dwReturn != MCI_MODE_PLAY)
            break;

        switch (wParam) {

        case SB_LINEUP:
        case SB_PAGEUP:
        case SB_LINEDOWN:
        case SB_PAGEDOWN:
            //
            // Move the scroll bar thumb and update the time indicator when
            // a scroll bar arrow or the scroll bar itself is clicked.
            //
            bScrolling = TRUE;
            nThumbPos = GetScrollPos (hwndScrollBar, SB_CTL);

            if (wParam == SB_LINEUP)
                nThumbPos--;
            else if (wParam == SB_PAGEUP)
                nThumbPos -= 8;
            else if (wParam == SB_LINEDOWN)
                nThumbPos++;
            else // wParam == SB_PAGEDOWN
                nThumbPos += 8;

            if (nThumbPos < 0)
                nThumbPos = 0;
            else if (nThumbPos > nTrackLength[nCurrentTrack - 1])
                nThumbPos = nTrackLength[nCurrentTrack - 1];

            SetWindowText (hwndTime, IntToMinSec (nThumbPos, szBuffer));
            SetScrollPos (hwndScrollBar, SB_CTL, nThumbPos, TRUE);
            return 0;

        case SB_ENDSCROLL:
            //
            // Jump to the location in the current track that corresponds
            // to the new position of the scroll bar thumb when the mouse
            // button is released.
            //
            if (bScrolling) {
                bScrolling = FALSE;
                nThumbPos = GetScrollPos (hwndScrollBar, SB_CTL);

                MCIPlay.dwFrom = MCI_MAKE_TMSF ((BYTE) nCurrentTrack,
                    (BYTE) (nThumbPos / 60), (BYTE) (nThumbPos -
                    ((nThumbPos / 60) * 60)), 30);

                mciSendCommand (MCIOpen.wDeviceID, MCI_PLAY,
                    MCI_FROM | MCI_NOTIFY, (DWORD) (LPVOID) &MCIPlay);
            }
            return 0;

        case SB_THUMBTRACK:
            //
            // Update the time indicator as the scroll bar thumb is dragged.
            //
            bScrolling = TRUE;
            nThumbPos = (int) LOWORD (lParam);
            SetWindowText (hwndTime, IntToMinSec (nThumbPos, szBuffer));
            return 0;

        case SB_THUMBPOSITION:
            //
            // Jump to the location in the current track that corresponds
            // to the new position of the scroll bar thumb when the thumb
            // is released.
            //
            if (bScrolling) {
                bScrolling = FALSE;
                nThumbPos = (int) LOWORD (lParam);
                SetWindowText (hwndTime, IntToMinSec (nThumbPos, szBuffer));
                SetScrollPos (hwndScrollBar, SB_CTL, nThumbPos, TRUE);

                MCIPlay.dwFrom = MCI_MAKE_TMSF ((BYTE) nCurrentTrack,
                    (BYTE) (nThumbPos / 60), (BYTE) (nThumbPos -
                    ((nThumbPos / 60) * 60)), 30);

                mciSendCommand (MCIOpen.wDeviceID, MCI_PLAY,
                    MCI_FROM | MCI_NOTIFY, (DWORD) (LPVOID) &MCIPlay);
            }
            return 0;
        }
        break;

    case MM_MCINOTIFY:
        //
        // Restart the CD if an MM_MCINOTIFY message arrives signaling
        // successful completion of the last play command and the continuous
        // play flag is set.
        //
        if (wParam == MCI_NOTIFY_SUCCESSFUL) {
            if (bContinuousPlay)
                BeginPlay (hwnd, 1);
            else
                nCurrentTrack = 0;
            return 0;
        }
        break;

    case WM_SYSCOMMAND:
        switch (wParam) {

        case IDM_SLEEP:
            //
            // Toggle the Sleep While Minimized option.
            //
            bSleepWhileMin = bSleepWhileMin ? FALSE : TRUE;
            CheckMenuItem (hSysMenu, IDM_SLEEP, bSleepWhileMin ?
                MF_CHECKED : MF_UNCHECKED);
            return 0;

        case IDM_ALWAYSONTOP:
            //
            // Toggle the Always on Top option.
            //
            bAlwaysOnTop = bAlwaysOnTop ? FALSE : TRUE;
            CheckMenuItem (hSysMenu, IDM_ALWAYSONTOP, bAlwaysOnTop ?
                MF_CHECKED : MF_UNCHECKED);
            SetWindowPos (hwnd, bAlwaysOnTop ? HWND_TOPMOST : HWND_NOTOPMOST,
                0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
            return 0;

        case IDM_ABOUT:
            //
            // Display the About CDPlayer dialog box.
            //
            lpfnAboutDlgProc = MakeProcInstance (AboutDlgProc, hInstance);
            DialogBox (hInstance, "AboutBox", hwnd, lpfnAboutDlgProc);
            FreeProcInstance (lpfnAboutDlgProc);
            return 0;
        }
        break;

    case WM_CLOSE:
        //
        // Save program settings to CDPLAYER.INI and call DestroyWindow to
        // terminate the program.
        //
        SaveSettings (hwnd);
        DestroyWindow (hwnd);
        return 0;

    case WM_ENDSESSION:
        //
        // Save program settings to CDPLAYER.INI before terminating.
        //
        if (wParam)
            SaveSettings (hwnd);
        return 0;

    case WM_DESTROY:
        //
        // Close the CD Audio device, kill the timer, deallocate GDI objects,
        // and terminate.
        //
        mciSendCommand (MCIOpen.wDeviceID, MCI_CLOSE, NULL, NULL);
        KillTimer (hwnd, nTimerID);
        DeleteObject (hGrayBrush);
        DeleteObject (hPatternBrush);
        PostQuitMessage (0);
        return 0;
    }
    return DefDlgProc (hwnd, message, wParam, lParam);
}

/*
 *  InitProgram initializes the program.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void InitProgram (HWND hwnd)
{
    int cxWindowPos, cyWindowPos, i;
    char szBuffer[32];
    RECT rect;

    //
    // Save the scroll bar's window handle and the handles of the windows
    // that display the status, track number, track length, and time.
    //
    hwndScrollBar = GetDlgItem (hwnd, IDD_SCROLLBAR);
    hwndTrack = GetDlgItem (hwnd, IDD_TRACK);
    hwndLength = GetDlgItem (hwnd, IDD_LENGTH);
    hwndTime = GetDlgItem (hwnd, IDD_TIME);
    hwndStatus = GetDlgItem (hwnd, IDD_STATUS);

    //
    // Destroy the text windows framing the control groups but save their
    // window coordinates.
    //
    for (i=0; i<4; i++)
        InitFrameRect (hwnd, IDD_RECT1 + i, &rectFrame[i]);

    //
    // Read program settings from CDPLAYER.INI.
    //
    if (GetPrivateProfileInt (szSection, szEntry[0], 0, szIniFile)) {
        bSleepWhileMin = TRUE;
        CheckMenuItem (hSysMenu, IDM_SLEEP, MF_CHECKED);
    }

    if (GetPrivateProfileInt (szSection, szEntry[1], 0, szIniFile)) {
        bAlwaysOnTop = TRUE;
        CheckMenuItem (hSysMenu, IDM_ALWAYSONTOP, MF_CHECKED);
        SetWindowPos (hwnd, HWND_TOPMOST, 0, 0, 0, 0,
            SWP_NOMOVE | SWP_NOSIZE);
    }

    if (GetPrivateProfileInt (szSection, szEntry[2], 0, szIniFile)) {
        bContinuousPlay = TRUE;
        SendMessage (GetDlgItem (hwnd, IDD_REPEAT), BM_SETCHECK, 1, 0L);
    }

    if (GetPrivateProfileString (szSection, szEntry[3], "", szBuffer,
        sizeof (szBuffer), szIniFile))
        sscanf (szBuffer, "%u,%u", &cxWindowPos, &cyWindowPos);
    else {
        GetWindowRect (hwnd, &rect);
        cxWindowPos = (GetSystemMetrics (SM_CXSCREEN) -
            (rect.right - rect.left)) >> 1;
        cyWindowPos = (GetSystemMetrics (SM_CYSCREEN) -
            (rect.bottom - rect.top)) >> 1;
    }
    SetWindowPos (hwnd, 0, cxWindowPos, cyWindowPos, 0, 0,
        SWP_NOSIZE | SWP_NOZORDER);
}

/*
 *  InitFrameRect saves the coordinates of the specified control window in
 *  a RECT structure, and then destroys the control window.  
 *
 *  Input parameters:
 *
 *    hwnd = Handle of the control's parent window
 *    nID = Control ID
 *    rect = Pointer to RECT structure that will receive the control's
 *           coordinates
 *
 *  Returns:
 *
 *    Nothing
 */

void InitFrameRect (HWND hwnd, int nID, RECT *rect)
{
    GetWindowRect (GetDlgItem (hwnd, nID), rect);
    ScreenToClient (hwnd, (POINT FAR *) &rect->left);
    ScreenToClient (hwnd, (POINT FAR *) &rect->right);
    DestroyWindow (GetDlgItem (hwnd, nID));
}

/*
 *  DrawFrame draws a recessed frame at the specified location. The frame's
 *  background is painted with the specified brush.
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    rect = Pointer to RECT structure containing the frame's coordinates
 *    hBrush = Handle of brush used to paint the frame's background
 *
 *  Returns:
 *
 *    Nothing
 */

void DrawFrame (HDC hdc, RECT *rect, HBRUSH hBrush)
{
    HPEN hPen;

    FillRect (hdc, rect, hBrush); 
    hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
    SelectObject (hdc, hPen);
    MoveTo (hdc, rect->left, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->top);
    LineTo (hdc, rect->right, rect->top);
    DeleteObject (SelectObject (hdc, GetStockObject (WHITE_PEN)));
    MoveTo (hdc, rect->right - 1, rect->top + 1);
    LineTo (hdc, rect->right - 1, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->bottom - 1);
}

/*
 *  SaveSettings saves program settings to CDPLAYER.INI.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void SaveSettings (HWND hwnd)
{
    RECT rect;
    WINDOWPLACEMENT wp;
    char szBuffer[32];

    WritePrivateProfileString (szSection, szEntry[0],
        _itoa (bSleepWhileMin ? 1 : 0, szBuffer, 10), szIniFile);

    WritePrivateProfileString (szSection, szEntry[1],
        _itoa (bAlwaysOnTop ? 1 : 0, szBuffer, 10), szIniFile);

    WritePrivateProfileString (szSection, szEntry[2],
        _itoa (bContinuousPlay ? 1 : 0, szBuffer, 10), szIniFile);

    if (!IsIconic (hwnd)) {
        GetWindowRect (hwnd, &rect);
        wsprintf (szBuffer, "%u,%u", rect.left, rect.top);
    }
    else {
        wp.length = sizeof (wp);
        GetWindowPlacement (hwnd, &wp);
        wsprintf (szBuffer, "%u,%u", (wp.rcNormalPosition).left,
            (wp.rcNormalPosition).top);
    }
    WritePrivateProfileString (szSection, szEntry[3], szBuffer, szIniFile);
}

/*
 *  BeginPlay starts play at the specified track.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    byTrackNumber = Track number (1 to 99)
 *
 *  Returns:
 *
 *    Nothing
 */

void BeginPlay (HWND hwnd, BYTE byTrackNumber)
{
    char szBuffer[8];

    bPaused = FALSE;
    nCurrentTrack = (int) byTrackNumber;

    //
    // Update the track, length, and time indicators.
    //
    SetWindowText (hwndTrack, _itoa ((int) byTrackNumber, szBuffer, 10));
    SetWindowText (hwndLength, IntToMinSec (nTrackLength[byTrackNumber - 1],
        szBuffer));
    SetWindowText (hwndTime, "0:00");

    //
    // Initialize the scroll bar for the current track.
    //
    SetScrollRange (hwndScrollBar, SB_CTL, 0,
        nTrackLength[byTrackNumber - 1], FALSE);
    SetScrollPos (hwndScrollBar, SB_CTL, 0, TRUE);

    //
    // Send an MCI_PLAY command to the CD Audio driver.
    //
    MCIPlay.dwCallback = (DWORD) hwnd;
    MCIPlay.dwFrom = MCI_MAKE_TMSF (byTrackNumber, 0, 0, 0);
    mciSendCommand (MCIOpen.wDeviceID, MCI_PLAY, MCI_FROM | MCI_NOTIFY,
        (DWORD) (LPVOID) &MCIPlay);
}

/*
 *  UpdateTrackButtons updates the numbers on the track button display for
 *  the current page.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void UpdateTrackButtons (HWND hwnd)
{
    int nBase, i;
    char szBuffer[3];

    //
    // Change the numbers on the button faces.
    //
    nBase = nPageNumber * 10;
    for (i=0; i<10; i++)
        SetDlgItemText (hwnd, IDD_1 + i, _itoa (i + nBase + 1, szBuffer, 10));

    //
    // If this is the final page and the number of tracks is not a multiple of
    // 10, disable some of the track buttons.
    //
    if ((nPageNumber == (nPageCount - 1)) && (nTrackCount % 10))
        for (i=(nTrackCount % 10); i<10; i++)
            EnableWindow (GetDlgItem (hwnd, IDD_1 + i), FALSE);
    //
    // Otherwise, enable all the track buttons.
    //
    else
        for (i=0; i<10; i++)
            EnableWindow (GetDlgItem (hwnd, IDD_1 + i), TRUE);
}

/*
 *  IntToMinSec converts a time value expressed in seconds to a string in
 *  minutes:seconds format.
 *
 *  Input parameters:
 *
 *    nTime = Time in seconds
 *    szResult = Pointer to results buffer
 *
 *  Returns:
 *
 *    Pointer to results buffer (szBuffer)
 */

char *IntToMinSec (int nTime, char *szResult)
{
    int nSeconds;
    char szBuffer[4];

    _itoa (nTime / 60, szResult, 10);
    lstrcat (szResult, ":");

    if ((nSeconds = (nTime - ((nTime / 60) * 60))) < 10)
        lstrcat (szResult, "0");

    lstrcat (szResult, _itoa (nSeconds, szBuffer, 10));
    return szResult;
}

/*
 *  TMSFtoMinSec converts a time value in TMSF format to a string in
 *  minutes:seconds format.
 *
 *  Input parameters:
 *
 *    dwTime = Time in TMSF format
 *    szResult = Pointer to results buffer
 *
 *  Returns:
 *
 *    Pointer to results buffer (szBuffer)
 */

char *TMSFtoMinSec (DWORD dwTime, char *szResult)
{
    int nSeconds;
    char szBuffer[4];

    _itoa ((int) MCI_TMSF_MINUTE (dwTime), szResult, 10);
    lstrcat (szResult, ":");

    if ((nSeconds = (int) MCI_TMSF_SECOND (dwTime)) < 10)
        lstrcat (szResult, "0");

    lstrcat (szResult, _itoa (nSeconds, szBuffer, 10)); 
    return szResult;
}

/*
 *  AboutDlgProc processes messages to the About CDPlayer dialog box.
 */

BOOL FAR PASCAL AboutDlgProc (HWND hwnd, WORD message, WORD wParam,
                              LONG lParam)
{
    static HPEN hPen;
    static HBRUSH hBrush;
    static RECT rectFrame;
    static HFONT hDlgFont;
    PAINTSTRUCT ps;
    LOGFONT lf;
    RECT rect;
    HDC hdc;

    switch (message) {

    case WM_INITDIALOG:
        InitFrameRect (hwnd, IDD_FRAME, &rectFrame);

        GetObject ((HFONT) SendMessage (hwnd, WM_GETFONT, 0, 0L),
            sizeof (LOGFONT), &lf);
        lf.lfHeight = (lf.lfHeight * 3) / 2;
        lf.lfWidth = (lf.lfWidth * 3) / 2;
        lf.lfItalic = 1;
        if ((hDlgFont = CreateFontIndirect ((LPLOGFONT) &lf)) != NULL)
            SendDlgItemMessage (hwnd, IDD_TITLE, WM_SETFONT, hDlgFont, 0L);

        hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
        hBrush = CreateSolidBrush (RGB (192, 192, 192));
        return TRUE;

    case WM_CTLCOLOR:
        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        if (GetDlgCtrlID (LOWORD (lParam)) == IDD_TITLE)
            SetTextColor ((HDC) wParam, RGB (255, 0, 0));
        return hBrush;

    case WM_PAINT:
        hdc = BeginPaint (hwnd, &ps);
        GetClientRect (hwnd, &rect);
        FillRect (hdc, &rect, hBrush);

        SelectObject (hdc, hPen);
        SelectObject (hdc, GetStockObject (NULL_BRUSH));
        Rectangle (hdc, rectFrame.left, rectFrame.top,
            rectFrame.right, rectFrame.bottom);

        SelectObject (hdc, GetStockObject (WHITE_PEN));
        Rectangle (hdc, rectFrame.left + 1, rectFrame.top + 1,
            rectFrame.right + 1, rectFrame.bottom + 1);

        EndPaint (hwnd, &ps);
        return TRUE;

    case WM_COMMAND:
        switch (wParam) {
            case IDOK:
            case IDCANCEL:
                EndDialog (hwnd, 0);
                return TRUE;
        }
        break;

    case WM_DESTROY:
        if (hDlgFont != NULL)
            DeleteObject (hDlgFont);
        DeleteObject (hBrush);
        DeleteObject (hPen);
        return TRUE;
    }
    return FALSE;
}
