/*
** Copyright (C) by Intratek Computing, 1994-1995. All rights reserved.
*/

/*
** SML, String Manipulation Library
**
**   This library contains a set of routines to deal with strings in ways
** not supported by the ANSI/ISO C standard library, although the routines
** only use standard functions.
**
** Version 1.0.0
**
** History:
**      - 1-15-95   completion of v1.0.0
**                  compiled... library formed... no errors
*/

#ifndef SML_H

 #if __cplusplus
  extern "C" {
 #endif

 #define SML_H   1
 #define SML_ALL (-1)

 #define SML_iMAX(a, b) ((a) < (b) ? b : a)
 #define SML_iMIN(a, b) ((a) < (b) ? a : b)

 #include <stddef.h>
 #include <stdarg.h>
 #include <string.h>
 #include <stdio.h>
 #include <stdlib.h>
 #include <ctype.h>
 #include <limits.h>

/*
** -------------------------------------------------------------------
** Enumerated types...
**
** * case sensitivity
** * error values
**
** -------------------------------------------------------------------
*/

/*
** CASE sensitivity.
*/

typedef enum _sml_case
{
        SML_SEN,              /* operation is case sen      */
        SML_ISEN              /* operation is not case sen  */
} SML_CASE;

/*
** ERROR values.
*/

typedef enum _sml_error
{
        SML_SUCCESS,          /* previous operation was successful   */
        SML_BADARG,           /* bad argument was passed             */
        SML_NOMEM,            /* not enough memory                   */
        SML_BADPOS,           /* position in string is not possible  */
        SML_NOSS,             /* a search for some sub-string failed */
        SML_NOINIT            /* SML has not been initialized        */
} SML_ERROR;

/*
** -------------------------------------------------------------------
** Interface
**
** * Notes
** * Prototypes with descriptions
**
** -------------------------------------------------------------------
*/

/*
** Notes
**
** Success Condition: the condition caused by a function fulfilling it's
** task. Denoted by the function returning a non-NULL pointer or returning
** an integer that is non-negative.
**
** Fail Condition: the condition caused by a function not fulfilling it's
** task. Denoted by the function returning a NULL pointer or negative int.
*/

/*
** Prototypes with descriptions
*/

/*
** sml_init()           Initialize SML package
**
** Input:
**      - none
**
** Output:
**      - none
**
** Notes:
**      - This function must be called before any SML routine is used.
**      - Should only be called *once*.
*/

extern void sml_init(void);

/*
** sml_rerr()           Return error condition
**
** Input:
**      - none
**
** Output:
**      - error value created by fail condition to call of immediately
**        previous sml function, of type SML_ERROR.
**
** Notes:
**      - Since the SML functions are concerned with returning only values
**        immediately usable to you, they set a variable to any error that
**        creates a fail condition, and return a NULL or negative int.
*/

extern SML_ERROR sml_rerr(void);

/*
** sml_create()         String Create
**
** Input:
**      - size = length of string
**              - 0 if size = strlen(init_str)
**      - init_str = string to initialize new string to
**              - NULL if new string == ""
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
**      - sml_create(0, NULL) will not perform any action
**      - sml_create(5, "TEST") will use SML_iMAX(5, strlen("TEST")) as size
*/

extern char * sml_create(const size_t size, const char *init_str);

/*
** sml_csize()          Create empty string
**
** Input:
**      - n = size of string to create
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_csize(const size_t n);

/*
** sml_ccopy()          Create Copy
**
** Input:
**      - str = string to be copied
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_ccopy(const char *str);

/*
** sml_free()           Free a dynamically created string
**
** Input:
**      - str = string to be deallocated from memory
**
** Output:
**      - none
*/

extern void sml_free(char *str);

/*
** sml_mcat()           Multiple string concatenation
**
** Input:
**      - str = first string to append
**      - ... = string(s) to append
**              - NULL must be last argument passed
**
** Output:
**      - pointer to str
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_mcat(char *str, ...);

/*
** sml_nleft()          Return n characters from left of string
**
** Input:
**      - str = string to return n characters from
**      - n = number of characters to return from left of str
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_nleft(const char *str, const size_t n);

/*
** sml_nright()         Return n characters from right of string
**
** Input:
**      - str = string to return n characters from
**      - n = number of characters to return from right of str
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_nright(const char *str, const size_t n);

/*
** sml_nmid()           Return n characters from some position
**
** Input:
**      - str = string to return n characters from
**      - pos = beginning position
**      - n = number of characters to return from str
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_nmid(const char *str, const size_t pos, const size_t n);

/*
** sml_ndel()           Remove n characters from some position
**
** Input:
**      - str = string to remove n characters from
**      - pos = beginning position
**      - n = number of characters to remove from str
**              - SML_ALL = remove all characters to end of string
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_ndel(char *str, const size_t pos, const size_t n);

/*
** sml_ssdel()          Remove sub-string from string
**
** Input:
**      - str = string to remove sub-string from
**              - NULL to remove next occurrance of del in previous str
**      - del = sub-string to remove
**      - cs = case sensitivity
**              - SML_SEN  = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
**
** Notes:
**      - to remove all occurances of a sub-string from a string, use:
**
**        sml_ssdel(string, substring, SML_?SEN);
**        while ( sml_ssdel(NULL, substring, SML_?SEN) != NULL );
*/

extern char * sml_ssdel(char *str, char *del, const SML_CASE cs);

/*
** sml_cdel()           Strip undesired characters from string
**
** Input:
**      - str = string to strip
**      - c = character to strip from str
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_cdel(char *str, const char c);

/*
** sml_insert()         Insert a sub-string into some position
**
** Input:
**      - str = string to be inserted into
**      - sub = string to be inserted
**      - pos = position to insert
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
**
** Notes:
**      - make sure string does not overstep it's memory boundry with insert
**      - concatenating a string to the end is illegal using sml_insert()
**              - use sml_mcat()
*/

extern char * sml_insert(char *str, const char *sub, const size_t pos);

/*
** sml_overlay()        Overlay a string with a sub-string
**
** Input:
**      - str = string to be overlayed
**      - sub = sub-string to overlay with
**      - pos = position to overlay at
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
**
** Notes:
**      - concatenating a string to the end is illegal using sml_overlay()
**              - use sml_mcat()
*/

extern char * sml_overlay(char *str, const char *sub, const size_t pos);

/*
** sml_str2array()      Parse a string into an array of strings
**
** Input:
**      - orig_str = string to be parsed
**      - tok = tokens used to parse string
**
** Output:
**      - array of strings
**              - last element being NULL
**              - NULL if fail condition
**
** Notes:
**      - sml_s2a_kill() to dispose of array
*/

extern char ** sml_str2array(const char *orig_str, const char *tok);

/*
** sml_s2a_kill()       Remove an array created by sml_str2array()
**
** Input:
**      - array = pointer to array of strings
**
** Output:
**      - none
**
** Notes:
**      - use on all sml_str2array() array's before exit
*/

extern void sml_s2a_kill(char **array);

/*
** sml_array2str()      Convert array of strings to a string
**
** Input:
**      - array = array of strings
**              - NULL being last element
**      - tok = token(s) string to pad substrings (in string)
**              - empty string for no padding
**
** Output:
**      - pointer to new string
**              - NULL if fail condition
**
** Notes:
**      - sml_free() to dispose of dynamically created string
*/

extern char * sml_array2str(const char **array, const char *tok);

/*
** sml_xchg()           String eXchange
**
** Input:
**      - str1 = string to be swapped with str2
**      - str2 = string to be swapped with str1
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
**
** Notes:
**      - make sure string(s) do not overstep memory boundrys with swap
*/

extern char * sml_xchg(char *str1, char *str2);

/*
** sml_fill()           Fill string with a character
**
** Input:
**      - str = string to be filled
**      - c = character to fill with
**      - n = bytes to fill
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_fill(char *str, const int c);

/*
** sml_nfill()          Fill string with a character n times at position
**
** Input:
**      - str = string to be filled
**      - c = character to fill with
**      - pos = position to begin filling at
**      - n = bytes to fill
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_nfill(char *str, const int c, const size_t pos,
        const size_t n);

/*
** sml_2upper()         Upper the case of a string
**
** Input:
**      - str = string to be uppered
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_2upper(char *str);

/*
** sml_n2upper()        Upper the case of n characters at some position
**
** Input:
**      - str = string to be uppered
**      - pos = beginning position
**      - n = characters to upper
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_n2upper(char *str, const size_t pos, const size_t n);

/*
** sml_2lower()         Lower the case of a string
**
** Input:
**      - str = string to be lowered
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_2lower(char *str);

/*
** sml_n2lower()        Lower the case of n characters at some position
**
** Input:
**      - str = string to be lowered
**      - pos = beginning position
**      - n = characters to lower
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_n2lower(char *str, const size_t pos, const size_t n);

/*
** sml_psearch()        String search with pointer return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to found sub-string
**              - NULL if fail condition
*/

extern char * sml_psearch(char *str1, char *str2, const SML_CASE cs);

/*
** sml_esearch()        String search with element return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - element sub-string begins at
**              - < 0 if fail condition
*/

extern int sml_esearch(char *str1, char *str2, const SML_CASE cs);

/*
** sml_jleft()          Left justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad right of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_jleft(char *str, const char pad, const size_t n);

/*
** sml_jright()         Right justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad left of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_jright(char *str, const char pad, const size_t n);

/*
** sml_jcenter()        Center justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad left and right of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_jcenter(char *str, const char pad, const size_t n);

/*
** sml_rrotate()        Rotate string right once
**
** Input:
**      - str = string to rotate
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_rrotate(char *str);

/*
** sml_nrrotate()       Rotate string right n times
**
** Input:
**      - str = string to rotate
**      - n = times to rotate characters in string right
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_nrrotate(char *str, const size_t n);

/*
** sml_lrotate()        Rotate string left once
**
** Input:
**      - str = string to rotate
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_lrotate(char *str);

/*
** sml_nlrotate()       Rotate string left n times
**
** Input:
**      - str = string to rotate
**      - n = times to rotate characters in string left
**
** Output:
**      - pointer to passed string
**              - NULL if fail condition
*/

extern char * sml_nlrotate(char *str, const size_t n);

 #if __cplusplus
  };
 #endif

#endif                  /* #ifndef SML_H         */
