/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	directory.c

	list pacsat directory

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:

	0.1	Initial version.			G0ORX
	0.2	Added view selection.			G0ORX
	0.3	Fixed selected entry after update.	G0ORX
	0.4	Added Fill button.			G0ORX
	0.5	Free file buffer.			G0ORX
	0.6	Converted to Xaw.			G4KLX
	0.7	Added double click for filling.		G4KLX
		Changed request format + cancel.
		Added "Logs" to view selection.
*/

#define VERSION_STRING "(version 0.7 by g0orx/n6lyt/g4klx)"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <dirent.h>
#include <ctype.h>
#include <time.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Viewport.h>

#include "xawutils.h"
#include "header.h"
#include "request.h"
#include "ftl0.h"

Display *dpy;
int scrn;

XtAppContext app_context;

typedef struct
{
	XFontStruct *bold_font, *button_font, *menu_font, *label_font, *list_font;
}
Resources;

Resources  resources;

Widget toplevel, compwindow, quitbutton, updatebutton, cancelbutton,
	filldirbutton, viewtypebutton, viewtypelabel, listlabel, viewport,
	dirlist;

static XtResource resource_list[] =
{
	{"boldFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, bold_font), XtRString, XtDefaultFont},
	{"buttonFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, button_font), XtRString, XtDefaultFont},
	{"menuFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, menu_font), XtRString, XtDefaultFont},
	{"labelFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, label_font), XtRString, XtDefaultFont},
	{"listFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, list_font), XtRString, XtDefaultFont}
};

static Arg shell_args[] =
{
	{XtNtitle,		(XtArgVal)NULL}
};

static Arg form_args[] =
{
	{XtNdefaultDistance,	(XtArgVal)0}
};

static Arg label_args[] =
{
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNfromHoriz,		(XtArgVal)NULL},
	{XtNlabel,		(XtArgVal)NULL},
	{XtNwidth,		(XtArgVal)0},
	{XtNfont,		(XtArgVal)NULL},
	{XtNborderWidth,	(XtArgVal)0},
	{XtNresize,		False},
	{XtNjustify,		XtJustifyLeft},
	{XtNvertDistance,	(XtArgVal)6},
	{XtNhorizDistance,	(XtArgVal)8},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainTop},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainLeft}
};

static Arg button_args[] =
{
	{XtNcallback,		(XtArgVal)NULL},
	{XtNlabel,		(XtArgVal)NULL},
	{XtNfont,		(XtArgVal)NULL},
	{XtNfromHoriz,		(XtArgVal)NULL},
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNvertDistance,	(XtArgVal)6},
	{XtNhorizDistance,	(XtArgVal)8},
	{XtNresize,		False},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainTop},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainLeft}
};

static Arg menu_button_args[] =
{
	{XtNlabel,		(XtArgVal)NULL},
	{XtNfont,		(XtArgVal)NULL},
	{XtNfromHoriz,		(XtArgVal)NULL},
	{XtNwidth,		(XtArgVal)70},
	{XtNresize,		False},
	{XtNvertDistance,	(XtArgVal)6},
	{XtNhorizDistance,	(XtArgVal)8},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainTop},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainLeft},
	{XtNjustify,		XtJustifyCenter}
};

static Arg viewport_args[] =
{
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNwidth,		(XtArgVal)600},
	{XtNheight,		(XtArgVal)150},
	{XtNforceBars,		True},
	{XtNallowVert,		True},
	{XtNresize,		True},
	{XtNvertDistance,	(XtArgVal)0},
	{XtNhorizDistance,	(XtArgVal)0},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainBottom},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainRight}
};

static Arg list_args[] =
{
	{XtNcallback,		(XtArgVal)NULL},
	{XtNfont,		(XtArgVal)NULL}
};

MenuEntry type_menu[] =
{
	{"All",			"type1",	0},
	{"My Mail",		"type2",	1},
	{"Broadcast",		"type3",	2},
	{"Logs",		"type4",	3},
	{"EIS",			"type5",	4}
};

#define	MAX_DIR_LIST_SIZE	2000

char myCall[16];
char satelliteId[16];

String dList[MAX_DIR_LIST_SIZE + 1];
int    nList = 0;

int lastIndex = 0;

int view;

int sock;
struct sockaddr_in loc_addr;
struct sockaddr_in rem_addr;

void LoadDirectory(char *dirName);

int dircomp(const void *first, const void *second)
{
	char **a = (char **)first;
	char **b = (char **)second;

	return(strncmp(*b + 9, *a + 9, 9));
}

void Update(void)
{
	DIR		*pDir;
	struct dirent	*pDirent;
	int i;
	Arg args[2];

	for (i = 0; i < nList; i++)
		if (dList[i] != NULL)
			XtFree(dList[i]);

	nList     = 0;
	lastIndex = -1;

	/* walk through the directory files */
	pDir = opendir(".");

	while ((pDirent = readdir(pDir)) != NULL && nList < MAX_DIR_LIST_SIZE)
	{
		if (strncmp(pDirent->d_name, "pb__", 4) == 0)
			LoadDirectory(pDirent->d_name);
	}

	closedir(pDir);

	dList[nList] = NULL;

	qsort(dList, nList, sizeof(String), dircomp);

	XawListChange(dirlist, dList, nList, 0, False);

	if (nList > 0) XawListHighlight(dirlist, 0);

	XtSetArg(args[0], XtNheight, 0);
	XtSetArg(args[1], XtNwidth,  0);
	XtSetValues(dirlist, args, 2);
}

void DirTypeCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	view = (int)client_data;

	XtVaSetValues(viewtypebutton, XtNlabel, type_menu[view].Label, NULL);

	Update();
}

void UpdateCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	Update();
}

void QuitCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtDestroyApplicationContext(app_context);
	
	exit(0);
}

void LoadDirectory(char * dirName)
{
	int hDir;
	int fileLength;
	int offset;
	char *pBuffer, *source, *destination;
	static struct tm * GMT;
	char szTemp[128];
	int headerSize;
	HEADER *pHeader;
	int j;
	int display;
	char *p;

	/* open the directory file */
	if ((hDir = open(dirName, O_RDONLY)) == -1)
	{
		perror(dirName);
		return;
	}

	/* get the file length */
	fileLength = lseek(hDir, 0L, SEEK_END);

	pBuffer = XtMalloc(fileLength);

	lseek(hDir, 0L, SEEK_SET);

	/* read in the complete file */
	read(hDir, pBuffer, fileLength);

	/* close the file */
	close(hDir);

	/* walk through the file extracting the headers */
	offset = 0;

	while (offset < fileLength)
	{
		if ((pHeader = ExtractHeader(&pBuffer[offset], fileLength-offset, &headerSize)) == NULL)
		{
			printf("invalid header entry (at offset %d (0x%X))\n", offset, offset);
			offset = fileLength;
		}
		else
		{
			source      = XtNewString(pHeader->source);
			destination = XtNewString(pHeader->destination);
		
			/* truncate the source and destination */
			j = 0;
			while (isalnum(pHeader->source[j]) != '\0') j++;
      			pHeader->source[j] = '\0';

			j = 0;
			while (isalnum(pHeader->destination[j]) != '\0') j++;
      			pHeader->destination[j] = '\0';

			GMT = gmtime(&pHeader->uploadTime);

			if (strlen(pHeader->title) == 0)
				sprintf(szTemp, "%8lx %02d%02d/%02d%02d %-8s %-8s %6ld %-40s",
					pHeader->fileId,
					GMT->tm_mon + 1, GMT->tm_mday, GMT->tm_hour, GMT->tm_min,
					pHeader->source,
					pHeader->destination,
					pHeader->fileSize,
					pHeader->fileName);
			else
				sprintf(szTemp, "%8lx %02d%02d/%02d%02d %-8s %-8s %6ld %-40s",
					pHeader->fileId,
					GMT->tm_mon + 1, GMT->tm_mday, GMT->tm_hour, GMT->tm_min,
					pHeader->source,
					pHeader->destination,
					pHeader->fileSize,
					pHeader->title);

			if (headerSize > pHeader->bodyOffset)
				offset += headerSize;
			else
				offset += pHeader->bodyOffset;

			for (p = source; *p != '\0'; p++)
				if (islower(*p)) *p=toupper(*p);

			for (p = destination; *p != '\0'; p++)
				if (islower(*p)) *p=toupper(*p);

			display = FALSE;

			switch (view)
			{
				case 0:
					display = TRUE;
					break;
				case 1:
					if (strstr(source, myCall)      != NULL ||
					    strstr(destination, myCall) != NULL)
						display = TRUE;
					break;
				case 2:
					if (strstr(destination, "ALL") != NULL)
						display = TRUE;
					break;
				case 3:
					if (pHeader->fileType >= 100  &&
					    pHeader->fileType != 221)
					     	display = TRUE;
					break;
				case 4:
					if (pHeader->fileType == 220 ||
					    pHeader->fileType == 221)
						display = TRUE;
					break;
				default:
					break;
			}

			if (display)
			{
				dList[nList] = XtNewString(szTemp);
				nList++;
			}

			XtFree((char *)pHeader);
			XtFree(source);
			XtFree(destination);
		}
	}

	XtFree(pBuffer);
}

void CancelFillCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	struct request request;


	request.type   = REQ_TYPE_CANCEL;
	request.fileId = 0L;

	if (sendto(sock, (char *)&request, sizeof(struct request), 0, (struct sockaddr *)&rem_addr, sizeof(rem_addr)) == -1)
	{
		perror("sendto");
	}
}

void SelectCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *selected;
	struct request request;

	selected = (XawListReturnStruct *)call_data;

	if (selected->list_index != lastIndex)
	{
		lastIndex = selected->list_index;
		return;
	}

	lastIndex = -1;

	request.type = REQ_TYPE_FILE;
	sscanf(selected->string, "%lx", &request.fileId);

	if (sendto(sock, (char *)&request, sizeof(struct request), 0, (struct sockaddr *)&rem_addr, sizeof(rem_addr)) == -1)
	{
		perror("sendto");
	}
}

void FillDirCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	struct request request;

	request.type   = REQ_TYPE_DIR;
	request.fileId = 0L;

	if (sendto(sock, (char *)&request, sizeof(struct request), 0, (struct sockaddr *)&rem_addr, sizeof(rem_addr)) == -1)
	{
		perror("sendto");
	}
}

int main(int argc, char **argv)
{
	static XtCallbackRec callback[2];
	char title[80], *s;

	if ((s = getenv("MYCALL")) == NULL)
		strcpy(myCall, "NOCALL");
	else
		strcpy(myCall, s);

	if ((s = getenv("SATELLITE")) == NULL)
	{
		printf("SATELLITE environment variable not set.\n");
		return(1);
	}

	strcpy(satelliteId , s);

	toplevel = XtAppInitialize(&app_context, "Xpb", NULL, 0, &argc, argv,
				NULL, shell_args, XtNumber(shell_args));

	sprintf(title, "directory:%s %s", satelliteId, VERSION_STRING);
	XtVaSetValues(toplevel, XtNtitle, title, NULL);

	dpy  = XtDisplay(toplevel);
	scrn = DefaultScreen(dpy);

	XtGetApplicationResources(toplevel, &resources,
				resource_list, XtNumber(resource_list),
				NULL, ZERO);

	compwindow = XtCreateManagedWidget("appForm", formWidgetClass,
				toplevel, form_args, XtNumber(form_args));

	callback[0].callback = QuitCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Quit";
	button_args[2].value = (XtArgVal)resources.button_font;
	quitbutton = XtCreateManagedWidget("quitButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	callback[0].callback = UpdateCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Update";
	button_args[3].value = (XtArgVal)quitbutton;
	updatebutton = XtCreateManagedWidget("updateButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	callback[0].callback = CancelFillCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Cancel Fill";
	button_args[3].value = (XtArgVal)updatebutton;
	cancelbutton = XtCreateManagedWidget("cancelButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	callback[0].callback = FillDirCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Fill Directory";
	button_args[3].value = (XtArgVal)cancelbutton;
	filldirbutton = XtCreateManagedWidget("dirButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	label_args[0].value = (XtArgVal)NULL;
	label_args[1].value = (XtArgVal)filldirbutton;
	label_args[2].value = (XtArgVal)"View";
	label_args[4].value = (XtArgVal)resources.label_font;
	viewtypelabel = XtCreateManagedWidget("viewtypeLabel", labelWidgetClass,
				compwindow, label_args, XtNumber(label_args));

	menu_button_args[0].value = (XtArgVal)type_menu[0].Label;
	menu_button_args[1].value = (XtArgVal)resources.button_font;
	menu_button_args[2].value = (XtArgVal)viewtypelabel;
	viewtypebutton = XtCreateManagedWidget("viewtypeButton", menuButtonWidgetClass,
				compwindow, menu_button_args, XtNumber(menu_button_args));

	createMenuPopup(viewtypebutton, resources.menu_font, type_menu,
				XtNumber(type_menu), DirTypeCb);

	label_args[0].value = (XtArgVal)quitbutton;
	label_args[1].value = (XtArgVal)NULL;
	label_args[2].value = (XtArgVal)"       Id Date/Time From     To         Size Title";
	label_args[3].value = (XtArgVal)500;
	label_args[4].value = (XtArgVal)resources.list_font;
	listlabel = XtCreateManagedWidget("listLabel", labelWidgetClass,
				compwindow, label_args, XtNumber(label_args));

	viewport_args[0].value = (XtArgVal)listlabel;
	viewport = XtCreateManagedWidget("directoryViewport", viewportWidgetClass,
				compwindow, viewport_args, XtNumber(viewport_args));
		
	callback[0].callback = SelectCb;
	callback[0].closure  = toplevel;
	list_args[0].value = (XtArgVal)callback;
	list_args[1].value = (XtArgVal)resources.list_font;
	dirlist = XtCreateManagedWidget("directoryList", listWidgetClass,
				viewport, list_args, XtNumber(list_args));

	/* setup to talk to xpb */
	memset((char *)&loc_addr, 0, sizeof(rem_addr));
	loc_addr.sin_family      = AF_INET;
	loc_addr.sin_addr.s_addr = INADDR_ANY;
	loc_addr.sin_port        = htons(0);

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
	}

	if (bind(sock, (struct sockaddr *)&loc_addr, sizeof(loc_addr)) == -1)
	{
		perror("bind");
	}

	memset((char *)&rem_addr, 0, sizeof(rem_addr));
	rem_addr.sin_family      = AF_INET;
	rem_addr.sin_port        = htons(5100);
	rem_addr.sin_addr.s_addr = htonl(0x7f000001);

	view = 0;
	Update();

	XtRealizeWidget(toplevel);

	XtAppMainLoop(app_context);

	return(0);
}
