/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	upload.c

	Pacsat Upload for Linux and X-Windows

	This program has been run using the 1.0 version of the
	Linux kernel with the patches from Alan Cox to provide AX.25
	encapsulation in the SLIP protocol(verion 0.12).

	The TNC must be setup for KISS protocol.

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:

	0.1	cloned from (x)upload.c
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/socket.h>
#include <linux/ax25.h>
#include <signal.h>
#include <ctype.h>

#include "header.h"
#include "ftl0.h"

extern char satellite[];
extern char myCall[];

#define IDLE			0
#define LOGGING_IN		1
#define REQUESTING_UPLOAD	2
#define UPLOADING		3
#define SENT_END_DATA		4

int state = IDLE;

int fd[];
int f, s;
unsigned long fileId;
long offset;
int fileLength;
UPLOAD_REQUEST *ul_request;
UPLOAD_RESPONSE *ul_response;
unsigned long uploadFileId;
unsigned char buffer[1024];
extern int maxFrame;
extern int T1;
	
#define MAXHEADER 1024
unsigned char header[MAXHEADER];
int nHeader;

char msg[256];

int convert_call(char *, struct full_sockaddr_ax25 *);

void update_status(char *msg)
{
	write(fd[1], msg, strlen(msg) + 1);
}

void error(char *msg, int errno)
{
	char text[128];

	if (errno != 0)
	{
		switch (errno)
		{
			case 106:
				sprintf(text, "%s: already connected", msg);
				break;
			case 107:
				sprintf(text, "%s: not connected", msg);
				break;
			case 108:
				sprintf(text, "%s: connection shutdown", msg);
				break;
			case 110:
				sprintf(text, "%s: connection timedout", msg);
				break;
			case 111:
				sprintf(text, "%s: connection refused", msg);
				break;
			case 112:
				sprintf(text, "%s: no route to host", msg);
				break;
			default:
				sprintf(text, "%s: error %d", msg, errno);
				break;
		}

		update_status(text);
	}
	else
	{
		update_status(msg);
	}

	close(s);
	close(f);

	sleep(1);
	exit(1);
}

void ReceivedFrame(char *fileName)
{
	int bytes;
	int type;
	int length;

	if ((bytes = recv(s, buffer, 512, 0)) < 0)
		error("recv failed", errno);

	length = buffer[0] + ((buffer[1] << 3) & 0xFF00);
	type   = buffer[1] & 0x1F;

	switch (state)
	{
		case LOGGING_IN:
			if (type != LOGIN_RESP)
			{
				sprintf(msg, "Expected LOGIN_RESP got %d", type);
				error(msg, 0);
			}
			
			update_status("Status: Logged in");

			/* request an upload */
			sprintf(msg, "Status: Requesting upload: file:%lx length:%d",
					fileId, fileLength);
			update_status(msg);

			buffer[0] = sizeof(UPLOAD_REQUEST);
			buffer[1] = UPLOAD_CMD;
			
			ul_request         = (UPLOAD_REQUEST *)&buffer[2];
			ul_request->file   = fileId;
			ul_request->length = fileLength;
	
			if (send(s, (char *)buffer, sizeof(UPLOAD_REQUEST) + 2, 0)
				!= (sizeof(UPLOAD_REQUEST) + 2))
				error("send failed", errno);

			state = REQUESTING_UPLOAD;
			break;

		case REQUESTING_UPLOAD:
			switch (type)
			{
				case UL_GO_RESP:
					break;
				case UL_ERROR_RESP:
					sprintf(msg, "Received UL_ERROR_RESP: %d", buffer[2]);
					error(msg, 0);
				default:
					sprintf(msg, "Received unexpected GO_RESP: %d", type);
					error(msg, 0);
			}

			ul_response = (UPLOAD_RESPONSE *)&buffer[2];
			offset       = ul_response->offset;
			uploadFileId = ul_response->file;

			sprintf(msg, "Status: Received GO_RESP: file:%lx offset:%ld",
				uploadFileId, offset);
			update_status(msg);

			/* update the file id in the header */
			/* incase it does not all make it */
			lseek(f, 5L, SEEK_SET);
			write(f, (char *)&uploadFileId, 4);

			/* now seek to the start of the data and start uploading */
			lseek(f, offset, SEEK_SET);
			state = UPLOADING;
			break;

		case SENT_END_DATA:
			switch (type)
			{
				case UL_ACK_RESP:
					break;
				case UL_NAK_RESP:
					sprintf(msg, "Received UL_NAK_RESP: %d", buffer[2]);
					error(msg, 0);
				default:
					sprintf(msg, "Received unexpected response: %d", type);
					error(msg, 0);
			}

			sprintf(msg, "Status: File %s uploaded as message %lx", fileName, uploadFileId);
			update_status(msg);

			unlink(fileName);

			close(s);
			sleep(1);

			exit(0);
	}
}

void SendFrame(char *fileName)
{
	int bytes;

	if ((bytes = read(f, &buffer[2], maxFrame)) > 0)
	{
		buffer[0] = bytes;
		buffer[1] = DATA;

		/* always send fileId of 0 in header */
		if (offset == 0)
		{
			buffer[7] = 0;
			buffer[8] = 0;
			buffer[9] = 0;
			buffer[10] = 0;
		}

		if (send(s, buffer, bytes + 2, 0) != (bytes + 2))
			error("send failed", errno);

		offset += bytes;
		return;
	}

	if (bytes < 0) error("read failed", errno);

	close(f);

	/* send a data end packet */
	buffer[0] = 0;
	buffer[1] = DATA_END;

	if (send(s, buffer, 2, 0) != 2)
		error("send failed", errno);

	state = SENT_END_DATA;
}

void upload(char *fileName)
{
	HEADER *pHeader;
	int opt;
	int addr_len;
	struct full_sockaddr_ax25 addr;
	int bytes;
	char temp[80];
	fd_set rx, tx;

	sprintf(msg, "Status: Uploading: %s", fileName);
	update_status(msg);

	if ((f = open(fileName, O_RDWR)) < 0)
		error("File open failed", 0);

	/* get the file length */
	fileLength = lseek(f, 0L, SEEK_END);

	/* extracting the header */
	lseek(f, 0L, SEEK_SET);
	bytes = read(f, buffer, 1024);
	lseek(f, 0L, SEEK_SET);
	pHeader = ExtractHeader(buffer, bytes, &bytes);
	fileId = pHeader->fileId;
	XtFree((char *)pHeader);

	if ((s = socket(AF_AX25, SOCK_SEQPACKET, PF_AX25)) < 0)
		error("Socket open failed", 0);

	opt = 1;

	if (setsockopt(s, SOL_AX25, AX25_WINDOW, &opt, sizeof(opt)) == -1)
		error("setsockopt (AX25_WINDOW) failed", 0);

	if (setsockopt(s, SOL_AX25, AX25_T1, &T1, sizeof(opt)) == -1)
		error("setsockopt (AX25_T1) failed", 0);

#ifdef DEBUG
	opt = 1;

	if (setsockopt(s, SOL_SOCKET, SO_DEBUG, &opt, sizeof(opt)) == -1)
		error("setsockopt (SO_DEBUG) failed", 0);
#endif

	addr_len = convert_call(myCall, &addr);

	if (bind(s, (struct sockaddr *)&addr, addr_len) == -1)
		error("bind failed", 0);

	addr_len = convert_call(satellite, &addr);

	update_status("Status: Connecting...");

	if (connect(s, (struct sockaddr *)&addr, addr_len))
	{
		switch (errno)
		{
			case ECONNRESET:
				error("Connection reset by peer", 0);
			case ECONNREFUSED:
				error("Connection refused", 0);
			case ETIMEDOUT:
				error("Connection timed out", 0);
			default:
				sprintf(temp, "Connect failed (%d)", errno);
				error(temp, 0);
		}
	}

	update_status("Status: Connected");

	/* setup state */
	state = LOGGING_IN;

	for (;;)
	{
		FD_ZERO(&rx);
		FD_ZERO(&tx);

		FD_SET(s, &rx);

		if (state == UPLOADING)
		{
			FD_SET(s, &tx);	
			select(s + 1, &rx, &tx, NULL, NULL);
		}
		else
		{
			select(s + 1, &rx, NULL, NULL, NULL);
		}
				
		if (FD_ISSET(s, &rx)) ReceivedFrame(fileName);
		
		if (FD_ISSET(s, &tx)) SendFrame(fileName);
	}
}

/*
 *	Library routine for callsign conversion.
 *
 */
 
void convert_call_entry(char *name,unsigned char *buf)
{
	int ct = 0;
	int ssid = 0;
	char *p = name;

	while (ct < 6)
	{
		if (*p == '-' || *p == '\0') break;

		if (islower(*p)) *p = toupper(*p);

		if (!isalnum(*p))
		{
			sprintf(msg, "Invalid symbol in callsign - %c\n", *p);
			error(msg, 0);
		}
		
		buf[ct] = (*p << 1);
		p++;
		ct++;
	}

	while (ct < 6)
	{
		buf[ct] = ' ' << 1;
		ct++;
	}

	if (*p == '-')
	{
		p++;

		if (sscanf(p, "%d", &ssid) != 1 || ssid < 0 || ssid > 15)
		{
			sprintf(msg, "Invalid SSID - %s - %s\n", name, p);
			error(msg, 0);
		}
	}

	buf[6] = ((ssid + '0') << 1) & 0x1E;
}

int convert_call(char *call, struct full_sockaddr_ax25 *sax)
{

	int len = 0;
	unsigned char *bp, *np;
	char *addrp;
	int n = 0;
	unsigned char *tmp = strdup(call);
	
	if (tmp == NULL) return -1;
		
	bp = tmp;
	
	addrp = sax->fsa_ax25.sax25_call.ax25_call;

	do
	{
		/* Fetch one callsign token */
		while (*bp && isspace(*bp)) bp++;
		np = bp;
		while (*np && !isspace(*np)) np++;
		if (*np) *np++ = 0;
	
		/* Check for the optional 'via' syntax */
		if (n == 1 && (strcasecmp(bp, "V") == 0 || strcasecmp(bp, "VIA") == 0))
		{
			bp = np;
			continue;
		}
		
		/* Process the token */
		convert_call_entry(bp, addrp);
			
		/* Move along */
		bp = np;
		n++;

		if (n == 1)
		{
			addrp = sax->fsa_digipeater[0].ax25_call;	/* First digipeater address */
			len  += sizeof(struct sockaddr_ax25);
		}
		else
		{
			addrp += sizeof(ax25_address);
			len   += sizeof(ax25_address);
		}
	}
	while (n < AX25_MAX_DIGIS && *bp);

	free(tmp);

	/* Tidy up */
	sax->fsa_ax25.sax25_ndigis = n - 1;
	sax->fsa_ax25.sax25_family = AF_AX25;	

	return len;
}

