.. computes the B-spline non-recursively

reset();

function lambda (x)
## For x_1,...,x_{m+2}, return l, s.t.
## sum l_j (t-x_j)^{m+1} =m+1.
	{X,Y}=field(x,x);
	m=cols(x)-2;
	return (((Y-X)^(m+1))\(dup(m+1,m+2)))';
endfunction

function lambdaval (l,x,t)
## Evaluate sum l_j (t-x_j)_+^m for x_1,...,x_{m+2}.
	{T,X}=field(t,x);
	m=cols(x)-2;
	A=max((T-X),0)^m;
	return l.A;
endfunction

function testlambda
	"Compute the B-Splines by their lambda"
    shrinkwindow();
    setplot(0,8,0,2); xplot(0,0); hold on;
    t=0:0.05:8;
    s=1|((2:6)+random(1,5)*0.2-0.1)|7;
    for i=3 to 7;
    	si=s[1:i];
        plot(t,lambdaval(lambda(si),si,t));
    end;
    mark(s,zeros(size(s))); hold off;
    return plot();
endfunction

function bsplines (x,m,t)
## Compute a field of B_Splines by their lambda and evaluate them
## at t. Return a (cols(x)-m-1) x cols(t) matrix.
	n=cols(x)-m-1;
	R=dup(t,n);
	loop 1 to n;
		xx=x[#:#+m+1];
		ll=lambda(xx);
		R[#]=lambdaval(ll,xx,t);
	end;
	return R;
endfunction

function nsplines (x,m,t)
## Compute a field of normalized B_Splines by their lambda and
## evaluate them at t. Return a (cols(x)-m-1) x cols(t) matrix.
	n=cols(x)-m-1;
	R=dup(t,n);
	loop 1 to n;
		xx=x[#:#+m+1];
		ll=lambda(xx)*(x[#+m+1]-x[#])/(m+1);
		R[#]=lambdaval(ll,xx,t);
	end;
	return R;
endfunction

function splinterpol (xk,m,x,y,t)
## Compute the spline interpolant to (x,y) with knots at xk and
## evaluate it at t.
	R=bsplines(xk,m,x);
	a=R'\y';
	R=bsplines(xk,m,t);
	return a'.R;
endfunction

function cubinterpol (xk,yp,d1,d2,t)
## Compute the cubic interpolant with knots in xk, values yp and
## derivative d1 in xk[1], d2 in xk[k+2] and evaluate in t.
	k=cols(xk)-8;
	h=sqrt(epsilon);
	x=(xk[4]-h)|xk[4:(5+k)]|(xk[5+k]+h);
	R=inv(bsplines(xk,3,x))';
	Rt=bsplines(xk,3,t);
	y=(yp[:,1]-h*d1)|yp|(yp[:,k+2]+h*d2);
	a=R.y';
	return a'.Rt;
endfunction

function cubtest
## The user clicks some points, which form the interpolation points.
## The program draws the interpolation with equidistant and chordal
## parametrization.
	setplot(-1,1,-1,1);
	xplot(-1,1); hold on;
	p=zeros(2,0);
	repeat
		m=mouse();
		if (cols(p)>0) && (m[2]>1); break; endif;
		p=p|m';
		mark(m[1],m[2]);
	end;
	t=linspace(0,cols(p)-1,300);
	color(2); s=cubinterpol(-3:(cols(p)+2),p,[0;1],[0;-1],t);
	plot(s[1],s[2]);
	color(1);
	hold off;
	return "";
endfunction

function interpnorm (xk,m,x)
## Compute the interpolation norm at knots xk and points x.
	n=cols(xk);
	t=linspace(xk[m+1],xk[n-m],100);
	n=cols(x);
	R=dup(t,n); N=1:n;
	loop 1 to n;
		R[#]=splinterpol(xk,m,x,N==#,t);
	end
	xplot(t,R); wait(180);
	r=sum(abs(R'))';
	xplot(t,r); title("sum |s_i|"); wait(180);
	return max(r);
endfunction

function testnorm (xk,m)
## Compute the interpolation norm for points at avarages of knots.
	n=cols(xk);
	xx=max(xk,xk[m+1]); xx=min(xx,xx[n-m]);
	s=0|cumsum(xx[2:n-1]);
	x=(s[m+1:n-1]-s[1:n-m-1])/m;
	return interpnorm(xk,m,x);
endfunction

function splineapprox1 (ffunction,x,m,t)
## Build the simple approximation s(x) = sum f(t_i) N_i(x)
## with t_i = (x_{i-m}+x_i)/2 and evaluate at t.
	k=cols(x)-(2*m+2);
	N=nsplines(x,m,t);
	ti=(x[1:k+m+1]+x[m+2:k+2*m+2])/2;
	ti=max(x[m+1],ti); ti=min(x[m+k+2],ti);
	return ffunction(ti).N;
endfunction

function test1 (ffunction)
	t=linspace(-0.3,1.3,100);
	s=splineapprox1(ffunction,-0.3:0.1:1.3,3,t);
	xplot(t,(ffunction(t))_s);
	title(ffunction|" and its simple approximation in S_3");
	wait(180);
	i=nonzeros(t>=0 && t<=1);
	xplot(t[i],ffunction(t[i])-s[i]);
	title(ffunction|" and its simple approximation in S_3 (error)");
	wait(180);
	return "";
endfunction

function splinecurve (p,t,m=3)
## Compute BS_{k,m}(p_0,...p_n)(t).
	n=cols(p)-1; kk=n-m;
	x=((-m:-1)/(1000*kk))|((0:(kk+1))/(kk+1))|(1+(1:m)/(1000*kk));
	N=nsplines(x,m,t);
	return p.N;
endfunction

function nsplinetest (mm=3)
## The user clicks some points, which form the B-Spline polygon.
## The program draws the B-Spline curve
	setplot(-1,1,-1,1);
	xplot(-1,1); hold on;
	p=zeros(2,0);
	repeat
		m=mouse();
		if (cols(p)>0) && (m[2]>1); break; endif;
		p=p|m';
		plot(p[1],p[2]);
	end;
	t=linspace(0,1,300);
	color(2); s=splinecurve(p,t,mm); plot(s[1],s[2]);
	color(1);
	hold off;
	return "";
endfunction

.. EOF
