.. Rationale Interpolation

function ratintp (x,y,m)
## Computes the rational interpolation to x(i),y(i)
## with denominator degree m. Returns {p,q}.
## Method used: Linear system of n+m equations.
	l=length(x);
	n=l-m-1; e=ones([l,1]);
	if n<0; error("Not enough data!"); endif;
	if m<1; error("Use polynomial interpolation!"); endif;
	{i,x}=field(1:n,x); 
	A=x^i;
	if m>1;
		{i,x}=field(1:m-1,x);
		B=dup(y',m-1)*x^i;
		C=e|A|-y'|-B;
	else
		C=e|A|-y';
	endif;
	b=(C\(y*x^m)')';
	return {b[1:n+1],b[(n+2):length(b)]|1}
endfunction

function ratval (p,q,x)
## Evaluates the rational p/q at x
	return polyval(p,x)/polyval(q,x);
endfunction 

function thiele (x,y)
## Computes the Thiele continued fraction, which interpolates (x_i,y_i).
	f=y; n=length(y);
	loop 1 to n-1;
		i=index()+1:n;
		f[i]=(x[i]-x[index()])/(f[i]-f[index()]);
	end;
	return f;
endfunction

function thieleval (x,f,t)
## Evaluates the Thiele continued fraction computed with "thiele" at t.
	n=length(f); r=f(n);
	for i=n-1 to 1 step -1;
		r=f[i]+(t-x[i])/r;
	end;
	return r
endfunction;

function test
## Compares the polynomial and rational interpolation.
	t=0:0.1:1; s=sqrt(t);
	x=linspace(min(t),max(t),100); y=sqrt(x);
	plot(x,interpval(t,interp(t,s),x)-y); color(1);
	hold on;
	plot(x,thieleval(t,thiele(t,s),x)-y); color(2);
	hold off;
	title("Rational interpolation is better than polynomial");
	wait(180);
	return plot();
endfunction

test()