/*****************************************************************************
 * $Id: rmt_tape.c,v 1.12 1994/07/06 21:19:34 ak Exp $
 *****************************************************************************
 * $Log: rmt_tape.c,v $
 * Revision 1.12  1994/07/06 21:19:34  ak
 * Bugfix.
 *
 * Revision 1.11  1994/06/02 13:40:31  ak
 * Stop reading when a filemark is encountered.
 *
 * Revision 1.10  1994/02/16 15:29:20  edvkai
 * Dummy checkin for CVS 1.3 crlf.
 *
 * Revision 1.9  1993/12/14 21:55:34  ak
 * Bugfix. Partition number must be -1 (no change).
 *
 * Revision 1.8  1993/11/29  16:59:25  edvkai
 * *** empty log message ***
 *
 * Revision 1.7  1993/11/26  21:59:37  edvkai
 * Added "rewind" for relative multi-volume QFA.
 *
 * Revision 1.6  1993/11/25  18:54:01  edvkai
 * Removed DLL import by number.
 * Changed return codes to avoid ambiguities.
 * Changed lseek into seek, parameter changes.
 *
 * Revision 1.3  1993/04/26  14:49:29  AK
 * AIX.
 *
 * Revision 1.2  1993/02/10  13:41:56  AK
 * Check for write-protected tapes.
 * Flag '+' for append to EOT.
 *
 * Revision 1.1.1.1  1993/02/08  21:32:17  AK
 * TAR device interface DLLs.
 *
 * Revision 1.1  1993/02/08  21:32:15  AK
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: rmt_tape.c,v 1.12 1994/07/06 21:19:34 ak Exp $";

/*
 *	rmt_tape.c
 *
 * Tape interface for GNU tar.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>

#include "scsi.h"
#include "tape.h"
#include "errtab.h"

#ifndef __IBMC__
# define _System
#endif

extern FILE *msg_file;

static int	errcode;
static int	wrflag;
static int	tape_no = 0;
static long	tapeblock = -1;
static long	last_error;
static int	filemark;

int _System
rmt_open(char *name, int mode, int prot)
{
	long r;
	char *cp;
	int eot = 0;

	if (*name == '+') {
		++name;
		eot = 1;
	}

	if (r = tape_open(name)) {
		last_error = r;
		return -2;
	}

	tape_ready();		/* skip "Cartridge Changed" */
	r = tape_ready();
	if (!r)
		r = tape_get_blocksize(&tapeblock);
	if (r) {
		last_error = r;
		return -2;
	}

	if (++tape_no == 1)
		switch (mode & 3) {
		case O_RDONLY:
			break;
		case O_WRONLY:
			r = tape_writable();
			if (r) {
				last_error = r;
				return -2;
			}
			wrflag = 1;
			if (eot)
				tape_space(TapeSpace_LogEndOfMedia, 0L, NULL);
			break;
		case O_RDWR:
			last_error = ErrNotSupported;
			return -2;
		}
	else
		tape_rewind(0);

	filemark = 0;

	return 0;
}

int _System
rmt_read(int fd, void *p, unsigned len)
{
	long actual = TapeUndefLength, r = 0;

	if (filemark)
		return 0;

	r = tape_read(p, len, &actual);
	if (actual == TapeUndefLength)
		actual = 0;
	if (r) {
		last_error = r;
		if (ErrorClass(r) == ErrclDeviceError
		  && ErrorType(r) == ErrtySenseByte) {
			if (r & FM)
				filemark = 1;
			if (ErrorCode(r) == MediaOverflow)
				return -3;
			if (ErrorCode(r) == FM+NoSense)
				return actual;
		}
		if (actual == 0)
			return -2;
	}
	return actual;
}

int _System
rmt_write(int fd, void *p, unsigned len)
{
	long actual = TapeUndefLength, r = 0;

	r = tape_write(p, len, &actual);
	if (actual == TapeUndefLength)
		actual = 0;
	if (r) {
		last_error = r;
		if (r == ErrclDeviceError+ErrtySenseByte+MediaOverflow)
			return -3;
		if (actual == 0)
			return -2;
	}
	return actual;
}

long _System
rmt_seek(int fd, long block, long blocksz, int mode)
{
	/* used:	0/1/2	-> standard lseek, return code 0
			3	-> physical block, return code 0
			4	-> return current phys block
			5	-> rewind tape
			logical block is TAR block, not tape block
	*/
	long r, actual;

	switch (mode) {
	case 5:
		filemark = 0;
		r = tape_rewind(0);
		if (r) {
			last_error = r;
			return -2;
		}
		return 0;
	case 4:
		r = tape_tell(&actual, NULL);
		if (r) {
			last_error = r;
			return -2;
		}
		return actual;
	case 3:
		filemark = 0;
		/* physical block, direct seek required */
		r = tape_seek(0, block, -1);
		if (r) {
			last_error = r;
			return -2;
		}
		return 0;
	case 2:
		if (!filemark)
			r = tape_space(TapeSpace_Filemarks, 1L, &actual);
		filemark = 0;
		break;
	case 1:
		r = 0;
		break;
	case 0:
		if (filemark)
			r = tape_space(TapeSpace_Filemarks, -1L, &actual);
		filemark = 0;
		if (!r)
			r = tape_space(TapeSpace_Filemarks, -1L, &actual);
		if (!r)
			tape_space(TapeSpace_Filemarks, 1L, &actual);
		break;
	default:
		return -4;
	}
	if (block && !r) {
		if (tapeblock)
			block *= blocksz / tapeblock;
		r = tape_space(TapeSpace_Blocks, block, &actual);
	}
	if (!r)
		return 0;
	last_error = r;
	return -2;
}

int _System
rmt_close(int fd)
{
	long r = 0;

	if (wrflag)
		r = tape_filemark(0, 1L, NULL);
	if (r)
		tape_close();
	else
		r = tape_close();
	wrflag = 0;
	if (!r)
		return 0;
	last_error = r;
	return -2;
}

int _System
rmt_ioctl(int fd, int code, void *arg)
{
	return -4;
}

int _System
rmt_block(int fd)
{
	return tapeblock;
}

long _System
rmt_error(void)
{
	return last_error;
}

char * _System
rmt_status(long rcode)
{
	return tape_status(rcode);
}

#if defined(__EMX__)

unsigned long
_DLL_InitTerm(unsigned long mod_handle, unsigned long flag)
{
	if (!flag) {
		_CRT_init();
	} else {
		_CRT_term();
	}
	return 1;
}

#elif defined(unix)

int
rmt_startup(void)
{
	tape_no = 0;
	tapeblock = 512;
	last_error;
	return 0;
}

#endif
