 Program ExIOCard;

 {$IFNDEF DPMI}
   Error! - compile for protected mode only!
 {$ENDIF}


   {-----------------------------------------------------------------------}
   {                                                                       }
   {  Program ExIOCard           Example Routines to Use EZDPMI to Handle  }
   {                             Execution of I/O Card Real Mode ROM Code  }
   {-----------------------------------------------------------------------}
   {                                                                       }
   {  This is a demonstration program to provide some example routines     }
   {  using the EZDPMI unit to call Real Mode code in the ROM of an        }
   {  I/O adapter card.                                                    }
   {                                                                       }
   {  For this example, the I/O card routines reside starting at           }
   {  $D000:$0400 in the DOS Real Mode address space.                      }
   {                                                                       }
   {  Issues addressed by these routines:                                  }
   {                                                                       }
   {    1. How to call Real Mode ROM code.                                 }
   {                                                                       }
   {    2. How to pass parameters to the EZDPMI RealCall routine.          }
   {                                                                       }
   {    3. How to pass real mode pointers to strings using var and         }
   {       const String parameters.                                        }
   {                                                                       }
   {    4. Setting up global variables in Real Mode memory for use         }
   {       as a data exchange buffer to be shared by Real Mode and         }
   {       Protected Mode routines.  Global variables are used and         }
   {       allocated at program startup, since they will be used           }
   {       throughout the program (we assume) as a communications          }
   {       data buffers. It would be too slow and inefficient to           }
   {       allocate and deallocate the buffer memory each time the         }
   {       program called an I/O routine.                                  }
   {                                                                       }
   {       Additionally, in case the program is operating in a             }
   {       multitasking environment, the memory for the I/O buffers        }
   {       should be allocated and kept throughout the program's life      }
   {       to keep it reserved. It would not do to have the routine fail   }
   {       because some other program allocated all the available free     }
   {       DOS memory causing an inability to allocate a buffer during     }
   {       the program run.                                                }
   {                                                                       }
   {    5. Separate Transmit and Receive routines are included, as well    }
   {       as a single function that sends a string and receives one       }
   {       back in the same call.                                          }
   {                                                                       }
   {                                                                       }
   {-----------------------------------------------------------------------}
   { This program provided "AS-IS" for The Pascal Magazine by Ray Bernard. }
   {-----------------------------------------------------------------------}

 Uses
   WinDOS, { for TRegisters type }
   EZDPMI;

 const
  { provide program test mode }
   ProgramTest : Boolean = False;  { set to True to test without I/O card }

  { values per the I/O card API }
   IO_Code_Seg   = $D000;        { I/O card code segment }
   IO_Init_Call  = $0400;        { init routine }
   IO_TX_Call    = $0440;        { transmit routine }
   IO_RX_Call    = $0480;        { receive routine }
   IO_TXRX_Call  = $04C0;        { transmit/receive routine }

   MaxStringLen      = 255;      { use for error checking }

   Success           =  0;       { indicates no errors }
   ParamDataError    = -1;       { Error values for I/O routine program  }
   IODataError       = -2;       { errors - use negative values for easy }
   DPMIRealCallError = -3;       { differentiation from I/O card errors  }
                                 { which are always positive.            }
 type
   StringPointer = ^String;  { Required to make this kind of assignment: }
                             {   PStringProtMode := gPDOSStrBufPmode;    }
                             { as in the procedure pmode_IE_OUTPUT.      }


 { global variables to point to DOS memory area string buffer }
 var
   gPDOSStrTXBufReal,     { pointer to hold transmit buffer real mode address }
   gPDOSStrTXBufPmode,                 { pointer to hold buffer pmode address }
   gPDOSStrRXBufReal,      { pointer to hold receive buffer real mode address }
   gPDOSStrRXBufPmode : StringPointer; { pointer to hold buffer pmode address }


 {------------------------------------------------------------}
 { The next two functions handle allocation and deallocation  }
 { of the String-sized data buffer set up in Real Mode memory }
 { for use by both Real Mode and Protected Mode code.         }
 {------------------------------------------------------------}

  function AllocateRealModeStringBuffers : Boolean;
  {-Create DOS real mode string buffers for use with       }
  { the real mode adapter card API I/O routines.           }
  { Allocate blocks of DOS real mode memory the size of    }
  { a string and get real mode and pmode pointers to it.   }
  { Set the four global variables to point to the buffers. }
  var
    Okay : Boolean;
  begin
    Okay := DOSGetMem(gPDOSStrTXBufReal,gPDOSStrTXBufPmode,SizeOf(String));
    if Okay then
      Okay := DOSGetMem(gPDOSStrRXBufReal,gPDOSStrRXBufPmode,SizeOf(String));
    AllocateRealModeStringBuffers := Okay;
  end;

  function DeallocateRealModeStringBuffers : Boolean;
  {-Deallocate the DOS real mode memory used for the     }
  { program's real mode adapter card I/O string buffers. }
  var
    Okay : Boolean;
  begin
    Okay := DOSFreeMem(gPDOSStrRXBufPmode);
    if Okay then
      Okay := DOSFreeMem(gPDOSStrTXBufPmode);
    DeallocateRealModeStringBuffers := Okay;
  end;



 {------------------------------------------------------------}
 { The next two procedures demonstrate calling Real Mode code }
 { in the I/O card's ROM.                                     }
 {------------------------------------------------------------}


  function pmode_IO_INIT(ioPort, ioMode, Setting : Integer) : Integer;
  {-Initializes the I/O card API returning 0 for success or the      }
  { error code passed back by the I/O card's initialization routine. }
  { Does not need to use any global variables, since its parameters  }
  { are word-sized data (integers) rather than pointers, and the     }
  { data itself is passed to the I/O card code on the stack.         }
  { Note that the I/O card's routine puts a 0 for success or an      }
  { error code value value on the stack in the position of ioMode.   }
  { Since the I/O card returns positive values for error codes, we   }
  { use negative values for program error codes.                     }
  var
    RealCallStack : record
      _Setting : Integer;     { These are listed in the record in the }
      _ioMode  : Integer;     { reverse order of the parameters as is }
      _ioPort  : Integer;     { required by RealCall. See EZDPMI.DOC. }
    end;
    Regs : TRegisters;
    CallAddress : RealProc;   { To pass as parameter to RealCall. }
    CallPointer : Pointer;    { Needed for Pointer value assignment which    }
  begin                       { is then assigned by typecast to CallAddress. }
    pmode_IO_INIT := Success;
    with RealCallStack do
      begin
        _Setting := Setting;
        _ioMode  := ioMode;
        _ioPort  := ioPort;
      end;
    CallPointer := Ptr(IO_Code_Seg,IO_Init_Call);  { assign I/O routine addr }
    CallAddress := RealProc(CallPointer);      { typecast to make assignment }
    FillChar(Regs, SizeOf(Regs), 0);           { no data passed in registers }

    if ProgramTest then   { allow test without I/O card installed }
      Exit;

    {-- Call the I/O card init code --------------------------------------}
    if not RealCall(CallAddress, Regs, 3, @RealCallStack) then
      pmode_IO_INIT := DPMIRealCallError  { report program error }
    else
      pmode_IO_INIT := RealCallStack._ioMode; { value set by I/O card }
    {---------------------------------------------------------------------}
  end; { function pmode_IO_INIT }



  function pmode_IO_OUTPUT(const S : String) : Integer;
  {-Transmits a string using the I/O card API.                             }
  {                                                                        }
  { The I/O card routine expects an integer and a pointer to the transmit  }
  { string buffer to be pushed on the stack, in that order, just prior to  }
  { calling the routine. It expects the string in that buffer to be in     }
  { Pascal format (using string[0] as the length byte).  The result code   }
  { is returned in the integer position on the stack as 0 for success or a }
  { positive integer value error code.                                     }
  {                                                                        }
  { Note: uses global variables gPStrTXBufReal and gPDOSStrTXBufPmode      }
  { which are real and protected mode pointers to a string buffer          }
  { located in DOS memory allocated at program startup.  Global buffer     }
  { pointer assignments are marked in comments as #1 and #2.               }
  {                                                                        }
  { I/O card routine error values are positive range numbers and are       }
  { returned as the function result.  If there is a program error this is  }
  { reported as a negative error code return value.                        }
  var
    RealCallStack : record
      PStringRealMode : StringPointer;  { assign gDOSPstrTXBufReal to this  }
      IOResultCode    : Integer;
    end;
    PStringProtMode : StringPointer;    { assign gPDOSStrTXBufPmode to this }
    Regs            : TRegisters;
    CallAddress     : RealProc;   { to pass as parameter to RealCall }
    CallPointer     : Pointer;    { needed for typecast }
    NumBytesToCopy  : Byte;
  begin
    pmode_IO_OUTPUT := Success;
    if Length(S) > MaxStringLen then                 { trap corrupted string }
      begin
        pmode_IO_OUTPUT := ParamDataError;         { return data error value }
        Exit;
      end;
    PStringProtMode := gPDOSStrTXBufPmode;         { #1 assign pmode pointer }
    with RealCallStack do
      begin
        PStringRealMode := gPDOSStrTXBufReal;  { #2 assign real mode pointer }
       { copy S into global transmit string buffer }
        NumBytesToCopy := Length(S)+1;        { include length byte in copy! }
        Move(S[0],PStringProtMode^[0],NumBytesToCopy); { fast data copy }
      end;
    CallPointer := Ptr(IO_Code_Seg,IO_TX_Call); { assign I/O routine addr }
    CallAddress := RealProc(CallPointer);  { assign addr to pass to RealCall }
    FillChar(Regs, SizeOf(Regs), 0);  { no data passed in registers }

    if ProgramTest then   { allow test without I/O card installed }
      Exit;

    {-- Call the I/O card transmit code ------------------------------------}
    if not RealCall(CallAddress, Regs,
                    SizeOf(RealCallStack) div 2, @RealCallStack) then
       pmode_IO_OUTPUT := DPMIRealCallError { report program error }
    else
       pmode_IO_OUTPUT := RealCallStack.IOResultCode { set by I/O card }
    {-----------------------------------------------------------------------}
  end; { function pmode_IO_OUTPUT }


  function pmode_IO_INPUT(var S : String) : Integer;
  {-Receives a string using the I/O card API and passes it back via the    }
  { S parameter, which returns an empty string if nothing is received.     }
  {                                                                        }
  { The I/O card routine expects an integer and a pointer to the receive   }
  { string buffer to be pushed on the stack, in that order, just prior to  }
  { calling the routine. It expects the string in that buffer to be in     }
  { Pascal format (using string[0] as the length byte).  The result code   }
  { is returned in the integer position on the stack as 0 for success or a }
  { positive integer value error code.                                     }
  {                                                                        }
  {                                                                        }
  { Note: uses global variables gPStrRXBufReal and gPDOSStrRXBufPmode      }
  { which are real and protected mode pointers to a string buffer          }
  { located in DOS memory allocated at program startup.  Global buffer     }
  { pointer assignments are marked in comments as #1 and #2.               }
  {                                                                        }
  { With ProgramTest set to True, this function will return whatever was   }
  { placed in the global receive string buffer prior to calling it.        }
  {                                                                        }
  { I/O card routine error values are positive range numbers and are       }
  { returned as the function result.  If there is a program error this is  }
  { reported as a negative error code return value.                        }
  var
    RealCallStack : record
      PStringRealMode : StringPointer;  { assign gDOSPstrRXBufReal to this  }
      IOResultCode    : Integer;
    end;
    PStringProtMode : StringPointer;    { assign gPDOSStrRXBufPmode to this }
    Regs            : TRegisters;
    CallAddress     : RealProc;   { to pass as parameter to RealCall }
    CallPointer     : pointer;    { needed for typecast }
    NumBytesToCopy  : byte;
  begin
    pmode_IO_INPUT := Success;
    S := '';  { blank in case nothing is received }
    PStringProtMode := gPDOSStrRXBufPmode;          {#1 assign pmode pointer }
    with RealCallStack do
      begin
        IOResultCode := Success;                 { set for Program Test mode }
        PStringRealMode := gPDOSStrRXBufReal;  { #2 assign real mode pointer }
      end;
    CallPointer := Ptr(IO_Code_Seg,IO_RX_Call);    { assign I/O routine addr }
    CallAddress := RealProc(CallPointer);  { assign addr to pass to RealCall }
    FillChar(Regs, SizeOf(Regs), 0);  { no data passed in registers }

    if not ProgramTest then   { allow test without I/O card installed }
    {-- Call the I/O board receive code ------------------------------------}
    if not RealCall(CallAddress, Regs,
                    SizeOf(RealCallStack) div 2, @RealCallStack) then
      begin
        pmode_IO_INPUT := DPMIRealCallError; { report program error }
        Exit;
      end
    else
      pmode_IO_INPUT := RealCallStack.IOResultCode; { set by I/O card }
    {-----------------------------------------------------------------------}

    if RealCallStack.IOResultCode <> Success then
      Exit;

    if Length(PStringProtMode^) > MaxStringLen then  { trap corrupted string }
      begin
        pmode_IO_INPUT := IODataError;             { return data error value }
        Exit;
      end;
    with RealCallStack do   { copy string value to S }
      begin
        NumBytesToCopy := Length(PStringProtMode^)+1;  { include length byte }
        Move(PStringProtMode^[0],S[0],NumBytesToCopy);      { fast data copy }
      end;
  end; { function pmode_IO_INPUT }


  function pmode_IO_INOUT(const TXStr : String; var RXStr : String) : Integer;
  {-Transmits TXStr and receives a string using the I/O card API, passing  }
  { the received string back via the TXStr parameter, or passing back an   }
  { an empty string if nothing is received or there is a program error.    }
  {                                                                        }
  { The I/O card routine expects pointers to an integer and the transmit   }
  { and receive string buffers to be pushed on the stack, in that order,   }
  { just prior to calling the routine. The I/O card expects the string     }
  { buffers to hold Pascal style strings.                                  }
  {                                                                        }
  { Note: uses global variables gPStrTXBufReal and gPDOSStrTXBufPmode      }
  { and gPStrRXBufReal and gPDOSStrRXBufPmode which are sets of real and   }
  { and protected mode pointers to string buffers located in DOS memory    }
  { allocated at program startup.  Global buffer pointer assignments are   }
  { marked in comments as #1, #2, #3 and #4.                               }
  {                                                                        }
  { With ProgramTest set to True, this function will return whatever was   }
  { placed in the global receive string buffer prior to calling it.        }
  {                                                                        }
  { I/O card routine error values are positive range numbers and are       }
  { returned as the function result.  If there is a program error this is  }
  { returned as a negative error code value.                               }
  var
    RealCallStack : record
      PRXStrRealMode : StringPointer;  { assign gDOSPstrRXBufReal to this  }
      PTXStrRealMode : StringPointer;  { assign gDOSPstrTXBufReal to this  }
      IOResultCode   : Integer;
    end;
    PTXStrProtMode : StringPointer;    { assign gPDOSStrTXBufPmode to this }
    PRXStrProtMode : StringPointer;    { assign gPDOSStrRXBufPmode to this }
    Regs            : TRegisters;
    CallAddress     : RealProc;   { to pass as parameter to RealCall }
    CallPointer     : pointer;    { needed for typecast }
    NumBytesToCopy  : byte;
  begin
    pmode_IO_INOUT := Success;
    RXStr := ''; { blank in case nothing received }
    if Length(TxStr) > MaxStringLen then  { trap corrupted string }
      begin
        pmode_IO_INOUT := ParamDataError; { return data error value }
        Exit;
      end;
    PTXStrProtMode := gPDOSStrTXBufPmode;      { #1 assign pmode pointer }
    PRXStrProtMode := gPDOSStrRXBufPmode;      { #2 assign pmode pointer }
    with RealCallStack do
      begin
        IOResultCode := Success;                 { set for Program Test mode }
        PTXStrRealMode := gPDOSStrTXBufReal;   { #3 assign real mode pointer }
        PRXStrRealMode := gPDOSStrRXBufReal;   { #4 assign real mode pointer }
       { copy TXStr into global Transmit string buffer }
        NumBytesToCopy := Length(TXStr)+1;    { include length byte in copy! }
        Move(TXStr[0],PTXStrProtMode^[0],NumBytesToCopy);   { fast data copy }
      end;
    CallPointer := Ptr(IO_Code_Seg,IO_TX_Call); { assign I/O routine addr }
    CallAddress := RealProc(CallPointer);  { assign addr to pass to RealCall }
    FillChar(Regs, SizeOf(Regs), 0);  { no data passed in registers }

    if not ProgramTest then   { allow test without I/O card installed }
    {-- Call the I/O card transmit/receive code ----------------------------}
    if not RealCall(CallAddress, Regs,
                    SizeOf(RealCallStack) div 2, @RealCallStack) then
      begin
        pmode_IO_INOUT := DPMIRealCallError; { report program error }
        Exit;
      end
    else
      pmode_IO_INOUT := RealCallStack.IOResultCode; { set by I/O card }
    {-----------------------------------------------------------------------}
    if RealCallStack.IOResultCode <> Success then
      Exit;

    if Length(PRXStrProtMode^) > MaxStringLen then  { trap corrupted string }
      begin
        pmode_IO_INOUT := IODataError;             { return data error value }
        Exit;
      end;
    with RealCallStack do   { copy string value to S }
      begin
        NumBytesToCopy := Length(PRXStrProtMode^)+1;  { include length byte }
        Move(PRXStrProtMode^[0],RXStr[0],NumBytesToCopy);   { fast data copy }
      end;
  end; { function pmode_IO_INOUT }




 { variables used for testing }
 var
   RxS,TxS : String;
   ResultCode : Integer;

 { Some code to test the string buffer handling }
 { using ProgramTest set to True.               }
 begin
   ProgramTest := True;
   TxS := 'Hello There!';
   if AllocateRealModeStringBuffers then
     begin
       Writeln('DOS Real Mode Buffers Allocated.');
       gPDOSStrRXBufPmode^ := '***Simulated Received String***';
     end
   else
     Writeln('ERROR: DOS Real Mode Buffers NOT Allocated.');
   Writeln('Sending: ['+TxS+']');
   ResultCode := pmode_IO_INOUT(TxS,RxS);
   if ResultCode = Success then
     begin
        Writeln('Successful test data transmission.');
        Writeln('Received: ['+RxS+']');
     end
   else
     Writeln('ERROR: Error Code ',ResultCode);
   if DeAllocateRealModeStringBuffers then
     Writeln('DOS Real Mode Buffers Deallocated.')
   else
     Writeln('ERROR: DOS Real Mode Buffers NOT Deallocated.')
 end.


