Unit Fields;

{ input fields }

interface

uses
   Global, Noiselib, Strlib, Datelib, EnumStrs,
   Objects, Drivers, Views, Dialogs, MsgBox;

const
   cmTabToNext = 1350;

type
   TMaskStr  = string [ 20 ];

type
   PFieldLabel = ^TFieldLabel;
   TFieldLabel = object ( TLabel )
      Procedure Draw; virtual;
      Function GetPalette : PPalette; virtual;
      end;

type
   PMaskedField = ^TMaskedField;
   TMaskedField = object ( tInputLine )
      Mask : PString;
      Negative : boolean;
      Constructor Init ( R: TRect; vMask : TMaskStr );
      Destructor Done; virtual;
      Constructor Load ( var S : TStream );
      Procedure Store ( var S : TStream ); virtual;
      Procedure SetState ( aState : word; Enable : boolean ); virtual;
      Function Valid ( Command : word ) : boolean; virtual;
      end;

   PIntegerField = ^TIntegerField;
   TIntegerField = object ( TMaskedField )
      Num : longint;
      Constructor Load ( var S : TStream );
      Procedure Store ( var S : TStream ); virtual;
      Procedure SetState ( aState : word; Enable : boolean ); virtual;
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      Procedure GetData ( var Rec ); virtual;
      Procedure SetData ( var Rec ); virtual;
      Function DataSize : word; virtual;
      end;

   PLongintField = ^TLongintField;
   TLongintField = object ( TIntegerField )
      Function DataSize : word; virtual;
      Procedure GetData ( var Rec ); virtual;
      Procedure SetData ( var Rec ); virtual;
      end;

   PRealField = ^TRealField;
   TRealField = object ( TMaskedField )
      Num : real;
      Constructor Load ( var S : TStream );
      Procedure Store ( var S : TStream ); virtual;
      Function NumDigits : integer;
      Function NumDecimals : integer;
      Function DataSize : word; virtual;
      Procedure GetData ( var Rec ); virtual;
      Procedure SetData ( var Rec ); virtual;
      Procedure SetState ( aState : word; Enable : boolean ); virtual;
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      end;

   PDateField = ^TDateField;
   TDateField = object ( TInputLine )
      JDate : TJulianDate;
      Constructor Init ( R: TRect );
      Constructor Load ( var S : TStream );
      Procedure Store ( var S : TStream ); virtual;
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      Procedure SetState ( AState : word; Enable : boolean ); virtual;
      Procedure GetData ( var Rec ); virtual;
      Procedure SetData ( var Rec ); virtual;
      Function DataSize : word; virtual;
      Function Valid ( Command : word ): boolean; virtual;
      end;

   PEnumeratorListBox = ^TEnumeratorListBox;
   TEnumeratorListBox = object ( TListBox )
      Constructor Init ( var R : TRect; ScrollBar : PScrollBar);
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      Function GetPalette : PPalette; virtual;
      end;

   PEnumeratorWindow = ^TEnumeratorWindow;
   TEnumeratorWindow = object ( TWindow )
      ListBox : PEnumeratorListBox;
      Constructor Init ( R : TRect; EnumId : TEnumId; Item : word );
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      Function GetPalette : PPalette; virtual;
      end;

   PEnumeratorField = ^TEnumeratorField;
   TEnumeratorField = object ( tInputLine )
      Id      : TEnumId;
      Current : byte;
      Constructor Init ( R : TRect; vId : TEnumId );
      Constructor Load ( var S : TStream );
      Procedure Store ( var S : TStream ); virtual;
      Procedure Draw; virtual;
      Procedure HandleEvent ( var Event : TEvent ); virtual;
      Function GetPalette: PPalette; virtual;
      Procedure GetData ( var Rec ); virtual;
      Procedure SetData ( var Rec ); virtual;
      Function DataSize : word; virtual;
      end;

Procedure RegisterFields;

Implementation

const
   RFieldLabel : TStreamRec = (
      ObjType : rnFieldLabel;
      VmtLink : Ofs(TypeOf(TFieldLabel)^);
      Load    : @TFieldLabel.Load;
      Store   : @TFieldLabel.Store );

   RMaskedField : TStreamRec = (
      ObjType : rnMaskedField;
      VmtLink : Ofs(TypeOf(TMaskedField)^);
      Load    : @TMaskedField.Load;
      Store   : @TMaskedField.Store );

   RIntegerField : TStreamRec = (
      ObjType : rnIntegerField;
      VmtLink : Ofs(TypeOf(TIntegerField)^);
      Load    : @TIntegerField.Load;
      Store   : @TIntegerField.Store );

   RLongintField : TStreamRec = (
      ObjType : rnLongintField;
      VmtLink : Ofs(TypeOf(TLongintField)^);
      Load    : @TLongintField.Load;
      Store   : @TLongintField.Store );

   RRealField : TStreamRec = (
      ObjType : rnRealField;
      VmtLink : Ofs(TypeOf(TRealField)^);
      Load    : @TRealField.Load;
      Store   : @TRealField.Store );

   RDateField : TStreamRec = (
      ObjType : rnDateField;
      VmtLink : Ofs(TypeOf(TDateField)^);
      Load    : @TDateField.Load;
      Store   : @TDateField.Store );

   REnumeratorField : TStreamRec = (
      ObjType : rnEnumeratorField;
      VmtLink : Ofs(TypeOf(TEnumeratorField)^);
      Load    : @TEnumeratorField.Load;
      Store   : @TEnumeratorField.Store );

{========================================================================}

Procedure RegisterFields;

Begin
   RegisterType ( RFieldLabel );
   RegisterType ( RMaskedField );
   RegisterType ( RIntegerField );
   RegisterType ( RLongintField );
   RegisterType ( RRealField );
   RegisterType ( RDateField );
   RegisterType ( REnumeratorField );
End;

{========================================================================}
{ Field Label Object                                                     }
{========================================================================}

Procedure TFieldLabel.Draw;

var
   Color : word;
   B : TDrawBuffer;

Begin
   if ( Link^.State and sfDisabled ) = 0 then
      begin
      inherited Draw;
      exit;
      end;

   Color := GetColor($0506);
   MoveChar ( B [ 0 ], ' ', byte ( Color ), Size.X );
   if Text <> nil then
      MoveCStr ( B [ 1 ], Text^, Color );
   WriteLine ( 0, 0, Size.X, 1, B );
End;

{========================================================================}

Function TFieldLabel.GetPalette : PPalette;

{ Palette layout        }
{ 1 = Text Normal       }
{ 2 = Text Selected     }
{ 3 = Shortcut Normal   }
{ 4 = Shortcut Selected }
{ 5 = Disabled Normal   }
{ 6 = Disabled Selected }

const
   CGrayFieldLabel = CLabel + #13 + #13;
   CBlueFieldLabel = CLabel + #31 + #31;
   CCyanFieldLabel = CLabel + #31 + #31;
   P: array[dpBlueDialog..dpGrayDialog] of string [ length ( CBlueFieldLabel ) ] =
      ( CCyanFieldLabel, CBlueFieldLabel, CGrayFieldLabel );

Begin
   GetPalette := @P[PDialog(Owner)^.Palette];
End;

{========================================================================}
{ Masked Field Object                                                    }
{========================================================================}

Constructor TMaskedField.Init ( R: TRect; vMask : TMaskStr );

Begin
   R.B.X := R.A.X + length ( vMask ) + 2; { force the field width }
   inherited Init ( R, length ( vMask ) );
   Mask := NewStr ( vMask );
   options := options or ofValidate;
End;

{========================================================================}

Destructor TMaskedField.Done;

Begin
   DisposeStr ( Mask );
   inherited Done;
End;

{========================================================================}

Constructor TMaskedField.Load ( var S : TStream );

Begin
   inherited Load ( S );
   Mask := S.ReadStr;
End;

{========================================================================}

Procedure TMaskedField.Store ( var S : TStream );

Begin
   inherited Store ( S );
   S.WriteStr ( Mask );
End;

{========================================================================}

Procedure TMaskedField.SetState ( aState : word; Enable : boolean );

Begin
   inherited SetState ( aState, Enable );

   { format the data when the field loses the focus }
   if ( aState = sfFocused ) and ( not Enable ) then
      begin
      Data^ := StrToMask ( Data^, Mask^ );
      end;
End;

{========================================================================}

Function TMaskedField.Valid ( Command : word ) : boolean;

var
   Ok : boolean;

Begin
   Data^ := StripMask ( Data^ );  { remove mask chars }
   Ok := inherited Valid ( Command );
   if Ok then
      Data^ := StrToMask ( Data^, Mask^ );
   Valid := Ok;
End;

{========================================================================}
{ Integer Field Object                                                   }
{========================================================================}

Constructor TIntegerField.Load ( var S : TStream );

Begin
   inherited Load ( S );
   S.Read ( Num, sizeof ( Num ) );
   Data^ := StrToMask ( IntToStr ( Num ), Mask^ );
End;

{========================================================================}

Procedure TIntegerField.Store ( var S : TStream );

Begin
   inherited Store ( S );
   S.Write ( Num, sizeof ( Num ) );
End;

{========================================================================}

Procedure TIntegerField.SetState ( aState : word; Enable : boolean );

Begin
   { update the number when the field loses the focus }
   if ( aState = sfFocused ) and ( not Enable ) then
      begin
      Data^ := StripMask ( Data^ );  { remove unwanted chars }
      Num := StrToInt ( Data^ );
      end;

   { unformat the data when the field gains the focus }
   if ( aState = sfFocused ) and ( Enable ) then
      Data^ := IntToStr ( Num );

   inherited SetState ( aState, Enable );
End;

{========================================================================}

Function TIntegerField.DataSize : word;

Begin
   DataSize := SizeOf ( integer );
End;

{========================================================================}

Procedure TIntegerField.SetData ( var Rec );

type
   TLongint = record
     Low, High : integer;
     end;

Begin
   move ( Rec, TLongint ( Num ).Low, DataSize );
   Data^ := StrToMask ( IntToStr ( Num ), Mask^ );
End;

{========================================================================}

Procedure TIntegerField.GetData ( var Rec );

type
   TLongint = record
     Low, High : integer;
     end;

Begin
   move ( TLongint ( Num ).Low, Rec, DataSize );
End;

{========================================================================}

Procedure TIntegerField.HandleEvent ( var Event : TEvent );

const
   ValidChars = [ '0'..'9', ',', '-', '+' ];
   AllChars = [ #32..#255 ];

Begin
   if ( Event.What = evKeyboard ) and
      ( Event.CharCode in ValidChars ) then
      begin
      if Event.CharCode = '-' then
         begin
         if pos ( '-', Mask^ ) = 0 then
            ClearEvent ( Event )
         else
         if SelEnd > SelStart then
            inherited HandleEvent ( Event )
         else
         if Data^ [ 1 ] = '-' then
            Data^ := copy ( Data^, 2, length ( Data^ ) )
         else
            Data^ := '-' + Data^;
         ClearEvent ( Event );
         DrawView;
         end;

      if Event.CharCode = '+' then
         begin
         if Data^ [ 1 ] = '-' then
            Data^ := copy ( Data^, 2, length ( Data^ ) );
         ClearEvent ( Event );
         DrawView;
         end;

      inherited HandleEvent ( Event );
      ClearEvent ( Event );
      end
   else
   if ( Event.What = evKeyboard ) and
      ( Event.CharCode in AllChars ) then
      ClearEvent ( Event )
   else
      inherited HandleEvent ( Event );
End;

{========================================================================}
{ Longint Field Object                                                   }
{========================================================================}

Function TLongintField.DataSize : word;

Begin
   DataSize := SizeOf ( longint );
End;

{========================================================================}

Procedure TLongintField.SetData ( var Rec );

Begin
   move ( Rec, Num, DataSize );
   Data^ := StrToMask ( IntToStr ( Num ), Mask^ );
End;

{========================================================================}

Procedure TLongintField.GetData ( var Rec );

Begin
   move ( Num, Rec, DataSize );
End;

{========================================================================}
{ Real Field Object                                                      }
{========================================================================}

Constructor TRealField.Load ( var S : TStream );

Begin
   inherited Load ( S );
   S.Read ( Num, sizeof ( Num ) );
   Data^ := StrToMask ( RealToStr ( Num, NumDigits, NumDecimals ), Mask^ );
End;

{========================================================================}

Procedure TRealField.Store ( var S : TStream );

Begin
   inherited Store ( S );
   S.Write ( Num, sizeof ( Num ) );
End;

{========================================================================}

Function TRealField.DataSize : word;

Begin
   DataSize := SizeOf ( real );
End;

{========================================================================}

Function TRealField.NumDigits : integer;

var
   i, Digits : byte;

Begin
   Digits := 0;
   for i := 1 to length ( Mask^ ) do
      if Mask^ [ i ] = '#' then
         inc ( Digits );
   NumDigits := Digits;
End;

{========================================================================}

Function TRealField.NumDecimals : integer;

var
   i, Decimals : byte;

Begin
   Decimals := 0;
   if pos ( '.', Mask^ ) <> 0 then
      for i := pos ( '.', Mask^ ) + 1 to length ( Mask^ ) do
         if Mask^ [ i ] = '#' then
            inc ( Decimals );
   NumDecimals := Decimals;
End;

{========================================================================}

Procedure TRealField.SetState ( aState : word; Enable : boolean );

Begin
   { format the data and update the num when the field loses the focus }
   if ( aState = sfFocused ) and ( not Enable ) then
      begin
      Data^ := StripMask ( Data^ );
      Num := StrToReal ( Data^, NumDigits, NumDecimals );
      Data^ := RealToStr ( Num, NumDigits, NumDecimals );
      end;

   { unformat the data when the field gains the focus }
   if ( aState = sfFocused ) and ( Enable ) then
      Data^ := LTrim ( RealToStr ( Num, NumDigits, NumDecimals ) );

   inherited SetState ( aState, Enable );
End;

{========================================================================}

Procedure TRealField.SetData ( var Rec );

Begin
   move ( Rec, Num, DataSize );
   Data^ := StrToMask ( RealToStr ( Num, NumDigits, NumDecimals ), Mask^ );
End;

{========================================================================}

Procedure TRealField.GetData ( var Rec );

Begin
   move ( Num, Rec, DataSize );
End;

{========================================================================}

Procedure TRealField.HandleEvent ( var Event : TEvent );

const
   ValidChars = [ '.', ',', '-', '+', '0'..'9' ];
   AllChars = [ #32..#255 ];

Begin
   if ( Event.What = evKeyboard ) and
      ( Event.CharCode in ValidChars ) then
      begin
      if ( Event.CharCode = '.' ) and ( pos ( '.', Data^ ) <> 0 ) then
         ClearEvent ( Event ) {. already keyed}
      else
      if ( Event.CharCode = '.' ) and ( pos ( '.', Mask^ ) = 0 ) then
         ClearEvent ( Event ) {. not allowed }
      else
      if ( Event.CharCode = '-' ) then
         begin
         if pos ( '-', Mask^ ) = 0 then
            ClearEvent ( Event )
         else
         if SelEnd > SelStart then
            inherited HandleEvent ( Event )
         else
         if Data^ [ 1 ] = '-' then
            Data^ := copy ( Data^, 2, length ( Data^ ) )
         else
            Data^ := '-' + Data^;
         ClearEvent ( Event );
         DrawView;
         end;

      if Event.CharCode = '+' then
         begin
         if Data^ [ 1 ] = '-' then
            Data^ := copy ( Data^, 2, length ( Data^ ) );
         ClearEvent ( Event );
         DrawView;
         end;

      inherited HandleEvent ( Event );
      ClearEvent ( Event );
      end
   else
   if ( Event.What = evKeyboard ) and
      ( Event.CharCode in AllChars ) then
      ClearEvent ( Event )
   else
      inherited HandleEvent ( Event );
End;

{========================================================================}
{ Date Field Object                                                      }
{========================================================================}

Constructor TDateField.Init ( R: TRect );

Begin
   R.B.X := R.A.X + 13; { force the field width }
   inherited Init ( R, 11 );
   Options := Options or ofValidate;
End;

{========================================================================}

Constructor TDateField.Load ( var S : TStream );

Begin
   inherited Load ( S );
   S.Read ( JDate, sizeof ( JDate ) );
End;

{========================================================================}

Procedure TDateField.Store ( var S : TStream );

Begin
   inherited Store ( S );
   S.Write ( JDate, sizeof ( JDate ) );
End;

{========================================================================}

Function TDateField.DataSize : word;

Begin
   DataSize := SizeOf ( TJulianDate );
End;

{========================================================================}

Procedure TDateField.SetData ( var Rec );

Begin
   move ( Rec, JDate, DataSize );
   Data^ := JDateToStr ( JDate );
End;

{========================================================================}

Function TDateField.Valid ( Command : word ) : boolean;

var
   Month, Day, Year : integer;

Begin
   if ( Command <> cmCancel ) or ( Command <> cmQuit ) then
      begin
      Month := ValidMonth ( copy ( Data^, 4, 3 ) );
      Day := StrToInt ( Data^ [ 1 ] + Data^ [ 2 ] );
      Year := StrToInt ( copy ( Data^, 8, 4 ) );
      if Month = 0 then
         begin
         Blip;
         CurPos := 3;
         Select;
         Valid := False;
         Draw;
         exit;
         end;

      Day := StrToInt ( Data^ [ 1 ] + Data^ [ 2 ] );
      if not ValidDay ( Day, Month ) then
         begin
         Blip;
         CurPos := 0;
         Select;
         Valid := False;
         Draw;
         exit;
         end;

      if ( Month = 2 ) and ( Day = 29 ) and ( not LeapYear ( Year ) ) then
         begin
         Blip;
         CurPos := 0;
         Select;
         Valid := False;
         Draw;
         exit;
         end;

      Valid := inherited Valid ( Command );
      end;
End;

{========================================================================}

Procedure TDateField.GetData ( var Rec );

Begin
   move ( JDate, Rec, DataSize );
End;

{========================================================================}

Procedure TDateField.SetState ( aState : word; Enable : boolean );

Begin
   { update the number when the field loses the focus }
   if ( aState = sfFocused ) and ( not Enable ) then
      begin
      JDate := StrToJDate ( Data^ );
      end;

   { reset the cursor when the field gains the focus}
   if ( aState = sfFocused ) and ( Enable ) then
      begin
      SelEnd := SelStart;
{      SetCursor ( Cursor.X, Cursor.Y );}
      end;

   inherited SetState ( aState, Enable );
End;

{========================================================================}

Procedure TDateField.HandleEvent ( var Event : TEvent );

const
   ValidNumKeys  : TCharSet = [ '0'..'9' ];
   ValidCharKeys : TCharSet = [ 'A'..'Z', 'a'..'z' ];
   AllChars = [ #32..#255 ];

var
   Day, Month, Year : integer;

Procedure AddChar;
Begin
   Data^ [ CurPos + 1 ] := UpCase ( Event.CharCode );
   inc ( CurPos );
   ClearEvent ( Event );
End;

Begin
   if ( Event.What = evKeyboard ) then
      begin
      { add the key to the input data string (overwrite mode) }
      if ( Event.CharCode in ValidNumKeys ) and
         ( ( CurPos < 2 ) or ( CurPos > 6 ) ) then
         AddChar
      else
      if ( Event.CharCode in ValidCharKeys ) and
         ( ( CurPos > 2 ) and ( CurPos < 6 ) ) then
         AddChar
      else
      if ( CtrlToArrow ( Event.KeyCode ) = ( kbLeft      ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbBack      ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbRight     ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbHome      ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbEnd       ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbDel       ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbCtrlLeft  ) ) or
         ( CtrlToArrow ( Event.KeyCode ) = ( kbCtrlRight ) ) then
         begin
         case CtrlToArrow ( Event.KeyCode ) of
            kbLeft,
            kbBack  : begin
                      dec ( CurPos );
                      if ( CurPos = 2 ) or ( CurPos = 6 ) then
                         dec ( CurPos );
                      end;
            kbRight : inc ( CurPos );
            kbCtrlRight : if CurPos < 2 then
                             CurPos := 3
                          else
                          if CurPos < 6 then
                             CurPos := 7
                          else
                             CurPos := Size.X - 2;
            kbCtrlLeft  : if CurPos > 7 then
                             CurPos := 7
                          else
                          if CurPos > 3 then
                             CurPos := 3
                          else
                             CurPos := 0;
            kbEnd   : CurPos := Size.X - 3;
            kbHome  : CurPos := 0;
            kbDel   : ;
            end; { case }
         ClearEvent ( Event );
         end
      else
      if Event.CharCode in AllChars then
         ClearEvent ( Event )
      else
         inherited HandleEvent ( Event );

      if CurPos < 0 then
         CurPos := 0;
      if CurPos > ( Size.X - 3 ) then
         CurPos := Size.X - 3;

      { test for valid day of the month after the 2nd digit }
      if CurPos = 2 then
         begin
         Day := StrToInt ( Data^ [ 1 ] + Data^ [ 2 ] );
         if Day <= 31 then
            inc ( CurPos )
         else
            begin
            Blip;
            CurPos := 0;
            end;
         end;

      { test for valid month after the 6th character }
      if CurPos = 6 then
         begin
         Month := ValidMonth ( copy ( Data^, 4, 3 ) );
         if Month = 0 then
            begin
            Blip;
            CurPos := 3;
            end
         else
            begin
            { test for valid day for this month }
            Day := StrToInt ( Data^ [ 1 ] + Data^ [ 2 ] );
            if ValidDay ( Day, Month ) then
               inc ( CurPos )
            else
               begin
               Blip;
               CurPos := 0;
               end;
            end;
         end;

      DrawView;
      end;

   inherited HandleEvent ( Event );
End;

{========================================================================}
{ Enumerator List Box object                                             }
{========================================================================}

Constructor TEnumeratorListBox.Init ( var R : TRect; ScrollBar : PScrollBar);

Begin
   inherited Init ( R, 1, ScrollBar );
End;

{========================================================================}

Procedure TEnumeratorListBox.HandleEvent;

Begin
   if ( ( Event.What = evMouseDown ) and ( Event.Double ) ) or
      ( ( Event.What = evKeyDown ) and ( Event.KeyCode = kbEnter ) ) then
      begin
      Event.What := evCommand;
      Event.Command := cmOK;
      PutEvent ( Event );
      ClearEvent ( Event );
      end;

   if ( ( Event.What = evKeyDown ) and ( Event.KeyCode = kbTab ) ) then
      begin
      Event.What := evCommand;
      Event.Command := cmTabToNext;
      PutEvent ( Event );
      ClearEvent ( Event );
      end;

   inherited HandleEvent ( Event );
End;

{========================================================================}

Function TEnumeratorListBox.GetPalette : PPalette;

const
   P : string [ Length ( CHistoryViewer ) ] = CHistoryViewer;

Begin
   GetPalette := @P;
End;

{========================================================================}
{ Enumerator Window Object                                               }
{========================================================================}

Constructor TEnumeratorWindow.Init ( R : TRect; EnumId : TEnumId; Item : word );

Begin
   inherited Init ( R, '', wnNoNumber );
   Flags := Flags and not ( wfGrow + wfMove + wfZoom );

   GetExtent ( R );
   R.Grow ( -1, -1 );
   ListBox := New ( PEnumeratorListBox, Init ( R, StandardScrollBar ( sbVertical or sbHandleKeyboard ) ) );
   ListBox^.NewList ( Enumerator [ EnumId ] );
   Insert ( ListBox );
   ListBox^.FocusItem ( Item );
End;

{========================================================================}

Function TEnumeratorWindow.GetPalette : PPalette;

const
   P : string [ Length ( CHistoryWindow ) ] = CHistoryWindow;

Begin
   GetPalette := @P;
End;

{========================================================================}

Procedure TEnumeratorWindow.HandleEvent;

Begin
   if ( ( Event.What = evKeyDown )   and ( Event.KeyCode = kbEsc ) ) or
      ( ( Event.What = evMouseDown ) and ( Event.Buttons = mbRightButton ) ) then
      begin
      Event.What := evCommand;
      Event.Command := cmCancel;
      end;

   if Event.What = evCommand then
      case Event.Command of
         cmOK, cmTabToNext, cmCancel : EndModal ( Event.Command );
      end;

   inherited HandleEvent ( Event );
End;

{========================================================================}
{ Enumerator Field object                                                }
{========================================================================}

Constructor TEnumeratorField.Init ( R : TRect; vId : TEnumId );

Begin
   R.B.X := R.A.X + Enumerator [ vId ]^.MaxLen + 5;
   inherited Init ( R, Enumerator [ vId ]^.MaxLen + 5 );
   Id := vId;
   Current := 0;
End;

{========================================================================}

Constructor TEnumeratorField.Load ( var S : TStream );

Begin
   inherited Load ( S );
   S.Read ( Id, SizeOf ( Id ) );
   S.Read ( Current, SizeOf ( Current ) );
End;

{========================================================================}

Procedure TEnumeratorField.Store ( var S : TStream );

Begin
   inherited Store ( S );
   S.Write ( Id, SizeOf ( Id ) );
   S.Write ( Current, SizeOf ( Current ) );
End;

{========================================================================}

Procedure TEnumeratorField.GetData ( var Rec );

Begin
   Move ( Current, Rec, DataSize );
End;

{========================================================================}

Procedure TEnumeratorField.SetData ( var Rec );

Begin
   Move ( Rec, Current, DataSize );
   Data^ := PString ( Enumerator [ Id ]^.At ( Current ) )^;
End;

{========================================================================}

Function TEnumeratorField.DataSize : word;

Begin
   DataSize := SizeOf ( Current );
End;

{========================================================================}

Function TEnumeratorField.GetPalette : PPalette;

const
   CEnumeratorField = CInputLine + #22 + #23;
   P : string [ length ( CEnumeratorField ) ] = CEnumeratorField;

Begin
   GetPalette := @P;
End;

{========================================================================}

Procedure TEnumeratorField.Draw;

var
   Color : byte;
   B     : TDrawBuffer;

Begin
   if ( State and sfFocused = 0 ) then
      Color := GetColor ( 1 )
   else
      Color := GetColor ( 2 );
   MoveChar ( B, ' ', Color, Size.X );
   MoveStr  ( B [ 1 ], Data^, Color ) ;
   MoveCStr ( B [ Size.X - 3 ], #222'~'#25'~'#221, GetColor ( $0506 ) );
   WriteLine ( 0, 0, Size.X, 1, B );
   SetCursor ( CurPos + 1, Cursor.Y );
End;

{========================================================================}

Procedure TEnumeratorField.HandleEvent ( var Event : TEvent );

var
   R, Bounds : TRect;
   Window    : PEnumeratorWindow;
   Result    : word;
   TheStr, CompStr : string[80];
   Match : boolean;
   PosCurrent : integer;
   TheChar : char;

Begin
   { convert all mouse events to local corordinates }
   if ( Event.What = evMouseDown ) then
      MakeLocal ( Event.Where, Event.Where );

   { scroll the enumerator field }
   if ( ( Event.What = evKeyboard  ) and ( Event.CharCode = ' ' ) ) or
      ( ( Event.What = evMouseDown ) and ( Event.Where.X <= Size.X - 3 ) ) and
      ( ( State and sfFocused <> 0 ) ) then
      begin
      inc ( Current );
      if Current > Enumerator [ Id ]^.Limit - 1 then
         Current := 0;
      Data^ := PString ( Enumerator [ Id ]^.At ( Current ) )^;
      CurPos := 0;
      DrawView;
      ClearEvent ( Event );
      end
   else

   { pop up the selection box }
   if ( ( Event.What = evKeyboard  ) and ( Event.KeyCode = kbDown ) )  or
      ( ( Event.What = evMouseDown ) and ( Event.Where.X > Size.X - 3 ) ) and
      ( ( State and sfFocused <> 0 ) ) then
      begin
      Owner^.GetExtent ( Bounds );
      GetBounds ( R );
      Inc ( R.A.Y );
      if ( R.A.Y + Enumerator [ Id ]^.Limit + 2 ) > Bounds.B.Y then
         R.B.Y := Bounds.B.Y
      else
         R.B.Y := R.A.Y + Enumerator [ Id ]^.Limit + 2;

      { move the box up until at least it can display 2 items }
      while ( ( R.B.Y - R.A.Y ) < 4 ) and ( R.A.Y > Bounds.A.Y ) do
         dec ( R.A.Y );

      Window := New ( PEnumeratorWindow, Init ( R, Id, Current ) );
      Result := Owner^.ExecView ( Window );
      if Result <> cmCancel then
         begin
         Current := Window^.ListBox^.Focused;
         Data^ := PString ( Enumerator [ Id ]^.At ( Current ) )^;
         DrawView;
         end;
      dispose ( Window, Done );
      if Result = cmTabToNext then
         Owner^.SelectNext ( False );
      ClearEvent ( Event );
      end;

   if ( Event.What = evKeyboard ) and
      ( Event.CharCode in [ #32..#127 ] ) then
      begin
        { store user selected string }
      CompStr := UpCaseStr ( copy ( Data^, 1, CurPos ) ) + upcase ( Event.charcode );
      PosCurrent := 0;
      Match := false;
      repeat
        TheStr := pString ( Enumerator [ Id ]^.At ( PosCurrent ) )^;
        if copy ( UpCaseStr ( TheStr ), 1, CurPos + 1 ) = CompStr then
        begin
          Data^ := TheStr;
          Current := PosCurrent;
          inc ( CurPos );
          Match := true;
          Draw;
        end;
        inc ( PosCurrent );
      until Match or ( PosCurrent > ( Enumerator [ Id ]^.Limit - 1 ) );
      ClearEvent ( Event );
   end;

   inherited HandleEvent ( Event );
End;

{========================================================================}

End.
