{==EZCrtErr===========================================================

A unit that provides a simple method to replace the interrupt 24
handler (critical error handler).

WARNING: a critical error handler is *extremely* limited in terms of
which DOS calls it can make. See a DOS programmers book for guidelines
on what you can or cannot do.

The unit can be compiled in real or protected mode.

EZINT24 is Copyright (c) 1994 by  Julian M. Bucknall

VERSION HISTORY
24Apr94 JMB 1.00 initial release
======================================================================}

unit EzCrtErr;

{------Common compiler switches---------------------------------------}
{$A+   Word align variables }
{$B-   Short-circuit boolean expressions }
{$F+   Force Far calls }
{$I-   No I/O checking }
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q-   No integer overflow checking }
{$R-   No range checking }
{$S-   No stack checking }
{$T-   @ operator is NOT typed }
{$V-   Disable var string checking }
{$X+   Enable extended syntax }
{$IFDEF DEBUG}
{$D+,L+,Y+  Enable debug information }
{$ENDIF}
{---------------------------------------------------------------------}

{------Real mode compiler switches------------------------------------}
{$IFDEF MSDOS}
{$E+   Enable coprocessor emulation }
{$G-   8086 type instructions }
{$O-   Do NOT allow overlays }
{$DEFINE RealMode}
{$UNDEF  ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Protected mode compiler switches-------------------------------}
{$IFDEF DPMI}
{$E+   Enable coprocessor emulation }
{$G+   80286+ type instructions }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Windows compiler switches--------------------------------------}
{$IFDEF WINDOWS}
{$G+   80286+ type instructions }
{$K+   Use smart callbacks
{$W-   No Windows realmode stack frame }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{$IFDEF Windows} Error - DOS real & protected mode only {$ENDIF}

INTERFACE

type
  PDeviceHeader = ^TDeviceHeader;            {The device driver header}
  TDeviceHeader = record
    Link : pointer;                          {Link to next device header}
    Attr : word;                             {Device attribute word}
    StrategyPtr : word;                      {Strategy routine}
    InterruptPtr : word;                     {Interrupt routine}
    case boolean of
      false : (Name : array [1..8] of char); {if char device: name}
      true  : (BlockDeviceNumber : byte);    {if block device: block number}
  end;

  TCritErrPosition = (               {Block device error occurred...}
                      cepMSDOS,      {..within MSDOS}
                      cepFAT,        {..accessing the FAT}
                      cepDirectory,  {..accessing the root dir}
                      cepDataArea);  {..accessing disk data area}

  TCritErrResponse = (               {Valid responses to critical error...}
                      cerIgnore,     {..ignore it}
                      cerRetry,      {..retry it}
                      cerAbort,      {..abort it (not allowed in pmode)}
                      cerFail);      {..fail it}

  TCritErrData = record                {Critical Error Data Block}
    IsWriteError  : boolean;           {..error on a write operation?}
    IsBlockDevice : boolean;           {..error on block device?}
    CanFail       : boolean;           {..allowed to fail?}
    CanRetry      : boolean;           {..allowed to retry?}
    CanIgnore     : boolean;           {..allowed to ignore?}
    DriveNumber   : byte;              {..0=A:, 1=B:, etc for block device}
    ErrorCode     : byte;              {..error code}
    ErrorPos      : TCritErrPosition;  {..if block device, where occurred}
    DeviceHeader  : PDeviceHeader;     {..device header block: READONLY}
  end;

  CritErrHandler = function (var CritErrData : TCritErrData) : TCritErrResponse;
                            {Template for critical error handler}

{=InstallCritErrHandler===============================================
Installs a new critical error handler. If one is already installed,
this will automatically unhook it and replace it with the new one.
24Apr94 JMB
======================================================================}
procedure InstallCritErrHandler(Handler : CritErrHandler);

{=RemoveCritErrHandler================================================
Deinstalls the critical error handler.
24Apr94 JMB
======================================================================}
procedure RemoveCritErrHandler;

IMPLEMENTATION

{$IFDEF DPMI}
uses
  EZDPMI, WinAPI, WinDOS;
{$ELSE}
uses
  DOS;
{$ENDIF}

const
  Hooked : boolean = false;
  ExitSave : pointer = nil;

var
  OldInt24 : pointer;
  UserHandler : CritErrHandler;

procedure UnitExitProc; far;
  begin
    ExitProc := ExitSave;
    RemoveCritErrHandler;
  end;

{$IFDEF MSDOS}

procedure NewInt24(Flags, CS, IP, AX, BX, CX, DX, SI, DI, DS, ES, BP: Word); interrupt;
  var
    CritErrData : TCritErrData;
  begin
    with CritErrData do
      begin
        IsWriteError := (AX and $0100) <> 0;
        IsBlockDevice := (AX and $8000) = 0;
        CanFail := (AX and $0800) <> 0;
        CanRetry := (AX and $1000) <> 0;
        CanIgnore := (AX and $2000) <> 0;
        DriveNumber := lo(AX);
        ErrorCode := lo(DI);
        ErrorPos := TCritErrPosition((AX shr 9) and $03);
        DeviceHeader := Ptr(BP, SI);
        if not IsBlockDevice then
          IsBlockDevice := (DeviceHeader^.Attr and $8000) = 0;
      end;
    AX := byte(UserHandler(CritErrData));
  end;

procedure InstallCritErrHandler(Handler : CritErrHandler);
  begin
    if not Assigned(Handler) then
      RemoveCritErrHandler
    else
      begin
        if not Hooked then
          begin
            GetIntVec($24, OldInt24);
            SetIntVec($24, @NewInt24);
            if not Assigned(ExitSave) then
              begin
                ExitSave := ExitProc;
                ExitProc := @UnitExitProc;
              end;
            Hooked := true;
          end;
        UserHandler := Handler;
      end;
  end;

procedure RemoveCritErrHandler;
  begin
    if Hooked then
      begin
        SetIntVec($24, OldInt24);
        Hooked := false;
      end;
  end;

{$ELSE} {protected mode}

var
  NewInt24 : RealProc;
  OurDeviceHeader : PDeviceHeader;

procedure Int24Handler(var Regs : TRegisters); far;
  var
    CritErrData : TCritErrData;
  begin
    with Regs, CritErrData do
      begin
        IsWriteError := (AH and $01) <> 0;
        IsBlockDevice := (AH and $80) = 0;
        CanFail := (AH and $08) <> 0;
        CanRetry := (AH and $10) <> 0;
        CanIgnore := (AH and $20) <> 0;
        DriveNumber := AL;
        ErrorCode := lo(DI);
        ErrorPos := TCritErrPosition((AH shr 1) and $03);
        ReassignMappedDPMIPtr(OurDeviceHeader, Ptr(BP, SI), sizeof(TDeviceHeader));
        DeviceHeader := OurDeviceHeader;
        if not IsBlockDevice then
          IsBlockDevice := (DeviceHeader^.Attr and $8000) = 0;
      end;
    Regs.AL := byte(UserHandler(CritErrData));
  end;

procedure InstallCritErrHandler(Handler : CritErrHandler);
  begin
    if not Assigned(Handler) then
      RemoveCritErrHandler
    else
      begin
        if not Hooked then
          if GetMappedDPMIPtr(OurDeviceHeader, nil, sizeof(TDeviceHeader)) then
            begin
              GetCallBack(Int24Handler, true, NewInt24);
              if not Assigned(NewInt24) then
                FreeMappedDPMIPtr(OurDeviceHeader)
              else
                begin
                  GetRealIntVec($24, OldInt24);
                  SetRealIntVec($24, @NewInt24);
                  if not Assigned(ExitSave) then
                    begin
                      ExitSave := ExitProc;
                      ExitProc := @UnitExitProc;
                    end;
                  Hooked := true;
                end;
            end;
        UserHandler := Handler;
      end;
  end;

procedure RemoveCritErrHandler;
  begin
    if Hooked then
      begin
        SetRealIntVec($24, OldInt24);
        DestroyCallBack(NewInt24);
        FreeMappedDPMIPtr(OurDeviceHeader);
        Hooked := false;
      end;
  end;

{$ENDIF}

end.
