{==EZCTLBRK===========================================================

A unit that traps the Ctrl-Break interrupt and interfaces a boolean
function to test for the user pressing Ctrl-Break.

The unit can be compiled in real or protected mode.

EZCTLBRK is Copyright (c) 1994 by  Julian M. Bucknall

VERSION HISTORY
24Apr94 JMB 1.00 initial release
======================================================================}

unit EzCtlBrk;

{------Common compiler switches---------------------------------------}
{$A+   Word align variables }
{$B-   Short-circuit boolean expressions }
{$F+   Force Far calls }
{$I-   No I/O checking }
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q-   No integer overflow checking }
{$R-   No range checking }
{$S-   No stack checking }
{$T-   @ operator is NOT typed }
{$V-   Disable var string checking }
{$X+   Enable extended syntax }
{$IFDEF DEBUG}
{$D+,L+,Y+  Enable debug information }
{$ENDIF}
{---------------------------------------------------------------------}

{------Real mode compiler switches------------------------------------}
{$IFDEF MSDOS}
{$E+   Enable coprocessor emulation }
{$G-   8086 type instructions }
{$O-   Do NOT allow overlays }
{$DEFINE RealMode}
{$UNDEF  ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Protected mode compiler switches-------------------------------}
{$IFDEF DPMI}
{$E+   Enable coprocessor emulation }
{$G+   80286+ type instructions }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Windows compiler switches--------------------------------------}
{$IFDEF WINDOWS}
{$G+   80286+ type instructions }
{$K+   Use smart callbacks
{$W-   No Windows realmode stack frame }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{$IFDEF Windows} Error - DOS real & protected mode only {$ENDIF}

{-Define this directive to automatically load the handler-}
{.$DEFINE AutoLoadHandler}

INTERFACE

{=CtrlBreakWasHit=====================================================
Returns true if Ctrl+Break has been pressed since the last time this
routine was called, or since installation of the handler.
24Apr94 JMB
======================================================================}
function CtrlBreakWasHit : boolean;

{=InstallCtrlBreakHandler=============================================
Installs a new Ctrl+Break handler. If it has already been installed,
this will do nothing.
24Apr94 JMB
======================================================================}
procedure InstallCtrlBreakHandler;

{=RemoveCtrlBreakHandler==============================================
Deinstalls the Ctrl+Break handler that was installed with
InstallCtrlBreakHandler. Note that this routine will be automatically
called on program termination.
24Apr94 JMB
======================================================================}
procedure RemoveCtrlBreakHandler;

IMPLEMENTATION

{$IFDEF DPMI}
uses
  EZDPMI, WinDOS;
{$ELSE}
uses
  DOS;
{$ENDIF}

const
  Hooked : boolean = false;    {Have we hooked int $1B yet?}
  ExitSave : pointer = nil;    {The exit chain}

var
  OldInt1B : pointer;          {The previous int $1B handler}
  KeyHit   : boolean;          {Key has been hit}

function CtrlBreakWasHit : boolean; assembler;
  asm
    xor ax, ax
    mov al, KeyHit
    mov KeyHit, ah
  end;

procedure UnitExitProc; far;
  {Exit procedure for this unit}
  begin
    ExitProc := ExitSave;
    RemoveCtrlBreakHandler;
  end;

{$IFDEF MSDOS}

procedure NewInt1B; far; assembler;
  asm
    push ax
    push ds
    mov ax, seg @Data
    mov ds, ax
    mov KeyHit, 1
    pop ds
    pop ax
    iret
  end;

procedure InstallCtrlBreakHandler;
  begin
    if not Hooked then
      begin
        GetIntVec($1B, OldInt1B);
        SetIntVec($1B, @NewInt1B);

        if not Assigned(ExitSave) then
          begin
            ExitSave := ExitProc;
            ExitProc := @UnitExitProc;
          end;

        Hooked := true;
      end;

    KeyHit := false;
  end;

procedure RemoveCtrlBreakHandler;
  begin
    if Hooked then
      begin
        SetIntVec($1B, OldInt1B);
        Hooked := false;
        KeyHit := false;
      end;
  end;

{$ELSE} {protected mode}

var
  NewInt1B : RealProc;

procedure Int1BHandler(var Regs : TRegisters); far;
  begin
    KeyHit := true;
  end;

procedure InstallCtrlBreakHandler;
  begin
    if not Hooked then
      begin
        GetCallBack(Int1BHandler, true, NewInt1B);
        if Assigned(NewInt1B) then
          begin
            GetRealIntVec($1B, OldInt1B);
            SetRealIntVec($1B, @NewInt1B);

            if not Assigned(ExitSave) then
              begin
                ExitSave := ExitProc;
                ExitProc := @UnitExitProc;
              end;

            Hooked := true;
          end;
      end;

    KeyHit := false;
  end;

procedure RemoveCtrlBreakHandler;
  begin
    if Hooked then
      begin
        SetRealIntVec($1B, OldInt1B);
        DestroyCallBack(NewInt1B);
        Hooked := false;
        KeyHit := false;
      end;
  end;

{$ENDIF}

{$IFDEF AutoLoadHandler}
begin
  InstallCtrlBreakHandler;
{$ENDIF}
end.
