{==EZDPMI=============================================================

A protected mode DPMI access unit. Provides an easy-to-use interface
to the common application-oriented requirements for DPMI: the ability
to interface real-mode drivers and TSRs from protected mode using DOS
memory.

EZDPMI is Copyright (c) 1993, 1994 by  Julian M. Bucknall

VERSION HISTORY
24Apr94 JMB 1.10 added callbacks; coalesced previous versions
24Jan94 JMB 1.03 added stack transfer to RealCall
05Sep93 JMB 1.02 split documentation from source, added interrupt
                 vector get/set routines
10Aug93 JMB 1.01 fixed GetMappedDPMIPtr
                 (thanx to Garr Updegraff for spotting this bug)
21Mar93 JMB 1.00 initial release
======================================================================}

unit EzDPMI;

{------Common compiler switches---------------------------------------}
{$A+   Word align variables }
{$B-   Short-circuit boolean expressions }
{$F+   Force Far calls }
{$I-   No I/O checking }
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q-   No integer overflow checking }
{$R-   No range checking }
{$S-   No stack checking }
{$T-   @ operator is NOT typed }
{$V-   Disable var string checking }
{$X+   Enable extended syntax }
{$IFDEF DEBUG}
{$D+,L+,Y+  Enable debug information }
{$ENDIF}
{---------------------------------------------------------------------}

{------Real mode compiler switches------------------------------------}
{$IFDEF MSDOS}
{$E+   Enable coprocessor emulation }
{$G-   8086 type instructions }
{$O-   Do NOT allow overlays }
{$DEFINE RealMode}
{$UNDEF  ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Protected mode compiler switches-------------------------------}
{$IFDEF DPMI}
{$E+   Enable coprocessor emulation }
{$G+   80286+ type instructions }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Windows compiler switches--------------------------------------}
{$IFDEF WINDOWS}
{$G+   80286+ type instructions }
{$K+   Use smart callbacks
{$W-   No Windows realmode stack frame }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{$IFDEF MSDOS} Error - protected mode only {$ENDIF}

INTERFACE

uses WinDOS,
{$IFDEF Windows}
     WinProcs
{$ELSE}
     WinAPI
{$ENDIF}
     ;

{$IFDEF Windows}
var
  Seg0040 : word;     { To access the BIOS data area in Windows }
{$ENDIF}

type
  RealProc = procedure;
  CallBackProc = procedure (var Regs : TRegisters);

{=DOSGetMem===========================================================
Allocates and returns the real and protected mode pointers to a DOS
memory block of Size bytes in the first 1Mb. Returns true if
successful, false otherwise.
21Mar93 JMB
======================================================================}
function DOSGetMem(var RealPtr, ProtPtr; Size : word) : boolean;

{=DOSFreeMem==========================================================
Deallocates a DOS memory block allocated with DOSGetMem. Returns true
if successful, false otherwise.
21Mar93 JMB
======================================================================}
function DOSFreeMem(ProtPtr : pointer) : boolean;

{=RealIntr============================================================
Calls the real mode interrupt IntNo. Unlike Intr this guarantees a
real mode interrupt. Intr performs a protected mode interrupt first,
which the DPMI server may pass thru to the real mode interrupt.
Returns true if successful, false otherwise.
21Mar93 JMB
======================================================================}
function RealIntr(IntNo : byte; var Regs : TRegisters) : boolean;

{=RealCall============================================================
Calls the real mode Routine procedure (must be a far procedure and
return with RETF). StackWords words will be transferred from the
memory block pointed to by StackToCopy to the stack prior to making
the real-mode call. If no stack needs to be copied use 0 for
StackWords and nil for StackToCopy.
Returns true if successful, false otherwise.
21Mar93 JMB
======================================================================}
function RealCall(Routine : RealProc;
                  var Regs : TRegisters;
                  StackWords  : word;
                  StackToCopy : pointer) : boolean;

{=GetMappedDPMIPtr====================================================
Given a real mode pointer to a DOS memory block, returns a protected
mode pointer mapped to the same block.
Returns true if successful, false otherwise.
21Mar93 JMB
======================================================================}
function GetMappedDPMIPtr(var ProtPtr; RealPtr : pointer; Size : word)
           : boolean;

{=FreeMappedDPMIPtr===================================================
Frees a protected mode pointer (ie selector) that was allocated by
GetMappedDPMIPtr.
Returns true if successful, false otherwise.
21Mar93 JMB
======================================================================}
function FreeMappedDPMIPtr(ProtPtr : pointer) : boolean;

{=ReassignMappedDPMIPtr===============================================
Given a protected mode pointer (that has already been mapped with
GetMappedDPMIPtr) and a real mode pointer to a DOS memory block,
returns a protected mode pointer mapped to the same block (ie a remaps
the protected mode pointer to another realmode memory block).
Returns true if successful, false otherwise.
24Apr94 JMB
======================================================================}
function ReassignMappedDPMIPtr(var ProtPtr; RealPtr : pointer; Size : word)
           : boolean;

{=GetRealIntVec=======================================================
Returns the real mode interrupt vector for interrupt Int. This is a
real-mode pointer, do not dereference it!
05Sep93 JMB version 1.02
======================================================================}
procedure GetRealIntVec(IntNo: Byte; var Vector: Pointer);

{=SetRealIntVec=======================================================
Sets the real mode interrupt vector for interrupt Int. This must be a
real-mode pointer to a real-mode routine.
Returns true if successful, false otherwise.
05Sep93 JMB version 1.02
======================================================================}
function SetRealIntVec(IntNo: Byte; Vector: Pointer) : boolean;

{=GetProtIntVec=======================================================
Returns the protected mode interrupt vector for interrupt Int. Note
that GetIntVec in the DOS/WinDOS unit returns this vector as well.
05Sep93 JMB version 1.02
======================================================================}
procedure GetProtIntVec(IntNo: Byte; var Vector: Pointer);

{=SetProtIntVec=======================================================
Sets the protected mode interrupt vector for interrupt Int. This must
be a protected mode pointer to a protected mode routine.
Returns true if successful, false otherwise.
05Sep93 JMB version 1.02
======================================================================}
function SetProtIntVec(IntNo: Byte; Vector: Pointer) : boolean;

{=GetCallBack=========================================================
Returns the address of a small realmode stub routine that will switch
to protected mode and call Handler. If the routine is to be called
via a realmode interrupt then IsForInterrupt must be true, else it
must be false. If any error occurred, the returned value of CallBack
is nil.
24Apr94 JMB version 1.10
======================================================================}
procedure GetCallBack(Handler : CallBackProc; IsForInterrupt : boolean;
                      var CallBack : RealProc);

{=DestroyCallBack=====================================================
Destroys a callback routine allocated with GetCallBack.
24Apr94 JMB version 1.10
======================================================================}
procedure DestroyCallBack(CallBack : RealProc);

IMPLEMENTATION

type
  OS = record O, S : word; end; {to split pointers into sel/seg & ofs}
  LH = record L, H : word; end; {to split longints into high & low words}

type
  TDPMIRegisters = record
    EDI, ESI, EBP, Res, EBX, EDX, ECX, EAX  : longint;
    Flags, ES, DS, FS, GS, IP, CS, SP, SS   : word;
  end;

type
  PCallBackData = ^TCallBackData;
  TCallBackData = record
    DPMIRegs    : TDPMIRegisters;       {This MUST be first}
    Regs        : TRegisters;
    UserHandler : CallBackProc;
    CallBackAddr: pointer;
    Next        : PCallBackData;
    DataSeg     : word;
    IsIntHandler: boolean;
  end;

var
  CallBackList : PCallBackData;
  ExitSave : pointer;

function DOSGetMem(var RealPtr, ProtPtr; Size : word) : boolean;
  var
    RealMode : pointer absolute RealPtr;
    ProtMode : pointer absolute ProtPtr;
    AllocResult : longint;
  begin
    AllocResult := GlobalDOSAlloc(Size);
    if (AllocResult <> 0) then
      begin
        RealMode := Ptr(LH(AllocResult).H, 0);
        ProtMode := Ptr(LH(AllocResult).L, 0);
        DOSGetMem := true;
      end
    else DOSGetMem := false;
  end;

function DOSFreeMem(ProtPtr : pointer) : boolean;
  begin
    DOSFreeMem := GlobalDOSFree(OS(ProtPtr).S) = 0;
  end;

function RealIntr(IntNo : byte; var Regs : TRegisters) : boolean;
assembler;
  var
    DPMIregs : TDPMIRegisters;
  asm
    push ds
    lds si, Regs
    mov ax, ss;  mov es, ax;  lea di, DPMIregs
    cld
    xor ax, ax
    add si, 12         { EDI }
    movsw;  stosw
    sub si, 4          { ESI }
    movsw;  stosw
    sub si, 4          { EBP }
    movsw;  stosw
    stosw;  stosw      { Res }
    sub si, 8          { EBX }
    movsw;  stosw
    add si, 2          { EDX }
    movsw;  stosw
    sub si, 4          { ECX }
    movsw;  stosw
    sub si, 6          { EAX }
    movsw;  stosw
    add si, 16         { Flags }
    movsw;
    sub si, 4          { ES }
    movsw;
    sub si, 4          { DS }
    movsw;
    mov cx, 6          { FS, GS, IP, CS, SP, SS }
    rep stosw
    lea di, DPMIregs
    mov ax, 0300h      { DPMI code to simulate intr }
    xor bx, bx         { Set BH to zero (and BL) }
    mov bl, IntNo      { Save interrupt number }
    xor cx, cx         { No stack words to copy }
    int 31h            { DPMI Services }
    mov ax, 0
    jc @@ExitPoint     { Error? - yes }
    les di, Regs
    mov ax, ss;  mov ds, ax;  lea si, DPMIregs
    cld
    add si, 28; movsw  { AX }
    sub si, 14; movsw  { BX }
    add si, 6;  movsw  { CX }
    sub si, 6;  movsw  { DX }
    sub si, 14; movsw  { BP }
    sub si, 6;  movsw  { SI }
    sub si, 6;  movsw  { DI }
    add si, 34; movsw  { DS }
    sub si, 4;  movsw  { ES }
    sub si, 4;  movsw  { Flags }
    mov ax, 1
  @@ExitPoint:
    pop ds
  end;

function RealCall(Routine : RealProc;
                  var Regs : TRegisters;
                  StackWords  : word;
                  StackToCopy : pointer) : boolean;
assembler;
  var
    DPMIregs : TDPMIRegisters;
  asm
    push ds
    mov cx, StackWords
    jcxz @@DoneWithStack
    lds si, StackToCopy
    mov bx, cx
    dec bx
    shl bx, 1
    add si, bx
    std
  @@StackLoop:
    lodsw
    push ax
    loop @@StackLoop
  @@DoneWithStack:
    cld
    lds si, Regs
    mov ax, ss;  mov es, ax;  lea di, DPMIregs
    xor ax, ax
    add si, 12         { EDI }
    movsw;  stosw
    sub si, 4          { ESI }
    movsw;  stosw
    sub si, 4          { EBP }
    movsw;  stosw
    stosw;  stosw      { Res }
    sub si, 8          { EBX }
    movsw;  stosw
    add si, 2          { EDX }
    movsw;  stosw
    sub si, 4          { ECX }
    movsw;  stosw
    sub si, 6          { EAX }
    movsw;  stosw
    add si, 16         { Flags }
    movsw;
    sub si, 4          { ES }
    movsw;
    sub si, 4          { DS }
    movsw;
    mov cx, 6          { FS, GS, IP, CS, SP, SS }
    rep stosw
    sub di, 8
    mov ax, Routine.Word[0]  { Routine's real IP }
    stosw
    mov ax, Routine.Word[2]  { Routine's real CS }
    stosw
    lea di, DPMIregs
    mov ax, 0301h      { DPMI code to simulate call }
    xor bx, bx         { Set BH to zero (and BL) }
    mov cx, StackWords { Num stack words to copy }
    int 31h            { DPMI Services }
    mov ax, 0
    jc @@ExitPoint     { Error? - yes }
    les di, Regs
    mov ax, ss;  mov ds, ax;  lea si, DPMIregs
    cld
    add si, 28; movsw  { AX }
    sub si, 14; movsw  { BX }
    add si, 6;  movsw  { CX }
    sub si, 6;  movsw  { DX }
    sub si, 14; movsw  { BP }
    sub si, 6;  movsw  { SI }
    sub si, 6;  movsw  { DI }
    add si, 34; movsw  { DS }
    sub si, 4;  movsw  { ES }
    sub si, 4;  movsw  { Flags }
    mov ax, 1
  @@ExitPoint:
    mov cx, StackWords
    shl cx, 1
    add sp, cx
    pop ds
  end;

procedure MapToRealAddress; near; assembler;
  {Entry: bx=selector, dx=segment value
   Exit:  bx=selector, ax & cx & dx trashed
          carry flag on if error}
  asm
    xor ax, ax
    mov al, dh
    mov cl, 4
    shr ax, cl
    shl dx, cl
    xchg ax, cx
    mov ax, 7
    int 31h
  end;

procedure SetLimit; near; assembler;
  {Entry: bx=selector, cx=offset value, dx=size
   Exit:  bx=selector, cx=offset value, ax & dx trashed
          carry flag on if error}
  asm
    mov ax, 8
    add dx, cx
    jnc @@1
    xor dx, dx
    dec dx
  @@1:
    int 31h
  end;


function GetMappedDPMIPtr(var ProtPtr; RealPtr : pointer; Size : word)
           : boolean;
assembler;
  asm
    xor ax, ax               { Get an LDT descriptor & selector for it }
    mov cx, 1
    int 31h
    jc @@Error
    xchg ax, bx              { Map to realmode address }
    mov dx, RealPtr.Word[2]
    call MapToRealAddress
    jc @@Error
    mov cx, RealPtr.Word[0]  { Size the selector }
    mov dx, Size
    call SetLimit
    jc @@Error
    cld                      { Save selector:offset in ProtPtr }
    les di, ProtPtr
    xchg ax, cx
    stosw
    xchg ax, bx
    stosw
    mov ax, 1
    jmp @@Exit
  @@Error:
    xor ax, ax
  @@Exit:
  end;

function ReassignMappedDPMIPtr(var ProtPtr; RealPtr : pointer; Size : word)
           : boolean;
assembler;
  asm
    les di, ProtPtr          { Get the current selector}
    mov bx, es:[di].Word[2]  { Map to realmode address }
    mov dx, RealPtr.Word[2]
    call MapToRealAddress
    jc @@Error
    mov cx, RealPtr.Word[0]  { Size the selector }
    mov dx, Size
    call SetLimit
    jc @@Error
    cld                      { Save offset in ProtPtr }
    xchg ax, cx              { ..the selector hasn't changed }
    stosw
    mov ax, 1
    jmp @@Exit
  @@Error:
    xor ax, ax
  @@Exit:
  end;

function FreeMappedDPMIPtr(ProtPtr : pointer) : boolean;
assembler;
  asm
    mov ax, 1
    mov bx, ProtPtr.Word[2]
    int 31h
    mov ax, 0
    jc @@Error
    inc ax
  @@Error:
  end;

procedure GetRealIntVec(IntNo: Byte; var Vector: Pointer);
assembler;
  asm
    mov bl, IntNo
    mov ax, 0200h
    int 31h
    {no errors - always succeeds}
    cld
    les di, Vector
    xchg ax, dx
    stosw
    xchg ax, cx
    stosw
  end;

function SetRealIntVec(IntNo: Byte; Vector: Pointer) : boolean;
assembler;
  asm
    mov bl, IntNo
    mov cx, Vector.Word[2]
    mov dx, Vector.Word[0]
    mov ax, 0201h
    int 31h
    mov ax, 0
    jc @@Error
    inc ax
  @@Error:
  end;

procedure GetProtIntVec(IntNo: Byte; var Vector: Pointer);
assembler;
  asm
    mov bl, IntNo
    mov ax, 0204h
    int 31h
    {no errors - always succeeds}
    cld
    les di, Vector
    xchg ax, dx
    stosw
    xchg ax, cx
    stosw
  end;

function SetProtIntVec(IntNo: Byte; Vector: Pointer) : boolean;
assembler;
  asm
    mov bl, IntNo
    mov cx, Vector.Word[2]
    mov dx, Vector.Word[0]
    mov ax, 0205h
    int 31h
    mov ax, 0
    jc @@Error
    inc ax
  @@Error:
  end;

{=MainEZDPMICallBack==================================================
Provides a single callback entry point. From the TCallBackData
structure data, it will call the user handler.
04Apr94 JMB
======================================================================}
procedure MainEZDPMICallBack; far;
assembler;
  const
    SizeOfDPMIRegs = sizeof(TDPMIRegisters);
  asm
    {ds:si ==> realmode stack;
     es:di ==> DPMI registers at top of TCallBackData struc}
    cld
    {set up return information}
    lodsw; mov es:[di].TDPMIRegisters.&IP, ax
    lodsw; mov es:[di].TDPMIRegisters.&CS, ax
    cmp es:[di].TCallBackData.IsIntHandler, 1
    je @@ManageInterrupt
    add es:[di].TDPMIRegisters.&SP, 4
    jmp @@CallUserHandler
  @@ManageInterrupt:
    add es:[di].TDPMIRegisters.&SP, 6
  @@CallUserHandler:
    {set up normal registers structure}
    push es; push di
    mov ax, es; mov ds, ax
    mov si, di
    add di, SizeOfDPMIRegs
    mov bx, si
    mov dx, di
    add si, 28; movsw  { AX }
    sub si, 14; movsw  { BX }
    add si, 6;  movsw  { CX }
    sub si, 6;  movsw  { DX }
    sub si, 14; movsw  { BP }
    sub si, 6;  movsw  { SI }
    sub si, 6;  movsw  { DI }
    add si, 34; movsw  { DS }
    sub si, 4;  movsw  { ES }
    sub si, 4;  movsw  { Flags }
    mov di, bx
    {set up the correct data segment}
    mov ax, es:[di].TCallBackData.DataSeg; mov ds, ax
    {call the user handler}
    push es
    push dx
    call es:[di].TCallBackData.UserHandler
    {set up return DPMI registers structure}
    pop di; pop es
    mov ax, es; mov ds, ax
    mov si, di
    add si, SizeOfDPMIRegs
    mov bx, di
    cld
    add di, 28; movsw  { AX }
    sub di, 14; movsw  { BX }
    add di, 6;  movsw  { CX }
    sub di, 6;  movsw  { DX }
    sub di, 14; movsw  { BP }
    sub di, 6;  movsw  { SI }
    sub di, 6;  movsw  { DI }
    add di, 34; movsw  { DS }
    sub di, 4;  movsw  { ES }
    sub di, 4;  movsw  { Flags }
    {return}
    mov di, bx
    iret
  end;

{=AllocCallBack=======================================================
Calls the DPMI server to get a realmode callback.
04Apr94 JMB
======================================================================}
function AllocCallBack(ProtRoutine : pointer;
                       Data : PCallBackData;
                       var RealRoutine : pointer) : boolean;
assembler;
  asm
    push ds
    mov ax, $0303
    mov si, ProtRoutine.Word[0]
    mov ds, ProtRoutine.Word[2]
    mov di, Data.Word[0]
    mov es, Data.Word[2]
    int $31
    mov ax, 0
    jc @@Exit
    cld
    les di, RealRoutine
    xchg ax, dx
    stosw
    xchg ax, cx
    stosw
    mov ax, 1
  @@Exit:
    pop ds
  end;

{=AllocCallBack=======================================================
Calls the DPMI server to free a realmode callback.
04Apr94 JMB
======================================================================}
function FreeCallBack(Routine : pointer) : boolean;
assembler;
  asm
    mov ax, $0304
    mov dx, Routine.Word[0]
    mov cx, Routine.Word[2]
    int $31
    mov ax, 0
    jc @@Error
    inc ax
  @@Error:
  end;

procedure GetCallBack(Handler : CallBackProc; IsForInterrupt : boolean;
                      var CallBack : RealProc);
  var
    CBData : PCallBackData;
  begin
    CallBack := nil;

    New(CBData);
    if not Assigned(CBData) then
      Exit;
    FillChar(CBData^, sizeof(TCallBackData), 0);

    with CBData^ do
      begin
        if not AllocCallBack(@MainEZDPMICallBack, CBData, CallBackAddr) then
          begin
            Dispose(CBData);
            Exit;
          end;

        UserHandler := Handler;
        Next := CallBackList;
        DataSeg := DSeg;
        IsIntHandler := IsForInterrupt;

        CallBack := RealProc(CallBackAddr);

        CallBackList := CBData;
      end;
  end;

procedure DestroyCallBack(CallBack : RealProc);
  var
    Dad, Temp : PCallBackData;
  begin
    Dad := CallBackList;
    Temp := CallBackList;

    while Assigned(Temp) and (Temp^.CallBackAddr <> @CallBack) do
      begin
        Dad := Temp;
        Temp := Temp^.Next;
      end;

    if (Temp <> nil) then
      with Temp^ do
        begin
          if (Temp = CallBackList) then
               CallBackList := Next
          else Dad^.Next := Next;

          if not FreeCallBack(CallBackAddr) then
            {nothing};

          Dispose(Temp);
        end;
  end;

{=CleanupEZDPMI=======================================================
Removes the Seg0040 selector for Windows. Destroys the callback list.
21Mar93 JMB
======================================================================}
procedure CleanupEZDPMI; far;
  var
    PP : pointer;
  begin
    ExitProc := ExitSave;
    {$IFDEF Windows}
    PP := Ptr(Seg0040, 0);
    FreeMappedDPMIPtr(PP);
    {$ENDIF}
    while Assigned(CallBackList) do
      DestroyCallBack(RealProc(CallBackList^.CallBackAddr));
  end;

{=Initialisation======================================================
Sets up the Seg0040 selector for Windows. Sets up the callback list.
21Mar93 JMB
======================================================================}
var
  PP : pointer;
begin
  {$IFDEF Windows}
  GetMappedDPMIPtr(PP, Ptr($40, 0), $400); {1024 byte limit}
  Seg0040 := OS(PP).S;
  {$ENDIF}
  CallBackList := nil;
  ExitSave := ExitProc;
  ExitProc := @CleanupEZDPMI;
end.
