{==EZPRTSC============================================================

A unit that traps the Print Screen (PrtSc) interrupt and interfaces a
boolean function to test for the user pressing the PrintScreen key.

The function is meant to be polled in your keyboard idle loop, and
if it returns true then you can call a routine to 'print the screen'
(or save it to disk, or just print the topmost window, or whatever).
An example:

  procedure KeyboardIdle;
    begin
      repeat
        if PrintScreenWasHit then
          PrintTheScreen;
      until KeyPressed;
    end;

The unit can be compiled in real or protected mode.

EZPRTSC is Copyright (c) 1994 by  Julian M. Bucknall

VERSION HISTORY
24Apr94 JMB 1.00 initial release
======================================================================}

unit EzPrtSc;

{------Common compiler switches---------------------------------------}
{$A+   Word align variables }
{$B-   Short-circuit boolean expressions }
{$F+   Force Far calls }
{$I-   No I/O checking }
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q-   No integer overflow checking }
{$R-   No range checking }
{$S-   No stack checking }
{$T-   @ operator is NOT typed }
{$V-   Disable var string checking }
{$X+   Enable extended syntax }
{$IFDEF DEBUG}
{$D+,L+,Y+  Enable debug information }
{$ENDIF}
{---------------------------------------------------------------------}

{------Real mode compiler switches------------------------------------}
{$IFDEF MSDOS}
{$E+   Enable coprocessor emulation }
{$G-   8086 type instructions }
{$O-   Do NOT allow overlays }
{$DEFINE RealMode}
{$UNDEF  ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Protected mode compiler switches-------------------------------}
{$IFDEF DPMI}
{$E+   Enable coprocessor emulation }
{$G+   80286+ type instructions }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{------Windows compiler switches--------------------------------------}
{$IFDEF WINDOWS}
{$G+   80286+ type instructions }
{$K+   Use smart callbacks
{$W-   No Windows realmode stack frame }
{$UNDEF  RealMode}
{$DEFINE ProtMode}
{$ENDIF}
{---------------------------------------------------------------------}

{$IFDEF Windows} Error - DOS real & protected mode only {$ENDIF}

{-Define this directive to automatically load the handler-}
{.$DEFINE AutoLoadHandler}

INTERFACE

{=PrintScreenWasHit===================================================
Returns true if PrintScreen has been pressed since the last time this
routine was called, or since installation of the handler.
24Apr94 JMB
======================================================================}
function PrintScreenWasHit : boolean;

{=InstallPrintScreenHandler===========================================
Installs a new Print Screen handler. If it has already been installed,
this will do nothing.
24Apr94 JMB
======================================================================}
procedure InstallPrintScreenHandler;

{=RemovePrintScreenHandler============================================
Deinstalls the PrintScreen handler that was installed with
InstallPrintScreenHandler. Note that this routine will be
automatically called on program termination.
24Apr94 JMB
======================================================================}
procedure RemovePrintScreenHandler;

IMPLEMENTATION

{$IFDEF DPMI}
uses
  EZDPMI, WinDOS;
{$ELSE}
uses
  DOS;
{$ENDIF}

const
  Hooked : boolean = false;    {Have we hooked int $05 yet?}
  ExitSave : pointer = nil;    {The exit chain}

var
  OldInt05 : pointer;          {The previous int $05 handler}
  KeyHit   : boolean;          {Key has been hit}

function PrintScreenWasHit : boolean; assembler;
  asm
    xor ax, ax
    mov al, KeyHit
    mov KeyHit, ah
  end;

procedure UnitExitProc; far;
  {Exit procedure for this unit}
  begin
    ExitProc := ExitSave;
    RemovePrintScreenHandler;
  end;

{$IFDEF MSDOS}

procedure NewInt05; far; assembler;
  asm
    push ax
    push ds
    mov ax, seg @Data
    mov ds, ax
    mov KeyHit, 1
    pop ds
    pop ax
    iret
  end;

procedure InstallPrintScreenHandler;
  begin
    if not Hooked then
      begin
        GetIntVec($05, OldInt05);
        SetIntVec($05, @NewInt05);

        if not Assigned(ExitSave) then
          begin
            ExitSave := ExitProc;
            ExitProc := @UnitExitProc;
          end;

        Hooked := true;
      end;

    KeyHit := false;
  end;

procedure RemovePrintScreenHandler;
  begin
    if Hooked then
      begin
        SetIntVec($05, OldInt05);
        Hooked := false;
        KeyHit := false;
      end;
  end;

{$ELSE} {protected mode}

var
  NewInt05 : RealProc;

procedure Int05Handler(var Regs : TRegisters); far;
  begin
    KeyHit := true;
  end;

procedure InstallPrintScreenHandler;
  begin
    if not Hooked then
      begin
        GetCallBack(Int05Handler, true, NewInt05);
        if Assigned(NewInt05) then
          begin
            GetRealIntVec($05, OldInt05);
            SetRealIntVec($05, @NewInt05);

            if not Assigned(ExitSave) then
              begin
                ExitSave := ExitProc;
                ExitProc := @UnitExitProc;
              end;

            Hooked := true;
          end;
      end;

    KeyHit := false;
  end;

procedure RemovePrintScreenHandler;
  begin
    if Hooked then
      begin
        SetRealIntVec($05, OldInt05);
        DestroyCallBack(NewInt05);
        Hooked := false;
        KeyHit := false;
      end;
  end;

{$ENDIF}

{$IFDEF AutoLoadHandler}
begin
  InstallPrintScreenHandler;
{$ENDIF}
end.

