#include 'achoice.ch'
#include 'inkey.ch'
#include 'common.ch'

// Defines for the scrollbar from CA's scrolbar.prg
#define  TB_ROWTOP         1
#define  TB_COLTOP         2
#define  TB_ROWBOTTOM      3
#define  TB_COLBOTTOM      4
#define  TB_COLOR          5
#define  TB_POSITION       6
#define  TB_ELEMENTS       6
#define  TB_UPARROW        CHR(  24 )
#define  TB_DNARROW        CHR(  25 )
#define  TB_HIGHLIGHT      CHR( 178 )
#define  TB_BACKGROUND     CHR( 176 )

// Statics that must be available to most functions in this
// PRG file.

static nElement
static nRow
static aBar
static afiles

/***
*
*  FileEval(<bBlock>, [cFileSpec], [nTop], [nLeft], [nRows], [cColor], [cTabColor])
*
*  Evaluates a code block on files tagged/picked from a list of files
*  with the space bar on pressing the enter or esc key.
*
*  <> = needed parameter
*  [] = optional parameter
*
*  <bBlock> is the code block to execute on the files picked
*  if bBlock is {|file| ferase(file) } then each file tagged
*  will be deleted.
*
*  [cFileSpec] is the skeleton of the files presented
*
*  [nTop], [nLeft], [nRows] are the top left corner of the picklist
*  and the number of rows in the list.
*
*  [cColor] is the color string for the list.  This follows standard
*  Clipper convention.  'Normal, Highlight,,,Unavailable'
*
*  [cTabColor] is the color for the scrollbar
*
*/
FUNCTION FileEval(bBlock, cFileSpec, nTop, nLeft, nRows, cColor, cTabColor)
   local cOldColor, nOldCurs
   local cPath
   local nChoice := 0
   local nLastkey := 0
   local lMore := .t.
   local I

   default cFileSpec to '*.*',;
           nTop to 7,;
           nLeft to 30,;
           nRows to 10,;
           cColor to 'w+/n,n/w,,,w/n',;
           cTabColor to 'n/w'

   cOldColor := setcolor(cColor)
   nOldCurs := setcursor(0)
   cPath := iif(rat('\',cFileSpec)>0,left(cFileSpec,rat('\',cFileSpec)),'')

   if !valtype(bBlock) == 'B'       ;
      .or. nLeft + 17 > maxcol()    ;
      .or. nTop+nRows-1 > maxrow()
      alert('Problem with parameters in FileEval')
      return .f.
   endif

   nElement := 1
   nRow := 1

   aBar := ScrollbarNew(nTop, nLeft+17, ntop+nRows-1, cTabColor, 1)
   ScrollBarDisplay(aBar)
   afiles := getfiles(cFileSpec)


   do while lmore

      ScrollBarUpdate(aBar, nElement, len(afiles),.t.)
      nChoice := achoice(nTop,nLeft,nTop+nRows-1,nLeft+16,afiles,.t.,"myfunc()",nElement, nRow)
      nLastKey := lastkey()

      do case
         case nLastKey == K_ESC
            lmore := .f.

         case nLastKey == K_ENTER
            lmore := .f.

         case nLastKey == K_SPACE
            if substr(afiles[nChoice], 2, 1) == ''
               afiles[nChoice] := stuff(afiles[nChoice],2,1,' ')
            else
               afiles[nChoice] := stuff(afiles[nChoice],2,1,'')
            endif

      endcase
   enddo

   // finally! evaluate the code block
   for I = 1 to len(afiles)
      if substr(afiles[I],2,1) == ''
         eval(bBlock, cPath+alltrim(right(afiles[I],13)))
      endif
   next I

   setcursor(nOldCurs)
   setcolor(cOldColor)
   return .t.

/***
*
*  MyFunc(nMode, nCurElement, nRowPos)
*
*  udf for achoice function
*  See Norton Guide under ACHOICE
*
*/
FUNCTION MyFunc(nMode, nCurElement, nRowPos)
   local nRetVal := AC_CONT

   // update statics for achoice
   nElement := nCurElement
   nRow := nRowPos

   do case
      case nMode == AC_IDLE
         //update display etc.
         ScrollBarUpdate(aBar, nCurElement, len(afiles))

      case nMode == AC_HITTOP .or. nMode == AC_HITBOTTOM
         // too far up or down
         tone(100,3)

      case nMode == AC_EXCEPT
         // handle the non movement keys
         do case
            case lastkey() == K_ESC
               nRetVal := AC_ABORT

            case lastkey() == K_ENTER
               nRetVal := AC_ABORT

            case lastkey() == K_SPACE
               nRetVal := AC_SELECT

            otherwise
               nRetVal := AC_GOTO

         endcase

   endcase
   return nRetVal

/***
*
*  GetFiles(cFileSpec)
*
*  Returns an array of files for FileEval
*
*/
STATIC FUNCTION getfiles(cFileSpec)
   local afiles := {}

   default cFileSpec to '*.*'

   aeval(directory(cfileSpec),;
         {|array| aadd(afiles,space(3)+padr(alignfile(array[1]),13))})

   return afiles

/***
*
*  AlignFile(cFilename)
*
*  returns cFilename padded such that filenames will be aligned
*  on the '.'
*
*/
STATIC FUNCTION AlignFile(cFilename)

   cFilename := lower(alltrim(cFilename))

   if at('.',cFilename) > 0
      cFilename := space(9-at('.',cFilename))+cFilename
   else
      cFilename := space(8-len(cFilename))+cFilename+'.'
   endif

   cFilename := padr(cfilename, 12)

   return cFilename

/***
*
*  The following functions are from the sample file scrolbar.prg
*  in CA-Clipper 5.xx
*
*/


/***
*
*  Scrolbar.prg
*
*  Implements a scroll bar that can be updated as the cursor moves down
*  in a TBrowse object, ACHOICE(), DBEDIT(), or MEMOEDIT()
*
*  Copyright (c) 1993, Computer Associates International Inc.
*  All rights reserved.
*
*  NOTE: Compile with /n /w
*
*/

/***
*
*  ScrollBarNew( <nTopRow>, <nTopColumn>, <nBottomRow>,
*                <cColorString>, <nInitPosition> ) --> aScrollBar
*
*  Create a new scroll bar array with the specified coordinates
*
*/
FUNCTION ScrollBarNew( nTopRow, nTopColumn, nBottomRow, ;
                        cColorString, nInitPosition )

   LOCAL aScrollBar := ARRAY( TB_ELEMENTS )

   aScrollBar[ TB_ROWTOP ]    := nTopRow
   aScrollBar[ TB_COLTOP ]    := nTopColumn
   aScrollBar[ TB_ROWBOTTOM ] := nBottomRow
   aScrollBar[ TB_COLBOTTOM ] := nTopColumn

   // Set the default color to White on Black if none specified
   IF cColorString == NIL
      cColorString := "W/N"
   ENDIF
   aScrollBar[ TB_COLOR ] := cColorString

   // Set the starting position
   IF nInitPosition == NIL
      nInitPosition := 1
   ENDIF
   aScrollBar[ TB_POSITION ] := nInitPosition

   RETURN ( aScrollBar )



/***
*
*  ScrollBarDisplay( <aScrollBar> ) --> aScrollBar
*
*  Display a scoll bar array to the screen
*
*/
FUNCTION ScrollBarDisplay( aScrollBar )

   LOCAL cOldColor
   LOCAL nRow

   cOldColor := SETCOLOR( aScrollBar[ TB_COLOR ] )

   // Draw the arrows
   @ aScrollBar[ TB_ROWTOP ], aScrollBar[ TB_COLTOP ] SAY TB_UPARROW
   @ aScrollBar[ TB_ROWBOTTOM ], aScrollBar[ TB_COLBOTTOM ] SAY TB_DNARROW

   // Draw the background
   FOR nRow := (aScrollBar[ TB_ROWTOP ] + 1) TO (aScrollBar[ TB_ROWBOTTOM ] - 1)
      @ nRow, aScrollBar[ TB_COLTOP ] SAY TB_BACKGROUND
   NEXT

   SETCOLOR( cOldColor )

   RETURN ( aScrollBar )



/***
*
*  ScrollBarUpdate( <aScrollBar>, <nCurrent>, <nTotal>,
*     <lForceUpdate> ) --> aScrollBar
*
*  Update scroll bar array with new tab position and redisplay tab
*
*/
FUNCTION ScrollBarUpdate( aScrollBar, nCurrent, nTotal, lForceUpdate )

   LOCAL cOldColor
   LOCAL nNewPosition
   LOCAL nScrollHeight := ( aScrollBar[TB_ROWBOTTOM] - 1 ) - ;
                          ( aScrollBar[TB_ROWTOP] )

   IF nTotal < 1
      nTotal := 1
   ENDIF

   IF nCurrent < 1
      nCurrent := 1
   ENDIF

   IF nCurrent > nTotal
      nCurrent := nTotal
   ENDIF

   IF lForceUpdate == NIL
      lForceUpdate := .F.
   ENDIF

   cOldColor := SETCOLOR( aScrollBar[ TB_COLOR ] )

   // Determine the new position
   nNewPosition := ROUND( (nCurrent / nTotal) * nScrollHeight, 0 )

   // Resolve algorythm oversights
   nNewPosition := IF( nNewPosition < 1, 1, nNewPosition )
   nNewPosition := IF( nCurrent == 1, 1, nNewPosition )
   nNewPosition := IF( nCurrent >= nTotal, nScrollHeight, nNewPosition )

   // Overwrite the old position (if different), then draw in the new one
   IF nNewPosition <> aScrollBar[ TB_POSITION ] .OR. lForceUpdate
      @ (aScrollBar[ TB_POSITION ] + aScrollBar[ TB_ROWTOP ]), ;
         aScrollBar[ TB_COLTOP ] SAY TB_BACKGROUND
      @ (nNewPosition + aScrollBar[ TB_ROWTOP ]), aScrollBar[ TB_COLTOP ] SAY ;
        TB_HIGHLIGHT
      aScrollBar[ TB_POSITION ] := nNewPosition
   ENDIF

   SETCOLOR( cOldColor )

   RETURN ( aScrollBar )
