//
// Finger Version 3.1, a Windows Sockets Finger Client
//
// Copyright 1992, 1993 Network Research Corporation
//
// Permission to use, modify, and distribute this software and its
// documentation for any purpose and without fee is hereby granted, provided
// that the above copyright notice appears in all copies and that both
// that copyright notice and this permission notice appear in supporting
// documentation.  NRC makes no claims as to the suitability of this software
// for any purpose.
//
// Module DSPLIST allows the NETWRK_ module to build a display list,
// and the FINGER module to retrieve one.  A display list is a linked
// list of text lines accompanied by a count of list entries.  The finger
// client uses display lists to represent the the remote finger server's
// return data in a form suitable for scrolling window display.
//
// 10/20/92  Lee Murach    Created.
// 12/02/92  Lee Murach    PushChars() replaces PushChar() as export, LF
//                         triggers newline, CRs discarded
//

#include <string.h>
#include <stdlib.h>
#include <windows.h>
#include "finger.h"

#define  LF 10                         // linefeed
#define  CR 13                         // carriage return
#define  TABSTRING "        "          // 8 character tab

VOID PushChar(char ch);
VOID PushLine(VOID);
VOID FreeLine(LINEITEM *pLine);

char LineBuf[132];                     // holds accumulating text line
int LineLen;                           // length of text line
LINEITEM *pLast;                       // last item on list
LINEITEM *pFirst;                      // root of line list
int LineCount;                         // n items in list

//
// PushChars -- pushes buffer of characters into the display list;
// expands tabs.
//
VOID PushChars(char *buf, int buflen)
{
   while (buflen--)
   {
      if (*buf == '\t')
         PushChars(TABSTRING, strlen(TABSTRING));
      else
         PushChar(*buf);
   
      buf++;
   }
}


//
// PushChar -- pushes a character into the LineBuf.  Pushes line into
// the line list if <cr> encountered, or LineBuf is full.  Discards
// linefeeds.
//
VOID PushChar(char ch)
{
   if (LineLen < sizeof(LineBuf))
      if (ch != LF)
         if (ch != CR)
            LineBuf[LineLen++] = ch;
         else;                         // discard carriage returns
      else                             // linefeeds signal newline
         PushLine();
   else
      PushLine();
}

//
// PushLine -- pushes a text line into the accumulating list of text lines.
//
VOID PushLine(VOID)
{
   LINEITEM *p;

   if (p = calloc(1, sizeof(LINEITEM)))
   {
      if (p->sztext = malloc(LineLen))
      {
         strncpy(p->sztext, LineBuf, p->len = LineLen);
         pLast->next = p;
         pLast = p;
         LineLen = 0;
         LineCount++;
      }
   }
}

//
// FreeDisplayList -- frees the display list, leaves a null list.
//
VOID FreeDisplayList(VOID)
{
   FreeLineList(pFirst);
   pFirst = NULL;
   LineCount = 0;
}

//
// FreeLineList -- frees a LINEITEM list
//
VOID FreeLineList(LINEITEM *pLine)
{
   if (pLine)
   {
      FreeLineList(pLine->next);
      FreeLine(pLine);
   }
}

//
// FreeLine -- frees a LINEITEM
//
VOID FreeLine(LINEITEM *pLine)
{
   free(pLine->sztext);
   free(pLine);
}

//
// OpenDisplayList -- initializes a display list.  Call before using
// PushChar() to build the list.
//
BOOL OpenDisplayList(VOID)
{
   LineCount = LineLen = 0;

   // initialize list with a dummy LINEITEM
   if (pLast = pFirst = calloc(1, sizeof(LINEITEM)))
      return TRUE;

   return FALSE;  // failure
}

//
// CloseDisplayList -- pushes final LINEITEM (if any) onto list, and
// releases dummy list header.  Call after last PushChar() to complete
// the display list.
//
VOID CloseDisplayList(VOID)
{
   LINEITEM *p = pFirst;

   if (LineLen > 0)
      PushLine();

   pFirst = pFirst->next;
   FreeLine(p);
}

//
// GetDispList -- retrieves the display list.
//
VOID GetDisplayList(LINEITEM **ppLine, int *pNLines)
{
   *ppLine = pFirst;
   *pNLines = LineCount;
}
