/*
 * ChkPro v1.0r1 - ProBoard Configuration Checking Utility
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, part of the PB-GNU Project
 *
 * For details, see the file "copying".
 *
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <pblib.h>
#include <proutil.h>
#include <crc32.h>
#include <fsys.h>
#include <memdbg.h>

/* these are defined in the main module */
extern void _pascal logexit(const char *format, ...);
extern void _pascal log(const char *format, ...);

/* local function prototypes, structures and globals */
static struct tree{
	struct tree  *left, *right;
		   dword  data;
} *treeCRC = NULL;             /* local global binary search tree root   */

static FILE *__fpUser = NULL;  /* local global pointer to userbase index */

static void _pascal tree_add(struct tree **, dword);
static void _pascal tree_free(struct tree *);
static bool _pascal in_tree(struct tree *, dword);
static bool _pascal isValidUser(const char *);

/***************************************************************************\
 ** Local functions for the binary search tree
\***************************************************************************/
	void _pascal
tree_add(struct tree **root, dword data)
{
	struct tree *p;

	while( NULL != (p = *root) ){
		if( data < p->data ) root = &p->left;
		else root = &p->right;
	}
	p = (struct tree *)mem_calloc(sizeof(struct tree));
	if( NULL == p ) logexit("tree_add(): memory allocation failure\n");
	*root = p;
}

	void _pascal
tree_free(struct tree *node)
{
	if( NULL != node ){
		tree_free(node->left);
		tree_free(node->right);
		mem_free(node);
	}
}

	bool _pascal
in_tree(struct tree *root, dword data)
{
	struct tree *p = root;

	while(NULL != p){
		if( p->data == data ) return TRUE;
		if( data < p->data ) p = p->left;
		else p = p->right;
	}
	return FALSE;
}

/************************************************************************\
 ** Main processing functions
\************************************************************************/

/*
 * Checks if a user is in the userbase.
 *
 * A 32-bit CRC of the name is calculated. If the CRC is not in the binary
 * search tree already (previously found), we search the userbase and, if
 * we find it there, we add it to the tree. This makes searches very
 * efficient because we only access and search the userbase index once for
 * each user. Note that we need to complement the CRC to get the result
 * stored in the userbase index file (RAM bug, not mine). Also, we need to
 * convert the name to uppercase before we do the calculations on it.
*/
	bool _pascal
isValidUser(const char *name)
{
	char     tmp[40];
	dword    crc;
	USERSIDX buf;

	/* get the CRC and search the tree */
	strcpy(tmp, name);
	crc = ~bufCRC32(strupr(tmp), strlen(tmp));
	if( in_tree(treeCRC, crc) ) return TRUE;

	/* ok, not in the tree, rewind the index file (make sure it's open) */
	assert(__fpUser); rewind(__fpUser);
	for( ;; ){
		/* read a record and see if we're at the end of file */
		if( 1 != fread(&buf, sizeof(buf), 1, __fpUser) ){
			if(feof(__fpUser)) return FALSE;
			else logexit("fread() failed on the user index file.\n");
		}
		/* see if the CRC value matches, if it does, add it to the tree */
		if( buf.nameCRC32 == crc ){
			tree_add(&treeCRC, crc);
			return TRUE;
		}
	}
}

/*
 * Main processing function for the BINLOG.PB
*/
	int _pascal
do_binlog(bool repair)
{
	char  srcPath[MAXPATH], destPath[MAXPATH];
	FILE *fpSrc, *fpDest;
	int   retval = 0;

	assert(mem_inited);

	/* open the userbase index file with a large IO buffer */
	if(NULL == (__fpUser = usrfopen( fnUSERSIDX, "rb")) )
		logexit("error opening the userbase index '%s'\n", fnUSERSIDX);
	setvbuf(__fpUser, NULL, _IOFBF, 0x1000);

	/* get the path to the binary log file and create a .TMP name */
	strcpy(srcPath, _mkSysPath(fnBINLOG));
	strcpy(destPath, _mkSysPath("~BINLOG.TMP"));

	/* open the binary log file and assign a large IO buffer */
	if( NULL == (fpSrc = fopen(srcPath, "rb")) )
		logexit("Fatal error opening '%s'\n", srcPath);
	setvbuf(fpSrc, NULL, _IOFBF, 0x1000);

	/* see if we want to repair and if so, open the temp file */
	if( TRUE == repair && NULL == (fpDest = fopen(destPath, "wb")) )
		logexit("Fatal error opening '%s'\n", destPath);

	for( ;; ){
		BINLOG buf;

		/* read a record and see if we're at the end of file */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) goto _eofBinlog;
			logexit("fread() failed on reading '%s'\n", srcPath);
		}

		/* user not in the userbase, write a warning */
		if( FALSE == isValidUser(buf.name) ){
			log("warning: user '%s' not found%s\n",
					buf.name, repair ? "...removed" : "" );
			retval = -1;
		}
		/* if in the userbase and we're reapiring, write the record */
		else if( repair ) fwrite(&buf, sizeof(buf), 1, fpDest);
	}

_eofBinlog:
	/* close files and free memory */
	fclose(fpSrc);
	fclose(__fpUser);
	tree_free(treeCRC);

	/* if we're repairing, close and rename temp, rename orig to .BAK */
	if( repair ){
		fclose(fpDest);
		chext(destPath, srcPath, ".BAK");
		rename(srcPath, destPath);
		rename(_mkSysPath("~BINLOG.TMP"), srcPath);
	}

	return retval;
}
