
/*-
 * Copyright (c) 1995 The Apache Group. All rights reserved.
 * 
 *
 * Apache httpd license
 * ====================
 * 
 *
 * This is the license for the Apache Server. It covers all the
 * files which come in this distribution, and should never be removed.
 * 
 * The "Apache Group" has based this server, called "Apache", on
 * public domain code distributed under the name "NCSA httpd 1.3".
 * 
 * NCSA httpd 1.3 was placed in the public domain by the National Center 
 * for Supercomputing Applications at the University of Illinois 
 * at Urbana-Champaign.
 * 
 * As requested by NCSA we acknowledge,
 * 
 *  "Portions developed at the National Center for Supercomputing
 *   Applications at the University of Illinois at Urbana-Champaign."
 *
 * Copyright on the sections of code added by the "Apache Group" belong
 * to the "Apache Group" and/or the original authors. The "Apache Group" and
 * authors hereby grant permission for their code, along with the
 * public domain NCSA code, to be distributed under the "Apache" name.
 * 
 * Reuse of "Apache Group" code outside of the Apache distribution should
 * be acknowledged with the following quoted text, to be included with any new
 * work;
 * 
 * "Portions developed by the "Apache Group", taken with permission 
 *  from the Apache Server   http://www.apache.org/apache/   "
 *
 *
 * Permission is hereby granted to anyone to redistribute Apache under
 * the "Apache" name. We do not grant permission for the resale of Apache, but
 * we do grant permission for vendors to bundle Apache free with other software,
 * or to charge a reasonable price for redistribution, provided it is made
 * clear that Apache is free. Permission is also granted for vendors to 
 * sell support for for Apache. We explicitly forbid the redistribution of 
 * Apache under any other name.
 * 
 * The "Apache Group" makes no promises to support "Apache". Users and
 * sellers of Apache support, and users of "Apache Group" code, should be 
 * aware that they use "Apache" or portions of the "Apache Group" code at 
 * their own risk. While every effort has been made to ensure that "Apache"
 * is robust and safe to use, we will not accept responsibility for any
 * damage caused, or loss of data or income which results from its use.
 * 
 */



#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#ifdef AIX
#include <sys/select.h>
#endif

#include "stream.h"

/*
 * This file contains semi-buffered file input routines.
 * It is designed for reading data which is divided into a head
 * and a body. The head comprises CRLF terminated lines, and is
 * terminated by an empty line. The body contains abitrary binary
 * data.
 *
 * Input from the stream is buffered whilst reading the headers;
 * the body will not be buffered. After a call to sgets(), the
 * amount of extra data removed from the kernel stream and stored
 * in the buffer is guaranteed to only extend up to (and including)
 * a blank line. (If one has been received). If no blank line is
 * sent on the socket, then the routines may buffer all of the
 * received data.
 *
 * Thus after reading all the headers with sgets(), the calling
 * program can discard the STREAMS buffer without loosing any data.
 */


static int nread(int fd, unsigned char *buff, int n, int *isnl,
		 unsigned int timeout);


/*
 * Allocates memory for a new stream structure.
 * Allocates a buffer of buffsize, or of a default size if
 * buffsize is zero.
 * 
 * Returns the structure, or NULL if insufficient memory.
 */
STREAM *
mkstream(size_t buffsize)
{
    STREAM *sp;

    if (buffsize == 0) buffsize = STREAM_BUFSIZE;

    sp = malloc(sizeof(STREAM));
    if (sp == NULL) return NULL;

    sp->base = malloc(buffsize);
    if (sp->base == NULL)
    {
	free(sp);
	return NULL;
    }
    sp->cnt = 0;
    sp->ptr = sp->base;
    sp->size = buffsize;
    sp->file = -1;
    sp->isnl = 1;

    return sp;
}

/*
 * Frees a stream previously allocated by mkstream
 */
void
freestream(STREAM *sp)
{
    free(sp->base);
    free(sp);
}

/*
 * Resets a stream, and connects it to file descriptor fd
 */
void
sdopen(int fd, STREAM *sp)
{
    sp->cnt = 0;
    sp->ptr = sp->base;
    sp->file = fd;
    sp->isnl = 1;
}

/*
 * Reads from the stream into the array pointed to by buff, until
 * a (CR)LF sequence is read, or end-of-file condition is encountered
 * or until n-1 bytes have been stored in buff. If a CRLF sequence is
 * read, it is replaced by a newline character.  The string is then
 * terminated with a null character.
 *
 * Returns the number of bytes stored in buff, or zero on end of
 * transmission, or -1 on an error.
 * If a blank line has been read, then no more data has been buffered.
 * Errors:
 *  The same as read(2), plus
 *  ETIMEDOUT      No data arrived before timeout.
 *
 * Notes:
 *  If null characters are exepected in the data stream, then
 * buff should not be treated as a null terminated C string; instead
 * the returned count should be used to determine the length of the
 * string.
 *  CR characters in the byte stream not immediately followed by a LF
 * will be preserved.
 */
int
sgets(unsigned char *buff, int n, unsigned int timeout, STREAM *sp)
{
    int i, j, r, done, ch;

    j = 0;
    done = 0;

    r = 0;

    for (;;)
    {
/* copy bytes, stop on CRLF, and leave last character as \n */
	for (i=0; i < sp->cnt; )
	{
	    ch = sp->ptr[i++];
	    if (ch == '\012')  /* got LF */
	    {
		if (j == 0) buff[j++] = '\n';
/* if just preceeded by CR, replace CR with LF */
		else if (buff[j-1] == '\015') buff[j-1] = '\n';
		else if (j < n-1) buff[j++] = '\n';
		else i--; /* no room for LF */
		done = 1;
		break;
	    }
	    if (j == n-1)
	    {
		i--;  /* push back ch */
		done = 1;
		break;
	    }
	    
	    buff[j++] = ch;
	}
	sp->cnt -= i;
	sp->ptr += i;
	if (done) break;
/* must have used up all of buffer */
	sp->ptr = sp->base;

/* want to wait with a timeout; use select */

/* read the data */
	r = nread(sp->file, sp->base, sp->size, &sp->isnl, timeout);
	if (r == -1) break;
	sp->cnt = r;
	if (r == 0) break; /* what about zero length messages ? */
    }

    buff[j] = '\0';
    if (r == -1) return -1;
    else return j;
}

/*
 * Reads data into the buffer, storing up to n bytes, or until a
 * blank line has been read, or until eof.
 * isnl is non-zero if a (CR)LF has just been read.
 * Returns -1 on error, 0 on eof, or number of bytes read.
 * isnl is updated to show whether the last character was a LF
 */
static int
nread(int fd, unsigned char *buff, int n, int *isnl, unsigned int timeout)
{
    fd_set readfds;
    struct timeval timeouts;
    int i, j, r;

    FD_ZERO(&readfds);

/* wait for data */
    do {
	timeouts.tv_sec = timeout;  /* linux updates timeouts.. */
	timeouts.tv_usec = 0;
	FD_SET(fd, &readfds);
	r = select(fd+1, &readfds, NULL, NULL, &timeouts);
    } while (r == -1 && errno == EINTR);
    if (r == -1) return -1;

    if (r == 0) /* timeout */
    {
	errno = ETIMEDOUT;
	return -1;
    }
    
/* have a look at the data */
    do r = recv(fd, (char *)buff, n, MSG_PEEK);
    while (r == -1 && errno == EINTR);
    if (r == -1) return -1;

/* Look for LFLF or LFCRLF */
    if (isnl) j = -1;
    else j = -3;

    for (i=0; i < r; i++)
    {
	if (buff[i] == '\012')
	{
/* the order of this test has been optimised for the likely data */
	    if (i - j <= 2 && (i - j == 1 || buff[i-1] == '\015'))
	    {
		i++;
		break;
	    } else
		j = i;
	}
    }
/* read that many bytes */
    for (j=0; j < i; )
    {
	r = read(fd, &buff[j], i-j);
	if (r == -1)
	{
	    if (errno != EINTR) return -1;
	} else if (r == 0)  /* just in case */
	    break;
	else
	    j += r;
    }
    if (j > 0) *isnl = (buff[j-1] == '\012');
    return j;
}
