/* isprime.c */
/* Copyright 1995 Willis E. Howard, III */
/* Willis E. Howard, III  email: WEHoward@aol.com  mail: POB 1473 Elkhart, IN  46515 */

/*
   function isprime() returns true (1) if the argument is a
   prime number between 1 and 65535, and false (0) otherwise.
   There are 6542 primes in that range.  The smallest is 2.
   The largest is 65521. A table is precomputed with these
   values just to make a fast function for 16 bit numbers.
*/

static unsigned long primes[2048]=
{
0xa08a28ac, 0x28208a20, 0x02088288, 0x800228a2, 
0x20a00a08, 0x80282088, 0x800800a2, 0x08028228, 
0x0a20a082, 0x22880020, 0x28020800, 0x88208082, 
0x02022020, 0x08828028, 0x8008a202, 0x20880880, 
0x20000a00, 0x0a082008, 0x82820802, 0x00800a20, 
0x0028208a, 0x20080822, 0x20808020, 0x02208088, 
0x20080022, 0x28a00a00, 0x8a200080, 0x008a2000, 
0x00808800, 0x02082202, 0x80820880, 0x28220020, 
0x0a008280, 0x800020a0, 0x20208228, 0x80000208, 
0x20080802, 0x20020208, 0x0020a082, 0xa0000802, 
0x08a20a08, 0x00008280, 0x20820000, 0x00800020, 
0x82288002, 0x80082880, 0x08228a00, 0x08080080, 
0x80822080, 0x20008808, 0x08282282, 0x82000020, 
0x20000028, 0x0a00200a, 0x82082020, 0x28820000, 
0x80080200, 0x00800080, 0x00a28820, 0x02082002, 
0xa0002800, 0x08200000, 0x0808a208, 0x00802802, 
0x00208020, 0x8008028a, 0xa20a0002, 0x00020200, 
0x88000008, 0xa0002020, 0x28000808, 0x02208202, 
0x200000a0, 0x00208828, 0x8220a208, 0x00820800, 
0x08008220, 0x00002208, 0x02000080, 0x20a08808, 
0x80080000, 0x22020002, 0x00800200, 0x8820808a, 
0x82880822, 0x20200a00, 0x02008002, 0x000a20a0, 
0x08220008, 0x80002208, 0x20820080, 0x08008020, 
0x0a082200, 0x02800000, 0x20008808, 0x08200202, 
0x00020880, 0x00028020, 0x88000002, 0x00882002, 
0x20220208, 0x0a280000, 0x00000080, 0x08820828, 
0x8008800a, 0xa0002802, 0x00208000, 0x02000200, 
0x000028a2, 0x20800808, 0x08282280, 0xa00800a0, 
0x20800200, 0x08208082, 0x22800800, 0x20220800, 
0x08208080, 0x02820000, 0x08220008, 0x0200a000, 
0x00802880, 0x00020220, 0x0a08a088, 0x80000880, 
0x00200000, 0x0828208a, 0x020a0000, 0x28008200, 
0x80008008, 0xa2000822, 0x00020000, 0x0a080200, 
0x200a00a0, 0x08028028, 0x00000202, 0x200a0080, 
0x00200820, 0x02002080, 0x820000a0, 0x00008208, 
0x0800200a, 0x000008a2, 0x00820028, 0x08208080, 
0xa0002000, 0x00820a08, 0x80080082, 0x220a0000, 
0x00808000, 0x82288000, 0x80022002, 0x20000000, 
0x02002080, 0x00802080, 0x20808228, 0x08002280, 
0xa00a0882, 0x08008000, 0x82200008, 0x82082800, 
0x08000000, 0x08088002, 0x02002020, 0x00228800, 
0x82002000, 0x20820002, 0x00200800, 0x00000088, 
0x80820820, 0x20800a20, 0x8000020a, 0xa00008a0, 
0x08828008, 0x08200000, 0x00802022, 0x00800000, 
0x0a288280, 0x22080020, 0x00208020, 0x0020a200, 
0x80880000, 0x20208200, 0x0a088008, 0x02802822, 
0x00008200, 0x00080088, 0x20000002, 0x08800008, 
0x88202000, 0x82080020, 0x00200a00, 0x80280202, 
0x20080080, 0x08a00000, 0x00202208, 0xa0820080, 
0x28008020, 0x02082080, 0x82820082, 0x20200820, 
0x08200000, 0x000a0000, 0x08020220, 0x02000000, 
0x02880002, 0x00220a08, 0x08008080, 0x20002000, 
0x00820028, 0x8020a00a, 0x00022000, 0x28000a00, 
0x00080280, 0x02022880, 0x00a08200, 0x88080008, 
0x00000020, 0x08828028, 0x02208082, 0x80080820, 
0x20020008, 0x80000080, 0x02820020, 0x00808000, 
0x02080200, 0x20082880, 0x00208820, 0x00080000, 
0x02002802, 0x00a00028, 0x00000200, 0x02080002, 
0x08000200, 0x8220000a, 0x20080802, 0x28220208, 
0x82200280, 0x008800a0, 0x08800020, 0x02200002, 
0x00880882, 0x20008820, 0x0000a200, 0x00022000, 
0x00208200, 0x08002002, 0x200008a2, 0x20808008, 
0x0800a002, 0x02000000, 0x00020a00, 0x08200080, 
0x22820020, 0x08208020, 0x00080000, 0x80080882, 
0x28020000, 0x00082280, 0x80022080, 0x20800228, 
0x00000200, 0x02020802, 0x00000028, 0x8800a088, 
0x00082000, 0x00200000, 0x8a008202, 0x20080000, 
0x00a00020, 0x00288002, 0x20800082, 0x08220220, 
0x08008208, 0x02020822, 0x00008800, 0x80280088, 
0xa00200a0, 0x20800008, 0x08000000, 0x00802022, 
0x00000a08, 0x00288280, 0x020a0020, 0x00000808, 
0x00208008, 0x80022080, 0x20200220, 0x02088008, 
0x02800820, 0x20000200, 0x8000200a, 0xa2000880, 
0x08000020, 0x08208002, 0xa2802820, 0x08a00000, 
0x02080082, 0x20020080, 0x00008808, 0x00080000, 
0xa0882002, 0x20020800, 0x0200a000, 0x02800002, 
0x00208820, 0x88200280, 0x02080800, 0x08008220, 
0x8800800a, 0x00082000, 0x20200a08, 0x00208000, 
0x02800000, 0x00a00000, 0x80282000, 0x80882800, 
0x28200000, 0x02088080, 0x00020022, 0x00008008, 
0x88082088, 0xa0020002, 0x28020200, 0x00200082, 
0x80800002, 0x28000000, 0x8a200002, 0x00802000, 
0x00020028, 0x8000000a, 0x20000080, 0x00208220, 
0x08008088, 0x80000882, 0x00a00008, 0x00082208, 
0x20080002, 0x00008020, 0x80208000, 0x00082820, 
0x20800a08, 0x0a000000, 0x20002020, 0x08020808, 
0x00080000, 0xa00a0080, 0x00208800, 0x0800a200, 
0x02802000, 0x20000228, 0x00002000, 0x22080880, 
0x00808200, 0x02208008, 0x02800002, 0x00800200, 
0x80080082, 0x00882000, 0x08008800, 0x8000820a, 
0xa0000080, 0x08000000, 0x00002208, 0x00020022, 
0x20000200, 0x00280202, 0x800800a0, 0x00000000, 
0x8200208a, 0x82882020, 0x08800000, 0x82008000, 
0x20880020, 0x00820028, 0x0008200a, 0x00000880, 
0x00020a20, 0x00000280, 0x00822822, 0x20208000, 
0x0000000a, 0x00080820, 0x28028008, 0x02202080, 
0x00002002, 0x08800208, 0x0a200000, 0x22020080, 
0x00220020, 0x82000208, 0x80820880, 0x28008020, 
0x02000208, 0x02002082, 0x20000828, 0x80082082, 
0x82020000, 0x00020200, 0x08200088, 0x80802000, 
0x20a00200, 0x00200200, 0x22000000, 0x08a08808, 
0x82002000, 0x00882080, 0x00028a00, 0x02082200, 
0x00000080, 0x00208820, 0x88000082, 0x82080880, 
0x00000008, 0x0200200a, 0x00080802, 0x20020800, 
0x0a088002, 0x00800000, 0x00a00020, 0x00002208, 
0x20820802, 0x08008020, 0x00080200, 0x80020002, 
0x20800200, 0x08082000, 0x00020002, 0x28828020, 
0x0a00a002, 0xa0800002, 0x20200008, 0x00008080, 
0x000200a0, 0x08a20008, 0x0208a000, 0x00002802, 
0x00008800, 0x0a00a000, 0x00022000, 0x00800208, 
0x0800008a, 0x80000000, 0x20008220, 0x80200002, 
0x00002000, 0x00000a08, 0x80200200, 0x008a0000, 
0x00000820, 0x00280002, 0x80880882, 0x20028020, 
0x02008000, 0x00800080, 0x20a08220, 0x8000000a, 
0x20000002, 0x20008228, 0x08002002, 0x22000800, 
0x08220808, 0x88088082, 0x20002000, 0x08008820, 
0x80280008, 0x00020880, 0x08020800, 0x02002088, 
0x00000080, 0x00800220, 0x00202000, 0x20080822, 
0x08020008, 0x0200200a, 0x82080020, 0x08020000, 
0x88000202, 0x028020a0, 0x00820808, 0x8020000a, 
0x80822002, 0x00200200, 0x08088008, 0x00020820, 
0x00800220, 0x80200080, 0xa0020080, 0x00820000, 
0x8a002008, 0x80882002, 0x00000008, 0x0a288000, 
0x20820020, 0x08208800, 0x00008202, 0x00080000, 
0x20028220, 0x00088208, 0x00800002, 0x00800028, 
0x00000082, 0xa2000002, 0x08800228, 0x80008082, 
0x00002800, 0x28800000, 0x82000002, 0x220000a0, 
0x00020800, 0x80002000, 0x20082000, 0x20020a00, 
0x08000000, 0x80020822, 0x20000028, 0x00002082, 
0x00020080, 0x20820208, 0x88000000, 0x02800800, 
0x00220a00, 0x88280002, 0x02000000, 0x08800820, 
0x00080000, 0xa0800800, 0x00000800, 0x02088080, 
0x820000a0, 0x00008208, 0x08280008, 0x20080002, 
0x20820228, 0x00200000, 0x20080820, 0x00020008, 
0x80008080, 0x028800a0, 0x00808000, 0x00000200, 
0x00082800, 0x20208220, 0x00008080, 0x02022882, 
0x00800a00, 0x88000208, 0x020a0820, 0x00008020, 
0x00008008, 0xa0002822, 0x08800200, 0x0a000000, 
0x000a0020, 0x00020800, 0x00280202, 0xa0800002, 
0x00008000, 0x0800a008, 0x02000080, 0x00a08008, 
0x0008020a, 0x220800a0, 0x20000028, 0x8a000000, 
0x20002802, 0x00020200, 0x80288280, 0x02000020, 
0x00808020, 0x02208208, 0x20022800, 0x20028000, 
0x02000288, 0x80020020, 0x20000808, 0x08002002, 
0x22080882, 0x20020000, 0x0a200080, 0x02002800, 
0x08800000, 0x02000002, 0x00002000, 0x00808020, 
0x82002008, 0x20800800, 0x00200020, 0x00082208, 
0x00800000, 0x00200800, 0x80080000, 0x000000a2, 
0x20008008, 0x02000008, 0x20800022, 0x20000800, 
0x00008282, 0x00020000, 0x08200028, 0x0000a000, 
0x80082880, 0x08020020, 0x02000208, 0x00802020, 
0x00000020, 0x88282000, 0xa2000800, 0x00820228, 
0x0820a082, 0x00082002, 0x00a00800, 0x82080082, 
0x02082000, 0x00000000, 0x00002000, 0x200a0082, 
0x08008020, 0x00088280, 0x02020002, 0x00000820, 
0x0820200a, 0x000a0000, 0x00808200, 0x0a002002, 
0x22800820, 0x20a00800, 0x80088000, 0x02000080, 
0x08220820, 0x00288002, 0x00822002, 0x08000200, 
0x02000280, 0x80002020, 0x00000200, 0x08202088, 
0x000200a2, 0x08808220, 0x0a008000, 0xa0000000, 
0x28000008, 0x80200200, 0x000800a0, 0x00820000, 
0x80080008, 0x00800002, 0x00200200, 0x0800a008, 
0x00822880, 0x00200028, 0x00000280, 0x02020000, 
0x00808200, 0x0200008a, 0x80080802, 0x00a00008, 
0x8a280200, 0x00000080, 0x00000828, 0x00082002, 
0x20880002, 0x08020820, 0x08000288, 0x80000820, 
0x20200000, 0x00000008, 0x200008a2, 0x00000008, 
0x8220808a, 0x02000800, 0x08000000, 0x88288200, 
0x020a2000, 0x08a00000, 0x80280202, 0x00022802, 
0x20028000, 0x00080200, 0x80020020, 0x20800a00, 
0x08002080, 0x22000880, 0x08028000, 0x02008088, 
0x00800020, 0x00820200, 0x02000200, 0x20082080, 
0x00a08800, 0x82088008, 0xa0002800, 0x00220800, 
0x0800a208, 0x00022000, 0x00200000, 0x80080080, 
0x800808a2, 0x00000008, 0x80002080, 0x20802002, 
0x00000200, 0x00208202, 0x000a0020, 0x00008020, 
0x0020a008, 0x00002880, 0x28200000, 0x0808a000, 
0x00802000, 0x20200a08, 0x80280082, 0x82020000, 
0x08800008, 0x00000002, 0x20802820, 0x08200000, 
0x82000200, 0x200a0000, 0x08020000, 0x80282208, 
0x808a2002, 0x08008200, 0x08002000, 0x80800080, 
0x20200020, 0x0800020a, 0x00080000, 0x28008020, 
0x82008002, 0x20000820, 0x00000a00, 0x02000080, 
0x00800080, 0x00020008, 0x00200200, 0x200a0800, 
0x20008a20, 0x02088200, 0x00022880, 0x20a08000, 
0x00000280, 0xa0000080, 0x20820208, 0x02002000, 
0x02880820, 0x20020000, 0x08208082, 0x020020a0, 
0x00200000, 0x02002208, 0x20880082, 0x20020800, 
0x08082080, 0x02022882, 0x00800220, 0x00280200, 
0x80000820, 0x08808020, 0x02000000, 0x20000000, 
0x00020200, 0x08280080, 0x20800080, 0x08800028, 
0x02200000, 0x20080002, 0x00208020, 0x08000088, 
0x00020000, 0x00808008, 0x80080000, 0x02080820, 
0x20000020, 0x88002000, 0x82002000, 0x00200200, 
0x08088200, 0x02002000, 0x08a00800, 0x02008000, 
0x200a0880, 0x00200000, 0x00002280, 0x80020002, 
0x20800200, 0x08000082, 0x800a0802, 0x00000020, 
0x88202008, 0x00002020, 0x28a00000, 0x88000200, 
0x20802020, 0x00008000, 0x8000800a, 0xa0822080, 
0x00000000, 0x00080000, 0x02000022, 0x00208a00, 
0x0020020a, 0x02000882, 0x20008208, 0x00200080, 
0x80082020, 0x00000808, 0x08000282, 0x02820000, 
0x00020800, 0x82008002, 0x80882800, 0x20020000, 
0x0008a000, 0x02002022, 0x00000020, 0x88080008, 
0x20000082, 0x08020200, 0x08002082, 0x20080002, 
0x08a00a00, 0x02200000, 0x00022020, 0x00200808, 
0x82080008, 0x80820080, 0x08020020, 0x02008000, 
0x00020022, 0x20000800, 0x80002002, 0x00000020, 
0x00820208, 0x8a002000, 0x22802820, 0x00020a08, 
0x82000080, 0x20020000, 0x00820020, 0x8000a002, 
0x00802802, 0x00008200, 0x00008288, 0x00020820, 
0x00800208, 0x08280008, 0x80080022, 0x00800028, 
0x00000080, 0x20800800, 0x00000800, 0x08200280, 
0x000a2080, 0x00000008, 0x00000202, 0x00000082, 
0x00028020, 0x02002088, 0x82800002, 0x00200000, 
0x80280280, 0xa0000000, 0x08000000, 0x80008002, 
0x02000020, 0x00220a00, 0x8a200000, 0x208800a0, 
0x08828008, 0x00088202, 0x00000880, 0x20200800, 
0x0000a000, 0x820208a0, 0x20008000, 0x88080002, 
0x02080000, 0x00820020, 0x0800a00a, 0x00800000, 
0x00a00008, 0x08080200, 0x00000020, 0x08a00000, 
0x80280202, 0xa0000000, 0x00000000, 0x0a080208, 
0x80822000, 0x20008000, 0x00082000, 0x22000022, 
0x28008028, 0x88208008, 0x00882800, 0x20a20208, 
0x82008000, 0x02080080, 0x00828000, 0x02200008, 
0x00022080, 0x08000820, 0x00008088, 0x82002020, 
0x00200020, 0x00080002, 0xa2020000, 0x00000208, 
0x00200088, 0x00002000, 0x08020a08, 0x08208002, 
0x02820080, 0x08008020, 0x00200008, 0x80820080, 
0x08208000, 0x00080008, 0x02802880, 0x20208008, 
0x00002008, 0x20000802, 0x08800000, 0x80000002, 
0x80082022, 0x20000800, 0x80000202, 0x000800a0, 
0x08200000, 0x82002000, 0x20000002, 0x08000800, 
0x08002200, 0x02820880, 0x20000000, 0x08000082, 
0x80000080, 0x28000020, 0x80002000, 0x00800800, 
0x20820a00, 0x0a200082, 0x02020000, 0x08200820, 
0x00008008, 0xa0082002, 0x00200000, 0x00088208, 
0x80000820, 0x00a00000, 0x88000008, 0x00020000, 
0x28000028, 0x00202002, 0xa2000802, 0x00020000, 
0x00000002, 0x02082080, 0x00208020, 0x02088202, 
0x00000080, 0x20200820, 0x02080000, 0x02802002, 
0x00a00828, 0x80002002, 0x22080000, 0x08000020, 
0x00000082, 0x82000820, 0x20800800, 0x00080280, 
0x000a0000, 0x00828808, 0x0220a200, 0x80880000, 
0x08228020, 0x00002088, 0x82800800, 0x20000220, 
0x88080000, 0x00000022, 0x00000000, 0x02202000, 
0x22800002, 0x08a00800, 0x80208000, 0x00820000, 
0x08000800, 0x00088202, 0x00080080, 0x20028a00, 
0x0a000080, 0x00820000, 0x00000200, 0x00080280, 
0x00080802, 0x00000008, 0x08200000, 0x00802020, 
0x28000208, 0x8a080000, 0x02082020, 0x00a00008, 
0x8000a000, 0xa0802000, 0x00200800, 0x08002200, 
0x00000022, 0x00208a00, 0x0020008a, 0x220a0882, 
0x28020000, 0x0a002080, 0x80080000, 0x08820800, 
0x02280000, 0x220a0000, 0x08000808, 0x0200200a, 
0x00002080, 0x00208000, 0x0000a208, 0x00002000, 
0x20208808, 0x8000000a, 0x00000882, 0x00800020, 
0x0020a000, 0x20002820, 0x08800a00, 0x80200082, 
0x22002020, 0x08028000, 0x82080008, 0xa0000000, 
0x28000200, 0x08088080, 0x80000080, 0x00000000, 
0x80200280, 0x800a0020, 0x00008228, 0x08008002, 
0x00080022, 0x20220000, 0x80208080, 0x22020000, 
0x08000800, 0x80282202, 0x208a2002, 0x00008220, 
0x00008000, 0x02020080, 0x20a00800, 0x80002280, 
0x000a0020, 0x08000028, 0x00008082, 0x00800022, 
0x00220808, 0x00008200, 0x00002000, 0x00828008, 
0x00002200, 0x00000000, 0x20028800, 0x0800a008, 
0x00000080, 0x00800228, 0x08282000, 0xa2000802, 
0x00808200, 0x82008002, 0x00002002, 0x20200008, 
0x02000080, 0x008800a0, 0x00828028, 0x00208200, 
0xa0800000, 0x00208800, 0x0a000080, 0x80820800, 
0x00008020, 0x08080202, 0x200008a0, 0x00800000, 
0x8a000008, 0x20800002, 0x00020008, 0x08000082, 
0x000a0020, 0x00000020, 0x00088000, 0x20082800, 
0x08020a00, 0x02002088, 0x00822020, 0x00000008, 
0x08200002, 0x02080000, 0x08028008, 0x88208000, 
0x02000800, 0x00200008, 0x88008202, 0x02802000, 
0x00000020, 0x00280008, 0x00822882, 0x00220000, 
0x00082280, 0x00000002, 0x20800000, 0x00000000, 
0x000a0000, 0x08008208, 0x0a200000, 0xa0800002, 
0x20800800, 0x0a000080, 0x20080080, 0x08008000, 
0x82008202, 0x00022880, 0x20000020, 0x00080208, 
0x00002802, 0x00800208, 0x88000002, 0x02020000, 
0x08020020, 0x00202080, 0x82080820, 0x00a00000, 
0x80200002, 0x00000020, 0x00020000, 0x82280208, 
0x00022000, 0x20028800, 0x08088080, 0x80800880, 
0x20200000, 0x80002008, 0x02080820, 0x00020220, 
0x82002008, 0x02882800, 0x00200200, 0x02008002, 
0x00802080, 0x08820028, 0x0008800a, 0x00082000, 
0x28200820, 0x02008000, 0x80000022, 0x20000200, 
0x00080008, 0x20080080, 0x00028020, 0x08000082, 
0x22000020, 0x28000000, 0x0a000202, 0x020a2000, 
0x00200008, 0x00008002, 0x00800082, 0x20220800, 
0x02002088, 0x82002802, 0x00a00828, 0x00080080, 
0x82020802, 0x28000000, 0x00200008, 0x00080000, 
0x00a00000, 0x00280080, 0x20800080, 0x00808020, 
0x00080202, 0x80000000, 0x00008820, 0x08002288, 
0x80800000, 0x00200020, 0x00000008, 0x00000082, 
0x00800220, 0x02202082, 0xa0800800, 0x00200000, 
0x88008202, 0x22822000, 0x00008000, 0x02000208, 
0x80000000, 0x28228000, 0x00000000, 0x00000020, 
0x20008000, 0x00000282, 0x000a0802, 0x28008020, 
0x0020a000, 0x02880820, 0x28020208, 0x80008000, 
0x00080080, 0x00820020, 0x02082008, 0x00800800, 
0x00200a20, 0x08008200, 0x00020002, 0x00008800, 
0x0000008a, 0x82020080, 0x20008208, 0x0a200008, 
0x80802000, 0x28020200, 0x00080282, 0x20000000, 
0x08220020, 0x8000a000, 0x00802880, 0x08020020, 
0x08088000, 0x00000020, 0x00a08800, 0x80200002, 
0x20020800, 0x00800000, 0x00008088, 0x80880800, 
0x20800000, 0x00208080, 0x02020020, 0x00808008, 
0x80282000, 0x20820082, 0x20008820, 0x02000000, 
0x00020002, 0x00208808, 0x80002200, 0x800000a0, 
0x20800000, 0x88008008, 0x02800002, 0x20020208, 
0x00200002, 0x20022080, 0x08200000, 0x00000002, 
0x80800000, 0x00000220, 0x00000008, 0x00022880, 
0x00800800, 0x08082088, 0x800a0080, 0x00008000, 
0x00208082, 0xa0080002, 0x00000000, 0x0a200000, 
0x00000020, 0x08808008, 0x00080008, 0x20882002, 
0x00228200, 0x00088008, 0x80000082, 0x00200820, 
0x00002002, 0x82080002, 0x08808000, 0x02200002, 
0x22000000, 0x20a00800, 0x8a000200, 0x200a0020, 
0x08028820, 0x00208200, 0x00080002, 0x20200000, 
0x00002080, 0x00802820, 0x00000028, 0x08002008, 
0x00080080, 0x00028208, 0x00002088, 0x22000800, 
0x08220a08, 0x08088200, 0x00820020, 0x00800000, 
0x80000202, 0x00020802, 0x00220200, 0x0a080000, 
0x80800080, 0x00808220, 0x08200080, 0xa00200a2, 
0x20008000, 0x0800008a, 0x02802000, 0x00200208, 
0x08080000, 0x02002000, 0x00220828, 0x8008a002, 
0x00020000, 0x08200a00, 0x08080008, 0x00802002, 
0x20800820, 0x00000288, 0xa2080822, 0x08800208, 
0x00002008, 0x20000020, 0x20020a00, 0x00080000, 
0x020a0000, 0x08200800, 0x8020a202, 0x80820080, 
0x00208000, 0x00002208, 0x02802020, 0x00008820, 
0x00280000, 0x82020800, 0x00820200, 0x08002008, 
0x20000820, 0x08800a00, 0x80200000, 0x20002000, 
0x00808000, 0x00200200, 0x00800082, 0x20000020, 
0x08080080, 0x02820000, 0x00000008, 0x80000000, 
0x000808a0, 0x08020008, 0x82000002, 0x20000800, 
0x00a00000, 0x0a200000, 0x22802080, 0x00020800, 
0x0020200a, 0x80000800, 0x00000220, 0x00008200, 
0x020228a2, 0x00000000, 0x08002000, 0x000a0000, 
0x20000200, 0x0a008080, 0x20880022, 0x28220000, 
0x08200200, 0x02800080, 0x08808020, 0x8000000a, 
0x20080800, 0x20008a00, 0x02000000, 0x02822882, 
0x00000020, 0x80000000, 0xa0080020, 0x00000020, 
0x00000002, 0x00000022, 0x08220808, 0x82280080, 
0x00080020, 0x00000808, 0x0200a200, 0x00020882, 
0x28008800, 0x02000080, 0x00020020, 0x20200228, 
0x00080200, 0x00080880, 0x20808008, 0x80202000, 
0x82000800, 0x08800808, 0x00208000, 0x20000000, 
0x08200020, 0x80000000, 0x000a0882, 0x28000200, 
0x00002080, 0x80020000, 0x00008800, 0x08002202, 
0xa20200a0, 0x00020000, 0x08002002, 0x82080020, 
0x00000000, 0x82080200, 0x20080000, 0x00220020, 
0x80280002, 0x80002000, 0x08000200, 0x08002208, 
0x00000802, 0x20800200, 0x00080088, 0xa2000000, 
0x20020208, 0x80002080, 0x00000020, 0x20800000, 
0x08080082, 0x00000020, 0x00000820, 0x0200200a, 
0x00882800, 0x20200000, 0x08000208, 0x008008a0, 
0x20200800, 0x80200000, 0x20000082, 0x00800028, 
0x0000a002, 0x80002000, 0x00000800, 0x02088280, 
0x02002080, 0x08008808, 0x80200000, 0xa0002080, 
0x08000200, 0x02088200, 0x82800080, 0x00008008, 
0x88000088, 0x000a0020, 0x08800000, 0x0000a00a, 
0x20082020, 0x00200200, 0x00200080, 0x00002000, 
0x00220000, 0x00000200, 0x000a0802, 0x00208000, 
0x02088080, 0x02002882, 0x00200000, 0x00200080, 
0x00080020, 0x20808020, 0x00202082, 0x02800800, 
0x28200008, 0x0a008000, 0x00020080, 0x00000008, 
0x02082208, 0x00802082, 0x20020800, 0x02008080, 
0x00802080, 0x00000a20, 0x80280002, 0xa2000022, 
0x00000000, 0x8020800a, 0x20000802, 0x20020008, 
0x80000200, 0x20002080, 0x08008808, 0x00002202, 
0x20820000, 0x00020000, 0x08008200, 0x02802000, 
0x20000008, 0x08080008, 0x20000022, 0x00808000, 
0x02008008, 0x02800800, 0x00220000, 0x80080002, 
0x22802000, 0x00200000, 0x82008002, 0x00020882, 
0x28208800, 0x08080000, 0x00022000, 0x00008808, 
0x00200280, 0x000000a2, 0x28000000, 0x82208000, 
0x80082800, 0x00a20200, 0x0a008000, 0x020800a0, 
0x00000800, 0x02080000, 0xa0002000, 0x08020800, 
0x00082280, 0x02002002, 0x20208000, 0x0008000a, 
0x020a0020, 0x08828200, 0x00000080, 0xa0002020, 
0x08020000, 0x08280000, 0x02880000, 0x00028000, 
0x0220020a, 0x00880800, 0x08000220, 0x0a080000, 
0x00802802, 0x20000000, 0x08002008, 0x80000082, 
0x00020028, 0x80008002, 0x20082800, 0x28800000, 
0x80088200, 0x20080080, 0x00208000, 0x80000008, 
0x008a0082, 0x00000020, 0x00008280, 0x800200a0, 
0x20200028, 0x80202080, 0x000a0880, 0x00008208, 
0x80000002, 0x00882000, 0x00a00000, 0x00088002, 
0x22020080, 0x08800020, 0x00088200, 0x00820800, 
0x28000200, 0x00000000, 0x80020082, 0x00a08800, 
0x88200000, 0x00000082, 0x08800008, 0x08008000, 
0x00080000, 0x00200008, 0x02208002, 0x00000020, 
0x00820020, 0x02080202, 0x20002800, 0x00020800, 
0x0a002080, 0x02022000, 0x00800000, 0x80280088, 
0x00000000, 0x28000020, 0x00208002, 0xa2000002, 
0x20020008, 0x82000280, 0x00880000, 0x00000800, 
0x0028a002, 0x00800002, 0x28208000, 0x0200a088, 
0x82820020, 0x20808000, 0x80080002, 0x22000800, 
0x20808200, 0x80208000, 0x82000800, 0x00000808, 
0x00000282, 0x02800020, 0x08200800, 0x80280200, 
0xa0082000, 0x08200000, 0x02000208, 0x00002020, 
0x00800228, 0x08200200, 0x02080000, 0x00020028, 
0x82000080, 0x02080000, 0x20a20008, 0x80000200, 
0x22002000, 0x00000808, 0x0200200a, 0x00000002, 
0x08020a00, 0x00000088, 0x00020000, 0x00000220, 
0x80200200, 0xa0020020, 0x08820200, 0x00000088, 
0x80880002, 0x08000000, 0x02080002, 0x208200a0, 
0x00200800, 0x00008002, 0x00022080, 0x28000020, 
0x02088000, 0x008000a0, 0x20800200, 0x00202088, 
0x02000880, 0x08800000, 0x80202000, 0x22002022, 
0x28000200, 0x00280200, 0x00022020, 0x00828000, 
0x80280000, 0x20080002, 0x00008800, 0x00008200, 
0x82000022, 0x20000808, 0x00002002, 0x80000080, 
0x00800008, 0x0000a000, 0x22800002, 0x20820a00, 
0x02200082, 0x20020000, 0x00a00008, 0x80000008, 
0x80822000, 0x08200000, 0x02000080, 0x82000022, 
0x20000a08, 0x80080088, 0x00000080, 0x28020208, 
0x00200000, 0x00000802, 0x00200808, 0x08200200, 
0x00880080, 0x00020020, 0x00000200, 0x20000800, 
0x00208020, 0x08088080, 0x82000800, 0x00000208, 
0x0028220a, 0x00020820, 0x20000020, 0x0220008a, 
0x80082800, 0x00a20008, 0x82000200, 0x008000a0, 
0x00820800, 0x02200000, 0x80000082, 0x08208000, 
0x00008008, 0x00022000, 0x20200008, 0x08000002, 
0x200000a0, 0x20008020, 0x02000088, 0x00002002, 
0x00220a00, 0x80000082, 0x00802020, 0x08000800, 
0x00000008, 0x200a2082, 0x08020800, 0x08000000, 
0x00020000, 0x00808a28, 0x00202000, 0x00080080, 
0x00028000, 0x80002002, 0x00802000, 0x08200200, 
0x08000000, 0x02080020, 0x00828028, 0x0208a000, 
0x80020802, 0x00200220, 0x00080000, 0x80820800, 
0x20208a20, 0x80000208, 0xa2020000, 0x28820000, 
0x80002000, 0x20080000, 0x00000208, 0x00200280, 
0x20820000, 0x08020028, 0x00200008, 0x00080880, 
0x00028020, 0x0200a000, 0x00002880, 0x00800800, 
0x00002000, 0x82020080, 0x00020200, 0x08208002, 
0xa2080020, 0x20a00200, 0x80288080, 0x02000000, 
0x00a08000, 0x02200008, 0x80880000, 0x20008200, 
0x08000280, 0x80000802, 0x00008820, 0x08202280, 
0x02000800, 0x28000200, 0x0a00a002, 0x20802002, 
0x20800008, 0x00000082, 0x00822000, 0x00220008, 
0x80080202, 0x20880800, 0x08200800, 0x00080280, 
0x020000a0, 0x00a00808, 0x00200088, 0x000a0000, 
0x20800028, 0x00202080, 0x00080020, 0x08020000, 
0x80208000, 0x02800000, 0x00a00000, 0x02080008, 
0x00000002, 0x00220220, 0x02008008, 0x80822002, 
0x00800208, 0x88202000, 0x00020802, 0x20808020, 
0x02000008, 0x20002800, 0x08800008, 0x02200000, 
0x20000080, 0x08008800, 0x02082202, 0xa0080000, 
0x00020000, 0x0a000288, 0x80822000, 0x20a00208, 
0x80000008, 0x00000000, 0x00020220, 0x88000002, 
0x02800802, 0x08020808, 0x00280280, 0x00022000, 
0x00008000, 0x00208200, 0x00082080, 0x00200a00, 
0x00000080, 0x80800022, 0x00800820, 0x00202080, 
0x00080820, 0x00000028, 0x00008000, 0x00000000, 
0x28220200, 0x82000202, 0x00880000, 0x00208008, 
0x0020a00a, 0x20802800, 0x00200820, 0x00088208, 
0x80802020, 0x20008000, 0x00280280, 0x80000882, 
0x00020200, 0x00000000, 0xa0802020, 0x08020200, 
0x0a008082, 0x208a0000, 0x00028800, 0x02200200, 
0x00080000, 0x20008220, 0x08000000, 0x00002822, 
0x00a08808, 0x08082000, 0x80020800, 0x08000208, 
0x00208008, 0x02000002, 0x00800000, 0x00008080, 
0x02020000, 0x00208000, 0x80002008, 0x20022000, 
0x00000220, 0x0a082000, 0x800008a0, 0x00200820, 
0x00000282, 0x82020000, 0x08000000, 0x0000800a, 
0x20000022, 0x00820800, 0x80200002, 0x20002000, 
0x08000800, 0x00000202, 0x20022000, 0x08008200, 
0x00000000, 0x82002000, 0x20808208, 0x00280280, 
0x80020080, 0x20808000, 0x0a200002, 0x02000820, 
0x00200800, 0x08000000, 0x02800020, 0x08a20028, 
0x02008200, 0x00000002, 0x00008200, 0x08080088, 
0x00000002, 0x00000808, 0x80200202, 0x00020002, 
0x08000000, 0x00008080, 0x00002020, 0x00020808, 
0x88280000, 0x00082000, 0x08000028, 0x00000202, 
0xa0800002, 0x00208800, 0x00008088, 0x02820880, 
0x00008000, 0x88002208, 0x02080820, 0x00820208, 
0x08200080, 0x20800002, 0x00200000, 0x08200000, 
0x00082000, 0x00808020, 0x00200000, 0x20002080, 
0x28008200, 0x08080088, 0x80800002, 0x00800220, 
0x00080002, 0xa20a0080, 0x08000000, 0x8200a000, 
0x02000820, 0x00000000, 0x8a000002, 0x00880080, 
0x00008800, 0x00008000, 0x00820002, 0x08000000, 
0x00082208, 0x00000822, 0x00200820, 0x80200008, 
0x02000820, 0x20008208, 0x88000008, 0x00002000, 
0x00800808, 0x00200202, 0x00020080, 0x00000808, 
0x02202008, 0x00820000, 0x08028200, 0x0a000000, 
0x000028a0, 0x00a00000, 0x08080000, 0x82000082, 
0x08800200, 0x00002000, 0xa0000800, 0x00000800, 
0x00008200, 0x00022000, 0x00228008, 0x02080208, 
0xa0020082, 0x28020020, 0x00088000, 0x80020880, 
0x00000820, 0x00202082, 0x02000080, 0x08828028, 
0x80002002, 0x00880802, 0x20820a00, 0x82008080, 
0x20020000, 0x08820020, 0x80008002, 0x00822002, 
0x20000000, 0x02000208, 0x00002000, 0x20000200, 
0x00082080, 0x80000022, 0x08020008, 0x08002000, 
0x22800000, 0x28000800, 0x82000000, 0x00002080, 
0x08808000, 0x8000a000, 0x00002080, 0x20200000, 
0x00088000, 0x00820000, 0x00000028, 0x00080208, 
0x00000002, 0x00800200, 0x0200800a, 0x02082022, 
0x08a00000, 0x00200080, 0x00002020, 0x08000800, 
0x0200a000, 0x00020800, 0x00220000, 0x0800a080, 
0x82800020, 0x00800200, 0x00000200, 0x00080820, 
0x20000228, 0x00008080, 0x80802802, 0x08200208, 
0x88288000, 0x20080000, 0x00800000, 0x00280200, 
0x20002080, 0x00008800, 0x08002200, 0x80020020, 
0x20008820, 0x00000280, 0x02000080, 0x00028000
};

static unsigned long index[32] = 
{
    0x1, 0x2, 0x4, 0x8, 
    0x10, 0x20, 0x40, 0x80, 
    0x100, 0x200, 0x400, 0x800, 
    0x1000, 0x2000, 0x4000, 0x8000, 
    0x10000, 0x20000, 0x40000, 0x80000, 
    0x100000, 0x200000, 0x400000, 0x800000, 
    0x1000000, 0x2000000, 0x4000000, 0x8000000, 
    0x10000000, 0x20000000, 0x40000000, 0x80000000
};

int
isprime(unsigned int i)
{
    return (primes[i >> 5] & index[i & 0x1f]) ? 1 : 0;
}

#ifdef TEST

#include <stdio.h>

main(int argc, char **argv)
{
    unsigned int x;

    if (argc != 2)
    {
        printf("Wrong number of arguments.\n");
        exit(1);
    }
    sscanf(argv[1], "%u", &x);
    printf("%u is %sprime.\n", x, isprime(x) ? "" : "not " );
    exit(0);
}

#endif

