/*************************************************************************/
/*                                                                       */
/*  REXX Control Script for Modem Handling and User Access Control       */
/*                                                                       */
/* (C) 1995, 1996   Axel Mueller (amueller@stargate.rz.fh-offenburg.de)  */
/*     Version 0.98   Last modified: 14.04.96                            */
/*************************************************************************/

/* ##################################################################### */
/* The value for "ScriptIndex" is the only difference between the copies */
/* of the program running for each modem; ScriptIndex = 1 means COM1     */   
/*                                        ScriptIndex = 2 means COM2     */
/*                                        ScriptIndex = 3 means COM3     */
/*                                        ScriptIndex = 4 means COM4     */
/* ##################################################################### */

parse arg ScriptIndex

call on error



/************************************************************************/
/*                                                                      */
/* Setup                                                                */
/*                                                                      */
/************************************************************************/

/* Number of modems installed */
ModemMax = 2

/* boolean variables */
True          = 1
False         = 0


/**************/
/* Path setup */
/**************/

/* Root directory */
Path           = 'c:\userctrl'

/* Directory under which all the POP mail directories are situated */
MailPath       = value('MAILDIR',,'OS2ENVIRONMENT')

/* Path to IBMs TCP/IP ETC directory */
ETCPath        = value('ETC',,'OS2ENVIRONMENT')


/* ##### Log files ##### */

/* Log file for all messages generated by this REXX script */
SystemLog      = Path'\Log\UserCtl'ScriptIndex'.log'


/* ##### Files needed for login procedure ##### */

/* File to be send to modem user after CONNECT but before login */
WelcomeFile	= Path'\welcome.msg'


/* ##### TCP/IP related ##### */

/* Configuration file for PPP driver of IBM's TCP/IP */
PPP_CFGFile    = ETCPath'\ppp'ScriptIndex'.cfg'


/******************/
/* COM port setup */
/******************/


PortName      = 'com'ScriptIndex
PortHandle    = ''

WriteLim      = 50
ReadLim       = 50                     /* 0,5 seconds timout */
Flags1        = '00001001'
Flags2        = '10100000'
Flags3        = '11010010'
ErrChar       = '00'
BrkChar       = '00'
XonChar       = '11'
XoffChar      = '13'
EnhParms      = '00000010'

crlf          = D2C(13)''D2C(10)
none_on       = '00'
dtr_on        = '01'
rts_on        = '02'
both_on       = '03'
none_off      = 'FF'
dtr_off       = 'FE'
rts_off       = 'FD'
both_off      = 'FC'

Timeout       = 15                     /* Wait 15 seconds for com port */
UserMaxLen    = 25
PwdMaxLen     = 25


'@echo off'
'cd' Path

call flagfile('START')



/************************************************************************/
/*********************                ***********************************/
/********************* Main function  ***********************************/
/*********************                ***********************************/
/************************************************************************/


do forever


	/* open COM port */
	call open_com_port

	/* set modified Device Control Block (dcb) */
	rc = RxIOCtlSetDcbParm( PortHandle, WriteLim, ReadLim, Flags1, Flags2, Flags3, ErrChar, BrkChar, XonChar, XoffChar )
	if rc <> 0 then
		call log('##ERROR## : modified Device Control Block could not be set')
	else
		call log('MAINFUNCT : modified Device Control Block set')


	/* set Enhanced Parameters */
	rc = RxIOCtlSetEnhParm( PortHandle, EnhParms )
	if rc <> 0 then
		call log('##ERROR## : Enhanced parameter could not be set')
	else
		call log('MAINFUNCT : Enhanced Parameter set')


	/* initialize modem */
	call modem_init
   
	/* create flag file indicating that the system is waiting for caller */
	call flagfile('WAIT')
   
	/* wait for input from buffer; RING will be assumed */
	Buffer = ""
	Ring = False
	call log('MAINFUNCT : waiting for RING ...' )
	rc = lineout(SystemLog)
	rc = RxIOCtlRead(PortHandle, 0, -1, 'Buffer')

	/* create flag file indicating that login is in process */
	call flagfile('LOGIN')

	/* check the buffer for the RING string */
	RxCount = 0
	RxSize = 0
	rc = RxIOCtlGetRxCount( PortHandle, 'RxCount', 'RxSize' )
	Buffer = ''
	if RxCount > 0 then
		do
			do Number = 1 to RxCount
				Character = ''
				rc = RxIOCtlReadChar( PortHandle, 0, 'Character' )
				if Character <> '' then
					Buffer = Buffer||Character
			end
			RingPos = pos('RING',Buffer)
			if  RingPos > 0 then
				do
					call log('MAINFUNCT : RING detected' )
					Ring = True
				end
			else
				call log('##ERROR## : No RING found in buffer: <'Buffer'>')
		end

	if Ring then
		do               /* pick up the line */
			Rest = 0
			rc = RxIOCtlWrite( PortHandle, 'ATA'||D2C(13), 'Rest' )
			if rc <> 0 then
				call log('##ERROR## : could not send ATA to modem')
       
			/* wait for CONNECT; Timeout = 60 sconds */
			Connect = False
			Cycle = 0
			MaxCycle = 60 * 100 / ReadLim
			do while (Connect == False) & (Cycle < MaxCycle)
				Cycle = Cycle + 1
				rc = RxIOCtlRead( PortHandle, 0, 1, 'Buffer' )
				if left(Buffer,7) == "CONNECT" then
					do
						ConString = strip(Buffer,,D2C(10))
						call log('MAINFUNCT :' ConString )
						Connect = True
					end
			end

			call check_carrier    
			if Carrier then
				call handle_login           /* start login procedure */
			else
				call log('MAINFUNCT : Timeout waiting for CONNECT')

		end

	/* close COM-Port */
	call close_com_port

	/* wait 30 seconds to force line down */
	call log('MAINFUNCT : Waiting for 15 seconds ...')
	call SysSleep 15

end

/* exit programm */
exit




/************************************************************************/
/*********************                ***********************************/
/*********************   Functions    ***********************************/
/*********************                ***********************************/
/************************************************************************/



/************************************************************************/
/* handle_login()                                                       */
/*                                                                      */
/* main fuction handling the login procedure                            */
/************************************************************************/

handle_login:
          
	call log( 'HNDLELOGIN: Start HANDLE_LOGIN function ...' )

	call check_carrier    
	if Carrier then
		call welcome_screen                    /* send login screen */

	call check_carrier
	LoginOkay = False
	Attempt = 1
	do while (LoginOkay = False) & (Attempt<=3) & Carrier

		call log('LOGIN     : --- 'Attempt'. Attempt ---' )

		call check_carrier
		if Carrier then
			User = get_user()
		else
			call log( 'HNDLELOGIN: No Connection. Skipping user request' )

		call check_carrier
		if Carrier then
			Password = get_password()
		else
			call log( 'HNDLELOGIN: No Connection. Skipping password request' )

		call flush_buffer

		call check_buffer
		call check_carrier
		if Carrier & BufferIsEmpty then
			LoginOkay = check( User, Password )
		else
			do
				call log( 'HNDLELOGIN: No connection or characters in input buffer.')
				call log( 'HNDLELOGIN: Skipping user/password check' )
			end

		Attempt = Attempt + 1

	end
    
	if LoginOkay = True then
		do
			call check_carrier
			if Carrier then
				do
					call send( '$%' )
					call send( crlf )

					/* built file name for user log */
					UserLog = Path'\log\'substr(date(european),4,2)||substr(date(european),7,2)'_'ScriptIndex'.log'

					call log('HNDLELOGIN: *** Startup PPP driver ***')
					StartTime = time(normal)
					StartDate = date(european)
					rc = time('R')

					'PPP file' PPP_CFGFile

					LoginTime = trunc(time('R'),0)

					EndTime = time(normal)

					call log('HNDLELOGIN: *** Shutdown PPP driver ***')

					/* Log connection data */
					LogString = substr(User,1,20)||StartDate'   'StartTime'   'EndTime
					rc = lineout( UserLog, LogString )
					rc = lineout( UserLog )

					rc = RxPrfAddLoginTime(User, LoginTime)

					/* create flag file indicating that a user is active */
					call flagfile('LGOUT')

				end
		end
	else
		call log( 'HNDLELOGIN: Login failed. PPP not started.' )

	call log( 'HNDLELOGIN: ... Finish HANDLE_LOGIN function' )

	return



/************************************************************************/
/* welcome_screen ()                                                    */
/*                                                                      */
/* sends a text file (Welcom Screen) to modem user BEFORE user and      */
/* password are requested                                               */
/************************************************************************/

welcome_screen:


	if length(stream(WelcomeFile, 'c', 'query exists')) > 0 then
		do
			Line = linein(WelcomeFile, 1, 1 )
			call check_carrier    
			do while (Line <> '$') & Carrier
				call check_carrier    
				if Carrier then
					do
						call send( Line )
						call send( crlf )
					end
				Line = linein(WelcomeFile)
			end
			/* close WelcomeFile */
			rc = lineout(WelcomeFile)
			if Carrier == False then
				call log( 'WELCSCREEN: not (fully) sent' )
		end

    return



/************************************************************************/
/* get_user()                                                           */
/*                                                                      */
/* requests the username from modem user; there is a timeout after      */
/* which the login procedure is canceled and the line is dropped        */
/* (set in "Timeout"); there is also a maximum number of characters     */
/* read from com port for username (set in "UserMaxLen")                */
/************************************************************************/

get_user:

	call check_buffer
	if BufferIsEmpty then
		call send('   Login:')

	Dummy = 0
	User = ''
	UserLen = 0
	Cycle = 0
	UserTimeout = False
	TooLong = False
	Buffer = ''
	do while (Buffer <> D2C(13)) & (UserTimeout == False) & (TooLong == False) & Carrier & BufferIsEmpty
		call check_carrier
		if Carrier then
			do
				Buffer = ''
				rc = RxIOCtlReadChar( PortHandle, 1, 'Buffer' )
				select
					when rc = 0 then
						if UserLen < UserMaxLen then
							do
								User = User||Buffer
								UserLen = UserLen + 1
								Cycle = 0
							end
						else
							TooLong = True
					when rc = -1 then                 /* send no echo for CR */
						Dummy = 1
					when rc = -2 then
						Cycle = Cycle + 1
				end
				MaxCycle = Timeout * 100 / ReadLim
				if Cycle > MaxCycle then
					UserTimeout = True

				call check_carrier
			end
	end

	if Carrier & BufferIsEmpty then
		call send( crlf )

	if UserTimeout = True then
		do
			call log( 'GETUSER   : Timeout waiting for USER. Timout: 'Timeout's')
			call check_carrier
			if Carrier then
				call hangup
		end
	if TooLong = True then
		do
			call log( 'GETUSER   : USER lenght exceeded. Maximum length: 'UserMaxLen )
			call check_carrier
			if Carrier then
				call hangup
		end
	else
		if Carrier then
			call log( 'GETUSER   : User =' User )

	return User



/************************************************************************/
/* get_password()                                                       */
/*                                                                      */
/* requests the password from modem user; there is a timeout after      */
/* which the login procedure is canceled and the line is dropped        */
/* (set in "Timeout"); there is also a maximum number of characters     */
/* read from com port for username (set in "PwdMaxLen")                 */
/************************************************************************/

get_password:

	call check_buffer
	if BufferIsEmpty then
		call send('Password:')

	Dummy = 0
	Password = ''
	PwdLen = 0
	Cycle = 0
	PwdTimeout = False
	TooLong = False
	Buffer = ''
	do while (Buffer <> D2C(13)) & (PwdTimeout == False) & (TooLong == False) & Carrier & BufferIsEmpty
		call check_carrier
		if Carrier then
			do
				Buffer = ''
				rc = RxIOCtlReadChar( PortHandle, -1, 'Buffer' )
				select
					when rc = 0 then
						if PwdLen < PwdMaxLen then
							do
								Password = Password||Buffer
								PwdLen = PwdLen + 1
								Cycle = 0
							end
						else
							TooLong = True
					when rc = -1 then                 /* send no echo for CR */
						Dummy = 1
					when rc = -2 then
						Cycle = Cycle + 1
				end
				MaxCycle = Timeout * 100 / ReadLim
				if Cycle > MaxCycle then
					PwdTimeout = True

				call check_carrier
			end
	end

	if Carrier & BufferIsEmpty then
		call send( crlf )

	if PwdTimeout = True then
		do
			call log( 'GETPWD    : Timeout waiting for PWD. Timeout: 'Timeout's' )
			call check_carrier
			if Carrier then
				call hangup
		end
	if TooLong = True then
		do
			call log( 'GETPWD    : PWD lenght exceeded. Maximum length: 'PwdMaxLen )
			call check_carrier
			if Carrier then
				call hangup
		end
	else
		if Carrier then
			call log( 'GETPWD    : Password =' copies('*',PwdLen) )

	return Password



/************************************************************************/
/* check(User,Password)                                                 */
/*                                                                      */
/* checks username and password given by modem user                     */
/************************************************************************/

check:

    parse arg User, Password

	LoginOkay = RxPrfCheckPassword(User, Password)
	if LoginOkay then
		do
			 call log( 'CHECK     : User identification passed' )
			 call send( crlf )
			 call send('Login OK.')
			 call send( crlf )

			 call send( crlf )
			 LastLogin = 'Last login on 'RxPrfGetLastLoginDate(User)
			 call send(LastLogin)
			 call send( crlf )

			 UserMail = MailPath'\'User'\*.msg'   
			 call SysFileTree UserMail, 'MailFiles', 'O'
			 call send( crlf )
			 if MailFiles.0 > 0 then
				call send('*** You have 'MailFiles.0' new mail(s) ***')
			 else
				call send('No new mail.')

			 call send( crlf )
			 call send( crlf )

			 UserRealName = RxPrfGetRealName(User)

			/* create flag file indicating that a user is active */
			call flagfile('ACTIV')


			ActiveUser = 0
			do ModemIndex=1 to ModemMax
				ActivFlagFile = Path'\L'ModemIndex'_ACTIV.FLG'
				if length(stream(ActivFlagFile, 'c', 'query exists')) > 0 then
					do
						ActiveUser = ActiveUser + 1
						Line = linein(ActivFlagFile)
						rc = lineout(ActivFlagFile)
					end
				else
					Line = 'not used'

				call send( 'Modem 'ModemIndex' : 'Line )
				call send( crlf )
			end
			call send( crlf )
		end
	else
		do
			call log( 'CHECK     : User identification failed' )
			call send('Login failed.')
			call send( crlf )

			FailLog = Path'\log\'substr(date(european),4,2)||substr(date(european),7,2)'_'ScriptIndex'XX.LOG'
			LogString = date(european)'   'time(normal)'   U='User'   P='Password
			rc = lineout( FailLog, LogString )
			rc = lineout( FailLog )
		end

    return LoginOkay



/************************************************************************/
/* send ( sendstring )                                                  */
/*                                                                      */
/* send a character string off to the modem                             */
/************************************************************************/

send:

	parse arg SendString

	Rest = 0
	call check_carrier
	if Carrier then
		rc = RxIOCtlWrite( PortHandle, SendString, 'Rest' )
	else
		if SendString == crlf then
			say '##ERROR## : crlf not sent'
		else
			say '##ERROR## : 'SendString' not sent'

	return



/************************************************************************/
/* log ( logstring )                                                    */
/*                                                                      */
/* logs the main activities of this REXX script                         */
/************************************************************************/

log:

	parse arg LogString
	LogString = strip(LogString,,D2C(13))

	rc = lineout( SystemLog, date(european)'-'time(normal)'' LogString )
	if rc <> 0 then
		say '##ERROR## : writing main log file'

	say date(european)' 'time(normal)' 'left(LogString,61)

	return



/************************************************************************/
/* flagfile ( Flag )                                                    */
/*                                                                      */
/* creates a flag file indicating current status of UserCtrl program    */
/************************************************************************/

flagfile:

	parse arg Flag

	/* delete old flag file */
	FlagFile = Path'\L'ScriptIndex'_START.FLG'
	if length(stream(FlagFile, 'c', 'query exists')) > 0 then
		rc = SysFileDelete(FlagFile)
	FlagFile = Path'\L'ScriptIndex'_WAIT.FLG'
	if length(stream(FlagFile, 'c', 'query exists')) > 0 then
		rc = SysFileDelete(FlagFile)
	FlagFile = Path'\L'ScriptIndex'_LOGIN.FLG'
	if length(stream(FlagFile, 'c', 'query exists')) > 0 then
		rc = SysFileDelete(FlagFile)
	FlagFile = Path'\L'ScriptIndex'_ACTIV.FLG'
	if length(stream(FlagFile, 'c', 'query exists')) > 0 then
		rc = SysFileDelete(FlagFile)
	FlagFile = Path'\L'ScriptIndex'_LGOUT.FLG'
	if length(stream(FlagFile, 'c', 'query exists')) > 0 then
		rc = SysFileDelete(FlagFile)


	FlagFile = Path'\L'ScriptIndex'_'Flag'.FLG'
	if Flag == "ACTIV" then
		rc = lineout(FlagFile, UserRealName)

	rc = lineout(FlagFile)

	return



/************************************************************************/
/* open_com_port()                                                      */
/*                                                                      */
/* opens the com port                                                   */
/************************************************************************/


open_com_port:

	rc = RxIOCtlOpenPort(PortName, 'PortHandle')
	if rc <> 0 then
		do
			call log('##ERROR## : COM port' PortName 'could not be opened')
			exit
		end
	else
		call log('COMOPEN   : COM-Port' PortName ' opened')

	return



/************************************************************************/
/* close_com_port()                                                     */
/*                                                                      */
/* closes the com port                                                  */
/************************************************************************/


close_com_port:

	rc = RxIOCtlClosePort( PortHandle )
	if rc <> 0 then
		do
			call log('##ERROR## : COM-Port' PortName 'could not be closed')
			exit
		end
	else
		call log('COMCLOSE  : COM port' PortName ' closed')

	return



/************************************************************************/
/* modem_init ()                                                        */
/*                                                                      */
/* initializes the modem; note that the modem should be set to NO       */
/* auto answer (ATS0=0) since the script picks up the line if it        */
/* detects a RING signal. Therefore the modem has to be set for         */
/* echo. (ATE=1)                                                        */
/************************************************************************/


modem_init:

	call flush_buffer

	Rest = 0
	rc = RxIOCtlWrite( PortHandle, 'ATZ'||D2C(13), 'Rest' )
	if rc <> 0 then
		call log('##ERROR## : sending ATZ to modem')

	Buffer = ''
	rc = RxIOCtlRead( PortHandle, 0, 1, 'Buffer' )
	if rc <> 0 then 
		call log('##ERROR## : reading modem echo for ATZ rc='rc)
	if left(Buffer,3) <> "ATZ" then
		call log('##ERROR## : Expected: ATZ  Echo: 'Buffer)

	Buffer = ''
	rc = RxIOCtlRead( PortHandle, 0, 1, 'Buffer' )
	if rc <> 0 then
		call log('##ERROR## : reading modem response for ATZ rc='rc)
	if left(Buffer,2) <> "OK" then
		call log('##ERROR## : Expected: OK  Response: 'Buffer)

	return



/************************************************************************/
/* hangup()                                                             */
/*                                                                      */
/* forces the modem to drop the line                                    */
/************************************************************************/

hangup:

	call flush_buffer

	Rest = 0
	rc = RxIOCtlWrite( PortHandle, '+++', 'Rest' )
	if rc <> 0 then
		call log('##ERROR## : beim Senden von +++ an Modem')
	else
		call log( 'HANGUP    : +++ sent to modem' )


	call SysSleep 2

	Rest = 0
	rc = RxIOCtlWrite( PortHandle, 'ATH0'||crlf, 'Rest' )
	if rc <> 0 then
		call log('##ERROR## : sending ATH0 to modem')
	else
		call log( 'HANGUP    : ATH0 sent to modem' )

	call SysSleep 10

	return


/************************************************************************/
/* check_buffer ()                                                      */
/*                                                                      */
/* checks the input buffer of the com port for characters               */
/************************************************************************/

check_buffer:

	/* there should be NO characters in the input buffer */
	RxCount = 0
	RxSize = 0
	rc = RxIOCtlGetRxCount( PortHandle, 'RxCount', 'RxSize' )
	if RxCount > 0 then
		do
			BufferIsEmpty = False
			call log( 'CHCKBUFFER: number of characters in input buffer: 'RxCount )
			call log( 'CHCKBUFFER: size of input buffer: 'RxSize )
		end

	return



/************************************************************************/
/* flush_buffer ()                                                      */
/*                                                                      */
/* flushes the input buffer                                             */
/************************************************************************/

flush_buffer:

	rc = RxIOCtlFlushInput( PortHandle )

	call log( 'FLUSHBUFFR: Input buffer flushed' )

	BufferIsEmpty = True

	return



/************************************************************************/
/* check_carrier ()                                                     */
/*                                                                      */
/* checks if the line is up and there is a carrier                      */
/************************************************************************/

check_carrier:

	rc = RxIOCtlDetectCarrier( PortHandle )
	if rc = 0 then
		Carrier = True
	else
		do
			Carrier = False
			call log( 'CHKCARRIER: !!! No Carrier !!!' )
		end

	return



/************************************************************************/
/* error ()                                                             */
/*                                                                      */
/*                                                                      */
/************************************************************************/

error:

	call log('##ERROR## : error() called')
	call RxIOCtlClosePort PortHandle

	exit
