/*****************************************************************************
  GENERVW.CPP

  Purpose:
  
    Handles the drawing and UI for the view.  Also has an associated timer
	to handle color animation.

  Functions:

    CGeneratorView::OnCreate  
    Create the view and initialize the timer.
                
	CGeneratorView::OnDestroy 
	Destroy the view and kill the timer.
	            
	CGeneratorView::OnDraw    
	Paint a message refecting what the RGB value of the color.
	            
	CGeneratorView::OnEraseBkgnd    
	Paints the background of the view the color of the RGB value contained
	in the document.
	      
	CGeneratorView::OnTimer           
	If the documents animate flag is set, call OnGenColor to generate a new
	color.
	    
	CGeneratorView::OnGenColor        
	Generate a new color.
	    
	CGeneratorView::OnUpdateGenColor  
	This menu choice will not be available if color is being animated.
	    
	CGeneratorView::OnAnimateColors      
	Set the ananimate flag in the doc and if there is a client, notify
	it to update it's menu.
	 
	CGeneratorView::OnUpdateAnimateColors 
	Check on or off the Animate Colors menu choice.

  Development Team: Robert Duke

  Written by Microsoft Product Support Services, Languages Developer Support
  Copyright (c) 1993 Microsoft Corporation. All rights reserved.
\****************************************************************************/

#include "stdafx.h"
#include "Generat.h"

#include "ClntDisp.h"

#include "Generdoc.h"
#include "Genervw.h"


#include "mainfrm.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CGeneratorView

IMPLEMENT_DYNCREATE(CGeneratorView, CView)

BEGIN_MESSAGE_MAP(CGeneratorView, CView)
	//{{AFX_MSG_MAP(CGeneratorView)
	ON_COMMAND(ID_GENERATOR_GENCOLOR, OnGenColor)
	ON_WM_ERASEBKGND()
	ON_COMMAND(ID_GENERATOR_ANIMATECOLORS, OnAnimateColors)
	ON_UPDATE_COMMAND_UI(ID_GENERATOR_ANIMATECOLORS, OnUpdateAnimateColors)
	ON_UPDATE_COMMAND_UI(ID_GENERATOR_GENCOLOR, OnUpdateGenColor)
	ON_WM_CREATE()
	ON_WM_TIMER()
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CGeneratorView construction/destruction

CGeneratorView::CGeneratorView()
{

}

CGeneratorView::~CGeneratorView()
{

}

/************************************************************************
  CGeneratorView::OnCreate

  Purpose:

    Create the view and initialize the timer.

  Parameters: 

    lpCreateStruct    Points to a CREATESTRUCT structure that contains 
    information about the CWnd object being created.

  Returns:

    OnCreate must return 0 to continue the creation of the CWnd object. 
    If the application returns -1, the window will be destroyed.

  Comments:

    This is a default OnCreate function with the addition of the timer
	initialization.

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

int CGeneratorView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;

	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Timers are not such a limited resource with 32 bit
	// code so this probably won't fail.  

	if(0 == (m_uTimerID = SetTimer(1, 500, NULL)))
		MessageBox("Amimate Colors will not work.", "No Timers Available");
	else
		pDoc->m_bCanAnimate = TRUE;
			
	return 0;
}

/************************************************************************
  CGeneratorView::OnDestroy

  Purpose:

    Kill the timer - call the base class.

  Parameters:  None

  Returns: None.

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/
														
void CGeneratorView::OnDestroy() 						
{
	// timer creation may have failed
	if(0 != m_uTimerID)
		KillTimer(m_uTimerID);

	CView::OnDestroy();	
}

/////////////////////////////////////////////////////////////////////////////
// CGeneratorView drawing

/************************************************************************
  CGeneratorView::OnDraw

  Purpose:

    Paint a message refecting what the RGB value of the color.

  Parameters:  
  
    pDC    Points to the device context to be used for rendering an 
    image of the document.

  Returns: None.

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

void CGeneratorView::OnDraw(CDC* pDC)
{
	CGeneratorDoc* pDoc = GetDocument();					 
	ASSERT_VALID(pDoc);										 

	CString strMessage;
	strMessage.Format( "Red = %d, Green = %d, Blue = %d",
						GetRValue(pDoc->m_dwColor), 
						GetGValue(pDoc->m_dwColor), 
						GetBValue(pDoc->m_dwColor));

	pDC->TextOut(10, 10, strMessage);
}

/************************************************************************
  CGeneratorView::OnEraseBkgnd

  Purpose:

    Paints the background of the view the color of the RGB value 
    contained in the document.

  Parameters:  
  
    pDC    Specifies the device-context object.

  Returns: 
  
    Nonzero if it erases the background; otherwise 0.

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

BOOL CGeneratorView::OnEraseBkgnd(CDC* pDC) 
{
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Set brush to desired background color
	CBrush backBrush(pDoc->m_dwColor);

	// Save old brush
	CBrush* pOldBrush = pDC->SelectObject(&backBrush);

	CRect rect;
	pDC->GetClipBox(&rect);     // Erase the area needed

	pDC->PatBlt(rect.left, rect.top, rect.Width(), rect.Height(),
	  PATCOPY);
	pDC->SelectObject(pOldBrush);
 		
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CGeneratorView diagnostics

#ifdef _DEBUG
void CGeneratorView::AssertValid() const
{
	CView::AssertValid();
}

void CGeneratorView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CGeneratorDoc* CGeneratorView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CGeneratorDoc)));
	return (CGeneratorDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CGeneratorView message handlers

/************************************************************************
  CGeneratorView::OnTimer

  Purpose:

    If the documents animate flag is set, call OnGenColor to generate a 
    new	color.

  Parameters:  
  
    nIDEvent    Specifies the identifier of the timer.

  Returns: 	void

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

void CGeneratorView::OnTimer(UINT nIDEvent) 
{
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// the the animate color flag is set gen a color
	if(pDoc->m_bAnimate)
		OnGenColor();										 
	
	// don't call the base class - doing so will cause problems if you are
	// working with very high timer rates 														 
	//CView::OnTimer(nIDEvent);								 
}															 

/************************************************************************
  CGeneratorView::OnGenColor

  Purpose:

    Cause the document to generate a new color, update the message 
	reflecting who requested the color, Cause the view to repaint,
	and if a client exists, notify it that a new color has been
	generated.

  Parameters:  None

  Returns: 	void

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/
															 
void CGeneratorView::OnGenColor() 							 
{															 
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	pDoc->MakeRandColor();	 // changes m_dwColor in the document

	Invalidate(TRUE);  // only view of doc or UpdateAllViews

	// if there is a client - notify that we have changed
	// our color
	if(NULL != pDoc->m_ClientDisp.m_lpDispatch)
		pDoc->DoNotifyColorChange();
}

/************************************************************************
  CGeneratorView::OnUpdateGenColor

  Purpose:

    This menu choice will not be available if color is being animated.

  Parameters:  
  
    pCmdUI points to a CCmdUI object to be used to update the menu.

  Returns: 	void

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

void CGeneratorView::OnUpdateGenColor(CCmdUI* pCmdUI) 
{
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	pCmdUI->Enable(!pDoc->m_bAnimate);
}

/************************************************************************
  CGeneratorView::OnAnimateColors

  Purpose:

    Toggle the ananimate flag in the doc and if there is a client, notify
	it to update it's menu to match that of the server.

  Parameters:  None

  Returns: 	void

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

void CGeneratorView::OnAnimateColors() 
{
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	pDoc->m_bAnimate = !pDoc->m_bAnimate;

	// if there is a client - notify that we have changed
	// our animation state
	if(NULL != pDoc->m_ClientDisp.m_lpDispatch)
		pDoc->m_ClientDisp.SetAnimateUI();
}

/************************************************************************
  CGeneratorView::OnUpdateAnimateColors

  Purpose:

    Check on or off the Animate Colors menu choice.

  Parameters:  
  
    pCmdUI points to a CCmdUI object to be used to update the menu.

  Returns: 	void

  Comments:

  History:

  Date     Comment                                           Initials
  ======== ================================================= ========
  10/10/95 Created                                             RED
\***********************************************************************/

void CGeneratorView::OnUpdateAnimateColors(CCmdUI* pCmdUI) 
{
	CGeneratorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// if we can animate - maintain the check
	if(!pDoc->m_bCanAnimate)
		pCmdUI->Enable(FALSE);
	else
		pCmdUI->SetCheck(pDoc->m_bAnimate);
}

													   
													   
													   
													   
													   
													   
													   
													   
													   
