/*********************************************************************\ 
*                                                                     *
*  coreduce.c: color reduction using single-criterion elimination     *
*  and multi-criteria elimination algorithms.                         *
*  Copyright 1995 Diana Gruber, permission granted to use and         *
*  modify this code.                                                  *
*                                                                     *
*  command line: coreduce <pcxfile> <outfile> <ncolors>               *
*                                                                     *
\*********************************************************************/

#include <fastgraf.h>
#include <stdio.h>
#include <stdlib.h>

int minx,maxx,miny,maxy;
int width,height;

void reduce(int colors,int offset,int quick);

/*********************************************************************/
void main(int argc,char *argv[])
{
   int mode;
   static char PCXheader[128];

   /* initialize for SVGA */
   fg_svgainit(0);

   /* determine mode based on PCX file header */
   fg_pcxhead(argv[1],PCXheader);
   mode = fg_pcxmode(PCXheader);
   fg_setmode(mode);

   /* calculate the size of the picture */
   fg_pcxrange(PCXheader,&minx,&maxx,&miny,&maxy);
   width = maxx-minx+1;
   height = maxy-miny+1;

   /* display the picture and do the reduction */
   fg_showpcx(argv[1],0);
   reduce(atoi(argv[3]),10,0);

   /* create a new pcx file */
   fg_makepcx(0,width-1,0,height-1,argv[2]);
   fg_waitkey();

   /* reset the video mode and exit to DOS */
   fg_setmode(3);
   fg_reset();
}

/*********************************************************************/
void reduce(int target,int offset,int quick)
{
   static long count[256];
   static long orig_count[256];
   static unsigned char colormap[256];
   static unsigned char RGBvalues[768];
   long   d, distance;
   long   red, green, blue;
   int    in_use;
   long   least, index, match;
   int    nearest;
   int    i, j, x, y;
   long   temp;

   /* initialize the pixel counter and color map */
   for (i = 0; i < 256; i++)
   {
      count[i] = 0;
      colormap[i] = (unsigned char)i;
   }

   /* grab the current DAC values */
   fg_getdacs(0,256,RGBvalues);

   /* count how many pixels of each color */
   miny = 0; maxy = height;
   minx = 0; maxx = width;
   for (y = miny; y <= maxy; y++)
   {
      for (x = minx; x <= maxx; x++)
      {
         i = fg_getpixel(x,y);
         count[i]++;
      }
   }

   /* count how many colors are currently in use */
   in_use = 0;
   for (i = 0; i < 256; i++)
   {
      if (count[i] > 0) in_use++;
      orig_count[i] = count[i];
   }

   /* loop until you reach the target value */
   while (in_use > target)
   {
      /* elimination method */
      if (quick)
      {
         /* find the color with the fewest pixels */
         least = 0xFFFF;
         for (i = 0; i < 256; i++)
         {
            if (count[i] > 0 && count[i] <= least)
            {
               least = count[i];
               index = i;
            }
         }

         /* calculate the RGB components of this color */
         j = (int)index * 3;
         red   = RGBvalues[j++];
         green = RGBvalues[j++];
         blue  = RGBvalues[j];

         /* find the closest match */
         j = 0;
         distance = 0x7FFFFFFF;
         for (i = 0; i < 256; i++)
         {
            if (i != (int)index && count[i] > 0)
            {
               /* least squares calculation */
               d = (red  - RGBvalues[j])  * (red  - RGBvalues[j]) 
                 + (green- RGBvalues[j+1])* (green- RGBvalues[j+1])
                 + (blue - RGBvalues[j+2])* (blue - RGBvalues[j+2]);
               if (d < distance)
               {
                  match = i;
                  distance = d;
               }
            }
            j += 3;
         }
      }
      /* multi-criteria elmination method */
      else
      {
         distance = 0x7FFFFFFF;
         for (i = 0; i < 255; i++)
         {
            if (count[i] > 0)
            {
               red   = RGBvalues[i*3];
               green = RGBvalues[i*3+1];
               blue  = RGBvalues[i*3+2];

               for (j = i+1; j < 256; j++)
               {
                  if (count[j] > 0)
                  {
                     /* least squares calculation */
                     d = (red - RGBvalues[j*3])*(red - RGBvalues[j*3]) 
                       + (green - RGBvalues[j*3+1])*(green - RGBvalues[j*3+1])
                       + (blue  - RGBvalues[j*3+2])*(blue  - RGBvalues[j*3+2]);

                     /* find 2 colors with smallest distance & lowest pixel count */
                     if (d < distance ||
                        (d == distance && count[j] < min(count[index],count[match])))
                     {
                        index = i;
                        match = j;
                        distance = d;
                     }
                  }
               }
            }
         }

         /* swap index and match, so index will have lowest count */
         if (count[index] > count[match])
         {
            temp  = index;
            index = match;
            match = temp;
         }
      }

      /* assign match color to current index */
      colormap[index] = (unsigned char) match;
      count[match] += count[index];
      count[index] = 0;

      /* remap all colors assigned to index to avoid leapfrog effect */
      for (i = 0; i < 256; i++)
      {
         if (colormap[i] == (unsigned char)index)
         {
            red   = RGBvalues[i*3];
            green = RGBvalues[i*3+1];
            blue  = RGBvalues[i*3+2];

            distance = 0x7FFFFFFF;
            for (j = 0; j < 256; j++)
            {
               if (count[j] > 0 && j!=i)
               {
                  d = (red   - RGBvalues[j*3])  * (red   - RGBvalues[j*3]) 
                    + (green - RGBvalues[j*3+1])* (green - RGBvalues[j*3+1])
                    + (blue  - RGBvalues[j*3+2])* (blue  - RGBvalues[j*3+2]);

                  if (d < distance)
                  {
                     nearest = j;
                     distance = d;
                  }
               }
            }
            colormap[i] = (unsigned char)nearest;

            /* adjust the pixel count */
            count[nearest]+= orig_count[i];
            count[match]-=orig_count[i];
         }
      } 
      in_use--;
   }

   /* make all the used colors contiguous */
   j = 0;
   for (i = 0; i < 256; i++)
   {
      if (count[i] > 0)
      {
         colormap[i] = (unsigned char)j;
         RGBvalues[j*3]   = RGBvalues[i*3];
         RGBvalues[j*3+1] = RGBvalues[i*3+1];
         RGBvalues[j*3+2] = RGBvalues[i*3+2];
         j++;
      }
   }

   /* put all the colors in the range [offset..target+offset] */
   if (offset > 0)
   {
      j = target - 1;
      for (i = offset+j; i >= offset; i--)
      {
         RGBvalues[i*3]   = RGBvalues[j*3];
         RGBvalues[i*3+1] = RGBvalues[j*3+1];
         RGBvalues[i*3+2] = RGBvalues[j*3+2];
         j--;       
      }
   }

   /* adjust the colors that were remapped twice */
   for (i = 0; i < 256; i++)
   {
      if (count[i] == 0)
      {
         colormap[i] = colormap[colormap[i]];
      }
   }

   /* set the DACS to the new color set */
   fg_setdacs(offset,target,&RGBvalues[offset*3]);

   /* change all the pixels to the new colors */
   for (y = miny; y <= maxy; y++)
   {
      for (x = minx; x <= maxx; x++)
      {
         i = fg_getpixel(x,y);
         fg_setcolor(colormap[i]+offset);
         fg_point(x,y);
      }
   }
}

