unit LaserFrm;

interface

uses
  Messages, WinTypes, WinProcs, Classes, Graphics, Controls, Forms,
  DLPoints, SvrUtils;

type
  TFrmLaser = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormClose( Sender: TObject;
                         var Action: TCloseAction);
    procedure FormKeyDown( Sender: TObject; var Key: Word;
                           Shift: TShiftState);
    procedure FormMouseDown( Sender: TObject; Button: TMouseButton;
                             Shift: TShiftState; X, Y: Integer);
    procedure FormMouseMove( Sender: TObject; Shift: TShiftState;
                             X, Y: Integer);
  private
    FMousePt : TPoint;
    FDone : Boolean;
    FOriginPoint : Longint;
    FOrigins : array[ TLaserOrigin ] of Longint;
    FCurrPoint : Integer;
    FFirstPass : Boolean;
    FGettingPassword : Boolean;
    FOrigX : Integer;
    FOrigY : Integer;
    procedure IdleProc( Sender : TObject; var Done : Boolean );
    procedure Animate;
    procedure CursorOff;
    procedure CursorOn;
    procedure PopulateOrigins;
    procedure GetPassword;
    procedure SetTopMost( TopMost : Boolean );
  end;

var
  FrmLaser: TFrmLaser;

implementation

{$R *.DFM}

uses
  GetPWFrm, MMSystem;

const
  Offset = 25;     { Distance from Screen Edge of Laser Source }


{=======================}
{== TFrmLaser Methods ==}
{=======================}

procedure TFrmLaser.FormCreate( Sender : TObject );
begin
  Left := 0;                       { Resize Form to fit screen }
  Top := 0;
  Height := Screen.Height;
  Width := Screen.Width;

  Randomize;              { Initialize Random Number Generator }
  FCurrPoint := 1;
  FFirstPass := True;
  FDone := False;

  FMousePt.X := -1;
  FMousePt.Y := -1;

  LoadSettings;          { Load Settings from Control.ini file }

  Canvas.Pen.Color := LaserColor;
  Canvas.Pen.Mode := pmXor;

  PopulateOrigins; { Set Source Locations based on Screen Size }
  FOriginPoint := FOrigins[ LaserOrigin ];

  SetTopMost( True );
  CursorOff;
  Application.OnIdle := IdleProc;
end; {= TFrmLaser.FormCreate =}


procedure TFrmLaser.FormClose( Sender : TObject;
                               var Action : TCloseAction );
begin
  Application.OnIdle := nil;
  CursorOn;                   { Don't forget to turn on cursor }
end;


procedure TFrmLaser.FormKeyDown( Sender : TObject; var Key : Word;
                                 Shift : TShiftState );
begin
  GetPassword;
end;


procedure TFrmLaser.FormMouseDown( Sender : TObject;
                                   Button : TMouseButton;
                                   Shift : TShiftState;
                                   X, Y : Integer );
begin
  GetPassword;
end;


procedure TFrmLaser.FormMouseMove( Sender : TObject;
                                   Shift : TShiftState;
                                   X, Y : Integer );
begin
  if FDone then
    Exit;

  if ( FMousePt.X = -1 ) and ( FMousePt.Y = -1 ) then
  begin
    FMousePt.X := X;                      { Record first point }
    FMousePt.Y := Y;
    Exit;
  end;

  if ( Abs( X - FMousePt.X ) > 3 ) or
     ( Abs( Y - FMousePt.Y ) > 3 ) then
  begin
    GetPassword;      { If mouse moved > 3 pixels, GetPassword }
    FMousePt.X := -1;                   { Reset mouse position }
    FMousePt.Y := -1;
  end
  else
  begin
    FMousePt.X := X;               { Record new mouse position }
    FMousePt.Y := Y;
  end;
end;


procedure TFrmLaser.IdleProc(Sender : TObject; var Done : Boolean);
begin
  Animate;
  Done := False;                 { Prevent call to WaitMessage }
end;


{= LaserLine is a Windows callback procedure that is called   =}
{= by the LineDDA procedure in the Animate method below.      =}

procedure LaserLine( X, Y : Integer; Origin : Longint ); export;
begin
  Application.ProcessMessages;      { Process pending messages }

  { If screen saver shut down, then don't draw any more lines  }
  if FrmLaser.FDone then
    Exit;

  with FrmLaser.Canvas do
  begin
    MoveTo( LoWord( Origin ), HiWord( Origin ) );
    LineTo( X, Y );
    WinDelay( LaserSpeed );
    MoveTo( LoWord( Origin ), HiWord( Origin ) );
    LineTo( X, Y );
    Pixels[ X, Y ] := LaserColor;
  end;
end;


procedure TFrmLaser.Animate;
begin
  if ( FCurrPoint = 1 ) and FFirstPass then
  begin
    FOrigX := Random( Screen.Width - 300 );
    FOrigY := Random( Screen.Height - 120 );
    if RandomColors then
    begin
      LaserColor := Colors[ Random( 15 ) + 1 ];
      Canvas.Pen.Color := LaserColor;
    end;
    if RandomOrigin then
    begin
      LaserOrigin := TLaserOrigin( Random( 9 ) + 1 );
      FOriginPoint := FOrigins[ LaserOrigin ];
    end;
  end;

  { LaserPoints of -1 indicate breaks in connecting the dots }
  if LaserPoints[ FCurrPoint + 1 ].X <> -1 then
  begin
    if LaserPoints[ FCurrPoint ].X = -1 then
       Canvas.MoveTo( LaserPoints[ FCurrPoint + 1 ].X + FOrigX,
                      LaserPoints[ FCurrPoint + 1 ].Y + FOrigY )
    else
    begin
      { LaserLine called for each point between }
      { ( X1, Y1 ) and ( X2, Y2 ) }
      LineDda( LaserPoints[ FCurrPoint ].X + FOrigX,
               LaserPoints[ FCurrPoint ].Y + FOrigY,
               LaserPoints[ FCurrPoint + 1 ].X + FOrigX,
               LaserPoints[ FCurrPoint + 1 ].Y + FOrigY,
               @LaserLine, Pointer( FOriginPoint ) );
    end;
  end;

  Inc( FCurrPoint );          { Get Ready for Next Laser Point }

  if FCurrPoint = MaxLaserPoints then
  begin
    WinDelay( PauseDelay * 100 );       { Pause before Erasing }
    FCurrPoint := 1;                     { Setup for next pass }
    FFirstPass := not FFirstPass;
  end;
end; {= TFrmLaser.Animate =}


procedure TFrmLaser.CursorOff;
var
  Count : Integer;
begin
  repeat
    Count := ShowCursor( False );
  until Count < 0;
end;


procedure TFrmLaser.CursorOn;
var
  Count : Integer;
begin
  repeat
    Count := ShowCursor( True );
  until Count >= 0;
end;


procedure TFrmLaser.PopulateOrigins;
begin
  FOrigins[ loUpperLeft ] := MakeLong( Offset, Offset );
  FOrigins[ loUpperCenter ] := MakeLong( Screen.Width div 2,
                                         Offset );
  FOrigins[ loUpperRight ] := MakeLong( Screen.Width - Offset,
                                        Offset );
  FOrigins[ loMidLeft ] := MakeLong( Offset,
                                     Screen.Height div 2 );
  FOrigins[ loMidCenter ] := MakeLong( Screen.Width div 2,
                                       Screen.Height div 2 );
  FOrigins[ loMidRight ] := MakeLong( Screen.Width - Offset,
                                      Screen.Height div 2 );
  FOrigins[ loLowerLeft ] := MakeLong( Offset,
                                       Screen.Height - Offset );
  FOrigins[ loLowerCenter ] := MakeLong( Screen.Width div 2,
                                         Screen.Height-Offset );
  FOrigins[ loLowerRight ] := MakeLong( Screen.Width - Offset,
                                        Screen.Height - Offset);
end;


procedure TFrmLaser.GetPassword;
var
  CanExit : Boolean;
begin
  CanExit := False;
  Application.OnIdle := nil;                  { Stop Animating }

  if not PWProtected or ( Password = '' ) then
    CanExit := True
  else
  begin
    SetTopMost( False );
    CursorOn;
    FrmGetPassword := TFrmGetPassword.Create( Application );
    try
      FrmGetPassword.ShowModal;          { Display Password Form }
      if FrmGetPassword.ModalResult = mrOK then
        CanExit := True;
    finally
      FrmGetPassword.Free;
    end;
  end;

  if CanExit then
  begin
    FDone := True;
    Close;
  end
  else
  begin
    CursorOff;
    SetFocus;
    SetTopMost( True );
    Application.OnIdle := IdleProc;         { Resume Animating }
  end;
end; {= TFrmLaser.GetPassword =}


procedure TFrmLaser.SetTopMost( TopMost : Boolean );
begin
  { Use SetWindowPos rather than the FormStyle property to }
  { avoid erasing current laser image. }
  if TopMost then
    SetWindowPos( Handle, hwnd_TopMost, 0, 0, 0, 0,
                  swp_NoSize + swp_NoMove )
  else
    SetWindowPos( Handle, hwnd_NoTopMost, 0, 0, 0, 0,
                  swp_NoSize + swp_NoMove );
end;

end.
