{==============================================================================}
{= RzTrkBar Unit                                                              =}
{=                                                                            =}
{= The TRzTrackBar component is a slider control that mimics the behavior of  =}
{= the Windows 95 TrackBar control. This control works with mouse -and-       =}
{= keyboard input.                                                            =}
{=                                                                            =}
{= Building Custom Delphi Components - Ray Konopka                            =}
{= Copyright  1995 by Raize Software Solutions, Inc.                         =}
{==============================================================================}

{$I RAIZE.INC}

unit RzTrkBar;

interface

uses
  Messages, WinTypes, WinProcs, Classes, Graphics, Controls, Menus,
  ExtCtrls, RzCommon;

type
  TTrackOrientation = ( toHorizontal, toVertical );
  TThumbSize = ( tsSmall, tsMedium, tsLarge );
  TThumbStyle = ( tsBox, tsPointer );
  TTickStyle = ( tkStandard, tkOwnerDraw );

  TRzTrackBar = class;                               { Forward class reference }

  TDrawTickEvent = procedure ( TrackBar : TRzTrackBar; Canvas : TCanvas;
                               Location : TPoint; Index : Integer ) of object;

  TRzTrackBar = class( TCustomControl )
  private
    FAboutInfo : TRzAboutInfo;
    FBorderWidth : Integer;
    FMax : Integer;
    FMin : Integer;
    FOrientation : TTrackOrientation;
    FPageSize : Word;
    FPosition : Integer;
    FTickStyle : TTickStyle;
    FShowTicks : Boolean;
    FSliding : Boolean;

    FThumbHeight : Integer;
    FThumbRct : TRect;
    FThumbSize : TThumbSize;
    FThumbStyle : TThumbStyle;
    FThumbWidth : Integer;
    FHalfWidth : Integer;

    FTrackColor : TColor;                               { Attributes for track }
    FTrackRct : TRect;
    FTrackWidth : Word;

    FDitherBmp : TBitmap;
    FThumbBmp : TBitmap;
    FMaskBmp : TBitmap;
    FBackgroundBmp : TBitmap;

    FOnChange : TNotifyEvent;                                  { Custom events }
    FOnDrawTick : TDrawTickEvent;

    procedure SetMax( Value : Integer );
    procedure SetMin( Value : Integer );
    procedure SetOrientation( Value : TTrackOrientation );
    procedure SetPosition( Value : Integer );
    procedure SetShowTicks( Value : Boolean );
    procedure SetThumbSize( Value : TThumbSize );
    procedure SetThumbStyle( Value : TThumbStyle );
    procedure SetTickStyle( Value : TTickStyle );
    procedure SetTrackWidth( Value : Word );
    procedure SetTrackColor( Value : TColor );
    procedure LoadThumbBitmaps;
    procedure UpdateDitherBitmap;
    procedure WMGetDlgCode( var Msg : TWMGetDlgCode ); message wm_GetDlgCode;
    procedure WMSize( var Msg : TWMSize); message wm_Size;
    procedure CMEnabledChanged( var Msg : TMessage ); message cm_EnabledChanged;
  protected
    procedure DrawTrack; virtual;
    procedure DrawTicks; virtual;
    procedure DrawThumb; virtual;
    procedure Paint; override;

    procedure Change; dynamic;
    procedure DrawTick( Canvas : TCanvas; Location : TPoint;
                        Index : Integer ); dynamic;
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure KeyDown( var Key : Word; Shift : TShiftState ); override;
    procedure MouseDown( Button : TMouseButton; Shift : TShiftState;
                         X, Y : Integer ); override;
    procedure MouseMove( Shift : TShiftState; X, Y : Integer ); override;
    procedure MouseUp( Button : TMouseButton; Shift : TShiftState;
                       X, Y : Integer ); override;
  public
    constructor Create( AOwner : TComponent ); override;
    destructor Destroy; override;
  published
    property About : TRzAboutInfo               { Read-Only Published Property }
      read FAboutInfo;

    property Max : Integer
      read FMax
      write SetMax
      default 10;

    property Min : Integer
      read FMin
      write SetMin
      default 0;

    property Orientation : TTrackOrientation
      read FOrientation
      write SetOrientation
      default toHorizontal;

    property PageSize : Word
      read FPageSize
      write FPageSize
      default 1;

    property Position : Integer
      read FPosition
      write SetPosition;

    property ShowTicks : Boolean
      read FShowTicks
      write SetShowTicks
      default True;

    property ThumbSize : TThumbSize
      read FThumbSize
      write SetThumbSize
      default tsMedium;

    property ThumbStyle : TThumbStyle
      read FThumbStyle
      write SetThumbStyle
      default tsPointer;

    property TickStyle : TTickStyle
      read FTickStyle
      write SetTickStyle
      default tkStandard;

    property TrackColor : TColor
      read FTrackColor
      write SetTrackColor
      default clWhite;

    property TrackWidth : Word
      read FTrackWidth
      write SetTrackWidth
      default 8;

    property OnChange : TNotifyEvent
      read FOnChange
      write FOnChange;

    property OnDrawTick : TDrawTickEvent
      read FOnDrawTick
      write FOnDrawTick;

    { Inherited Properties & Events }
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property HelpContext;
    property Hint;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop default True;
    property Visible;

    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;


procedure Register;

implementation

{$IFDEF WIN32}                                    { Link in Bitmaps for Thumbs }
{$R RZTBAR32.RES}                                        { Win32 Resource File }
{$ELSE}
{$R RZTRKBAR.RES}                                        { Win16 Resource File }
{$ENDIF}

uses
  SysUtils;

{=========================}
{== TRzTrackBar Methods ==}
{=========================}

constructor TRzTrackBar.Create( AOwner : TComponent );
begin
  inherited Create( AOwner );
  Width := 200;
  Height := 50;
  FTrackWidth := 8;
  FOrientation := toHorizontal;
  FTrackColor := clWhite;
  FMin := 0;
  FMax := 10;
  FPosition := 0;
  FBorderWidth := 4;
  FPageSize := 1;
  TabStop := True;
  FShowTicks := True;
  FSliding := False;

  FThumbBmp := TBitmap.Create;                { Create internal bitmap objects }
  FMaskBmp := TBitmap.Create;
  FBackgroundBmp := TBitmap.Create;
  FThumbStyle := tsPointer;
  FThumbSize := tsMedium;
  FDitherBmp := TBitmap.Create;
  FDitherBmp.Width := 8;
  FDitherBmp.Height := 8;
  UpdateDitherBitmap;
  LoadThumbBitmaps;

  FAboutInfo := TRzAboutInfo.Create;                { Create About Info Object }
  FAboutInfo.CopyrightDate := '1995';
  FAboutInfo.Company := 'Raize Software Solutions, Inc.';
  FAboutInfo.Description := 'The TRzTrackBar component is a slider control ' +
                            'that mimics the behavior of the Windows 95 ' +
                            'TrackBar control.';
end;


destructor TRzTrackBar.Destroy;
begin
  FDitherBmp.Free;            { Be sure to free internally allocated objects }
  FThumbBmp.Free;
  FMaskBmp.Free;
  FBackgroundBmp.Free;

  FAboutInfo.Free;                                 { Destroy About Info Object }
  inherited Destroy;
end;


procedure TRzTrackBar.SetMax( Value : Integer );
begin
  if Value <> FMax then
  begin
    FMax := Value;
    if FPosition > FMax then     { If new max is less than current Position... }
      Position := FMax;                         { Update the Position property }
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetMin( Value : Integer );
begin
  if Value <> FMin then
  begin
    FMin := Value;
    if FPosition < FMin then  { If new min is greater than current Position... }
      Position := FMin;                         { Update the Position property }
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetOrientation( Value : TTrackOrientation );
begin
  if Value <> FOrientation then
  begin
    FOrientation := Value;
    LoadThumbBitmaps;                 { Get new bitmaps if Orientation changes }
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetPosition( Value : Integer );
begin
  if Value <> FPosition then
  begin
    if Value < FMin then                                      { Range Checking }
      Value := FMin
    else if Value > FMax then
      Value := FMax;

    FPosition := Value;

    { No need to be fancy in Design-mode.  Simply invalidate the control.      }
    { Besides, it is not appropriate to call the Change event at design-time.  }

    if csDesigning in ComponentState then
      Invalidate
    else
    begin
                          { Erase old thumb image by drawing background bitmap }
      Canvas.Draw( FThumbRct.Left, FThumbRct.Top, FBackgroundBmp );

      DrawThumb;                                  { Draw thumb at new location }
      Change;                                           { Trigger Change event }
    end;
  end;
end; {= TRzTrackBar.SetPosition =}


procedure TRzTrackBar.SetShowTicks( Value : Boolean );
begin
  if Value <> FShowTicks then
  begin
    FShowTicks := Value;
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetThumbSize( Value : TThumbSize );
begin
  if Value <> FThumbSize then
  begin
    FThumbSize := Value;
    LoadThumbBitmaps;                   { Reload bitmaps if thumb size changes }
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetThumbStyle( Value : TThumbStyle );
begin
  if Value <> FThumbStyle then
  begin
    FThumbStyle := Value;
    LoadThumbBitmaps;                  { Reload bitmaps if thumb style changes }
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetTickStyle( Value : TTickStyle );
begin
  if Value <> FTickStyle then
  begin
    FTickStyle := Value;
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetTrackColor( Value : TColor );
begin
  if Value <> FTrackColor then
  begin
    FTrackColor := Value;
    UpdateDitherBitmap;
    Invalidate;
  end;
end;


procedure TRzTrackBar.SetTrackWidth( Value : Word );
begin
  if FTrackWidth <> Value then
  begin
    FTrackWidth := Value;
    if FTrackWidth < 6 then
      FTrackWidth := 6;
    Invalidate;
  end;
end;


{ Array Constants hold all bitmap resource names for easy access }

const
  ThumbBitmapNames : array[ TTrackOrientation, TThumbSize ] of PChar =
    ( ( 'SmHorzThumb', 'MedHorzThumb', 'LgHorzThumb' ),
      ( 'SmVertThumb', 'MedVertThumb', 'LgVertThumb' ) );
  MaskBitmapNames : array[ TTrackOrientation, TThumbSize ] of PChar =
    ( ( 'SmHorzThumbMask', 'MedHorzThumbMask', 'LgHorzThumbMask' ),
      ( 'SmVertThumbMask', 'MedVertThumbMask', 'LgVertThumbMask' ) );
  BoxBitmapNames : array[ TTrackOrientation, TThumbSize ] of PChar =
    ( ( 'SmHorzBox', 'MedHorzBox', 'LgHorzBox' ),
      ( 'SmVertBox', 'MedVertBox', 'LgVertBox' ) );


procedure TRzTrackBar.LoadThumbBitmaps;
begin
  if FThumbStyle = tsPointer then
  begin
    FThumbBmp.Handle := LoadBitmap( HInstance,
                                  ThumbBitmapNames[ FOrientation, FThumbSize ]);
    FMaskBmp.Handle := LoadBitmap( HInstance,
                                 MaskBitmapNames[ FOrientation, FThumbSize ] );
  end
  else
  begin
    FThumbBmp.Handle := LoadBitmap( HInstance,
                                    BoxBitmapNames[ FOrientation, FThumbSize ]);
  end;

  if FOrientation = toVertical then
  begin
    FThumbHeight := FThumbBmp.Width;
    FThumbWidth := FThumbBmp.Height;
  end
  else
  begin
    FThumbHeight := FThumbBmp.Height;
    FThumbWidth := FThumbBmp.Width;
  end;
  FHalfWidth := FThumbWidth div 2;
end; {= TRzTrackBar.LoadThumbBitmaps =}


procedure TRzTrackBar.UpdateDitherBitmap;
var
  C : TColor;
  I, J : Integer;
begin
  C := clSilver;
  if ColorToRGB( FTrackColor) = clSilver then
    C := clGray;

  with FDitherBmp.Canvas do
  begin
    Brush.Color := FTrackColor;
    FillRect( Rect( 0, 0, FDitherBmp.Width, FDitherBmp.Height ) );
    for I := 0 to 7 do
      for J := 0 to 7 do
        if ( I + J ) mod 2 <> 0 then
          Pixels[ I, J ] := C;
  end;
end; {= TRzTrackBar.UpdateDitherBitmap =}


procedure TRzTrackBar.DrawTrack;
begin
  { Calculate the Size of the Track }
  if FOrientation = toVertical then
  begin
    FTrackRct.Top := FHalfWidth + FBorderWidth;
    FTrackRct.Bottom := Height - FBorderWidth - FHalfWidth;
    FTrackRct.Left := ( Width - FTrackWidth ) div 2;
    FTrackRct.Right := FTrackRct.Left + FTrackWidth;
  end
  else
  begin
    FTrackRct.Top := ( Height - FTrackWidth ) div 2;
    FTrackRct.Bottom := FTrackRct.Top + FTrackWidth;
    FTrackRct.Left := FHalfWidth + FBorderWidth;
    FTrackRct.Right := Width - FBorderWidth - FHalfWidth;
  end;

  { Draw the Track }
  Canvas.Brush.Color := FTrackColor;

  if not Enabled then
    Canvas.Brush.Bitmap := FDitherBmp;

  Canvas.FillRect( FTrackRct );
  DrawCtl3DBorder( Canvas, FTrackRct );                   { From RzCommon unit }
end; {= TRzTrackBar.DrawTrack =}


procedure TRzTrackBar.DrawTicks;
var
  Delta : Real;
  I, X, Y : Integer;
begin
  Canvas.Pen.Color := clBlack;
  with FTrackRct do
  begin
    if FOrientation = toVertical then
    begin
                                         { Delta is spacing between tick marks }
      Delta := ( Height - FThumbWidth - 2 * FBorderWidth ) / ( FMax - FMin );

      for I := FMin to FMax do
      begin
        Y := Trunc( Delta * ( I - FMin ) ) + FBorderWidth;

        if FTickStyle = tkStandard then
        begin
          Canvas.MoveTo( FBorderWidth, Y + FHalfWidth );
          Canvas.LineTo( 10, Y + FHalfWidth );
          if FThumbStyle = tsBox then               { Draw Ticks on Other Side }
          begin
            Canvas.MoveTo( Width - 10, Y + FHalfWidth );
            Canvas.LineTo( Width - FBorderWidth, Y + FHalfWidth );
          end;
        end
        else                                { Provide hook to owner draw ticks }
          DrawTick( Canvas, Point( 0, Y + FHalfWidth ), I );
      end;
    end
    else
    begin
      Delta := ( Width - FThumbWidth - 2 * FBorderWidth ) / ( FMax - FMin );

      for I := FMin to FMax do
      begin
        X := Trunc( Delta * ( I - FMin ) ) + FBorderWidth;

        if FTickStyle = tkStandard then
        begin
          Canvas.MoveTo( X + FHalfWidth, FBorderWidth );
          Canvas.LineTo( X + FHalfWidth, 10 );
          if FThumbStyle = tsBox then               { Draw Ticks on Other Side }
          begin
            Canvas.MoveTo( X + FHalfWidth, Height - 10 );
            Canvas.LineTo( X + FHalfWidth, Height - FBorderWidth );
          end;
        end
        else                                { Provide hook to owner draw ticks }
          DrawTick( Canvas, Point( X + FHalfWidth, 0 ), I );
      end;
    end;
  end;
end; {= TRzTrackBar.DrawTicks =}


procedure TRzTrackBar.DrawThumb;
var
  Offset : Longint;
  WorkBmp : TBitmap;
  WorkRct : TRect;
begin
  { Calculate new location of thumb based on Position }
  if FOrientation = toVertical then
  begin
    Offset := ( Longint( Height ) - FThumbWidth - 2 * FBorderWidth ) *
              ( FPosition - FMin ) div ( FMax - FMin );
    FThumbRct.Left := ( Width - FThumbHeight ) div 2;
    FThumbRct.Right := FThumbRct.Left + FThumbHeight;
    FThumbRct.Bottom := Height - Offset - FBorderWidth;
    FThumbRct.Top := FThumbRct.Bottom - FThumbWidth;
  end
  else
  begin
    Offset := ( Longint( Width ) - FThumbWidth - 2 * FBorderWidth ) *
              ( FPosition - FMin ) div ( FMax - FMin );
    FThumbRct.Left := Offset + FBorderWidth;
    FThumbRct.Right := FThumbRct.Left + FThumbWidth;
    FThumbRct.Top := ( Height - FThumbHeight ) div 2;
    FThumbRct.Bottom := FThumbRct.Top + FThumbHeight;
  end;

  { Save background image of new thumb location }
  FBackgroundBmp.Width := FThumbBmp.Width;
  FBackgroundBmp.Height := FThumbBmp.Height;
  FBackgroundBmp.Canvas.CopyRect( Rect(0, 0, FThumbBmp.Width, FThumbBmp.Height),
                                  Canvas, FThumbRct );

  { Draw the thumb by displaying the thumb bitmap }

  { WorkBmp is used to combine the Thumb bitmap and the background so that the }
  { background of the track appears in the corners of the Thumb image. }

  { If ThumbStyle is tsBox, there is no need to mask out the background, so }
  { just copy the thumb image to the control canvas. }

  WorkBmp := TBitmap.Create;
  try
    { Don't forget to set working bitmap size to that of thumb bitmap }
    WorkBmp.Height := FThumbBmp.Height;
    WorkBmp.Width := FThumbBmp.Width;

    { WorkRct specifies the Width and Height of the region we are dealing with }
    WorkRct := Rect( 0, 0, FThumbBmp.Width, FThumbBmp.Height );

    if FThumbStyle = tsPointer then
    begin
      { Copy the FBackgroundBmp image to WorkBmp }

      WorkBmp.Canvas.CopyMode := cmSrcCopy;
      WorkBmp.Canvas.CopyRect( WorkRct, FBackgroundBmp.Canvas, WorkRct );

      { Combine the FBackgroundBmp and the FMaskBmp images using the cmSrcAnd }
      { CopyMode. White pixels in mask have no effect. Background shows thru. }

      WorkBmp.Canvas.CopyMode := cmSrcAnd;
      WorkBmp.Canvas.CopyRect( WorkRct, FMaskBmp.Canvas, WorkRct );

      { Copy the Thumb bitmap onto the Working bitmap using the cmSrcPaint }
      { mode. Black pixels in Thumb bitmap let background show through. }

      WorkBmp.Canvas.CopyMode := cmSrcPaint
    end
    else
      WorkBmp.Canvas.CopyMode := cmSrcCopy;

    WorkBmp.Canvas.CopyRect( WorkRct, FThumbBmp.Canvas, WorkRct );

    if not Enabled then
    begin
      { If control is disabled, dither the thumb as well as the track }
      WorkBmp.Canvas.Brush.Bitmap := FDitherBmp;
      WorkBmp.Canvas.FloodFill( WorkRct.Right - 3, WorkRct.Bottom - 3,
                                clSilver, fsSurface );
    end;

    { Copy the working bitmap onto the control's Canvas at thumb position }
    Canvas.CopyRect( FThumbRct, WorkBmp.Canvas, WorkRct );
  finally
    WorkBmp.Free;
  end;
end; {= TRzTrackBar.DrawThumb =}


procedure TRzTrackBar.Paint;
begin
  with Canvas do
  begin
    if Focused then      { Indicate focus by drawing dotted box around control }
      DrawFocusRect( ClientRect );
    DrawTrack;
    if FShowTicks then
      DrawTicks;
    DrawThumb;
  end;
end;


procedure TRzTrackBar.Change;
begin
  if Assigned( FOnChange ) then
    FOnChange( Self );
end;


{= TRzTrackBar.DrawTick
{=   This method is the event dispatch method for the OnDrawTick event.       =}
{=   The parameters are:                                                      =}
{=     Canvas - The Canvas for the TrackBar Control                           =}
{=     Location - Point record indicating X or Y coordinates of tick mark     =}
{=     Index - Position index of tick mark to be drawn                        =}

procedure TRzTrackBar.DrawTick( Canvas : TCanvas; Location : TPoint;
                              Index : Integer );
begin
  if Assigned( FOnDrawTick ) then            { Allow user to draw custom ticks }
    FOnDrawTick( Self, Canvas, Location, Index );
end;


procedure TRzTrackBar.DoEnter;
begin
  inherited DoEnter;
  Refresh;      { When control gets focus, update display to show focus border }
end;


procedure TRzTrackBar.DoExit;
begin
  inherited DoExit;
  Refresh;   { When control loses focus, update display to remove focus border }
end;


procedure TRzTrackBar.KeyDown( var Key : Word; Shift : TShiftState );
begin
  inherited KeyDown( Key, Shift );

  case Key of
    vk_Prior:                                  { PgUp Key - increases Position }
      Position := FPosition + FPageSize;

    vk_Next:                                   { PgDn Key - decreases Position }
      Position := FPosition - FPageSize;

    vk_End:
      if FOrientation = toVertical then       { End is at Right for horizontal }
        Position := FMin              { TrackBar, Bottom for vertical TrackBar }
      else
        Position := FMax;

    vk_Home:
      if FOrientation = toVertical then       { Home is at Left for horizontal }
        Position := FMax                 { TrackBar, Top for vertical TrackBar }
      else
        Position := FMin;

    vk_Left:                                               { Decrease Position }
      if FPosition > FMin then
        Position := FPosition - 1;

    vk_Up:                                                 { Increase Position }
      if FPosition < FMax then
        Position := FPosition + 1;

    vk_Right:                                              { Increase Position }
      if FPosition < FMax then
        Position := FPosition + 1;

    vk_Down:                                               { Decrease Position }
      if FPosition > FMin then
        Position := FPosition - 1;
  end; { case }
end; {= TRzTrackBar.KeyDown =}


procedure TRzTrackBar.MouseDown( Button : TMouseButton; Shift : TShiftState;
                               X, Y : Integer );
var
  PtX, PtY : Integer;
  Delta : Real;
begin
  inherited MouseDown( Button, Shift, X, Y );
  SetFocus;                                           { Move focus to TrackBar }

  if ( Button = mbLeft ) and PtInRect( FThumbRct, Point( X, Y ) ) then
  begin
    { User pressed the left mouse button while on the thumb }
    FSliding := True;
  end
  else if ( Button = mbLeft ) and PtInRect( FTrackRct, Point( X, Y ) ) then
  begin
    { User pressed left mouse button inside the track on either side of thumb. }
    { Determine which side of thumb user clicked, and then update position     }

    if FOrientation = toVertical then
    begin
      Delta := ( Height - FThumbWidth - 2 * FBorderWidth ) / ( FMax - FMin );
      PtY := Trunc( Delta * ( ( FMax - FPosition ) - FMin ) ) + FBorderWidth;
      if Y < PtY then
        Position := FPosition + FPageSize
      else
        Position := FPosition - FPageSize;
    end
    else
    begin
      Delta := ( Width - FThumbWidth - 2 * FBorderWidth ) / ( FMax - FMin );
      PtX := Trunc( Delta * ( FPosition - FMin ) ) + FBorderWidth;
      if X < PtX then
        Position := FPosition - FPageSize
      else
        Position := FPosition + FPageSize;
    end;
  end;
end; {= TRzTrackBar.MouseDown =}


procedure TRzTrackBar.MouseMove( Shift : TShiftState; X, Y : Integer );
var
  P, W, H : Integer;
begin
  inherited MouseMove( Shift, X, Y );

  { If mouse is over thumb, then change cursor to either crSizeNS or crSizeWE  }
  { depending on whether the orientation is vertical or horizontal.            }

  if PtInRect( FThumbRct, Point( X, Y ) ) then
  begin
    if FOrientation = toVertical then
      Cursor := crSizeNS
    else
      Cursor := crSizeWE;
  end
  else
    Cursor := crDefault;

  { If in Sliding state, then move the thumb to the closest tick mark. }
  if FSliding then
  begin
    if FOrientation = toVertical then
    begin
      H := Height - FHalfWidth;
      P := Round( ( ( H - Y ) / H ) * ( FMax - FMin ) + FMin );
    end
    else
    begin
      W := Width - FHalfWidth;
      P := Round( ( ( X - FHalfWidth ) / W ) * ( FMax - FMin ) + FMin  );
    end;

    if P > FMax then
      P := FMax;
    if P < FMin then
      P := FMin;
    Position := P;
  end;
end; {= TRzTrackBar.MouseMove =}


procedure TRzTrackBar.MouseUp( Button : TMouseButton; Shift : TShiftState;
                             X, Y : Integer );
begin
  inherited MouseUp( Button, Shift, X, Y );

  if ( Button = mbLeft ) then
    FSliding := False;
end;


procedure TRzTrackBar.WMGetDlgCode( var Msg : TWMGetDlgCode );
begin
  inherited;
  Msg.Result := dlgc_WantArrows;          { So TrackBar can process arrow keys }
end;


procedure TRzTrackBar.WMSize( var Msg : TWMSize );
begin
  inherited;
  if Height > Width then
    Orientation := toVertical
  else
    Orientation := toHorizontal;
end;


procedure TRzTrackBar.CMEnabledChanged( var Msg : TMessage );
begin
  inherited;
  Invalidate;
end;


{========================}
{== Register Procedure ==}
{========================}

procedure Register;
begin
  RegisterComponents( RaizePage, [ TRzTrackBar ] );
end;

end.
