/**********************************************************************
 * LITTLECD.EXE Version 1.1                                           *
 * by Mike Ferris                                                     *
 * August 1995                                                        *
 *                                                                    *
 * A CD player for MicroSoft Windows 3.1. Uses MCI to control the     *
 * Windows CD audio device.                                           *
 *                                                                    *
 * Compiled with Borland C++ 4.5                                      *
 **********************************************************************/

#include <owl\owlcore.h>
#include <owl\framewin.h>
#include <owl\static.h>
#include <mmsystem.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#define TIMER_INTERVAL 1000  // 1 second

//Time Mode Constants
#define TM_TRACKPLAYED 1
#define TM_TRACKTIME   2
#define TM_TRACKREMAIN 3
#define TM_TOTALPLAYED 4
#define TM_TOTALREMAIN 5

// -------------------------------------------------------------------------------------- //
// ---------------- AboutDlg Class Definition ------------------------------------------- //
// -------------------------------------------------------------------------------------- //
class AboutDlg : public TDialog
{
public:
	AboutDlg(TWindow * AParent, LPSTR AName) : TDialog(AParent, AName) {};
	virtual void SetupWindow();
};

void AboutDlg::SetupWindow()
{
	int a,b,c,d;

	TDialog::SetupWindow();

	// Center the window on the screen
	c = (284+2*GetSystemMetrics(SM_CXDLGFRAME));
	d = (184+GetSystemMetrics(SM_CYCAPTION)+2*GetSystemMetrics(SM_CYDLGFRAME));
	a = (GetSystemMetrics(SM_CXSCREEN)-c)/2;
	b = (GetSystemMetrics(SM_CYSCREEN)-d)/2.5;
	::MoveWindow(HWindow,a,b,c,d,FALSE);
}

// -------------------------------------------------------------------------------------- //
// ---------------- Main Window Class Definition ---------------------------------------- //
// -------------------------------------------------------------------------------------- //
class TMainWindow : public TFrameWindow
{
public:
	//class variables
	WORD wDeviceID;
	char DeviceName[128];
	DWORD TotalLength,Track1Pos;
	int NumTracks;
	DWORD TrackTimes[100];
	int CurrentTrack;
	DWORD CurrentPosition;
	BOOL bTimerSet,bCDDeviceOpen,bCDPaused,bCDin,bTopMost,bTracksList,bCDStopped,bFocus,bMinimized;
	int TimeMode;
	TStatic *TrackTimeStatic[30],*DirectionsStatic;
	HFONT hStaticControlFont;
	RECT Rect[30];

	TMainWindow(TWindow *AParent, LPSTR ATitle);
	~TMainWindow();
	virtual void SetupWindow();
	virtual LPSTR GetClassName();
	virtual void GetWindowClass(WNDCLASS &AWndClass);

	// Timer message handler
	void EvTimer( UINT );

	// Message response functions
	HBRUSH EvCtlColor( HDC, HWND, UINT );
	void EvLButtonDown( UINT, TPoint& );
	void EvSize( UINT, TSize& );
	void EvSetFocus( HWND );
	void EvKillFocus( HWND );

	// Menu response functions
	void FileExit();
	void FileAbout();
	void ControlPlay();
	void ControlPause();
	void ControlStop();
	void ControlTimeTrackPlayed();
	void ControlTimeTrackTime();
	void ControlTimeTrackRemain();
	void ControlTimeTotalPlayed();
	void ControlTimeTotalRemain();
	void ControlAlwaysOnTop();
	void TracksFirst();
	void TracksLast();
	void TracksNext();
	void TracksPrevious();
	void TracksList();

  // class functions
	void GetDeviceName();
  BOOL CheckDisk();
	void SetTimeFormat(DWORD tf);
	BOOL OpenCDDevice();
	BOOL CloseCDDevice();
	BOOL QueryDeviceStatus();
	BOOL QueryCDPlaying();
	int GetNumTracks();
	DWORD GetTotalLength();
	void GetTrackTimes();
	void SetTrackNumber(int TrackNum);
	int GetTrackNumber();
	void SetPosition(DWORD pos);
	DWORD GetPosition();
	BOOL PauseCD();
	BOOL PlayCD();
	BOOL StopCD();
	void PlayFromTrack(int TrackNum);
DECLARE_RESPONSE_TABLE( TMainWindow );
};

//---------------------------------------------------------------------------------------------//
// Constructor
TMainWindow::TMainWindow(TWindow * AParent, LPSTR ATitle)
		: TFrameWindow(AParent, ATitle)
{
	int i;

	// Setup the window attributes
	Attr.Style = WS_POPUP|WS_BORDER|WS_SYSMENU|WS_MINIMIZEBOX|WS_CAPTION;
	Attr.W = 280;
	Attr.H = GetSystemMetrics(SM_CYCAPTION)+GetSystemMetrics(SM_CYMENU)+2*GetSystemMetrics(SM_CYBORDER);

	AssignMenu("MENU_1");

	// check for saved window position in .INI file and use if found
	// a return value of 30000 means no saved info
	i = GetPrivateProfileInt("Config","WindowPosX",30000,"LITTLCD.INI");
	if(i==30000) Attr.X = (GetSystemMetrics(SM_CXSCREEN)-Attr.W)/2; // center window
  else Attr.X = i;
	i = GetPrivateProfileInt("Config","WindowPosY",30000,"LITTLCD.INI");
	if(i==30000) Attr.Y = (GetSystemMetrics(SM_CYSCREEN)-Attr.H)/2.5; // center window
  else Attr.Y = i;

	bTimerSet = bCDDeviceOpen = bCDPaused = bCDin = bTopMost = bTracksList = FALSE;
  bCDStopped = TRUE;
	CurrentTrack = 1;
	CurrentPosition = MCI_MAKE_TMSF(CurrentTrack,0,0,0);
	TimeMode = TM_TRACKPLAYED;

	// Set up the static text controls for the track list display
	// Rect[] is used to hold bounds of controls -- used when the user
	// clicks the mouse to determine which control wwas clicked.
	DirectionsStatic = new TStatic(this,199,"Click a track to play it ...",5,5,200,15,29);
	for(i=0;i<14;i++){
		TrackTimeStatic[i] = new TStatic(this,200+i," ",5,20+i*15,100,15,32);
		Rect[i].left = 5;
		Rect[i].top = 20+i*15;
		Rect[i].right = Rect[i].left+100;
		Rect[i].bottom = Rect[i].top+15;
	}
	for(i=14;i<28;i++){
		TrackTimeStatic[i] = new TStatic(this,200+i," ",20+100+10,20+(i-14)*15,100,15,32);
		Rect[i].left = 20+100+10;
		Rect[i].top = 20+(i-14)*15;
		Rect[i].right = Rect[i].left+100;
    Rect[i].bottom = Rect[i].top+15;
	}

  // The font to use for the track list display
	hStaticControlFont = CreateFont(-12,0,0,0,0,0,0,0,0,0,0,0,0,"MS Sans Serif");
}

//---------------------------------------------------------------------------------------------//
// destructor
TMainWindow::~TMainWindow()
{
	char junk[8];

	// close the CD audio device
	if(bCDDeviceOpen)	CloseCDDevice();
  // remove the timer
	if(bTimerSet) ::KillTimer(HWindow,1);
  // remove the track list font
	DeleteObject(hStaticControlFont);

	// Save the window position in the .INI file
	wsprintf(junk,"%i",Attr.X);
	WritePrivateProfileString("Config","WindowPosX",junk,"LITTLCD.INI");
	wsprintf(junk,"%i",Attr.Y);
	WritePrivateProfileString("Config","WindowPosY",junk,"LITTLCD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::SetupWindow()
{
	HMENU hMenu;
	int i;

	TFrameWindow::SetupWindow();

	// get a handle to the window menu
	hMenu = ::GetMenu(HWindow);

	// Set the font for the static controls in the track list display
	for(i=200;i<229;i++){
		::SendDlgItemMessage(HWindow,i,WM_SETFONT,(WPARAM)hStaticControlFont,(LPARAM)MAKELONG((WORD)FALSE,0));
	}
	// Open the CD audio device
	if(OpenCDDevice()){
		bCDDeviceOpen = TRUE;
		GetDeviceName();
	}else{
		SetCaption("Little CD Player: No Disk");
		return;
	}
	// set up the timer
	if(::SetTimer(HWindow,1,TIMER_INTERVAL,NULL) != 0) bTimerSet = TRUE;

	::DrawMenuBar(HWindow);

	// Get saved settings from the .INI file
	i = GetPrivateProfileInt("Config","AlwaysOnTop",0,"LITTLECD.INI");
  if(i==1){ // Make Window topmost 
		::SetWindowPos(HWindow,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);
		::CheckMenuItem(hMenu,209,MF_BYCOMMAND|MF_CHECKED);
    bTopMost = TRUE;
	}
	i = GetPrivateProfileInt("Config","TimeMode",0,"LITTLECD.INI");
	switch(i){ // set the Time Mode
		case 0:
			TimeMode = TM_TRACKPLAYED;
			::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_CHECKED);
			break;
		case 1:
			TimeMode = TM_TRACKTIME;
			::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_CHECKED);
			break;
		case 2:
			TimeMode = TM_TRACKREMAIN;
			::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_CHECKED);
			break;
		case 3:
			TimeMode = TM_TOTALPLAYED;
			::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_CHECKED);
			break;
		case 4:
			TimeMode = TM_TOTALREMAIN;
			::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_CHECKED);
			break;
	}
	i = GetPrivateProfileInt("Config","TrackList",0,"LITTLECD.INI");
	if(i==1){ // Show the track list
		::CheckMenuItem(hMenu,305,MF_BYCOMMAND|MF_CHECKED);
		bTracksList = TRUE;
		::MoveWindow(HWindow,Attr.X,Attr.Y,Attr.W,5+15*15+5+GetSystemMetrics(SM_CYCAPTION)+GetSystemMetrics(SM_CYMENU)+2*GetSystemMetrics(SM_CYBORDER),TRUE);
	}
}

//---------------------------------------------------------------------------------------------//
LPSTR TMainWindow::GetClassName()
{
	return "LittleCDMainWindow";
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::GetWindowClass(WNDCLASS &AWndClass)
{
	TFrameWindow::GetWindowClass(AWndClass);

	AWndClass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH); // set the background window color
	AWndClass.hIcon = LoadIcon((HINSTANCE)GetModule(),"ICON_1"); // set the window icon
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::EvTimer( UINT )
{

	char junk[64],scrap[64];
	int Minutes,Seconds;
	HMENU hMenu;

	QueryDeviceStatus();

	// if the window is hidden (i.e. NOT TOPMOST and doesn't have the focus)
	// don't bother updating the display. This way other applications
	// will not have their performance unduly affected.
	if(!bFocus && !bTopMost) return;

	hMenu = ::GetMenu(HWindow);
	if(bCDin){ // if a CD is in the CD-ROM player ...
		switch(TimeMode){ // calculate the proper time to display based on Time Mode
			case TM_TRACKPLAYED:
				Minutes = MCI_TMSF_MINUTE(CurrentPosition);
				Seconds = MCI_TMSF_SECOND(CurrentPosition);
				break;
			case TM_TRACKTIME:
				Minutes = MCI_MSF_MINUTE(TrackTimes[CurrentTrack-1]);
				Seconds = MCI_MSF_SECOND(TrackTimes[CurrentTrack-1]);
				break;
			case TM_TRACKREMAIN:
				Minutes = MCI_MSF_MINUTE(TrackTimes[CurrentTrack-1])-MCI_TMSF_MINUTE(CurrentPosition);
				Seconds = MCI_MSF_SECOND(TrackTimes[CurrentTrack-1])-MCI_TMSF_SECOND(CurrentPosition);
				if(Seconds<0){ // subtract a minute if Seconds turns out less than zero
					Minutes--;
					Seconds += 60;
				}
				break;
			case TM_TOTALPLAYED:
				Minutes = MCI_MSF_MINUTE(CurrentPosition);
				Seconds = MCI_MSF_SECOND(CurrentPosition);
      	break;
			case TM_TOTALREMAIN:
				Minutes = MCI_MSF_MINUTE(TotalLength)-MCI_MSF_MINUTE(CurrentPosition);
				Seconds = MCI_MSF_SECOND(TotalLength)-MCI_MSF_SECOND(CurrentPosition);
				if(Seconds<0){ // subtract a minute if Seconds turns out less than zero
					Minutes--;
					Seconds += 60;
        }
				break;
		}
		// Set the Window caption to display the number of tracks and total time
		// if the window is not minimized
    if(!bMinimized){
			wsprintf(junk,"Little CD Player: %i Tracks [%02i:%02i]",
					NumTracks,
					MCI_MSF_MINUTE(TotalLength),
					MCI_MSF_SECOND(TotalLength));
			// Only resets the caption if it has changed -- no need to reset every time
			if(_fstrcmp(Title,junk)!= 0)	SetCaption(junk);
		}
		if(bCDStopped){ // if CD is not playing, don't display the time
			wsprintf(junk,"Trk %i",CurrentTrack);
    }else{ // if the CD is playing, display the time calculated above
			wsprintf(junk,"Trk %i [%02i:%02i]",
					CurrentTrack,
					Minutes,
					Seconds);
		}
		// Only display if the caption has changed. When not minized,
		// the current track number and time (based on TimeMode) is displayed
		// as a menu item on the right had side of the menu bar (ID=99, MF_HELP
		// constant means to create a HELP BREAK -- display right justified menu
		// item). When minimized, this information is shown in the window caption
		if(bMinimized){
			if(_fstrcmp(Title,junk)!= 0)	SetCaption(junk);
    }else{
			::GetMenuString(hMenu,99,scrap,127,MF_BYCOMMAND);
			if(_fstrcmp(scrap,junk) != 0){
				::ModifyMenu(hMenu,99,MF_BYCOMMAND|MF_STRING|MF_HELP,99,(LPSTR)junk);
				::DrawMenuBar(HWindow);
			}
		}
	}else{ // No disk in player so say so
		wsprintf(junk,"Little CD Player: No Disk");
		if(_fstrcmp(Title,junk) != 0)	SetCaption(junk);  // again only if changed
		::GetMenuString(hMenu,99,scrap,127,MF_BYCOMMAND);
		if(_fstrcmp(scrap,junk) != 0){
			::ModifyMenu(hMenu,99,MF_BYCOMMAND|MF_STRING|MF_HELP,99,(LPSTR)" "); // no track info/time
			::DrawMenuBar(HWindow);
		}
	}
}

//---------------------------------------------------------------------------------------------//
HBRUSH TMainWindow::EvCtlColor( HDC hdc, HWND hwnd, UINT )
{
	if(hwnd != DirectionsStatic->HWindow){ // if NOT the DirectionsStatic control...
		SetTextColor(hdc,RGB(0,0,255)); // ... set the text color to blue.
	}
	SetBkColor(hdc,RGB(192,192,192)); // set the text background color to light gray
	return (HBRUSH)GetStockObject(LTGRAY_BRUSH); // return (HBRUSH)Msg.Result a brush to use to paint the control
																						 // background -- in this case light gray.
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::EvLButtonDown( UINT, TPoint& tpt )
{
	POINT pt;
	int z,i;
	z = -1;

	pt = (POINT)tpt; // Make a POINT from the mouse position returned by the message
	// Determine which rectangle (Static Control bounds) the mouse is in, if any
	for(i=0;i<30;i++){
		if(PtInRect(&Rect[i],pt) != 0) z = i+1;
	}
	// was the mouse clicked on a valid control?
	if(z>0 && z<=NumTracks){
  	// play the track corresponding to the control
		bCDStopped = FALSE;
		SetTrackNumber(z);
	}
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::EvSize( UINT SizeType, TSize& )
{
	// When the window is minimized or restored, set the BOOLEAN variable accordingly
	if(SizeType == SIZE_MINIMIZED){
		bMinimized = TRUE;
	}else{
		bMinimized = FALSE;
  }
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::EvSetFocus( HWND )
{
	bFocus = TRUE;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::EvKillFocus( HWND )
{
	bFocus = FALSE;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::FileExit()
{
	// Tell Windows we want to quit
	PostQuitMessage(0);
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::FileAbout()
{
	// Show the About Dialog Box
	GetModule()->ExecDialog(new AboutDlg(this,"DIALOG_1"));
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlPlay()
{
	PlayCD();
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlPause()
{
	PauseCD();
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlStop()
{
	StopCD();
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlTimeTrackPlayed()
{
	HMENU hMenu;

	// Check the menu item corresponding to this selection
	hMenu = ::GetMenu(HWindow);
	::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_CHECKED);
	::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_UNCHECKED);

  // Set the Time Mode and save it to the .INI file
	TimeMode = TM_TRACKPLAYED;
	WritePrivateProfileString("Config","TimeMode","0","LITTLECD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlTimeTrackTime()
{

	HMENU hMenu;

	// Check the menu item corresponding to this selection
	hMenu = ::GetMenu(HWindow);
	::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_CHECKED);
	::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_UNCHECKED);

	// Set the Time Mode and save it to the .INI file
	TimeMode = TM_TRACKTIME;
	WritePrivateProfileString("Config","TimeMode","1","LITTLECD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlTimeTrackRemain()
{

	HMENU hMenu;

	// Check the menu item corresponding to this selection
	hMenu = ::GetMenu(HWindow);
	::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_CHECKED);
	::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_UNCHECKED);

	// Set the Time Mode and save it to the .INI file
	TimeMode = TM_TRACKREMAIN;
	WritePrivateProfileString("Config","TimeMode","2","LITTLECD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlTimeTotalPlayed()
{

	HMENU hMenu;

	// Check the menu item corresponding to this selection
	hMenu = ::GetMenu(HWindow);
	::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_CHECKED);
	::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_UNCHECKED);

	// Set the Time Mode and save it to the .INI file
	TimeMode = TM_TOTALPLAYED;
	WritePrivateProfileString("Config","TimeMode","3","LITTLECD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlTimeTotalRemain()
{

	HMENU hMenu;

	// Check the menu item corresponding to this selection
	hMenu = ::GetMenu(HWindow);
	::CheckMenuItem(hMenu,208,MF_BYCOMMAND|MF_CHECKED);
	::CheckMenuItem(hMenu,204,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,205,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,206,MF_BYCOMMAND|MF_UNCHECKED);
	::CheckMenuItem(hMenu,207,MF_BYCOMMAND|MF_UNCHECKED);

	// Set the Time Mode and save it to the .INI file
	TimeMode = TM_TOTALREMAIN;
	WritePrivateProfileString("Config","TimeMode","4","LITTLECD.INI");
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::ControlAlwaysOnTop()
{

  HMENU hMenu;

  hMenu = ::GetMenu(HWindow);
	if(!bTopMost){ // if window was previously NOT TOPMOST ...
  	// ... make it TOPMOST ...
		::SetWindowPos(HWindow,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);
    // ... check the menu item ...
		::CheckMenuItem(hMenu,209,MF_BYCOMMAND|MF_CHECKED);
    // ... and save the setting to the .INI file.
		WritePrivateProfileString("Config","AlwaysOnTop","1","LITTLECD.INI");
	}
	if(bTopMost){ // if window was previously TOPMOST ...
  	// ... make NOT TOPMOST ...
		::SetWindowPos(HWindow,HWND_NOTOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);
    // ... uncheck the menu item ...
		::CheckMenuItem(hMenu,209,MF_BYCOMMAND|MF_UNCHECKED);
    // and save the setting to the .INI file.
		WritePrivateProfileString("Config","AlwaysOnTop","0","LITTLECD.INI");
	}
  // make the BOOLEAN variable reflect ethe current state
	bTopMost = !bTopMost;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::TracksFirst()
{

	// Play Track #1
	bCDStopped = FALSE;
	SetTrackNumber(1);
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::TracksLast()
{

	// Play last track
	bCDStopped = FALSE;
	SetTrackNumber(NumTracks);
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::TracksNext()
{

	// Play next track ...
	bCDStopped = FALSE;
	CurrentTrack++;
  // ... as long as it is a valid track
  if(CurrentTrack>NumTracks) CurrentTrack = NumTracks;
	SetTrackNumber(CurrentTrack);
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::TracksPrevious()
{

  // Play previous track ...
	bCDStopped = FALSE;
	CurrentTrack--;
  // ... as long as it is a valid track
  if(CurrentTrack<1) CurrentTrack = 1;
	SetTrackNumber(CurrentTrack);
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::TracksList()
{

	HMENU hMenu;

  hMenu = ::GetMenu(HWindow);

	if(bTracksList){ // If the Tracks List is shown ...
  	// ... uncheck the menu item ...
		::CheckMenuItem(hMenu,305,MF_BYCOMMAND|MF_UNCHECKED);
    // ... hide the list by shrinking the window size ...
		::MoveWindow(HWindow,Attr.X,Attr.Y,Attr.W,GetSystemMetrics(SM_CYCAPTION)+GetSystemMetrics(SM_CYMENU)+2*GetSystemMetrics(SM_CYBORDER),TRUE);
    // and save the setting to the .INI file.
		WritePrivateProfileString("Config","TrackList","0","LITTLECD.INI");
	}
	if(!bTracksList){ // If the Tracks List is not shown ...
  	// ... check the menu item ...
		::CheckMenuItem(hMenu,305,MF_BYCOMMAND|MF_CHECKED);
    // ... show the list by expanding the size of the window ...
		::MoveWindow(HWindow,Attr.X,Attr.Y,Attr.W,5+15*15+5+GetSystemMetrics(SM_CYCAPTION)+GetSystemMetrics(SM_CYMENU)+2*GetSystemMetrics(SM_CYBORDER),TRUE);
    // ... and save the setting to the .INI file.
		WritePrivateProfileString("Config","TrackList","1","LITTLECD.INI");
	}
  // make the BOOLEAN variable reflect the current state
	bTracksList = !bTracksList;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::GetDeviceName()
{
	MCI_INFO_PARMS mip;

  // Use the MCI_INFO command to get the name of the device we are using
	mip.lpstrReturn = (LPSTR)DeviceName;
  mip.dwRetSize = 127;
	mciSendCommand(wDeviceID,MCI_INFO,MCI_INFO_PRODUCT|MCI_WAIT,(DWORD)(LPVOID)&mip);
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::CheckDisk()
{
	MCI_STATUS_PARMS msp;

  // Use the MCI_STATUS command to see if there is a disk in the player
	msp.dwItem = MCI_STATUS_MEDIA_PRESENT;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);
  return (BOOL)msp.dwReturn;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::SetTimeFormat(DWORD TimeFormat)
{
  MCI_SET_PARMS msp;

  // Use the MCI_SET command to set the time format
	msp.dwTimeFormat = TimeFormat;
	mciSendCommand(wDeviceID,MCI_SET,MCI_SET_TIME_FORMAT|MCI_WAIT,(DWORD)(LPVOID)&msp);
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::OpenCDDevice()
{
  MCI_OPEN_PARMS mop;

  // Use the MCI_OPEN command to open the CD Audio device
  mop.lpstrDeviceType = "cdaudio";
	if(mciSendCommand(NULL,MCI_OPEN,MCI_OPEN_TYPE|MCI_WAIT,(DWORD)(LPVOID)&mop))
		return FALSE;
  wDeviceID = mop.wDeviceID; // save the device ID
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::CloseCDDevice()
{
  // Use the MCI_CLOSE command to close the CD audio device
	if(mciSendCommand(wDeviceID,MCI_CLOSE,0,NULL))
		return FALSE;
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::QueryDeviceStatus()
// Checks for the presence of a disk, gets tracks,times, and position
{
  char junk[64];

  if(!bCDin){ // if there is no disk in the player (as far we know) ... 
		if(CheckDisk()){ // ... check for the presence of a disk ...
    	// ... and get Track and Time information if present
			NumTracks = GetNumTracks();
			TotalLength = GetTotalLength();
			GetTrackTimes();
			bCDin = TRUE;
		}
  }
	if(bCDin){ // if there is a disk in the player (again as far as we know) ...
		if(CheckDisk()){ // ... check to be sure the disk is still in ...
		  // .. and get current position information if it is. 
			CurrentTrack = GetTrackNumber();
			CurrentPosition = GetPosition();
			// Also check to see if the disk in the player is actually playing
      // It could have been started by another application
			bCDStopped = !QueryCDPlaying();
			if(bCDPaused) bCDStopped = FALSE;
		}else{
    	bCDin = FALSE;
    }
	}
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::QueryCDPlaying()
{
	MCI_STATUS_PARMS msp;

  // Use the MCI_STATUS command to check if the disk in the player is playing
	msp.dwItem = MCI_STATUS_MODE;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);
	if(msp.dwReturn == MCI_MODE_PLAY) return TRUE;
	return FALSE;
}

int TMainWindow::GetNumTracks()
{
  MCI_STATUS_PARMS msp;

  msp.dwItem = MCI_STATUS_NUMBER_OF_TRACKS;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);
  return msp.dwReturn;
}

//---------------------------------------------------------------------------------------------//
DWORD TMainWindow::GetTotalLength()
{
	MCI_STATUS_PARMS msp;
	MCI_SET_PARMS msetp;

  // Set the time format to Minutes/Seconds/Frames ...
  SetTimeFormat(MCI_FORMAT_MSF);

  // ... and use the MCI_STATUS command to get the total length of the disk
  msp.dwItem = MCI_STATUS_LENGTH;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);

	return msp.dwReturn;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::GetTrackTimes()
{
	int i;
	MCI_STATUS_PARMS msp;
	char junk[31];
	DWORD TrackLength;
	DWORD z[100];
  int Minutes,Seconds,Frames;

  // Set the time format to Minutes/Seconds/Frames ...
	SetTimeFormat(MCI_FORMAT_MSF);
	// ... and for each track, get the track position. The 'z' array holds the track positions
	for(i=1;i<=NumTracks;i++){
		msp.dwItem = MCI_STATUS_POSITION;
		msp.dwTrack = i;
		mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_TRACK|MCI_WAIT,(DWORD)(LPVOID)&msp);
		z[i-1] = msp.dwReturn;
    if(i==1) Track1Pos = msp.dwReturn;
	}
	// The track positions are the times at which each track STARTS
	// To get the LENGTH of each track, we need to subtract the start of the
	// track we are interested in from the start of the NEXT track. This requires
  // us to know the TOTAL LENGTH in order to get the length of the LAST track
  z[NumTracks] = TotalLength;
  for(i=1;i<=NumTracks;i++){
		if(i<30){ // Only do the first thirty tracks since the Track List can only disply thirty
			// subtract the current track start from the next track start
      // NOTE: 'z' is zero offset so 'i' is NEXT track and 'i-1' is current track
			Minutes = MCI_MSF_MINUTE(z[i])-MCI_MSF_MINUTE(z[i-1]);
			Seconds = MCI_MSF_SECOND(z[i])-MCI_MSF_SECOND(z[i-1]);
			if(Seconds<0){ // if Seconds is less than zero, subtract a minute
				Minutes--;
				Seconds += 60;
			}
      // Use MCI to create DWORD holding the track length in Minutes/Seconds/Frames format. 
			TrackTimes[i-1] = MCI_MAKE_MSF(Minutes,Seconds,0);
			// Set the text of the corresponding static control in the track list
      // to show the track length
			wsprintf(junk,"Trk %02i = %02i:%02i",i,MCI_MSF_MINUTE(TrackTimes[i-1]),MCI_MSF_SECOND(TrackTimes[i-1]));
			TrackTimeStatic[i-1]->SetText(junk);
		}
  }
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::SetTrackNumber(int TrackNum)
{
	CurrentTrack = TrackNum;
	if(!bCDPaused){ // if the CD is not paused, play the track
		CurrentPosition = MCI_MAKE_TMSF(CurrentTrack,0,0,0); // Tracks/Minutes/Seconds/Frames format
    PlayCD();
  }
}

//---------------------------------------------------------------------------------------------//
int TMainWindow::GetTrackNumber()
{
	MCI_STATUS_PARMS msp;

  // Use the MCI_STATUS command to get the number of the track currently playing 
  msp.dwItem = MCI_STATUS_CURRENT_TRACK;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);
  return (int)msp.dwReturn;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::SetPosition(DWORD pos)
{
	MCI_SEEK_PARMS msp;

  // Use the MCI_SEEK command to set the position in Tracks/Minutes/Seconds/Frames format
  SetTimeFormat(MCI_FORMAT_TMSF);
  msp.dwTo = pos;
  mciSendCommand(wDeviceID,MCI_SEEK,MCI_TO|MCI_WAIT,(DWORD)(LPVOID)&msp);
}

//---------------------------------------------------------------------------------------------//
DWORD TMainWindow::GetPosition()
{
	MCI_STATUS_PARMS msp;
  DWORD tf;

	// Depending on the current Time Mode, we want to get the
	// current position in various formats so that we can
	// display the correct information and do subtractions
  // if necessary.
	switch(TimeMode){
		case TM_TRACKPLAYED:
			tf = MCI_FORMAT_TMSF;
      break;
		case TM_TRACKTIME:
			tf = MCI_FORMAT_MSF;
      break;
		case TM_TRACKREMAIN:
			tf = MCI_FORMAT_TMSF;
      break;
		case TM_TOTALPLAYED:
			tf = MCI_FORMAT_MSF;
      break;
		case TM_TOTALREMAIN:
			tf = MCI_FORMAT_MSF;
      break;
	}
  // Set the time format as determined by the above code
  SetTimeFormat(tf);

  // Use the MCI_STATUS command to get the current position 
	msp.dwItem = MCI_STATUS_POSITION;
	mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM|MCI_WAIT,(DWORD)(LPVOID)&msp);
  return msp.dwReturn;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::PauseCD()
{
	// Use the MCI_PAUSE command to pause the CD
	mciSendCommand(wDeviceID,MCI_PAUSE,MCI_WAIT,NULL);
	bCDPaused = TRUE;
	bCDStopped = FALSE;
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::PlayCD()
{
	MCI_PLAY_PARMS mpp;

  // Set the time format to Tracks/Minutes/Seconds/Frames
	SetTimeFormat(MCI_FORMAT_TMSF);

  // Use the MCI_PLAY command to play the CD
  // if the CD is stopped, start playing at the first track
	if(bCDStopped){
		mpp.dwFrom = MCI_MAKE_TMSF(1,0,0,0); // Start of track number one
    // Use MCI_FROM to play from the position set in the dwFrom variable
		if(mciSendCommand(wDeviceID,MCI_PLAY,MCI_FROM,(DWORD)(LPVOID)&mpp))
			return FALSE;
		return TRUE;
	}
  // if the CD is paused, restart it
	if(bCDPaused){
  	// Play -- the CD will automatically start from where it was paused
		if(mciSendCommand(wDeviceID,MCI_PLAY,0,(DWORD)(LPVOID)&mpp))
			return FALSE;
	}else{ // if not paused -- used primarily with the TRACKS menu and
				 // and WM_LBUTTONDOWN message to play a specified track
		mpp.dwFrom = CurrentPosition; // Set to the track to play by other functions
    // Use MCI_FROM to play from the position specified in the dwFrom variable
		if(mciSendCommand(wDeviceID,MCI_PLAY,MCI_FROM,(DWORD)(LPVOID)&mpp))
			return FALSE;
	}
	bCDPaused = FALSE;
	bCDStopped = FALSE;
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
BOOL TMainWindow::StopCD()
{
  // Use the MCI_STOP command to stop the CD
	mciSendCommand(wDeviceID,MCI_STOP,MCI_WAIT,NULL);
	bCDPaused = FALSE;
  bCDStopped = TRUE;
  return TRUE;
}

//---------------------------------------------------------------------------------------------//
void TMainWindow::PlayFromTrack(int TrackNum)
{
	// Set the Current Postion to the start of the desired track and play 
	CurrentPosition = MCI_MAKE_TMSF(TrackNum,0,0,0);
	PlayCD();
}


// -------------------------------------------------------------------------------------- //
// ---------------- Application Class Definition ---------------------------------------- //
// -------------------------------------------------------------------------------------- //
class TApp : public TApplication
{
public:
  // Inline constructor
	TApp(LPSTR AName, HINSTANCE hInstance, HINSTANCE hPrevInstance,
    LPSTR lpCmdLine, int nCmdShow)
		: TApplication(AName, hInstance, hPrevInstance, lpCmdLine, nCmdShow) {};
	// Main window initialization
	virtual void InitMainWindow();
};

void TApp::InitMainWindow()
{
	// Set up the application's window
	SetMainWindow(new TMainWindow(NULL, "Little CD Player"));
}

// ------------------------------------------------------------------- //
// ---------------- WINMAIN() ---------------------------------------- //
// ------------------------------------------------------------------- //
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
	LPSTR lpCmdLine, int nCmdShow)
{
	// Start the application
	TApp App("Little CD Player", hInstance, hPrevInstance, lpCmdLine, nCmdShow);
	App.Run();
	// return the App return value
	return App.Status;
}

DEFINE_RESPONSE_TABLE1( TMainWindow, TFrameWindow )
		EV_WM_TIMER,
		EV_WM_CTLCOLOR,
		EV_WM_LBUTTONDOWN,
    EV_WM_SIZE,
    EV_WM_SETFOCUS,
		EV_WM_KILLFOCUS,
    EV_COMMAND( 101, FileExit ),
    EV_COMMAND( 102, FileAbout ),
    EV_COMMAND( 201, ControlPlay ),
    EV_COMMAND( 202, ControlPause ),
    EV_COMMAND( 203, ControlStop ),
    EV_COMMAND( 204, ControlTimeTrackPlayed ),
    EV_COMMAND( 205, ControlTimeTrackTime ),
    EV_COMMAND( 206, ControlTimeTrackRemain ),
    EV_COMMAND( 207, ControlTimeTotalPlayed ),
    EV_COMMAND( 208, ControlTimeTotalRemain ),
    EV_COMMAND( 209, ControlAlwaysOnTop ),
    EV_COMMAND( 301, TracksFirst ),
    EV_COMMAND( 302, TracksLast ),
    EV_COMMAND( 303, TracksNext ),
    EV_COMMAND( 304, TracksPrevious ),
    EV_COMMAND( 305, TracksList ),
END_RESPONSE_TABLE;
