/****************************************************************************
                       TSE Named Clipboard System
 ****************************************************************************
                                  Version 4
                               **************

  Copyright: 1993 David Marcus

          Permission is granted to all persons for non-commercial
          distribution of this file and the macros and ideas contained
          herein provided (a) credit is given to the author and (b) all
          changes not made by the author are attributed.

          Permission is granted to SemWare for commercial distribution
          provided (a) and (b) above.

  Author:  David Marcus

  Date:    4-27-93

  Description:

          Nameclip is a macro system that implements named clipboards
          and related functions in The Semware Editor.

     ClipBoard Menu  ͻ   Ŀ
    Copy             <Grey+>                           v
    Copy Append <Ctrl Grey+>          ClipBoard Settings  Ŀ
   Ķ        Clipboard Type           [  Named] 
    Cut              <Grey->         Mode                    [Picklist] 
    Cut Append  <Ctrl Grey->         Include Text on Picklists    [Yes] 
   Ķ       Ĵ
    Paste            <Grey*>         Backup Clipboards       [      On] 
    Paste Over  <Ctrl Grey*>        Ĵ
   Ķ        Use Current Line If No Block [Yes] 
    View Named Clipboard             UnMark After Copy            [ No] 
    Rename Clipboard                Ĵ
    Delete Named Clipboard           UnMark After Paste           [ No] 
   Ķ        Delete After Paste           [ No] 
    Settings...              >    
   Ķ
    Global Actions...        >Ŀ
   ͼ               
                                              v
                               ClipBoard Global Actions  Ŀ
                              Fill Clipboards From File     
                              Save All Clipboards to a File 
                             Ĵ
                              Delete All Named Clipboards   
                             

  Installation:

  1. Place file this in the directory where your configuration
  (interface) files reside.  Add this line to your TSE.S (or other
  configuration) file:

     #include ['nameclip.s']

  Place this line anywhere the file as long as it comes before your
  menus.

  2. Delete any definitions you already have in your TSE.KEY file for
  these keys:

     <f11>          ClipBoardMenu()
     <Grey*>        ClipBoard_Menu_Action( PASTE_APPEND )
     <Ctrl Grey*>   ClipBoard_Menu_Action( Paste_OVERWRITE )
     <Ctrl PrtSc>   ClipBoard_Menu_Action( Paste_OVERWRITE )
     <Grey+>        ClipBoard_Menu_Action( Copy_OverWrite )
     <Ctrl Grey+>   ClipBoard_Menu_Action( Copy_APPEND )
     <Grey->        ClipBoard_Menu_Action( Cut_OverWrite )
     <Ctrl Grey->   ClipBoard_Menu_Action( Cut_APPEND )

  Or, change any of the key definitions included at the end of this
  file to use keys you prefer.

  Or, delete the keydef() at the end of this file and run NameClip only
  from menus.

  3. If you have a menu named ClipBoardMenu in your user interface file,
  as you probably do, delete it or comment it out. This file contains a
  substitution for it.

  4. Make sure that your main menu has a listing for ClipBoardMenu on
  it; it should look something like this:

        "&Clipboard",   ClipboardMenu()

  5. Delete the main() procedure from this file; it is only useful if
  you want to compile and use this file as an external macro file.

  6. Advanced Users: In NAMECLIP.S, the string ':U:' appears on each
  line that contains the initial setting of a toggle or mode. Search for
  :U: and customize as you like.  (Note: The initial settings for the
  UnMarkAfterPaste and UseCurrentLineifNoBlock toggles comes from your
  TSE configuration.)

  7. Re-bind the editor using the -b switch of sc.  If you receive any
  "duplicate name" errors, delete the specified procedure from your
  configuration file [NOT from nameclip.s] or comment it out, and
  re-bind.

  8. (optional) Delete or comment out any procedures that sc reports as
  "unused."

  9. (optional) Update your help file with any of the keys listed above.


  Usage Notes:

     * "Delete after paste" option -- turn this ON when you have moved a
       lot of stuff into clipboards and you want to make sure you move
       it _all_ back into your file.

     * In NAMECLIP.S file, the string ':U:' appears on each line that
       contains the initial setting of a toggle or mode.

       Search for :U: and customize as you like.  (Note: The initial
       settings for the UnMarkAfterPaste and UseCurrentLineifNoBlock
       toggles comes from your TSE configuration.)

     * "Mode" for Cut/Copy/Paste toggles between:

          QUICK - No picklists, no verification of overwrites of
               existing clipboards. Clipboard name entries must be
               typed.

          PICKLIST - Clipboards are selected from picklists. If <new> is
               selected and an existing name is entered, overwrite
               confirmation is requested.

     * Setting for automatic backup of clipboards can be:

          ON - If the clipboard to which you are doing a Copy or Cut
               (without the append option) exists and has text, the text
               is first copied to another clipboard. The backup
               clipboard name is 'BAK' + the name of the source
               clipboard. If the backup clipboard already has text on
               it, it is first deleted.

          ADDITIVE - Same as above except text on the backup clipboard
               is not deleted. New text being copied to the clipboard is
               added at the top of the clipboard. OFF - No backups.

       NOTE: Backups are not made when loading clipboards from a file.

     * When you use a clipboard with a zero-length name (i.e., just
       press <enter> at the name prompt), NameClip uses the default
       system clipboard.

       This means that if you start with the clipboard type set to
       UNNAMED and then change to NAMED, the default TSE clipboard
       contents are available. You'll see a blank name in picklists;
       this is it.

     * If you Fill Clipboards from File, NameClip() does not check for
       duplicate names. This may create multiple clipboards with the
       same name; you can use the rename function to change them.

     * When the UseCurrentLineIfNoBlock optionis used, and block is
       marked, the currentline is cut/copied. If UnMarkAfterCopy/Cut is
       set to NO, it is left as the marked block.

     * Press <F11> for Clipboard Menu.

     * Use of a blank (null) name is allowed. This is handy when you
       want to keep named clipboards turned on, but not use them much.
       That's what I generally do.

     * NameClip integrates Tim Farley's WinClip macro to add Windows
       clipboard support:

          1. The 'clipboard type' toggle goes NAMED/UNNAMED when TSE is
             not being run in a Windows DOS Window. When it is in a DOS
             Window, it toggles to WINDOWS, too.

          2. WINCLIP must be compiled and available as WINCLIP.MAC.

             If it is not available, NameCLip will still run fine but
             you will not see WINDOWS as a possible clipboard type to
             use. WINCLIP is available from the SemWare BBS.

             ͻ
              NOTE: This proc _must_ be added to WINCLIP:              
                                                                       
              public proc IWCP()                                       
                   SetGlobalInt( 'IWCP', IsWindowsClipboardPresent() ) 
              end                                                      
             ͼ

 **************************************************************************/

/**************************************************************************
          Variable & Constant Declarations
 **************************************************************************/

CONSTANT                           // for operation
     COPY_OVERWRITE     = 10,           // NOTE: The order of these is
     COPY_APPEND        = 20,           // critical because some comparisons
     CUT_OVERWRITE      = 30,           // use checks such as
     CUT_APPEND         = 40,           //   "IF FOO >= PASTE_APPEND ..."
     GETTING_FROM_FILE  = 50,           // GETTING from file
     PASTE_APPEND       = 60,
     PASTE_OVERWRITE    = 70,
     DELETE             = 80,
     VIEW               = 90,
     RENAME             = 100

CONSTANT                           // others
     ADDITIVE = 2,                           // for backup types
     NORMAL = 0,                             // for store_name_on_list mode
     MAKING_BACKUP = 1,
     UNNAMED = 0,                             // for clipboard_type
     NAMED = 1,
     WINDOWS = 2,
     REPORT_STATUS = 0,                      // arg for toggles
     CHANGE_STATUS = 1,
     PICKLIST= 1,                            // mode
     QUICK = 0

STRING
     backup_str[5] = 'BAK',                // :U: string prepended to buffer
                                             //     name for backups. Must end
                                             //     with other than a space.
     new_option[34] = '99999        < new clipboard > ',
                                             // :U: Text added to top of list
     title[20] = "Named ClipBoards",         // :U: Picklist title
     buffer_flag[10]='o',             // :U: Separates buffers in
                                             //     saved file
                                             //     Arbitrary, but unique
     saveclipfilename[50]='!clipbrd.sav',    // :U: default file name
     buffertext[53]='',                      // added to viewing list
//     status[254] = '',                       // audit
     BufferName[25] = '',                    // name of each buffer
     new_BufferName[25] = '',                // for rename and backup
     msg[32]=''

INTEGER
     mode = PICKLIST,                        // :U: QUICK, PICKLIST
     backup_type = ON,                       // :U: ON, OFF, ADDITIVE
     clipboard_type = 1,                     // :U: NAMED, UNNAMED, WINDOWS
     textflag = TRUE,                        // :U: include text on picklist?
                                             //     TRUE, FALSE
     unmark_flag = FALSE,                    // :U: unmark after copy
                                             //     TRUE, FALSE
     delete_after_paste = FALSE,             // :U: just what it says!
                                             //     TRUE, FALSE
     bid_width = 5,                          // chars allocated for bufferid
     original_clip_ID,                       // original clip ID
     buffer_to_use,                          // for named clipboard
     sid,                                    // starting buffer
     prompt_hist = 0,                        // history
     clip_list_ID = 0,                       // buffer clipboard name list
     max_width = 78,
     operation,                              // menu-selected operation
     msgl                                    // to hold starting msglevel

forward proc                  PopMessage( string title, integer mattr, integer xdelay )

/****************************************************************************
     INSERTBUFFER()
 ***************************************************************************/
integer proc                  insertbuffer(integer id)
//     status = 'ins_buff() ' + status
     PushPosition()
     PushBlock()         UnMarkBlock()
     if GotoBufferID(id) and (NumLines() > 1 or CurrLineLen())
          BegFile()      MarkLine()
          EndFile()      MarkLine()
     else
          PopPosition()
          PopBlock()
          return(FALSE)
     endif
     PopPosition()
     CopyBlock()
     PopBlock()
     return(TRUE)
end

/****************************************************************************
     BUFFERLINES()
 ***************************************************************************/
integer proc                  bufferlines(integer id)
//     status = 'bufflines() ' + status
     PushPosition()
     if GotoBufferID(id) and (NumLines() > 1 or CurrLineLen())
          PopPosition()
          return(NumLines())
     else
          PopPosition()
          return(0)
     endif
     return(TRUE)
end

/****************************************************************************
     mYESNOmenu()
           A primitive YesNo menu that works under all circumstances.
 ***************************************************************************/
/****
menu                          mYesNoMenu()
     noEscape
     "&No"    ,  , CloseBefore
     "&Yes"   , , CloseBefore
     "&Cancel", , CloseBefore
end

/****************************************************************************
     mYESNO(STRING question)
 ***************************************************************************/
/***
integer proc                  mYesNo(string arg)
     integer mba = Set( MenuBorderAttr, Query(MsgAttr) )
     integer maxX, maxY, maxL, x1, y1
     maxX = Query( ScreenCols )
     maxY = Query( ScreenRows )
     maxL = length( arg ) + 4
     x1 = (maxX - maxL) / 2
     Set(x1, iif( x1<0, 0, x1))
     y1 = (maxY - SizeOf(myesnomenu) -2 )/ 2
     Set(y1, iif( y1<0, 0, y1))
     mYesNomenu(arg)
     Set ( MenuBorderAttr, mba)
     Return(MenuOption() - 1)           // return 0, 1, 2
end

/****************************************************************************
     X(INTEGER ARG)
          A quickie helper to do a push or pop block -- but only when the
          unmark after copy/cut flag is ON.
****************************************************************************/
proc                          x(integer arg)
//     status = 'x() ' + status

     if unmark_flag
          return()
     else
          case arg
               when 1 PushBlock()
               when 2 PopBlock()
          endcase
     endif
end

/****************************************************************************
     SETUP_LIST_BUFFER()
     Helper to set up the buffer for list of
     clipboards.
****************************************************************************/
proc                          setup_list_buffer()
//     status = 'setup_list_buffer() ' + status
     if clip_list_ID == 0
          PushPosition()
          clip_list_ID = CreateTempBuffer()
          AddLine(new_option)
          PopPosition()
     endif
end

/****************************************************************************
     IF_LIST_EXISTS()
          Helper to determine if clipboard list exists
          and return buffer id
****************************************************************************/
integer proc                  if_list_exists()
//     status = 'if_list_exists() ' +  status
     /*
          Halts w/ msg if none exist
          Else returns the id of the list buffer
     */

     PushPosition()
     if clip_list_ID and GotoBufferID(clip_list_ID)
          if (operation >= PASTE_APPEND) and (NumLines() < 1)
               goto bailout
          else
               PopPosition()
               return(clip_list_ID)
          endif
     else
          setup_list_buffer()
          PopPosition()
          if operation <> GETTING_FROM_FILE
               goto bailout
          endif
     endif

     bailout:
               msg = 'No named clipboards exist'
               PopMessage( '', Query(MsgAttr), 18 )
               PopPosition()
               UpdateDisplay()
               halt
    return(42)
end

/****************************************************************************
     MARK_NAMES()
          Helper to mark name area of cliplist
****************************************************************************/
proc                          mark_names()
//     status = 'mark_names() ' + status
     UnMarkBLock()
     Gotoline(2)    GotoPos(bid_width + 1)                    MarkColumn()
     EndFile()      GotoPos(bid_width + Sizeof(buffername))   MarkColumn()
end

/****************************************************************************
     SORT_NAME_LIST()
          Helper to sort list of names of clipboards
****************************************************************************/
proc                          sort_name_list()
//     status = 'sort_name_list() ' + status
     PushPosition()
     GotoBufferID(clip_list_ID)
     PushBlock()
     mark_names()
     msgl = Set(MsgLevel, _NONE_)
     sort(_IGNORE_CASE_)
     Set(MsgLevel, msgl)
     PopBlock()
     PopPosition()
end

/****************************************************************************
     EXPANDED_ID()
          Returns zero-filled buffer_id
 ****************************************************************************/
string  proc                  expanded_id()
     return(Format(buffer_to_use:bid_width:'0'))
end

/****************************************************************************
     ADD_CONTENTS_TO_LIST()
          Adds opening contents of buffer to the clipboard list buffer
***************************************************************************/
proc                          add_contents_to_list()
//     status = 'add_contents_to_list() ' + status
     buffertext = '  '
     PushPosition()
     GotoBufferID(buffer_to_use)
     BegFile()
     while Length(buffertext) < (SizeOf(buffertext)-3)
          if PosLastNonWhite() == 0     // skip line if blank
               if not Down()
                    goto ending         // breaks out of while
               endif
          else                          // add line break indicator
               if Length(buffertext) > 3
                    buffertext = buffertext + '  '
               endif
               buffertext = buffertext + GetText( PosFirstNonWhite(),
                              PosLastNonWhite() - PosFirstNonWhite() + 1 )
               if not Down()
                    goto ending
               endif
          endif
     endwhile
   ending:
     GotoBufferID(clip_list_ID)
     if not lFind(expanded_id(), '^g')
          warn('hmmmmm....')
     endif
     GotoPos(bid_width + sizeof(buffername) + 1)
     DeltoEol()
     InsertText(buffertext, _OverWrite_)
     PopPosition()
end

/****************************************************************************
     TRIM_BUFFERNAME() removes trailing spaces
****************************************************************************/
proc                          trim_buffername()
//     status = 'trim_buffername() ' + status
     while length(buffername) AND buffername[length(buffername)] == ' '
          buffername = substr(buffername, 1, length(buffername) -1)
     endwhile
end

/****************************************************************************
     EXPAND_BUFFERNAME() removes trailing spaces
****************************************************************************/
proc                          expand_buffername()
//     status = 'expand_buffername() ' + status
     buffername = format(buffername : - sizeof(buffername) : ' ')
end

/****************************************************************************
     GET_BUFF
          Quickie to return the buffer number for the buffer on the
          current line of clipboard list.
****************************************************************************/
integer proc                  get_buff_num()
//     status = 'gbn() ' + status
     return( val(gettext(1,bid_width)) )
end

/****************************************************************************
     ADD_ENTRY_LINE()
          Adds an entry to the clip_list buffer
 ****************************************************************************/
proc                          add_entry_line()
     PushPosition()
     GotoBufferID(clip_list_ID)
     BegFile()
     AddLine(expanded_id()+buffername)
     PopPosition()
end

/****************************************************************************
     STORE_NAME_ON_LIST(INTEGER STORE_MODE)
          Helper to store names of clipboards;
          Create clipboard if does not exist;
          Returns buffer id either way.
****************************************************************************/
integer proc                  store_name_on_list(integer store_mode)
     // store_mode = NORMAL for all operations except
     // store_mode = BACKUPS when creating backup buffers

//     status = 'store_name_on_list() ' + status
     sid = getbufferid()
     GotoBufferID(if_list_exists())          // Goto list buffer
                                             // warns & halts if no exist
     /*
          Mark for Local Search (to exclude IDs)
     */
     PushBlock()
     mark_names()
   xtop:
     expand_buffername()
     if not lFind(buffername, '^gli')              // if not find 'name'
          PopBlock()
          if not (buffer_to_use == original_clip_id) or
             (store_mode == 1)
               buffer_to_use = CreateTempBuffer()  // create buffer
          endif                                    // if not original
          if not buffer_to_use
               message('Err 2a')              // Create buffer failed
               halt
          endif
          add_entry_line()
          sort_name_list()
          lFind(buffername, 'gin')           // position to last add
          goto ending
     endif
     trim_buffername()

     if ( store_mode <> MAKING_BACKUP ) and  // i.e., isn't a backup buffer
        ( mode <> QUICK ) and                // picklists are being used
        (Operation < VIEW ) and
        length(buffername)                   // no query to overwrite unnamed

          case YesNo(buffername+ ' exists; overwrite?')
               when 2                        // re-ask
                    if not Ask(Msg, BufferName, prompt_hist) and Length(buffername)
                         PopBlock()
                         GotoBufferID(sid)
                         halt
                    endif
                    goto xtop
               when 1                        // overwrite
                    buffer_to_use = get_buff_num()
                    PopBlock()
               otherwise                     // bail out for esc, cancel
                    PopBlock()
                    GotoBufferID(sid)
                    halt
          endcase
     else                                    // is a backup buffer
          buffer_to_use = get_buff_num()     // so no overwrite prompt
          PopBlock()
     endif
   ending:
     GotoBufferID(sid)
     return(buffer_to_use)
end

/****************************************************************************
     MARK_IDS()
          Help to mark id section of cliplist as a block.
****************************************************************************/
proc                          mark_ids()

//     status = 'mark_ids() ' + status
     PushPosition()
     PushBlock()    UnMarkBLock()
     BegFile()      GotoPos(1)             MarkColumn()
     EndFile()      GotoPos(bid_width)     MarkColumn()
     PopPosition()
end

/****************************************************************************
     LISTCLIPS():
          Helper to list named clipboards
          and return the bufferid of the selected one, if any
          or return 0 if <esc>
****************************************************************************/
integer proc                  ListClips()
     integer
          deleted = FALSE                    // flag
//     status = 'ListClips() ' + status

     if mode == QUICK
          return(99999)
     endif

     sid = getbufferid()                     // starting bufferid

     if GotoBufferID( if_list_exists() )
          if NumLines() <= 1                 // no named clipboards
               GotoBufferID(sid)
               return(99999)                 // 'create new' code
          endif
     endif

     if operation > GETTING_FROM_FILE
          lfind(expanded_id(),'g')
     endif

     /*
          Delete bufferIDs before listing
     */
     PushBlock()
     mark_ids()
     DelBlock()

     /*
          Delete 'new clipboard' option for certain ops
     */
     if operation > GETTING_FROM_FILE
          PushPosition()
          BegFile()
          DelLine()
          PopPosition()
          deleted = TRUE
     else
          BegFile()                     // set default to <new>
                                        // for copy & cut

     endif
                                             // set list location
     Set(Y1,2)                               // line two
     Set(X1,                                 // right justified
         iif(textflag, Query (ScreenCols) - max_width - 1,
                       Query (ScreenCols) - SizeOf(buffername) - 2))
     if lList( title,
               iif(textflag, max_width + 1, SizeOf(buffername) + 2),
               iif(NumLines() > 20, NumLines(), 20), _ENABLE_SEARCH_ )
          PushPosition()
          BegFile()
          /*
               Restore 'new clipboard' option
          */
          if deleted
               UnDelete()
          endif

          /*
               Restore buffer IDs
          */
          UnDelete()

          PopPosition()
          PopBlock()     //----------
          buffer_to_use = get_buff_num()          // get the bufferid of
                                                  // the selected clipboard
          if buffer_to_use <> 99999               // 99999 is 'create new' flag
               buffername = gettext(bid_width+1, sizeof(buffername))
               trim_buffername()
               AddHistoryStr(buffername,prompt_hist)
          endif
     else                                    // <esc> was pressed
          buffer_to_use = 0
          PushPosition()
          BegFile()
          /*
               Restore 'new clipboard' option
          */
          if deleted
               UnDelete()
          endif
          /*
               Restore buffer IDs
          */
          UnDelete()
          PopPosition()
     endif
     PopBlock()
     GotoBufferID(sid)
     return(buffer_to_use)                       // 0 if <esc> for list
end

/****************************************************************************
     DELETE_NAMED_CLIPBOARD(INTEGER xMODE)
****************************************************************************/
     // mode = 0 for all operations except
     // mode = 1 when deleting after Paste
proc                          delete_named_clipboard(integer xmode)
//     status = 'delete_named_clipboard() ' + status
     sid = getbufferid()
     operation = DELETE
     PushPosition()
     GotoBufferID( if_list_exists() )

     if xmode == 0
          ListClips()                        // set buffer_to_use
     endif

     GotoBufferID(sid)

     if buffer_to_use
          if GotoBufferID( if_list_exists() )
               mark_ids()
               if lFind(expanded_id(), 'glin')
                  DelLine()
                  if NumLines() == 1 and CurrLineLen() == 0
                    AbandonFile()
                    clip_list_ID = 0
                  endif
                  if GotoBufferID(buffer_to_use)
                      AbandonFile()
                  else
                      message('Err 3')       // got buffer id but Goto
                      halt                   // failed
                  endif
               else
                  message('Err 4')           // got id but can't find
                  halt
               endif
           endif
     endif
     PopPosition()
     PopBlock()
end

/****************************************************************************
     GET_NAME_FROM_ID()
          Helper to get name
****************************************************************************/
string proc                   get_name_from_id()
//     status = 'get_name_from_id() ' + status
     PushPosition()
     GotoBufferID(clip_list_ID)
     lFind(expanded_id() , '^g')
     buffername = GetText(bid_width+1, PosLastNonWhite())
     PopPosition()
     return(buffername)
end

/****************************************************************************
     DELETE_ALL_NAMED_CLIPBOARDS()
***************************************************************************/
proc                          delete_all_named_clipboards()
     integer
          id
     sid = getbufferid()
     GotoBufferID(if_list_exists())
     BegFile()
     GotoLine(2)
     while NumLines() >= CurrLine() AND CurrLineLen() > 0
          buffer_to_use = get_buff_num()
          msg = get_name_from_id()
          PopMessage('Deleting...', Query(MsgAttr), 12)
          AbandonFile(get_buff_num())
          DelLine()
     endwhile
     AbandonFile()
     clip_list_id = 0
     gotobufferid(sid)
end

/****************************************************************************
     VIEW_NAMED_CLIPBOARD()
****************************************************************************/
proc                          view_named_clipboard()
     operation = VIEW

     PushPosition()
     GotoBufferID( if_list_exists() )
     buffer_to_use = ListClips()                        // get list of clips
     if buffer_to_use
          buffername = get_name_from_ID()
          trim_buffername()
          if GotoBufferID(buffer_to_use)
               list(buffername,80)
          else
               warn('Err 4')
          endif
     endif
     PopPosition()
end

/****************************************************************************
     BACKUP()
          This is the macro that handles backup before cut and copy.
****************************************************************************/

proc                          backup()
     integer
          primary_buffer = 0
     if backup_type and
        bufferlines(buffer_to_use)                // if there are contents
          PushPosition()
          new_buffername = buffername
          primary_buffer=buffer_to_use
          buffername = backup_str + buffername    // setup .bak name
          store_name_on_list(MAKING_BACKUP)       // sets buffer_to_use to
                                                  // .bak buffer
          GotoBufferID(buffer_to_use)
          if backup_type <> ADDITIVE              // if not additive
               PushBlock()                        // delete contents
               BegFile() MarkLine()
               EndFile() MarkLine()
               DelBlock()
               PopBlock()
          endif
          insertbuffer(primary_buffer)
          add_contents_to_list()
          PopPosition()
          buffer_to_use = primary_buffer          // reset for the copy/cut
          buffername = new_buffername             //     from the text file
     endif
end

/****************************************************************************
     NAMED_CLIPBOARD()
          This is the macro that handles all named clipboard operations.
****************************************************************************/
integer proc                  named_clipboard()
     integer marked = FALSE                  // was curr line marked?
     /*
          Save current ClipBoard Id
     */
     original_clip_ID = GetClipBoardId()

     /*
          Make sure the history buffer has been setup
     */
     if prompt_hist == 0
          prompt_hist = GetFreeHistory()
     endif

     /*
          Make sure the list of names buffer has been setup
     */
     setup_list_buffer()

     /*
          Saves (Cut/Copy) require a marked block in the current buffer
          unless UseCurrLineIfNoBlock is On
     */
     if Operation <= CUT_APPEND
        AND Query(BlockId) <> GetBufferId()
          if query(usecurrlineifnoblock) == TRUE
               UnMarkBlock()
               MarkLine()
               MarkLine()
               marked = true
          else
               msg = 'Block must be in current file'
               popmessage('', query(MsgAttr), 0)
               return (FALSE)
          endif
     endif

     /*
          Set up user prompt/message used when prompting for name of CB
     */
     case Operation
          when COPY_OVERWRITE    Msg = "Copy"
          when COPY_APPEND       Msg = "Copy/Append"
          when CUT_OVERWRITE     Msg = "Cut"
          when CUT_APPEND        Msg = "Cut/Append"
     endcase
     msg = msg +' to clipboard:'

     BufferName = ""
     buffer_to_use = ListClips()
     if buffer_to_use == 99999               // if 'create new'
          trim_buffername()
          if not Ask(Msg, BufferName, prompt_hist)
               if marked
                    UnMarkBlock()
               endif
               return(FALSE)                 // return if <esc>
          elseif not Length(buffername)
               buffer_to_use = original_clip_id
               SetClipBoardId( store_name_on_list(NORMAL) )
          else
               SetClipBoardId( store_name_on_list(NORMAL) )
          endif
     elseif buffer_to_use == 0               // if returns 0 (<esc>)
          if marked
               UnMarkBlock()
          endif
          return(FALSE)
     elseif buffer_to_use                    // else set ID
          SetClipBoardId( buffer_to_use )
     endif

     case Operation
          when COPY_OVERWRITE    x(1)
                                 backup()
                                 Copy()
                                 x(2)
          when COPY_APPEND       x(1)
                                 Copy(_APPEND_)
                                 x(2)
          when CUT_OVERWRITE     backup()
                                 Cut()
          when CUT_APPEND        Cut(_APPEND_)

          when PASTE_APPEND      if Paste() and delete_after_paste
                                   delete_named_clipboard(1)
                                 endif
          when PASTE_OVERWRITE   if Paste(_OVERWRITE_) and delete_after_paste
                                   delete_named_clipboard(1)
                                 endif
     endcase

     /*
          Add text to clipboard listing
     */
     if operation >= COPY_OVERWRITE AND
        operation <= CUT_APPEND
          add_contents_to_list()
     endif

     // Restore ClipBoard
     SetClipBoardId( original_clip_ID )
     set(BREAK,OFF)
     return(TRUE)
end

/****************************************************************************
     TOGGLE_TEXTFLAG():
          Toggles whether to include first bit of text in list of clipboards
****************************************************************************/
proc                          toggle_textflag()
     textflag = NOT textflag
end

/****************************************************************************
     TOGGLE_UNMARK():
          Toggles whether to unmark after Cut/Copy
****************************************************************************/
proc                          toggle_unmark()
     unmark_flag = NOT unmark_flag
end

/****************************************************************************
     TOGGLE_DELETE_AFTER_PASTE():
          Toggles whether to delete the clipboard after Cut/Copy
****************************************************************************/
proc                          toggle_delete_after_paste()
     delete_after_paste = NOT delete_after_paste
end

/****************************************************************************
     TOGGLE_MODE(INTEGER ARG)
          Toggles picklist vs quick
****************************************************************************/
string proc                   toggle_mode(integer arg)
     case arg
          when REPORT_STATUS
               case mode
                    when PICKLIST return ('Picklist')
                    when QUICK return ('Quick')
                    when ADDITIVE return('Additive')
               endcase
          when CHANGE_STATUS
               mode = not mode
     endcase
     return('')
end

/****************************************************************************
     TOGGLE_backup_type(INTEGER ARG)
          Toggles backup type.
****************************************************************************/
string proc                   toggle_back_type(integer arg)
     case arg
          when REPORT_STATUS
               case backup_type
                    when OFF return('Off')
                    when ON return('On')
                    when ADDITIVE return('Additive')
               endcase
          when CHANGE_STATUS
               backup_type = (backup_type + 1) Mod 3
     endcase
     return('')
end

/****************************************************************************
     TOGGLE_CLIP_TYPE(INTEGER ARG)
          Toggles clipboard type.
          If WINCLIP.MAC is present it is loaded so that
               presence of Windows clipboard can be tested
               to see if OK to toggle type to WINDOWS.
****************************************************************************/
string proc                   toggle_clip_type(integer arg)
     case arg
          when REPORT_STATUS
               case clipboard_type
                    when UNNAMED return('UnNamed')
                    when NAMED return('Named')
                    when WINDOWS return('Windows')
               endcase
          when CHANGE_STATUS
               msgl = set(msglevel,_NONE_)
               if loadmacro('winclip')
                    Set(MsgLevel,msgl)
                    ExecMacro('IWCP')
                    clipboard_type =
                    (clipboard_type + 1)
                         Mod ( iif(GetGlobalInt('IWCP'), 3, 2) )
               else
                    Set(MsgLevel,msgl)
                    clipboard_type = NOT clipboard_type
               endif
     endcase
     Set(MsgLevel,msgl)
     if clipboard_type == NAMED                     // just changed to NAMED
          PushPosition()                         // so add system clipboard
          original_clip_id = GetClipboardID()    // to clip_list
          gotobufferid(original_clip_id)
          if NumLines() > 0
               buffer_to_use = original_clip_id
               buffername = ''
               setup_list_buffer()
               store_name_on_list(NORMAL)
               add_contents_to_list()
          endif
          PopPosition()
     endif

     return('')
end

/****************************************************************************
     CLIPBOARD_MENU_ACTION(INTEGER ACTION)
          This macro handles unnamed and windows clipboard operations.
          It branches to named_clipboard() for named clipboard ops.
          It also takes care of the UnMarkAfterPaste operation, if set,
               for all clipboard types.
****************************************************************************/
proc                          clipboard_menu_action(integer Action)
    operation = action                 // set global integer with action

    if clipboard_type == 2 AND (NOT IsMacroLoaded('winclip'))
        LoadMacro('winclip')
    endif

    if Query(unmarkafterpaste) AND (action == PASTE_APPEND or
                                    action == PASTE_OVERWRITE)
        PushBlock()
    endif

    case clipboard_type
        when NAMED
             named_clipboard()
        when UNNAMED
             case action
                 when COPY_OVERWRITE    x(1) Copy() x(2)
                 when COPY_APPEND       x(1) Copy(_APPEND_)  x(2)
                 when PASTE_APPEND      Paste()
                 when PASTE_OVERWRITE   Paste(_OVERWRITE_)
                 when CUT_OVERWRITE     Cut()
                 when CUT_APPEND        Cut(_APPEND_)
             endcase
         when WINDOWS
             case action
                 when COPY_OVERWRITE    x(1) Copy() ExecMacro('CopytoWindows') x(2)
                 when COPY_APPEND       message('Cannot APPEND to Windows clipboard')
                 when PASTE_APPEND      ExecMacro('PasteFromWindows') Paste()
                 when PASTE_OVERWRITE   message('Cannot OVERWRITE from Windows clipboard')
                 when CUT_OVERWRITE     Cut() ExecMacro('CopytoWindows')
                 when CUT_APPEND        message('Cannot APPEND to Windows clipboard')
             endcase
     endcase

     if Query(unmarkafterpaste) and (action == PASTE_APPEND OR
                                     action == PASTE_OVERWRITE)
        PopBlock()
     endif

    // Done
    return ()
end

/****************************************************************************
     SAVECLIPS()
****************************************************************************/
proc                          SaveClips()
     integer
          scf=0                       // working buffer
     clip_list_id = if_list_exists()  // will halt w/error if no name clips
     original_clip_ID = GetClipBoardId()
     PushPosition()

     /*
          Get file name
     */
     askfile:
     if Ask('File name for clipboard storage?', saveclipfilename)
          and Length(saveclipfilename)
     else
          goto ending
     endif

     /*
          Verify replacement if exists
     */
     if FileExists(saveclipfilename)
          case YesNo('Overwrite '+saveclipfilename+ '?')
               when 2 goto askfile      // No
               when 0,3 goto ending     // Esc, Cancel
          endcase
     endif

     /*
          If file is loaded, get rid of it
     */
     msgl = set(msgLevel, _NONE_)
     if EditFile(saveclipfilename)
          AbandonFile()
     endif
     set(msgLevel, msgl)

     scf = CreateBuffer(saveclipfilename)
     GotoBufferID(clip_list_id)
     BegFile()
     while Down()
          SetClipBoardId( get_buff_num() )
          buffername = GetText(bid_width+1, CurrLineLen() - bid_width)
          msg = buffername
          PopMessage('Processing...', Query(MsgAttr), 12)
          GotoBufferID(scf)
          AddLine(buffer_flag + buffername)
          AddLine()
          Paste()
          EndFile()
          GotoBufferID(clip_list_id)
     endwhile
     GotoBufferID(scf)
     msgl = set(MsgLevel, _NONE_)
     if SaveFile()
          AbandonFile()
     else
          msg = "Couldn't save; keeping file"
          PopMessage('', query(MsgAttr), 0)
     endif
     set(MsgLevel, msgl)
     ending:
          SetClipBoardID(original_clip_ID)
          PopPosition()
end

/****************************************************************************
     RENAME_IT(STRING OLD_NAME, STRING NEW_NAME)
****************************************************************************/
proc                          rename_it(STRING old_name, STRING new_name)
//     status = 'rename_it() ' + status
     GotoBufferID(clip_list_ID)
     buffername = old_name
     expand_buffername()
     if not lFind(expanded_id() + buffername, '^g')
          msg = "Err 5"
          PopMessage('', query(MsgAttr), 0)
          halt
     endif
     GotoPos(bid_width + 1)
     buffername = new_name
     expand_buffername()
     InsertText(buffername, _OVERWRITE_)
end

/****************************************************************************
     RENAME_CLIPBOARD()
****************************************************************************/
proc                          rename_clipboard()
     new_buffername=''
     sid = GetBufferID()
     operation = RENAME
     buffer_to_use = ListClips()
     if not buffer_to_use
          goto ending
     endif
     buffername = get_name_from_ID()
     trim_buffername()
     if Ask('New name for ' +  buffername + '?', new_buffername, prompt_hist) AND
          Length(new_buffername)
          rename_it(buffername, new_buffername)
          sort_name_list()
     endif
     ending:
          GotoBufferID(sid)

end

/****************************************************************************
     STORE_IT()
****************************************************************************/
proc                          store_it()
     PushPosition()
//   GotoBufferID(if_list_exists())          // Goto list buffer

     /*
          Create Buffer and Copy
     */
     buffer_to_use = CreateTempBuffer() // create buffer
     if not buffer_to_use
          message('Err 2b')              // Create buffer failed
          halt
     endif
     CopyBlock()

     /*
          Add to list
     */
     GotoBufferID(clip_list_ID)         // return to list
     BegFile()

     /*
          Check for Dupes ------ NOT WORKING; COMMENTED OUT
     PushBlock()
     mark_names()
     if lFind(buffername, 'gl')
          trim_buffername()
          case YesNo('Replace current ' + buffername + '?')
               when 2                        // NO
                    ask('Alternate name?', buffername, prompt_hist)
//                  AbandonFile(get_buff_num())
  //                DelLine()
               when 1                        // YES
                    warn(buffername)
                    buffer_to_use = get_buff_num()
                    rename_it(buffername, buffername + ''+ saveclipfilename)
                    buffername = substr(buffername, 1, pos('', buffername)-1)
               otherwise
                    popposition()
                    halt
          endcase
     endif
     PopBlock()
     */
     add_entry_line()
     add_contents_to_list()
     PopPosition()
end

/****************************************************************************
     GETCLIPS()
****************************************************************************/
proc                          GetClips()
     integer
          temp,                              // temp buffer holds get file
          buffer                             // used to hold buffer #s

     original_clip_ID = GetClipBoardId()
     operation = GETTING_FROM_FILE

     setup_list_buffer()                     // just in case?
     PushPosition()
     PushBlock()

     if Ask('File name for clipboards?', saveclipfilename)
          AND Length(saveclipfilename)
     else
          msg = 'bad file name'
          PopMessage('', Query(MsgAttr), 19)
          goto ending
     endif

     temp = CreateTempBuffer()
     msgl = set(msglevel, _NONE_)
     InsertFile(saveclipfilename)
     set(msglevel, msgl)

     while lFind(buffer_flag, '^')
          UnMarkBLock()
          buffername = GetText(Length(buffer_flag)+1,
                               CurrLineLen()-Length(buffer_flag))
          Down()
          MarkLine()
          if lFind(buffer_flag, '^')
               Up()
               MarkLine()
          else
               EndFile()
               MarkLine()
          endif
          store_it()
          GotoBufferID(temp)
          msg = buffername
          PopMessage('Processing...', Query(MsgAttr), 12)
     endwhile

     sort_name_list()

     ending:
          AbandonFile(temp)
          PopBlock()
          SetClipBoardID(original_clip_ID)
          PopPosition()
end

/****************************************************************************
     CLIPBOARD_SETTINGS_MENU()
****************************************************************************/
Menu                          Clipboard_Settings_Menu()
      History
      Title = ' ClipBoard Settings '

     "Clipboard &Type"    [toggle_clip_type(REPORT_STATUS):7] ,
          toggle_clip_type(CHANGE_STATUS)                        , DontClose
     "&Mode"              [toggle_mode(REPORT_STATUS):8],
          toggle_mode(CHANGE_STATUS)                             , DontClose
     "&Include Text on Picklists" [ iif(textflag,'Yes', 'No') : 3] ,
          toggle_textflag()                                      , DontClose
      ""                                                ,        , Divide
     "&Backup Clipboards" [toggle_back_type(REPORT_STATUS):8] ,
          toggle_back_type(CHANGE_STATUS)                        , DontClose
     ""                                                 ,         , Divide
     "&Use Current Line If No Block" [iif(Query(UseCurrLineIfNoBlock),'Yes', 'No') : 3] ,
          toggle(UseCurrLineIfNoBlock)                           , DontClose
     "UnMa&rk After Copy"[ iif(unmark_flag,'Yes', 'No') : 3] ,
          toggle_unmark()                                        , DontClose
     ""                                                         , , Divide
     "UnMar&k After Paste"[iif(query(unmarkafterpaste),'Yes', 'No') : 3] ,
          toggle(unmarkafterpaste)                               , DontClose
     "&Delete After Paste"[ iif(delete_after_paste,'Yes', 'No') : 3] ,
          toggle_delete_after_paste()                            , DontClose end

/****************************************************************************
     GLOBAL_CLIPBOARD_MENU()
****************************************************************************/
Menu                          Global_Clipboard_Menu()
          History
          Title = ' ClipBoard Global Actions '
    "&Fill Clipboards From File",     GetClips()                    , CloseBefore
    "&Save All Clipboards to a File", SaveClips()                   , CloseBefore
     ""                 ,                                           , Divide
    "&Delete All Named Clipboards"
                        ,   delete_all_named_clipboards()           , CloseBefore
end

/****************************************************************************
     CLIPBOARDMENU()
****************************************************************************/
Menu                          ClipBoardMenu()
          History
          Title = ' ClipBoard Menu '
    "&Copy"             ,   clipboard_menu_action( COPY_OVERWRITE ) , CloseAllBefore
    "Copy &Append"      ,   clipboard_menu_action( COPY_APPEND )    , CloseAllBefore
    ""                  ,                                           , Divide
    "C&ut"              ,   clipboard_menu_action( CUT_OVERWRITE )  , CloseAllBefore
    "Cu&t Append "      ,   clipboard_menu_action( CUT_APPEND )     , CloseAllBefore
    ""                  ,                                           , Divide
    "&Paste"            ,   clipboard_menu_action( PASTE_APPEND )   , CloseAllBefore
    "Paste &Over"       ,   clipboard_menu_action( PASTE_OVERWRITE ), CloseAllBefore
    ""                  ,                                           , Divide
    "&View Named Clipboard"
                        ,   view_named_clipboard()                  , DontClose
    "R&ename Clipboard"
                        ,   rename_clipboard()                      , DontClose
    "&Delete Named Clipboard"
                        ,   delete_named_clipboard(0)               , DontClose
    ""                  ,                                           , Divide
    "&Settings..."      ,   Clipboard_Settings_Menu()               , DontClose
    ""                  ,                                           , Divide
    "&Global Actions...",   Global_Clipboard_Menu()                 , DontClose
end

/****************************************************************************
     POPMESSAGE( STRING TITLE, STRING MSG, INTEGER MATTR, INTEGER XDELAY )
****************************************************************************/
proc                          PopMessage( string title, integer mattr, integer xdelay )
          // by Richard Hendricks
  integer maxX, maxY, maxL, saveattr, boxtype, x1, y1, x2, y2, tlen, mlen
  if xdelay == 0
     msg = msg + ' ... press any key to continue.'
  endif
  maxX = Query( ScreenCols )
  maxY = Query( ScreenRows )
  tlen = length( title )
  mlen = length( msg )
  maxL = iif( tlen < mlen, mlen, tlen )
  x1 = (maxX - maxL - 3) / 2
  x1 = iif( x1<0, 0, x1)
  y1 = (maxY - 3) / 2
  y1 = iif( y1<0, 0, y1)
  x2 = x1 + maxL + 3
  y2 = y1 + 2
  boxtype = 6
  PopWinOpen( x1, y1, x2, y2, boxtype, title, query(hiliteattr) )
  saveattr = Query( Attr )
  Set( Attr, mattr )
  VHomeCursor()
  ClrScr()
  Set( Cursor, Off )
  Write( Format(" ":((maxL-length(msg)+3)/2))+msg ) // center msg
  if xdelay == 0
     GetKey()
  else
     delay(xdelay)
  endif
  Set( Attr, saveattr )
  PopWinClose()
  Set( Cursor, On )
end PopMessage

proc                          main()
     ClipBoardMenu()
end

/****************************************************************************
     KEY DEFS
****************************************************************************/

<f11>          ClipBoardMenu()
<Grey*>        ClipBoard_Menu_Action( PASTE_APPEND )
<Ctrl Grey*>   ClipBoard_Menu_Action( Paste_OVERWRITE )
<Ctrl PrtSc>   ClipBoard_Menu_Action( Paste_OVERWRITE )
<Grey+>        ClipBoard_Menu_Action( Copy_OverWrite )
<Ctrl Grey+>   ClipBoard_Menu_Action( Copy_APPEND )
<Grey->        ClipBoard_Menu_Action( Cut_OverWrite )
<Ctrl Grey->   ClipBoard_Menu_Action( Cut_APPEND )