/*
   echorep.c

   echolink report generator for squish.cfg and/or areas.bbs systems
   written by Carl Declerck, May 1993

   this source is 100% ANSI compliant, except for strcmpi()

   small memory model DOS executables may run into trouble when processing
   echo-data for big systems (ie. hubs etc.). recompile for a larger model
   (or just dump DOS :) when you experience this.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#define  ERR_OK                     0
#define  ERR_NOROOM                -1

#define  CFG_AREASBBS               0
#define  CFG_SQUISHCFG              1

#define  ADD_ATSTART                0         /* add to list, fastest */
#define  ADD_ATEND                  1         /* slowest */
#define  ADD_SORTED                 2         /* depends */

#define  VERSION "0.05"

typedef struct _address
        {
            int  zone;                   /* TC++ doesn't like long ints :( */
            int  net;
            int  node;
            int  point;
            char domain[16];
        } ADDRESS;

typedef struct _address_list
        {
            ADDRESS              address;
            struct _address_list *next;
        } ADDRESS_LIST;

typedef struct _echo
        {
            char         *name;
            ADDRESS_LIST *list;
        } ECHO;

typedef struct _echo_list
        {
            ECHO              echo;
            struct _echo_list *next;
        } ECHO_LIST;


ECHO_LIST *echo_list = NULL;
ECHO echo;
ADDRESS_LIST *in_list = NULL, *ex_list = NULL, *process = NULL;
ADDRESS addr, us = { 0, 0, 0, 0, "fidonet" };


void show_help (void)
{
    fputs (
            "Usage syntax:\n\n"
            "EchoRep <address> -a<our-address> -b<areas-bbs> -c<squish-cfg> -x<exclude>\n\n"
            "  where:   <address>     = address to report links for (*)\n"
            "           <our-address> = our primary address\n"
            "           <areas-bbs>   = path to AREAS.BBS\n"
            "           <squish-cfg>  = path to SQUISH.CFG\n"
            "           <exclude>     = address NOT to report links for (*)\n\n"
            "  (*) may be replicated with different values\n"
            "      wild-cards may replace any zone/net/node/point/domain\n\n"
            "  eg: EchoRep 2:*/*@dudenet -x2:234/*@dudenet -cG:\\FIDO\\SQUISH\\SQUISH.CFG\n", stderr
          );

    exit (1);
}

int  addrcmp (ADDRESS *a1, ADDRESS *a2)
{
    if (*a1->domain == '*' || *a2->domain == '*' || strcmp(a1->domain, a2->domain) == 0)
    {
        if (a1->zone == -1 || a2->zone == -1 || a1->zone == a2->zone)
        {
            if (a1->net == -1 || a2->net == -1 || a1->net == a2->net)
            {
                if (a1->node == -1 || a2->node == -1 || a1->node == a2->node)
                {
                    if (a1->point == -1 || a2->point == -1 || a1->point == a2->point)
                     return (0);
                    else return (a1->point - a2->point);
                }
                else return (a1->node - a2->node);
            }
            else return (a1->net - a2->net);
        }
        else return (a1->zone - a2->zone);
    }
    else return (strcmp(a1->domain, a2->domain));
}

int  add_address (ADDRESS_LIST **list, ADDRESS *addr, int mode)
{
    ADDRESS_LIST **tmp, *next;

    switch (mode)
    {
        case ADD_ATSTART : tmp = list;
                           break;
        case ADD_ATEND     : for (tmp = list; *tmp != NULL; tmp = &(*tmp)->next);
                           break;
        case ADD_SORTED  : for (tmp = list; *tmp != NULL && addrcmp(&(*tmp)->address, addr) < 0; tmp = &(*tmp)->next);
                           break;
    }

    next = *tmp;

    if ((*tmp = (ADDRESS_LIST *) malloc(sizeof(ADDRESS_LIST))) != NULL)
    {
        (*tmp)->address = *addr;
        (*tmp)->next    = next;
        return (ERR_OK);
    }
    else *tmp = next;

    return (ERR_NOROOM);
}

void free_address_list (ADDRESS_LIST **list)
{
    ADDRESS_LIST *tmp;

    for (; *list != NULL; *list = tmp)
    {
        tmp = (*list)->next;
        free (*list);
    }
}

int  find_address (ADDRESS_LIST **list, ADDRESS *addr)
{
    ADDRESS_LIST *walk;
    int count;

    for (count = 1, walk = *list; walk != NULL; count++, walk = walk->next)
     if (addrcmp(&walk->address, addr) == 0) return (count);

    return (0);
}

int  add_echo_sorted (ECHO_LIST **list, ECHO *echo)
{
    ECHO_LIST **tmp, *next;

    for (tmp = list; *tmp != NULL && strcmpi((*tmp)->echo.name, echo->name) < 0; tmp = &(*tmp)->next);

    next = *tmp;

    if ((*tmp = (ECHO_LIST *) malloc(sizeof(ECHO_LIST))) != NULL)
    {
        (*tmp)->echo = *echo;
        (*tmp)->next = next;
        return (ERR_OK);
    }
    else *tmp = next;

    return (ERR_NOROOM);
}

void free_echo_list (ECHO_LIST **list)
{
    ECHO_LIST *tmp;

    for (; *list != NULL; *list = tmp)
    {
        tmp = (*list)->next;
        free ((*list)->echo.name);
        free_address_list (&(*list)->echo.list);
        free (*list);
    }
}

int  xatoi (char *str)
{
    if (*str == '*') return (-1);
    return (atoi(str));
}

void cleanup_domain (char *ddom, char *sdom)
{
    int i;

    for (i = 0; sdom[i] && sdom[i] != '.' && i < 15; i++)
     ddom[i] = tolower(sdom[i]);

    ddom[i] = 0;
}

int  parse_address (ADDRESS *addr, ADDRESS *defaddr, char *str)
{
    int  pnet = -1, pnode = -1, ppoint = -1, pdomain = -1, i;

    for (i = 0; str[i]; i++)
    {
        if (str[i] == ':' && pnet    == -1) pnet    = i;
        if (str[i] == '/' && pnode   == -1) pnode   = i;
        if (str[i] == '.' && ppoint  == -1) ppoint  = i;
        if (str[i] == '@' && pdomain == -1) pdomain = i;
    }

    addr->zone = (pnet != -1) ? xatoi(str) : defaddr->zone;
    addr->net  = (pnode != -1) ? xatoi(str + pnet + 1) : defaddr->net;
    addr->node = (ppoint == 0) ? defaddr->node : xatoi(str + pnode + 1);
    addr->point = (ppoint != -1) ? xatoi(str + ppoint + 1) : 0;
    cleanup_domain (addr->domain, (pdomain != -1) ? str + pdomain + 1 : defaddr->domain);
    if (*addr->domain == 0) strcpy (addr->domain, "fidonet");

    return (ERR_OK);
}

static char _scrap_str[64];

char *verbalise_address (ADDRESS *addr)
{
    if (addr->point)
     sprintf (_scrap_str, "%d:%d/%d.%d@%s", addr->zone, addr->net, addr->node, addr->point, addr->domain);
    else
     sprintf (_scrap_str, "%d:%d/%d@%s", addr->zone, addr->net, addr->node, addr->domain);

    return (_scrap_str);
}

char *strip_crlf (char *str)
{
    int i;

    for (i = 0; str[i] && str[i] != 10 && str[i] != 13; i++)
     _scrap_str[i] = str[i];
    _scrap_str[i] = 0;

    return (_scrap_str);
}

int  read_line (char *buff, FILE *f)
{
    int b, i;

    *buff = 0;
    do
    {
        while ((b = fgetc(f)) != EOF && (b == ' ' || b == 9 || b == 10 || b == 13));
        if (b == EOF) return (0);
        for (i = 0; b != EOF && b != 10 && b != 13; i++)
        {
            buff[i] = (b == 9) ? ' ': b;
            b = fgetc(f);
        }
        buff[i] = 0;
    }
    while (*buff == ';' || *buff == '%');

    return (1);
}

char *parse_arg (char *arg, char *buff)
{
    int i = 0, j = 0;

    if (buff == NULL) return (NULL);
    while (buff[i] && buff[i] == ' ') i++;
    while (buff[i] && buff[i] != ' ') arg[j++] = buff[i++];
    arg[j] = 0;

    return (j ? buff + i : NULL);
}

void process_areasbbs (char *path)
{
    FILE *f;
    char arg[256], line[8192], *tail;

    if ((f = fopen(path, "r")) == NULL)
    {
        fprintf (stderr, "Error opening %s !!!\n", path);
        show_help ();
    }

    read_line (line, f);                      /* ignore first ID line */

    while (read_line(line, f))
    {
        addr = us;

        tail = parse_arg(arg, line);         /* HMB# or path */
        tail = parse_arg(arg, tail);         /* tagname */
        echo.name = malloc(strlen(arg) + 1);
        strcpy (echo.name, arg);
        echo.list = NULL;
        do tail = parse_arg(arg, tail);      /* skip any options */
        while (*arg == '-');

        while (tail != NULL)
        {
            parse_address (&addr, &addr, arg);
            add_address (&echo.list, &addr, ADD_ATEND);
            if (find_address(&in_list, &addr) && !find_address(&ex_list, &addr) && !find_address(&process, &addr))
                                     add_address (&process, &addr, ADD_SORTED);
            tail = parse_arg(arg, tail);
        }

        add_echo_sorted (&echo_list, &echo);
    }

    fclose (f);
}

void process_squishcfg (char *path)
{
    FILE *f;
    int areasbbs = 0;
    char arg[256], line[8192], *tail;

    if ((f = fopen(path, "r")) == NULL)
    {
        fprintf (stderr, "Error opening %s !!!\n", path);
        show_help ();
    }

    while (read_line(line, f))
    {
        tail = parse_arg(arg, line);             /* directive */

        if (strcmpi(arg, "echoarea") == 0 && !areasbbs)
        {
            addr = us;
            tail = parse_arg(arg, tail);         /* tagname */
            echo.name = malloc(strlen(arg) + 1);
            strcpy (echo.name, arg);
            echo.list = NULL;
            tail = parse_arg(arg, tail);         /* pathname */
            do tail = parse_arg(arg, tail);
            while (*arg == '-');                 /* skip optional switches */

            while (tail != NULL)
            {
                parse_address (&addr, &addr, arg);
                add_address (&echo.list, &addr, ADD_ATEND);
                if (find_address(&in_list, &addr) && !find_address(&ex_list, &addr) && !find_address(&process, &addr))
                 add_address (&process, &addr, ADD_SORTED);
                tail = parse_arg(arg, tail);
            }

            add_echo_sorted (&echo_list, &echo);
        }
        else if (strcmpi(arg, "address") == 0)
        {
            if (us.net == 0)
            {
                tail = parse_arg(arg, tail);     /* actual address */
                parse_address (&us, &us, arg);
            }
        }
        else if (strcmpi(arg, "areasbbs") == 0)
        {
            tail = parse_arg(arg, tail);
            process_areasbbs (arg);
            areasbbs = 1;
        }
    }

    fclose (f);
}

int  main (int argc, char *argv[])
{
    ECHO_LIST *echo_walk;
    ADDRESS_LIST *addr_walk;
    ADDRESS defaddr = { 2, 292, 500, 10, "fidonet" };
    time_t tnow;
    int  i, cfgtype = CFG_SQUISHCFG, count, feed, found;
    char cfgpath[256];

    strcpy (cfgpath, "squish.cfg");

    fprintf (stderr, "EchoRep %s -- Written by Carl Declerck, %s.\n\n", VERSION, __DATE__);

    for (i = 1; i < argc; i++)
    {
        if (*argv[i] == '-')
        {
            switch (tolower(argv[i][1]))
            {
                case 'a' : parse_address (&us, &defaddr, argv[i] + 2);
                           defaddr = us;
                           break;
                case 'b' : strcpy (cfgpath, argv[i] + 2);
                           cfgtype = CFG_AREASBBS;
                           break;
                case 'c' : strcpy (cfgpath, argv[i] + 2);
                           cfgtype = CFG_SQUISHCFG;
                           break;
                case 'h' : show_help ();
                case 'x' : parse_address (&addr, &defaddr, argv[i] + 2);
                           add_address (&ex_list, &addr, ADD_ATSTART);
                           break;
                default  : fprintf (stderr, "Invalid option: %s\n", argv[i]);
                           break;
            }
        }
        else
        {
            parse_address (&addr, &defaddr, argv[i]);
            add_address (&in_list, &addr, ADD_ATSTART);
        }
    }

    if (in_list == NULL)
    {
        parse_address (&addr, &defaddr, "*:*/*.*@*");
        add_address (&in_list, &addr, ADD_ATSTART);
    }

    switch (cfgtype)
    {
        case CFG_AREASBBS  : process_areasbbs (cfgpath);
                             break;
        case CFG_SQUISHCFG : process_squishcfg (cfgpath);
                             break;
    }

    printf ("Echo link report for nodes connected with %s :\n\n", verbalise_address(&us));

    for (addr_walk = process; addr_walk != NULL; addr_walk = addr_walk->next)
    {
        printf ("Connected echoes for address %s :\n\n", verbalise_address(&addr_walk->address));

        for (count = feed = 0, echo_walk = echo_list; echo_walk != NULL; echo_walk = echo_walk->next)
        {
            if ((found = find_address(&echo_walk->echo.list, &addr_walk->address)) != 0)
            {
                feed += (found == 1); count++;
                printf ("%c %-24s%s", (found == 1) ? '+' : ' ', echo_walk->echo.name, (count % 3) ? "" : "\n");
            }
        }

        printf ("%s\n  Total of %d echoes. ", (count % 3) ? "\n" : "", count);
        if (feed) printf ("Feeds %d of these.", feed);
        printf ("\n\n");
    }

    time (&tnow);
    printf ("\nReport generated by EchoRep %s on %s.\n", VERSION, strip_crlf(asctime(localtime(&tnow))));

    free_address_list (&in_list);       /* clean up after playing */
    free_address_list (&ex_list);
    free_address_list (&process);
    free_echo_list (&echo_list);

    return (0);
}

