//===================================================================
// FINDEXEC - FAPI Path search utility
// Copyright 1993 Douglas Boling
//===================================================================

#define MAXFNAMELEN      256
#define MAXPATHLEN       260
#define BUFFSIZE         1024
#define CBUFFSIZE        16384

#define ERR_SYNTAX       10
#define ERR_NOFILESFOUND 11
#define ERR_OUTOFMEM     12
#define ERR_NOOPENEXE    13
#define ERR_NOLIBPATH    14
#define ERR_NOWINDIR     15
#define ERR_NOWININI     16
#define ERR_2SWITCHES    17
#define ERR_LIMTOHIGH    18
#define ERR_HELP         100

#define INCL_VIO
#define INCL_KBD
#define INCL_NOPM
#define INCL_DOS

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <dos.h>

#include "findexec.h"
typedef struct find_t FIND_T;

//
// Function Prototypes
//
INT	Search4EXE (char *, INT);
BOOL	DOSSearch (char *, char *, INT);
BOOL	WinSearch (char *, char *, INT);
INT	GetPIFRefs (char *);
INT	GetNERefs (char *, INT, ULONG, INT);
INT	GetLXRefs (char *, INT, ULONG);
INT	GetEXEType (char *, INT *, LONG *);
char	*FindEnvVar (PSZ, char *, char *,INT);
BOOL	FindFile (char *, char *, INT);
void	CombinePath (char *, char *, INT);
BOOL	SearchPath (char *, char *, char *, INT);
BOOL	SearchPath2 (char *, char *, char *, INT, INT);
BOOL	GetPathDir (char *, char *, INT);
BOOL	GetItem (char *, char *, char *, INT);
BOOL	AddExt (char *, char *, INT);
void	PrintFName (char *, INT, INT);
INT 	ParseCmdLine (INT, char **, char *, INT *);
void	DirQCur (char *, INT);
HFILE	FileOpen (char *);
INT	FileClose (HFILE);

//
// Global Data
//
char	szProgram[] = "\nFINDEXEC 1.0 Copyright 1993 Douglas Boling\n";
char	szProgram1[] = "First Published in PC Magazine, August 1993\n\n";

char	*szErrMsgs[] = {"Syntax error.  Type FINDEXEC /? for help\n",
                      "No executable files found\n",
                      "\nOut of Memory\n",
                      "\nCan\'t open executable file\n",
                      "\nCan\'t find LIBPATH parameter in CONFIG.SYS\n",
                      "\nCan\'t find Windows directory\n",
                      "\nCan\'t find Programs= parameter in WIN.INI\n",
                      "\nMore than 1 search option requested\n",
                      "\nLevel must be between 0 and 9\n",
   	               }; 

char	*szHelpMsgs[] = {"\nUSAGE: FINDEXEC [/?][/W] [/V | /Ln] program_name\n",
                    "\nSearch for program name using proper program search ",
                    "method.\nUses DOS search method under DOS and ",
                    "OS/2 search method under OS/2.\n\n",
                    "/W  - Perform Windows search.  Use only in a ",
                    "Windows DOS Box.\n",
                    "/V  - Verbose output.  Print all included DLLs\n",
                    "/Ln - Display to level n.  Displays included DLLs ",
                    "to the nth level.\n      n must be between 0 and 9.\n",
                    "/?  - Display this help message.\n",
   	             }; 

char	szDOSExts[] = "COM EXE BAT";
char	szWinExts[128] = "";
char	szWinLExts[] = "DLL EXE DRV";
char	szOS2Exts[] = "COM EXE CMD BAT";
char	szOS2LExts[] = "DLL EXE";

BOOL		fExt, fPathInc;
PSZ		lpszEnv;
char		szProgDir[MAXFNAMELEN] = "";
INT		sLevel = 0, sMaxLevel = 1;
char		szLibPath[MAXPATHLEN] = "";
char		szPath[MAXPATHLEN] = "";
USHORT	usDosVer;

//===================================================================
// Program Entry Point
//===================================================================
main (int argc, char *argv[]) {

	char	szExeName[MAXFNAMELEN] = "";
	USHORT	usEnvSel, usCmdOffset;
	PSZ		lpszCmdLine;
	PSZ		lpszPtr;
	INT		i, rc, sSrchType = 0;

	printf (szProgram);               //Print Copyright
	printf (szProgram1);

	DosGetVersion (&usDosVer);        //If OS/2, def to OS/2 search
	if (usDosVer >= 0xa00)
		sSrchType = 2;
	//
	// Get Ptrs to environment and Cmd line
	//
	DosGetEnv (&usEnvSel, &usCmdOffset);
	lpszEnv = MAKEP (usEnvSel, 0);
	lpszCmdLine = MAKEP (usEnvSel, usCmdOffset);
	//
	// Get program's directory from program name
	//
	lpszPtr = lpszEnv;
	while (*lpszPtr != 0) {           //Find end of env strings
		for (;*lpszPtr != 0; lpszPtr++)
			;
		lpszPtr++;
	}
	lpszPtr++;
	for (i = 0; i < sizeof (szProgDir) && *lpszPtr != 0; i++)
		szProgDir[i] = *lpszPtr++;
	*strrchr (szProgDir, '\\') = '\0';
	strupr (szProgDir);
	//
	//Get PATH env var
	//
	FindEnvVar (lpszEnv, "PATH", szPath, sizeof (szPath));
	//
	//Parse the command line
	//
	rc = ParseCmdLine (argc, argv, szExeName, &sSrchType);
	if (rc) {
		if (rc == ERR_HELP)
			for (i = 0; i < 10; i++)
				printf (szHelpMsgs[i]);
		else
			printf (szErrMsgs[rc-10]);
		DosExit (1, rc);
	}
	//
	//Search for the EXE
	//
	rc = Search4EXE (szExeName, sSrchType);
	if (rc > 10)
		printf (szErrMsgs[rc-10]);
	printf("\n");
	DosExit (1, rc);
	return 0;
}
//-------------------------------------------------------------------
// Search4EXE - Searches the system for an executable file
//
//-------------------------------------------------------------------
INT Search4EXE (char *szFName, INT sSrchType) {

	BOOL		fFound = FALSE;
	SHORT		sExeType, sNumRefEnt;
	ULONG		ulFPtr;
	char		szWinDir[128];

	switch (sSrchType) {
		//
		// DOS
		//
		case 0:
			if (usDosVer < 0x400)
				if (fExt) {
					*strrchr (szFName, '.') = '\0';
					fExt = FALSE;
				}	
			fFound = DOSSearch(szFName, szDOSExts, -1);
			if (fFound) {
				PrintFName (szFName, GetEXEType (szFName, 0, 0), sLevel);
				return 0;
			}
			break;
		//
		// Windows
		//
		case 1:
			if (!FindEnvVar (lpszEnv, "windir", szWinDir, sizeof (szWinDir)))
				return ERR_NOWINDIR;
			strcat (szWinDir, "\\WIN.INI");
			if (!GetItem (szWinDir, "Programs=", szWinExts, sizeof (szWinExts)))
				return ERR_NOWININI;
			fFound = WinSearch(szFName, szWinExts, -1);
			if (fFound) {
				fExt = FALSE;
				fPathInc = FALSE;
				sExeType = GetEXEType (szFName, &sNumRefEnt, &ulFPtr);
				PrintFName (szFName, sExeType, sLevel);

				if ((sExeType == 1) && (sNumRefEnt)) {
					if (sMaxLevel) {
						printf ("Loads:\n");
					 	return GetNERefs (szFName, sNumRefEnt, ulFPtr, 1);
					}
					return 0;	
				}
				if (sExeType == 7) 
				 	return GetPIFRefs (szFName);
			}
			break;
		//
		// OS/2
		//
		case 2:
			fFound = DOSSearch(szFName, szOS2Exts, 4);
			if (!fFound)
				fFound = DOSSearch(szFName, szDOSExts, 2);
			if (!fFound)
				fFound = DOSSearch(szFName, szDOSExts, -1);
			if (fFound) {
				fExt = FALSE;
				fPathInc = FALSE;
				sExeType = GetEXEType (szFName, &sNumRefEnt, &ulFPtr);
				PrintFName (szFName, sExeType, sLevel);

				if ((sNumRefEnt) && sMaxLevel && 
				    ((sExeType == 2) || (sExeType == 4))) {

					if (!GetItem ("c:\\CONFIG.SYS", "LIBPATH=",
					              szLibPath, sizeof (szLibPath)))
						return ERR_NOLIBPATH;
					printf ("Loads:\n");
					if (sExeType == 4) 
						return GetLXRefs (szFName, sNumRefEnt, ulFPtr);
					return GetNERefs (szFName, sNumRefEnt, ulFPtr, 2);
				}
				return 0;
			}
			break;
	}
	return ERR_NOFILESFOUND;
}
//-------------------------------------------------------------------
// DOSSearch - DOS program search method
//
// DOS search method
//   current dir
//   path
//
//-------------------------------------------------------------------
BOOL DOSSearch (char *pszFName, char *pszExts, INT sFType) {
	char	szDir[MAXFNAMELEN];
	BOOL	fFound;
 	
	if (fPathInc) {
		fFound = FindFile (pszFName, pszExts, sFType);
		return fFound;
	}	
	
	DirQCur (szDir, sizeof (szDir));
	CombinePath (szDir, pszFName, sizeof (szDir));
	if (FindFile (szDir, pszExts, sFType)) {
		strcpy (pszFName, szDir);
		return TRUE;
	}
	return SearchPath (pszFName, szPath, pszExts, sFType);
}
//-------------------------------------------------------------------
// WinSearch - Windows executable search method
//
// Windows search method
//   current dir
//   Win directory
//   Win System directory
//   EXE file's directory
//   path
//   net path
//
//-------------------------------------------------------------------
BOOL WinSearch (char *pszFName, char *pszExts, INT sFType) {
	char	szDir[MAXFNAMELEN];

	if (fPathInc) 
		return FindFile (pszFName, pszExts, sFType);
	//
	// Current dir
	//
	DirQCur (szDir, sizeof (szDir));
	CombinePath (szDir, pszFName, sizeof (szDir));
	if (FindFile (szDir, pszExts, sFType)) {
		strcpy (pszFName, szDir);
		return TRUE;
	}
	//
	// Windows dir
	//
	FindEnvVar (lpszEnv, "windir", szDir, sizeof (szDir));
	CombinePath (szDir, pszFName, sizeof (szDir));
	if (FindFile (szDir, pszExts, sFType)) {
		strcpy (pszFName, szDir);
		return TRUE;
	}
	//
	// Windows System dir
	//
	FindEnvVar (lpszEnv, "windir", szDir, sizeof (szDir));
	strcat (szDir, "\\SYSTEM");
	CombinePath (szDir, pszFName, sizeof (szDir));
	if (FindFile (szDir, pszExts, sFType)) {
		strcpy (pszFName, szDir);
		return TRUE;
	}
	//
	// This program's dir
	//
	strcpy (szDir, szProgDir);
	CombinePath (szDir, pszFName, sizeof (szDir));
	if (FindFile (szDir, pszExts, sFType)) {
		strcpy (pszFName, szDir);
		return TRUE;
	}

	// Search DOS's PATH
	//
	return SearchPath (pszFName, szPath, pszExts, sFType);
}
//-------------------------------------------------------------------
// GetPIFRefs - Prints the DOS file the PIF file loads
//-------------------------------------------------------------------
INT GetPIFRefs (char *szFName) {

	USHORT	usSel, usBytesRead;
	SHORT		i;
	PBYTE		lpbData;
	HFILE		hFile;
	char		szPIFName[128];

	//
	//Read PIF file
	//
	if (DosAllocSeg (BUFFSIZE, &usSel, 0))
		return ERR_OUTOFMEM;
	lpbData = MAKEP (usSel, 0);
	hFile = FileOpen (szFName);
	DosRead (hFile, lpbData, BUFFSIZE, &usBytesRead);
	FileClose (hFile);
	//
	//Copy program name
	//
	lpbData += 0x24;
	for (i = 0; i < sizeof (szPIFName)-1 && *lpbData != '\0'; i++) {
		szPIFName[i] = *lpbData++;
		if (szPIFName[i] == '.')
			fExt = TRUE;
		else if (szPIFName[i] == '\\') {
			fExt = FALSE;
			fPathInc = TRUE;
		} else if (szPIFName[i] == ':')
			fPathInc = TRUE;
	}
	szPIFName[i] = '\0';
	DosFreeSeg (usSel);

	if (DOSSearch (szPIFName, szDOSExts, -1)) {
		printf ("\n Runs ");
		PrintFName (szPIFName, GetEXEType (szPIFName, 0, 0), sLevel);
		return 0;
	}
	printf ("\nWarning!  Program file: ");
	printf (szPIFName);
	printf (" not found.\n\n");
	return 2;
}
//-------------------------------------------------------------------
// GetNERefs - Prints a list of an NE file's refs and their refs.
//-------------------------------------------------------------------
INT GetNERefs (char *szFName, INT sNumRefEnt, ULONG ulFPtr, INT sType) {

	USHORT	usSel, usBytesRead;
	SHORT		i, j, sSubRef, rc = 0;
	PBYTE		lpbData;
	HFILE		hFile;
	PSZ		lpszName;
	BYTE		bNameLen;
	BOOL		fGetRef;

	if (sLevel >= sMaxLevel)
		return 0;
	sLevel++;
	//
	//Read Mod Ref table
	//
	if (DosAllocSeg (BUFFSIZE, &usSel, 0))
		return ERR_OUTOFMEM;
	lpbData = MAKEP (usSel, 0);
	hFile = FileOpen (szFName);
	DosChgFilePtr (hFile, (LONG) ulFPtr, 0, &ulFPtr);
	DosRead (hFile, lpbData, BUFFSIZE, &usBytesRead);
	FileClose (hFile);
	//
	//Find file for each entry in the table.
	//
	strcpy (szProgDir, szFName);
	*strrchr (szProgDir, '\\') = '\0';
	for (i = 0; i < sNumRefEnt; i++) {
		//
		// Copy file name in ref table
		//
		lpszName = lpbData + *((PUINT)lpbData+i) + (sNumRefEnt*2);
		bNameLen = (BYTE) *lpszName++;
		for (j = 0; j < (SHORT) bNameLen; j++)
			szFName[j] = *lpszName++;
		szFName[j] = '\0';
		//
		fGetRef = TRUE;
		switch (sType) {
			case 1:
				if ((strcmp (szFName, "KERNEL") == 0) || 
				    (strcmp (szFName, "DISPLAY") == 0)) {
					PrintFName (szFName, 0x101, sLevel);
					fGetRef = FALSE;
				}
				break;
			case 2:
				if (strcmp (szFName, "DOSCALLS") == 0) {
					PrintFName (szFName, 0x106, sLevel);
					fGetRef = FALSE;
				}
				break;
		}
		if (fGetRef) {
			switch (sType) {
				case 1:
					fGetRef = WinSearch(szFName, szWinLExts, sType + 0x100);
					break;
				case 2:
					fGetRef = SearchPath2 (szFName, szLibPath, szOS2LExts, 
					                       0x102, 0x104);
					break;
				default:
					fGetRef = FALSE;
			}
			if (fGetRef) {
				j = GetEXEType (szFName, &sSubRef, &ulFPtr);
				PrintFName (szFName, j, sLevel);
				if ((sType) && (j-0x100 == sType)) {
					if (sNumRefEnt) 
						GetNERefs (szFName, sSubRef, ulFPtr, sType);
				}
			} else {
				printf ("\nWarning!  Library file: ");
				printf (szFName);
				printf (" not found.\n\n");
				rc = 2;
			}
		}
	}
	DosFreeSeg (usSel);
	sLevel--;
	return rc;
}
//-------------------------------------------------------------------
// GetLXRefs - Prints a list of an NE file's refs and their refs.
//-------------------------------------------------------------------
INT GetLXRefs (char *szFName, INT sNumRefEnt, ULONG ulFPtr) {

	USHORT	usSel, usBytesRead;
	SHORT		i, j, sSubRef, rc = 0;
	PBYTE		lpbData;
	HFILE		hFile;
	PSZ		lpszName;
	BYTE		bNameLen;
	BOOL		fGetRef;

	if (sLevel >= sMaxLevel)
		return 0;
	sLevel++;
	//
	//Read Mod Ref table
	//
	if (DosAllocSeg (BUFFSIZE, &usSel, 0))
		return ERR_OUTOFMEM;
	lpbData = MAKEP (usSel, 0);
	hFile = FileOpen (szFName);
	DosChgFilePtr (hFile, (LONG) ulFPtr, 0, &ulFPtr);
	DosRead (hFile, lpbData, BUFFSIZE, &usBytesRead);
	FileClose (hFile);
	//
	//Find file for each entry in the table.
	//
	strcpy (szProgDir, szFName);
	*strrchr (szProgDir, '\\') = '\0';
	lpszName = lpbData;
	for (i = 0; i < sNumRefEnt; i++) {
		// 
		// Copy entry table file name
		//
		bNameLen = (BYTE) *lpszName++;
		for (j = 0; j < (SHORT) bNameLen; j++)
			szFName[j] = *lpszName++;
		szFName[j] = '\0';
		//	
		// Don't look up kernel file.
		//
		fGetRef = TRUE;
		if (strcmp (szFName, "DOSCALLS") == 0) {
			PrintFName (szFName, 0x104, sLevel);
			fGetRef = FALSE;
		} else {
			//
			// Find File
			//
			if (SearchPath (szFName, szLibPath, szOS2LExts, 0x104)) {
				PrintFName (szFName, GetEXEType (szFName, &sSubRef, &ulFPtr),
				            sLevel);
				if (sSubRef) 
					GetLXRefs (szFName, sSubRef, ulFPtr);
			} else {
				printf ("\nWarning!  Library file: ");
				printf (szFName);
				printf (" not found.\n\n");
				rc = 2;
			}
		}
	}
	DosFreeSeg (usSel);
	sLevel--;
	return rc;
}
//-------------------------------------------------------------------
// GetEXEType - Loads a file and returns its dest OS.
//-------------------------------------------------------------------
INT GetEXEType (char *szFName, INT *psNumRefEnt, LONG *plFPtr) {

	PBYTE		lpbData;
	USHORT	usSel, usBytesRead;
	INT		sTargOS;
	HFILE		hFile;

	if (DosAllocSeg (BUFFSIZE, &usSel, 0))
		return ERR_OUTOFMEM;
	lpbData = MAKEP (usSel, 0);

	hFile = FileOpen (szFName);
	if (hFile == -1) {
		DosFreeSeg (usSel);
		return (ERR_NOOPENEXE);
	}					
	DosRead (hFile, lpbData, BUFFSIZE, &usBytesRead);
	if (usBytesRead < 0x40) {				
		FileClose (hFile);
		DosFreeSeg (usSel);
		return 0;
	}
	//Check for "MZ"
	if (*(PUSHORT)lpbData != 0x5a4d) {
		FileClose (hFile);
		sTargOS = 0;                 //DOS			
		if (*(PLONG)(lpbData+0x171) == 0x5243494d) 
			sTargOS = 7;              //Window PIF			
		DosFreeSeg (usSel);
		return sTargOS;
	}
	//Check for New EXE header
	if (*((PSHORT)(lpbData+0x18)) < 0x40) {
		FileClose (hFile);
		DosFreeSeg (usSel);
		return 0;
	}
	//Read New EXE header
	*plFPtr = (LONG) *((PLONG)(lpbData+0x3C));
	DosChgFilePtr (hFile, (LONG) *plFPtr, 0, plFPtr);

	DosRead (hFile, lpbData, BUFFSIZE, &usBytesRead);
	FileClose (hFile);

	switch (*(PUSHORT)lpbData) {

		//Check for NewEXE (NE)
		case 0x454E:
			if (*(lpbData+0x36) & 2) 
				sTargOS = 1;                 //Windows
			else if (*(lpbData+0x36) & 1) 
				sTargOS = 2;                 //OS/2 1.x

			if (*(PUSHORT)(lpbData+0x0C) & 0x8000) 
				sTargOS += 0x100;            //Library

			if (psNumRefEnt != 0) 			
				*psNumRefEnt = *((PSHORT)(lpbData+0x1E));
			if (plFPtr != 0) 
				*plFPtr += (LONG) *((PUINT)(lpbData+0x28));

			break;

		//Check for LinearEXE (LE)
		case 0x454C:
			sTargOS = 3;                    //Win 3.x Enh mode
			if (*(PULONG)(lpbData+0x10) & 0x8000) 
				sTargOS += 0x100;            //Library

			if (psNumRefEnt != 0)
				*psNumRefEnt = (INT) *((PULONG)(lpbData+0x74));
			if (plFPtr != 0) 
				*plFPtr += (LONG) *((PULONG)(lpbData+0x70));
			break;

		//Check for LinearEXE (LX)
		case 0x584C:
			sTargOS = 4;                    //OS/2 2.x
			if (*(PULONG)(lpbData+0x10) & 0x8000) 
				sTargOS += 0x100;            //Library

			if (psNumRefEnt != 0)
				*psNumRefEnt = (INT) *((PULONG)(lpbData+0x74));
			if (plFPtr != 0) 
				*plFPtr += (LONG) *((PULONG)(lpbData+0x70));
			break;

		//Check for LinearEXE (PE)
		case 0x4550:
			sTargOS = 5;                    //Win/NT
			if (psNumRefEnt != 0) 			
				*psNumRefEnt = 0;
			break;
	
	}
	DosFreeSeg (usSel);
	return sTargOS;
}

//-------------------------------------------------------------------
// CombinePath - Appends a file name to a Path string
//-------------------------------------------------------------------
void CombinePath (char *pszOut, char *pszIn, INT sOutSize) {

	if ((INT) strlen (pszIn) > sOutSize - strlen (pszOut))
		return;

	if (pszOut[strlen (pszOut)-1] != '\\')
		strcat (pszOut, "\\");
	strcat (pszOut, pszIn);
	return;
}
//-------------------------------------------------------------------
// SearchPath - Searches for a file along a path 
//-------------------------------------------------------------------
BOOL SearchPath (char *pszFName, char *szPath, char *pszExts, INT sFType) {
	char	szDir[MAXPATHLEN];
	INT	j;

	j = 0;		
	while (GetPathDir (szPath, szDir, j++)) {
		CombinePath (szDir, pszFName, sizeof (szDir));
		if (FindFile (szDir, pszExts, sFType)) {
			strcpy (pszFName, szDir);
			return TRUE;
		}
	}
	return FALSE;
}
//-------------------------------------------------------------------
// SearchPath2 - Searches for a file along a path.  Looks for 2
// types of files.
//-------------------------------------------------------------------
BOOL SearchPath2 (char *pszFName, char *szPath, char *pszExts, 
                  INT sFType1, INT sFType2) {
	char	szDir[MAXPATHLEN];
	INT	j;

	j = 0;		
	while (GetPathDir (szPath, szDir, j++)) {
		CombinePath (szDir, pszFName, sizeof (szDir));
		if (FindFile (szDir, pszExts, sFType1)) {
			strcpy (pszFName, szDir);
			return TRUE;
		} else if (FindFile (szDir, pszExts, sFType2)) {
			strcpy (pszFName, szDir);
			return TRUE;
		}
	}
	return FALSE;
}

//-------------------------------------------------------------------
// FindFile - Searches for a file in a directory
//-------------------------------------------------------------------
BOOL FindFile (char *pszDir, char *pszExts, INT sFType) {
	char	szTemp[MAXFNAMELEN];
	INT	i, rc = 0, sFHdl = 1;
	char	*pszNameEnd;
	FILEFINDBUF	ffb;
	
	strcpy (szTemp, pszDir);
	if (fExt) {
		rc = 1;
		sFHdl = 1;
		DosFindFirst (szTemp, &sFHdl, 0x27, &ffb, sizeof (ffb), &rc, 0);
	} else {
		pszNameEnd = &szTemp[strlen (szTemp)];
		for (i = 0; AddExt (pszNameEnd, pszExts, i); i++) {
			rc = 1;
			sFHdl = 1;
			DosFindFirst (szTemp, &sFHdl, 0x27, &ffb, sizeof (ffb), &rc, 0);
			if (rc != 0)
				break;
		}
	}
	if (rc != 0) {
		if ((sFType != -1) && (GetEXEType (szTemp, 0, 0) != sFType)) 
			return FALSE;
		strcpy (pszDir, szTemp);
		strupr (pszDir);
		return TRUE;
	} else
		return FALSE;
}
//-------------------------------------------------------------------
// AddExt - Appends an extension from a list of extensions to a filename.
//-------------------------------------------------------------------
BOOL AddExt (char *pszOut, char *pszExt, INT sNum) {
	
	while (*pszExt != '\0') {
		//
		// Find non-space char
		//		
		while (*pszExt <= ' ') {
			if (*pszExt == '\0')
				break;
			pszExt++;
		}
		if (sNum == 0)
			break;
		pszExt += min (3, strlen (pszExt));
		sNum--;
	}
	if (*pszExt == '\0')
		return FALSE;

	strcpy (pszOut, ".");
	strncpy (pszOut+1, pszExt, 3);
	*(pszOut+4) = '\0';
	return TRUE;
}	
//-------------------------------------------------------------------
// GetPathDir - Returns a drectory from the path
//-------------------------------------------------------------------
BOOL GetPathDir (char *pszPath, char *pszDir, INT sDirNum) {
	char	*pszEnd;
	char	chChar;
	INT	i;

	pszEnd = strchr (pszPath, ';');
	for (i = 0; i < sDirNum && pszEnd != 0; i++) {
		pszPath = pszEnd+1;
		pszEnd = strchr (pszPath, ';');
	}
	if (i == sDirNum) {
		chChar = 0;
		if (pszEnd != 0) {
			chChar = *pszEnd;
			*pszEnd = '\0';
		}
		strcpy (pszDir, pszPath);
		if (chChar)
			*pszEnd = chChar;
		return TRUE;
	}
	return FALSE;
}
//-------------------------------------------------------------------
// FindEnvVar - Returnss the environment for a variable
//-------------------------------------------------------------------
char *FindEnvVar (PSZ lpszEnv, char *pszVarName, char *pszOut, 
                  INT sOutSize) {

	INT	i, sVarSize;
	char	*pszPtr;
	char	*pszOutSave;

	sVarSize = strlen (pszVarName);
	pszOutSave = pszOut;
	while (*lpszEnv != 0) {
		pszPtr = pszVarName;
		for (i = 0; i < sVarSize && *lpszEnv != '\0'; i++) {
			if (*pszPtr++ != (char) *lpszEnv++)
				break;
		}
		if ((i == sVarSize) && (*lpszEnv == '=')) {
			lpszEnv++;
			for (i = 0; i < sOutSize-1 && *lpszEnv != '\0'; i++)
				*pszOut++ = *lpszEnv++;
			*pszOut = '\0';
			return pszOutSave;
		} else {
			for (i = 0; i < MAXPATHLEN && *lpszEnv != '\0'; i++)
				lpszEnv++;
 			lpszEnv++;
		}
	}
	return 0;
}
//-------------------------------------------------------------------
// GetItem - Returns an item from a file
//-------------------------------------------------------------------
BOOL GetItem (char *pszFName, char *pszSrchStr, char *pszOut, INT sOutSize ) {

	PSZ		lpszData, lpszSrc, lpszName;
	USHORT	i, usSel, usBytesRead, usSrchLen;
	HFILE		hFile;

	usSrchLen = strlen (pszSrchStr);
	if (DosAllocSeg (CBUFFSIZE, &usSel, 0))
		return ERR_OUTOFMEM;
	lpszData = MAKEP (usSel, 0);

	hFile = FileOpen (pszFName);
	if (hFile == -1) {
		DosFreeSeg (usSel);
		return FALSE;
	}
	DosRead (hFile, lpszData, CBUFFSIZE, &usBytesRead);
	FileClose (hFile);
	if (usBytesRead < usSrchLen) {
		DosFreeSeg (usSel);
		return FALSE;
	}
	while (usBytesRead >= usSrchLen) {
		//
		// Find nonspace character
		//
		while (usBytesRead >= usSrchLen) {
			if (*lpszData > ' ')
				break;
			usBytesRead--;	
		}	
		//
		// See if 1st char matches
		//			
		if ((char) *lpszData == pszSrchStr[0]) {
			lpszSrc = lpszData;
			lpszName = (PSZ) pszSrchStr;
			i = 0;
			while ((*lpszSrc == *lpszName) && (i < usBytesRead)	&&
			       (*lpszSrc != 0) && (*lpszName != 0)) {
				lpszName++;
				lpszSrc++;
				i++;
			}
			if (*lpszName == 0)
				break;
		}
		//
		// Skip to end of line
		//
		for (i = 0; i < usBytesRead; i++) {
			usBytesRead--;
			if (*lpszData++ == 0x0d)
				break;
		}			
		usBytesRead--;
		lpszData++;
	}
	if (*lpszName == 0)
		for (i = 0; i < (USHORT) sOutSize && *lpszSrc != 0x0d; i++)
			*pszOut++ = *lpszSrc++;
	*pszOut = 0;

	DosFreeSeg (usSel);
	if (*lpszName == 0)
		return TRUE;
	else
		return FALSE;
}
//-------------------------------------------------------------------
// ParseCmdLine - Parses the command line.
//-------------------------------------------------------------------
INT ParseCmdLine (INT argc, char **argv, char *pszExeName, INT *psSrchType) {
	BOOL	fFirst = TRUE, fSwitch = FALSE;
	INT	i, j;
	char	szFName[MAXFNAMELEN];

	fExt = FALSE;
	fPathInc = FALSE;
	szFName[0] = '\0';
	
	if (argc < 2)
		return ERR_SYNTAX;

	for (i = 1; i < argc; i++) {

		if (*argv[i] == '/' || *argv[i] == '-') {
			switch (*(argv[i]+1)) {
				case '?':
					return ERR_HELP;

				case 'd':
				case 'D':
					if (fSwitch)
						return ERR_2SWITCHES;
					*psSrchType = 0;
					fSwitch = TRUE;
					break;

				case 'w':
				case 'W':
					if (fSwitch)
						return ERR_2SWITCHES;
					*psSrchType = 1;
					fSwitch = TRUE;
					break;

				case '2':
					if (fSwitch)
						return ERR_2SWITCHES;
					*psSrchType = 2;
					fSwitch = TRUE;
					break;

				case 'v':
				case 'V':
					sMaxLevel = 20;
					break;

				case 'l':
				case 'L':
					if ((*(argv[i]+2) >= '0') && (*(argv[i]+2) <= '9')) {
						sMaxLevel = *(argv[i]+2) - '0';
						if ((*(argv[i]+3) >= '0') && (*(argv[i]+3) <= '9')) 
							return ERR_LIMTOHIGH;
					} else
						return ERR_SYNTAX;
					break;

				default:
					return ERR_SYNTAX;
			}
		} else {
			if (!fFirst)
				return ERR_SYNTAX;
			fFirst = FALSE;
			for (j = 0; (j < MAXFNAMELEN) && (argv[i][j] > ' ') &&				
				         !strchr ("?*-+/<>", argv[i][j]); j++) {

				if (argv[i][j] == '.')
					fExt = TRUE;
				else if (argv[i][j] == '\\') {
					fExt = FALSE;
					fPathInc = TRUE;
				} else if (argv[i][j] == ':')
					fPathInc = TRUE;
					
				szFName[j] = argv[i][j];
			}
			szFName[j] = '\0';
		}
	}
	if (szFName[0] == 0)
		return ERR_SYNTAX;
	if (!fFirst && fPathInc)
		_fullpath (pszExeName, szFName, MAXFNAMELEN);
	else
		strcpy (pszExeName, szFName);

	return 0;
}
//-------------------------------------------------------------------
// PrintFName - Long ptr string cmp
//-------------------------------------------------------------------
void PrintFName (char *pszFName, INT sExeType, INT sLevel) {

	INT	i;

 	for (i = 0; i < sLevel; i++)
 		printf("   ");

	switch (sExeType & 0xff) {
		case 0:
			printf ("DOS ");
			break;
		case 1:
			printf ("Windows ");
			break;
		case 2:
			printf ("OS/2 1.x ");
			break;
		case 3:
			printf ("Windows Enhanced Mode ");
			break;
		case 4:
			printf ("OS/2 2.x ");
			break;
		case 5:
			printf ("Windows/NT ");
			break;
		case 6:
			printf ("OS/2 ");
			break;
		case 7:
			printf ("Windows PIF ");
			sExeType = 0x207;
			break;
	}
	if (sExeType > 0x200)
		printf ("File: ");
	else if (sExeType > 0x100)
		printf ("DLL: ");
	else
		printf ("Program: ");

	printf (pszFName);
	printf ("\n");
}

//-------------------------------------------------------------------
// DirQCur - Returns the current drive and dir
//-------------------------------------------------------------------
void DirQCur (char *pszDir, INT sMaxLen) {

	DosQCurDisk ((PUSHORT)pszDir, (PLONG)&pszDir[2]);
	pszDir[0] += 0x40;
	pszDir[1] = ':';
	pszDir[2] = '\\';
	sMaxLen -= 3;
	DosQCurDir (0, &pszDir[3], &sMaxLen);
	if (pszDir[3] == '\\')
		pszDir[3] = '\0';
	return;
}
//-------------------------------------------------------------------
// FileOpen - Opens a file for editing
//-------------------------------------------------------------------
HFILE FileOpen (char *szFileName) {
	UINT	usAction, rc;
	HFILE	hFile;

	rc = DosOpen (szFileName, &hFile, &usAction, 0, 0, FILE_OPEN,
					  OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,  0L);

	if (rc == 0)
	   return hFile;
	else
		return -1;
}	                
//-------------------------------------------------------------------
// FileClose - Closes a file
//-------------------------------------------------------------------
INT FileClose (HFILE hFile) {

	return DosClose (hFile);
}	                
