/*
    uiomanip.h - User defined manipulators
    ----------

    Copyright 1993 by Sam Edge.
    All rights reserved.

    Credits
    -------

    The logic of the xManip<T> was taken from:-
    "The C++ Programming Language, Second Edition"
     by Bjarne Stroustrup, Addison Wesley Publishing Company

    The provision of s,o,i and io versions along with the logic of the
    xApply<T> class was inspired by the macros in IOMANIP.H in
    Borland C++ 3.1.

    The provision of two arg versions was my own idea and acts as
    an example in case you wish to extend the templates to include three
    or more arg versions. Alternatively, you could pass a structure
    to the one arg version for more complex manipulators.

    Usage
    -----

    The template classes sApply<T>, oApply<T>, iApply<T> and ioApply<T>
    allow you to define your own mainipulators which take a single arg
    of any type for use with the << and >> operators on ios, ostream,
    istream and iostream objects respectively.

    For example, if you wish to define an output manipulator that inserts
    any number of the pad character, you could define a function to implement
    the manipulator as follows:-

            ostream & OutPad(ostream & os,int n)
            {
                while ( os && n-- )
                    os << os.fill();
                return os;
            }

    You would then create a manipulator (usually with file scope):-

            oApply<int> Pad(OutPad);

    The manipulator can then be used:-

            cout << setfill('-') << 'A' << Pad(10) << 'B';

    to give output:-

            A----------B

    You are not limited to using int arguments. In fact any predefined or
    user-defined type or class may be used, provided that it has meaningful
    copy semantics and a default constructor.

    Note that you don't ever have to use the sManip, oManip, iManip or
    ioManip classes directly in your code.

    The classes sApply2, oApply2, iApply2 and ioApply2 allow two arg
    manipulators, e.g:-

            ostream & OutChPad(ostream & os,char c,int n)
            {
                while ( os && n-- )
                    os << c;
                return os;
            }
            oApply2<char,int> ChPad(OutChPad);
            cout << 'A' << ChPad('-',10) << 'B';

    xManip0 - a reminder
    --------------------
    If you wish to implement your own manipulators that take no arguments,
    like the standard library ones ws, hex, dec, etc., remember that the
    operators << and >> are already overloaded to support these directly:-

            ostream & Tab(ostream & os)
            {
                return os << '\t';
            }
            cout << 'A' << Tab << 'B';

    gives:-
            A       B

    Licence
    -------

    This file is freeware! However, if you do use it, or have any
    comments, feel free to contact me. If you wish to show your
    appreciation, please send a donation the the charity Greenpeace.

    I can be contacted as follows:-

    Compuserve : 100117,515

    Internet   : 100117.515@COMPUSERVE.COM

    Mail       : Sam Edge
                 26 College Road
                 Ringwood
                 Hampshire BH24 1NX
                 Great Britain


*/

#ifndef __cplusplus
#error Must use C++ for the user defined io stream manipulators.
#endif

#if ! defined(__UIOMANIP_H)
#define __UIOMANIP_H

#if ! defined(___DEFS_H)
#include <_defs.h>
#endif

#if ! defined(__IOSTREAM_H)
#include <iostream.h>
#endif

//
// Ios manipulators
//
template <class T> class sManip {
    typedef ios & (*FT)(ios &,T);
    FT fun;
    T arg;
public:
    sManip(FT f,T t) : fun(f), arg(t) { }
    friend ios & operator << (ios & s, sManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
    friend ios & operator >> (ios & s, sManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
}
template <class T> class sApply {
    typedef ios & (*FT)(ios &,T);
    FT fun;
public:
    sApply(FT f) : fun(f) { }
    sManip<T> operator ()(T v) {
        return sManip<T>(fun,v);
    }
}

//
// Ostream manipulators
//
template <class T> class oManip {
    typedef ostream & (*FT)(ostream &,T);
    FT fun;
    T arg;
public:
    oManip(FT f,T t) : fun(f), arg(t) { }
    friend ostream & operator << (ostream & s, oManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
}
template <class T> class oApply {
    typedef ostream & (*FT)(ostream &,T);
    FT fun;
public:
    oApply(FT f) : fun(f) { }
    oManip<T> operator ()(T v) {
        return oManip<T>(fun,v);
    }
}

//
// Istream manipulators
//
template <class T> class iManip {
    typedef istream & (*FT)(istream &,T);
    FT fun;
    T arg;
public:
    iManip(FT f,T t) : fun(f), arg(t) { }
    friend istream & operator >> (istream & s, iManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
}
template <class T> class iApply {
    typedef istream & (*FT)(istream &,T);
    FT fun;
public:
    iApply(FT f) : fun(f) { }
    iManip<T> operator ()(T v) {
        return iManip<T>(fun,v);
    }
}

//
// Iostream manipulators
//
template <class T> class ioManip {
    typedef iostream & (*FT)(iostream &,T);
    FT fun;
    T arg;
public:
    ioManip(FT f,T t) : fun(f), arg(t) { }
    friend iostream & operator << (iostream & s, ioManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
    friend iostream & operator >> (iostream & s, ioManip & manip) {
        return (manip.fun)(s,manip.arg);
    }
}
template <class T> class ioApply {
    typedef iostream & (*FT)(iostream &,T);
    FT fun;
public:
    ioApply(FT f) : fun(f) { }
    ioManip<T> operator ()(T v) {
        return ioManip<T>(fun,v);
    }
}

// -------- Two arg versions

//
// Ios manipulators
//
template <class T1,class T2> class sManip2 {
    typedef ios & (*FT)(ios &,T1,T2);
    FT fun;
    T1 arg1;
    T2 arg2;
public:
    sManip2(FT f,T1 t1,T2 t2) : fun(f), arg1(t1), arg2(t2) { }
    friend ios & operator << (ios & s, sManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
    friend ios & operator >> (ios & s, sManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
}
template <class T1,class T2> class sApply2 {
    typedef ios & (*FT)(ios &,T1,T2);
    FT fun;
public:
    sApply2(FT f) : fun(f) { }
    sManip2<T1,T2> operator ()(T1 v1,T2 v2) {
        return sManip2<T1,T2>(fun,v1,v2);
    }
}

//
// Ostream manipulators
//
template <class T1,class T2> class oManip2 {
    typedef ostream & (*FT)(ostream &,T1,T2);
    FT fun;
    T1 arg1;
    T2 arg2;
public:
    oManip2(FT f,T1 t1,T2 t2) : fun(f), arg1(t1), arg2(t2) { }
    friend ostream & operator << (ostream & s, oManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
}
template <class T1,class T2> class oApply2 {
    typedef ostream & (*FT)(ostream &,T1,T2);
    FT fun;
public:
    oApply2(FT f) : fun(f) { }
    oManip2<T1,T2> operator ()(T1 v1,T2 v2) {
        return oManip2<T1,T2>(fun,v1,v2);
    }
}

//
// Istream manipulators
//
template <class T1,class T2> class iManip2 {
    typedef istream & (*FT)(istream &,T1,T2);
    FT fun;
    T1 arg1;
    T2 arg2;
public:
    iManip2(FT f,T1 t1,T2 t2) : fun(f), arg1(t1), arg2(t2) { }
    friend istream & operator >> (istream & s, iManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
}
template <class T1,class T2> class iApply2 {
    typedef istream & (*FT)(istream &,T1,T2);
    FT fun;
public:
    iApply2(FT f) : fun(f) { }
    iManip2<T1,T2> operator ()(T1 v1,T2 v2) {
        return iManip2<T1,T2>(fun,v1,v2);
    }
}

//
// Iostream manipulators
//
template <class T1,class T2> class ioManip2 {
    typedef iostream & (*FT)(iostream &,T1,T2);
    FT fun;
    T1 arg1;
    T2 arg2;
public:
    ioManip2(FT f,T1 t1,T2 t2) : fun(f), arg1(t1), arg2(t2) { }
    friend iostream & operator << (iostream & s, ioManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
    friend iostream & operator >> (iostream & s, ioManip2 & manip) {
        return (manip.fun)(s,manip.arg1,manip.arg2);
    }
}
template <class T1,class T2> class ioApply2 {
    typedef iostream & (*FT)(iostream &,T1,T2);
    FT fun;
public:
    ioApply2(FT f) : fun(f) { }
    ioManip2<T1,T2> operator ()(T1 v1,T2 v2) {
        return ioManip2<T1,T2>(fun,v1,v2);
    }
}

#endif // ! defined(__UIOMANIP_H)
