/**********************************************************************/
/*                                                                    */
/*    WALTIO32.C -- 32-bit interface module for Universal Thunk       */
/*                  access to WALTIO.DLL                              */
/*                                                                    */
/*    Copyright (C) 1993 by Walter Oney                               */
/*    All rights reserved                                             */
/*                                                                    */
/**********************************************************************/

#include <windows.h>

/* Define W32SUT_32 to get the 32-bit declarations from w32sut.h: */

#define W32SUT_32                // indicate 32-bit side of interface
#include "w32sut.h"              // from \mstools\win32s\ut on CD

#define VERMAJOR 1
#define VERMINOR 0

static UT32PROC lpThunk;         // stepdown thunk

static unsigned char isfmt[256]; // is char a format code?

/**********************************************************************/

/* 32-bit DLL initialization and termination routine. During
   initialization, register a Universal Thunk that will let us
   reach the 16-bit WALTIO.DLL functions "vprintf" and "puts". */

BOOL WINAPI LibMain(HANDLE hInstance, DWORD dwReason, LPVOID junk)
   {                             // LibMain
   switch (dwReason)
      {                          // select on reason code

/*--------------------------------------------------------------------*/

   case DLL_PROCESS_ATTACH:
      {                          // DLL_PROCESS_ATTACH
      WORD version[2] = {VERMINOR, VERMAJOR};

   /* Build an array of printf format codes. This could be done
      better with a static initializer, but the result would be
      unreadable for an example program. */

      isfmt['d'] = TRUE;
      isfmt['i'] = TRUE;
      isfmt['u'] = TRUE;
      isfmt['o'] = TRUE;
      isfmt['x'] = TRUE;
      isfmt['X'] = TRUE;
      isfmt['c'] = TRUE;
      isfmt['f'] = TRUE;
      isfmt['E'] = TRUE;
      isfmt['g'] = TRUE;
      isfmt['G'] = TRUE;
      isfmt['s'] = TRUE;
      isfmt['n'] = TRUE;
      isfmt['p'] = TRUE;

   /* Register a universal thunk for access to the 16-bit DLL. If
      this fails, also fail the load of the 32-bit DLL. */

      return UTRegister(hInstance, // module handle for 32-bit DLL
         "WALTIO16.DLL",         // name of 16-bit thunking DLL
         MAKEINTRESOURCE(2),     // special initialization
         MAKEINTRESOURCE(3),     // interface routine
         &lpThunk,               // where to put stepdown thunk addr
         NULL,                   // no 16-to-32 callback routine
         &version);              // data for init routine
      }                          // DLL_PROCESS_ATTACH

/*--------------------------------------------------------------------*/

   case DLL_PROCESS_DETACH:
      UTUnRegister(hInstance);   // would happen automatically anyway
      break;
      }                          // select on reason code

   return TRUE;                  // (only matters for process attach)
   }                             // LibMain

/**********************************************************************/

/* Interface to "puts" routine: This function has one pointer argument
   that will be automatically translated by the Universal Thunk
   interface. */

int _cdecl puts(const char *string)
   {                             // puts
   return (*lpThunk)((LPVOID) string, 1, NULL);
   }                             // puts

/**********************************************************************/

/* Interface to "vprintf" routine: This function has two pointer
   arguments, as follows:

   +------------+
   |  lpFormat  | --> format string
   +------------+
   |  substargs | -->+--------+
   +------------+    |  arg 1 |
                     +--------+
                     |  arg 2 |   => vector of substitution args,
                     +--------+      some of which may be pointers
                     |   ...  |
                     +--------+

   The strategy for stepping down this call is to copy and revise
   the format string so that escapes get translated to a 16-bit
   version that accomplishes the same thing as the 32-bit version.
   (E.g., "%d" becomes "%ld".) At the same time, we build a
   translation list for the thunk to process. This will take care
   of "%s" substitutions, for example.
*/

int _cdecl vprintf(const char *lpFormat, LPVOID substargs)
   {                             // vprintf
   const char *ofp;              // old-format string pointer
   int nsubs;                    // # substitutions in format
   char ch;                      // current character
   int nbytes;                   // # bytes needed for xlation stuff
   struct
      {                          // stepdown args
      char *format;              // revised format string
      LPVOID substargs;          // array of subst args
      } stepdownargs;            // stepdown args
   LPVOID *xlist;                // translation list
   LPVOID *xlcur;                // current entry in translation list
   char *rfp;                    // revised format string
   struct
      {                          // pointer to anything
      VOID *vp;
      DWORD *dwp;
      WORD *wp;
      LPVOID *pp;
      } svp;                     // pointer to anything

   static LPVOID data;           // translation data area
   static int datasize;          // current size of data area

/* Scan the format string to count the number of percent signs. This
   gives an overestimate of the number of escape sequences (since
   a % might be doubled to indicate a literal percent), and therefore
   doubly overestimages the number of pointers in the vector of
   substitution values (since not all escapes refer to pointers). */

   for (nsubs = 0, ofp = lpFormat; (ch = *ofp); ++ofp)
      if (ch == '%')
         ++nsubs;

/* Allocate memory to hold the revised format string and the
   translation list. To avoid excessive free storage overhead, try
   to reuse a single buffer area from one call to the next */

   nbytes = strlen(lpFormat)     // original format string
      + 1                        // null terminator for format string
      + nsubs                    // an "l" for each escape
      + nsubs * sizeof(LPVOID)   // xlation for each subst arg
      + 2 * sizeof(LPVOID)       // xlation for lpFormat & substargs
      + sizeof(LPVOID);          // NULL at end of xlation list

   if (nbytes > datasize)
      {                          // need new data area
      VirtualFree(data, 0, MEM_RELEASE);
      datasize = (nbytes + 4095) & ~4095;
      data = VirtualAlloc(NULL, datasize, MEM_COMMIT, PAGE_READWRITE);
      if (!data)
         return 0;               // can't get data buffer
      }                          // need new data area

/* Build the revised format string and the translation list. */

   xlist = (LPVOID *) data;
   xlist[0] = &stepdownargs.format;
   xlist[1] = &stepdownargs.substargs;
   xlcur = xlist + 2;

   svp.vp = stepdownargs.substargs = substargs;
   rfp = stepdownargs.format = (char *) (xlist + (nsubs + 3));
   ofp = lpFormat;

   while ((ch = *ofp++))
      {                          // scan & revise format string
      BOOL hadlmod;              // did we see an "l"?

      *rfp++ = ch;               // add char to revised buffer
      if (ch != '%')
         continue;               // raw text char
      if (*ofp == '%')
         {                       // doubled %
         *rfp++ = ch;
         ++ofp;
         continue;
         }                       // doubled %

   /* Scan the format escape looking for the trailing character and
      for an existing "l" modifier. */

      hadlmod = FALSE;
      while ((ch = *ofp++) && !isfmt[ch])
         {                       // copy format string
         if (ch == 'l')
            hadlmod = TRUE;
         *rfp++ = ch;
         }                       // copy format string
      if (!ch)
         break;                  // end of string before fmt char
      if (!hadlmod)
         *rfp++ = 'l';           // add an "l" modifier
      *rfp++ = ch;               // add format character

   /* If this is a %s escape, the corresponding vector entry is
      a pointer that needs translation. */

      if (ch == 's')
         *xlcur++ = svp.vp;
      svp.dwp++;                 // up to next vector entry

      }                          // scan & revise format string

   *rfp = 0;                     // add trailing null
   *xlcur = NULL;                // fence at end of xlation list

/* Call through the thunk to the 16-bit version of vprintf. */

   return (*lpThunk)(&stepdownargs, 0, xlist);
   }                             // vprintf

/**********************************************************************/

/* Interface to "printf" routine: This function has a variable number
   of arguments. It's implemented by calling into the 32-bit version
   of "vprintf", which will do the actual work. */

int _cdecl printf(const LPSTR lpFormat, ...)
   {                             // printf
   return vprintf(lpFormat, (LPVOID *) (&lpFormat) + 1);
   }                             // printf

/**********************************************************************/

/* Interface to benchmarking "FortyTwo" routine. */

int _cdecl FortyTwo()
   {                             // FortyTwo
   return (*lpThunk)(NULL, 2, NULL);
   }                             // FortyTwo
